function papco_spec_fit,energy,flux,err, output,FITFUN=FITFUN,FITSEP=FITSEP,FITLIM=FITLIM, WEIGHT = weight, PLOT = plot, $
  REM_NEG = rem_neg, STDDEV = STDDEV, _EXTRA = _E
;***********************************************************************************************
;* Function: papco_spec_fit
;* 
;* Description: Generalized routine to fit a selected function to experimental data  
;* 
;* Inputs:   
;*           energy - 1 (central energy) or 2 (start and end energies)
;*                    dimensional array. Dimension: (1/2,flux_vals)
;*                    
;*           flux   - array of measured flux at given energies (energy
;*                    bins) Dimension: flux_vals
;*           err    - array of measurement errors, Dimensions:
;*                    flux_vals
;*        
;*           output - array of energies for which fitted fluxes should
;*                    be returned
;* 
;* Outputs:  Structure with fit Parameters and description
;*
;* Keywords: FITFUN - string, or array of strings giving function to use for fit (MAXW,KAPPA,SINGEXP,
;*                    DBLEXP, GAUSS)
;*           FITSEP -  separation between different fits. Must have one
;*                    less elements than FITFUN (or none if FITFUN has
;*                    only one element) 
;*           FITLIM - the absolute limits for the fit in energy 
;*           PLOT -   plot the flux and fit
;*           REM_NEG -remove negative values in the flux
;*
;* Modification History: 
;*           Written by Arne Aasnes at LANL, Jan. 2004
;*
;***********************************************************************************************

; check dimensions and elements of input

   flux_vals=size(flux,/n_elements)
   energy_dims=size(energy,/n_dimensions)
   if energy_dims ne 1 and energy_dims ne 2 then message,'Incorrect dimension in energy array!',/continue
   if energy_dims eq 2 then energy_vals=size(energy,/n_elements)/2 else energy_vals=size(energy,/n_elements)
   if energy_vals ne flux_vals then message,'Not equal length of flux and energy arrays!',/CONTINUE

   flux_dims = size(flux, /n_dimensions)
   IF flux_dims GT 1 THEN flux = reform(flux, flux_vals)

   if n_elements(fitsep) lt n_elements(fitfun)-1 or n_elements(fitsep) ge n_elements(fitfun) then $
     message,' Incorrect number of energy separators compared to fit functions! !C ' + $
     'FITSEP must have FITFUN-1 elements!',/CONTINUE

   IF n_params() LT 3 THEN err = flux*0+1
   


; rebin the energy and flux arrays to n*double length if energy_dims
; eq 2

   if energy_dims eq 2 then BEGIN
      
;       energy=reform(rotate(energy,4),2*flux_vals) ; stretch energy array out in one dimension
      energy =reform(energy, 2*flux_vals)
      so_en=sort(energy) 
      flux=congrid(flux,flux_vals*2,/minus_one)
      err = congrid(err, flux_vals*2, /minus_one)
      flux_vals=flux_vals*2
;       energy = energy(so_en)
;       flux = flux(so_en)
;       err = err(so_en)
      
   endif 

; check for negative flux values and remove them
   IF keyword_set(rem_neg) AND min(flux) LT 0 THEN BEGIN 
      pos_ind = where(flux gt 0, pos_cnt)
      flux_vals = pos_cnt
      flux = flux(pos_ind)
      err = err(pos_ind)
      energy = energy(pos_ind)

   ENDIF 
   
   
; start the fitting

for i=0,n_elements(fitsep) do begin 
   status = 3
;   rep:


    ; find the energy range for this fit

    if not keyword_set(fitlim) then fitlim=[energy(0),energy(flux_vals-1)]

    if NOT keyword_set(fitsep) then fit_ind=where(energy ge fitlim(0) and energy le fitlim(1)) else begin
        if i eq 0  then fit_ind=where(energy le fitsep(0) and energy ge fitlim(0))
        if i gt 0 and i lt n_elements(fitsep) then fit_ind=where(energy ge fitsep(i-1) and energy le fitsep(i))
        if i eq n_elements(fitsep) then fit_ind=where(energy gt fitsep(i-1) and energy le fitlim(1))
    endelse 

   
    
    case strupcase(fitfun(i)) of 
        'MAXWELLIAN': BEGIN
 ;          startflux = flux(fit_ind(0))*energy(fit_ind(0))
;            IF status NE 6 THEN 
           start_params=[1e5,0.1]
            params=2

            IF status EQ 6 THEN BEGIN 
               new_par1 = ''
               new_par2 = ''
               print, 'Failed with these start params:', start_params
               read, 'give better stat param (1): ',new_par1
               read, 'give better stat param (2): ',new_par2
               start_params = [float(new_par1), float(new_par2)]
            endIF 
            ;parinfo=replicate({fitparams,value:0,fixed:0},2)
            funstr='p(0)*x*exp(-p(1)*x)'            
         END  
        'KAPPAFUN': begin 
            start_params=[1e6,0.1,0.1]
            params=3 
            funstr='p(0)*(1+x^2/p(1))^(-p(2)-1)' 
         END  
        'SINGEXP': begin
            start_params=[1e5,10.0]
            params=2
            funstr='p(0)*exp(-x/p(1))'
         END  
        'DBLEXP': begin
            start_params=[1e5,10,1e5,30]
            params=4
            funstr='p(0)*exp(-x/p(1))+p(2)*exp(-x/p(3))'
         END  
         
      ENDCASE
;      weight = 0*flux(fit_ind)+1
      weight = 1/flux(fit_ind)
;      weight = 1/sqrt(flux(fit_ind))
      
      IF keyword_set(_e) THEN BEGIN 
         isparv = where(tag_names(_E) EQ 'PARINFO')
         IF isparv(0) NE -1 THEN BEGIN
            isval = where(tag_names(_E.parinfo) EQ 'VALUE')
            IF isval(0) NE -1 THEN $
            p = mpfitfun(fitfun(i),energy(fit_ind),flux(fit_ind),err(fit_ind), PERROR = perror, STATUS = status, weight = weight, /quiet, _EXTRA = _E )
         ENDIF 
      ENDIF ELSE $
      p=mpfitfun(fitfun(i),energy(fit_ind),flux(fit_ind),err(fit_ind),start_params, PERROR = perror, STATUS = status, weight = weight, /quiet, _EXTRA = _E )
      
;    IF status EQ 6 THEN GOTO, rep
    if n_elements(funname) eq 0 then begin 
        funname=strupcase(fitfun(i))
        funstring=funstr
        parnr=[0, params] ; array giving index of this fit's parameters
        parval= p
        perrarr = perror
    endif else begin
        funname=[funname,strupcase(fitfun(i))]
        funstring=[funstring,funstr]
        parnr=[parnr,params+parnr(i)]
        parval=[parval,p]
        perrarr = [perrarr, perror]
    endelse 
   
 endfor
 all_fit_info={funname:funname,funstring:funstring,parnr:parnr,parval:parval, perror:perrarr}

 IF keyword_set(PLOT) THEN BEGIN 
    org_x = !x &  org_y = !y &  org_p = !p

    cleanplot, /silent
    window, title = 'Fit and Data'
    plot, energy, flux, /xlog, /ylog, psym = 4
    IF energy_dims EQ 2 THEN BEGIN 
       FOR i = 0,flux_vals/2-1 DO BEGIN 
          plots, energy(i*2), flux(i*2)
          plots, energy(i*2+1), flux(i*2+1), /CONTINUE
       endFOR 
    ENDIF
    sum = make_array(flux_vals)
    FOR i=0,n_elements(fitsep) DO BEGIN
       plotcall = all_fit_info.funname(i)
       p = all_fit_info.parval(all_fit_info.parnr(i):all_fit_info.parnr(i+1)-1)
       vals = call_FUNCTION(plotcall,energy,p)
       oplot, energy, vals
       sum = sum+vals
    ENDFOR
    oplot, energy, sum
    FOR i = 0, n_elements(fitsep)-1 DO BEGIN 
       plots, fitsep(i), min(flux)
       plots, fitsep(i), max(flux),linestyle = 2, /CONTINUE
    ENDFOR 
    IF keyword_set(fitlim) THEN BEGIN 
       plots, fitlim(0), min(flux)
       plots, fitlim(0), max(flux), linestyle = 2, /continue
       plots, fitlim(1), min(flux)
       plots, fitlim(1), max(flux), linestyle = 2, /CONTINUE
    endIF 
    !x = org_x &  !y = org_y &  !p = org_p
 ENDIF

 ;calculate the standard deviation between data and (total) fit
 IF keyword_set(stddev) AND keyword_set(plot) THEN BEGIN 
    ind = where(energy GT fitlim(0) AND energy LT fitlim(1))
    mean_dist = stddev(flux(ind)-sum(ind))
    print, 'standard deviation:'
    print, mean_dist
 endIF 


return,all_fit_info
 
end  


