;; papco_time is a simpler, more complete time library.  Drop T90,
;; mjdt is a time location, timeOffset is time offset structure,
;; and timeRange is a time range structure.  timeRanges are a useful
;; abstraction, when working with interval selections, describing when
;; data is available, etc.


function papco_timeDatum::Init, mjd, seconds, nanos
  if n_elements( nanos ) eq 0 then nanos=0
  self.mjd= mjd
  self.seconds= seconds
  self.nanos= nanos
  return, 1
end

function pt_isdigits, str
  b= byte(strtrim(str,2))
  r= where( b lt 48 or b ge 58 )
  return, r[0] eq -1
end

; returns { err:0, errtext:strarr() } and fills year, month, ...,
;       second, nanos
function papco_time_parse, atimeString, yy, doy, hh, mm, ss, nanos

  retVal={err:-1, $
          errtext:['Enter a time in one of these formats:', $
                   'date time, date', $
                   'where time is in one of those formats:', $
                   'h:m:s, h:m', $
                   'and date is in one of those formats:', $
                   'm/d/y, doy/y, d.m.y, m/d/y, yyyymmdd', $
                   'where d is day and doy is day of year, h:m:s is the time', $
                   '(year > 1990 if you use T90 format)', $
                   'e.g: 1/90 14:11, 1/1990 1:1:1, 350/1991 01:00:03', $
                   '     15.1.90 14:11, 1/15/1990 15:3:1']}

  addToTime=LONG(0)            ; this variable contains the seconds that have
                               ; to be added to seconds specified by the
                               ; time-part of the string in order to get the
                               ; seconds for the correct day.

  timeString=strtrim(aTimeString, 2)+' '

;--- parse a T90-string -------------------------------------------------------
  t90=1

  sarr=strarr(3)            ; the first words of the string
  n_s=0                     ; the number of words in the date-part
  d_type=-1                 ; the type of date -
                              ;     if the words in 'strarr' were seperated
                              ;        by '.', or d_type=0
                              ;        by '/', d_type=1
                              ;        by somehing else : d_type=-1

    ; find the first words of 'timeString' and extract them into 'sarr'
  end_reached=0
  FOR i=0, 2 DO BEGIN
      ind1=strpos(timeString, '/')
      IF ind1 EQ -1 THEN BEGIN
          ind1=strpos(timeString, '.')
          IF ind1 GT -1 THEN BEGIN
              IF d_type EQ 1 THEN $ ; in this case, the date-part is something
                return, retVal      ; like this: '1/5.1990' => produce error
              d_type=0
          ENDIF
      ENDIF ELSE BEGIN
          IF d_type EQ 0 THEN $    ; in this case, the date-part is something
            return, retVal         ; like this: '1.5/1990' => produce error
          d_type=1
      ENDELSE

      if not end_reached then begin
          IF ind1 EQ -1 THEN BEGIN ; timeString contains neither '.' nor '/'
              IF (d_type EQ 0 AND n_s EQ 2) OR (d_type EQ 1 AND n_s GE 1) THEN $
                ind1=strpos(timeString, ' ')
              ; find the end of the date by searching for ' '. Some lines
              ; before, timeString=strtrim(aTimeString, 2)+' ' was executed.
              ; so, ind1 contains the last position of aTimeString or the
              ; index where the time-part begins
              end_reached=1
          ENDIF

          IF ind1 GT -1 THEN BEGIN ; found a word...
              val=strtrim(strmid(timeString, 0, ind1), 2)
              IF val NE '' THEN BEGIN
                  sarr(n_s)=val ; store the n_s-th part.
                  timeString=strtrim(strmid(timeString, ind1+1), 2)+' '
                  IF NOT pt_isDigits(sarr(n_s)) THEN RETURN, retVal
                  n_s=n_s+1
              ENDIF
          ENDIF
      endif
  ENDFOR  ; end parsing the first words of 'timeString'

    ;now check for first word format of type yyyymmdd
  result=str_sep(strtrim(atimeString,2), ' ')
  pos=strpos(atimeString,'/')  ; ddd/yyyy format is also 8 characters long..
  if pos ne -1 then goto, skip1
  if strlen(result(0)) eq 8 then begin
      sarr(2)=strmid(result(0),0,4)
      sarr(1)=strmid(result(0),4,2)
      sarr(0)=strmid(result(0),6,2)
      d_type=0  &  n_s=3  &  end_reached=1
      IF n_elements(result) EQ 2 THEN TimeString= result(1) ELSE TimeString = '' ;<- A. Aasnes, put in time, if available
  endif
  skip1:

    ; now, sarr contains the first n_s words of 'aTimeString', and timeString
    ; does not contain the date any more
  IF n_s EQ 2 THEN BEGIN           ; 'day/year' - format
      day=long(sarr(0))
      year=long(sarr(1))
  ENDIF ELSE BEGIN
      IF n_s EQ 3 THEN BEGIN         ; 'd.m.y' - format
          IF d_type EQ 1 THEN BEGIN    ;english ?
              day_om=long(sarr(1))
              month=long(sarr(0))
          ENDIF ELSE BEGIN
              day_om=long(sarr(0))
              month=long(sarr(1))
          ENDELSE
          year=long(sarr(2))
          day=julday(month, day_om, year)-julday(1,1,year)+1
      ENDIF ELSE return, retVal
  ENDELSE

  day=LONG(day)
  year=LONG(year)

  IF year LE 99 THEN year=year+1900

;--- now, the format-specific part of the string is stripped, and the
;    string should consist only of a time.
  timeString=strtrim(timeString, 2)
  if timeString EQ '' THEN BEGIN ; no time specified
      hour=0L
      minutes=0L
      seconds=0L
  ENDIF ELSE BEGIN
      ind1=strpos(timeString, ':')
      ind2=strpos(timeString, ':', ind1+1)
      IF ind1 EQ -1 THEN RETURN, retVal

      hour=strtrim(strmid(timeString, 0, ind1), 2)
      IF NOT pt_isDigits(hour) THEN return, retVal

      IF ind2 EQ -1 THEN ind2=strlen(timeString)+1
      minutes=strtrim(strmid(timeString, ind1+1, ind2-ind1-1))
      IF NOT pt_isDigits(minutes) THEN return, retVal

      seconds='0'
      IF ind2 LT strlen(timeString)+1 THEN $
        seconds=strtrim(strmid(timeString, ind2+1, strlen(timeString)-ind2))
      IF NOT pt_isDigits(seconds) THEN return, retVal
  ENDELSE

  retVal.err=0

  ; change from Reiner's names to Jeremy's
  yy= year
  doy= day
  hh= hour
  mm= minutes
  ss= seconds
  nanos= 0L

  retVal.errtext=['ok']
  return, retVal
end

function papco_new_time, string, $
                         year=year, month=month, dom=dom, doy=doy, $
                         julian=julian, mjd=mjd, $
                         hours=hours, minutes=minutes, seconds=seconds, $
                         milliseconds=milliseconds, microseconds=microseconds, $
                         nanoseconds= nanoseconds
  if ( ( n_elements( string ) eq 1 ) and size( string, /type ) eq 7 ) then begin
      r= papco_time_parse( string, year, doy, hh, mm, ss, nanos )
      if ( r.err ne 0 ) then begin
          print, 'Invalid time format: '+string
      endif
      mjd= julday( 1, doy, year ) - 2400001L
      secs= hh*3600L + mm*60L + ss
      return, obj_new( 'papco_timeDatum', mjd, secs, nanos )
  endif else begin
      nanos= 0L
      secs= 0L
      if n_elements( hours ) gt 0 then secs= secs + 3600L * hours
      if n_elements( minutes ) gt 0 then secs= secs + 60L * minutes
      if n_elements( seconds ) gt 0 then secs= secs + seconds
      if n_elements( milliseconds ) gt 0 then nanos= nanos + long( milliseconds*1e6 )
      if n_elements( microseconds ) gt 0 then nanos= nanos + long( microseconds*1e3 )
      if n_elements( nanoseconds ) gt 0 then nanos= nanos + nanoseconds
      if ( n_elements( julian ) eq 1 ) then caldat, julian, month, dom, year
      if ( n_elements( mjd ) eq 1 ) then caldat, mjd+2400001L, month, dom, year
      if ( n_elements( doy ) eq 1 ) then caldat, julday( 1,doy,year ), month, dom
      mjd= julday( month, dom, year ) - 2400001L
      return, obj_new( 'papco_timeDatum', mjd, secs, nanos )
  endelse
end

function papco_new_timeoffset, string
  s= str_sep( string, ' ' )
  r= where( s ne '' )
  s= s[r]
  result= { days:0L, seconds:0L, nanos:0L }
  rseconds= 0.d;
  rnanos= 0.d;
  for i=0,n_elements(s)-1,2 do begin
      value= double( s[i] )
      unit= s[i+1]
      case unit of
          'days': begin
              rseconds= rseconds + ( value mod 1 * 86400 )
              result.days= result.days + long( value )
          end
          'hours': begin
              value= value * 3600
              rnanos= rnanos + ( value mod 1 * 1000000000L )
              result.seconds= result.seconds + long( value )
          end
          'min': begin
              value= value * 60
              rnanos= rnanos + ( value mod 1 * 1000000000L )
              result.seconds= result.seconds + long( value )
          end
          'sec': begin
              rnanos= rnanos + ( value mod 1 * 1000000000L )
              result.seconds= result.seconds + long( value )
          end
          'msec': begin
              value= value * 1000000L
              rnanos= rnanos + ( value mod 1 * 1000000L )
              result.nanos= result.nanos + long( value )
          end
          'usec': begin
              value= value * 1000L
              rnanos= rnanos + ( value mod 1 * 1000L )
              result.nanos= result.nanos + long( value )
          end
          'nanos': begin
              rnanos= rnanos + ( value mod 1 )
              result.nanos= result.nanos + long( value )
          end
          default: begin
              message, 'unrecognized unit: '+unit
          end
      endcase
  endfor

  value= rseconds
  rnanos= rnanos + ( value mod 1 * 1000000000L )
  result.seconds= result.seconds + long( value )

  if ( rnanos mod 1 gt 0.1 ) then begin
      message, 'round off errors'
  endif

  result.nanos= result.nanos + long( rnanos )

  return, obj_new( 'papco_timeoffset', result.days, result.seconds, result.nanos )

end

function papco_is_time, object
  t= tag_names( object )
  return, t[0] eq 'MJD'
end

function papco_is_timeoffset, object
  t= tag_names( object )
  return, t[0] eq 'DAYS'
end

function papco_is_timerange, object
  t= tag_names( object )
  return, t[0] eq 'T1'
end

function papco_timeoffset, mjdt1, mjdt2
  rseconds= mjdt2.t-mjdt1.t
  nanos= long( ( rseconds mod 1 ) * 1000000000L )
  seconds= long( rseconds )
  return, { days:mjdt2.mjd-mjdt1.mjd, $
            seconds: seconds, $
            nanos: nanos }
end

function papco_timeoffset::Init, days, seconds, nanos
  self.days=days
  self.seconds= seconds
  self.nanos= nanos
  return,1
end

function papco_timeDatum::subtract, time
  days= self.mjd - time.mjd
  seconds= self.seconds - time.seconds
  nanos= self.nanos - time.nanos
  return, obj_new( 'papco_timeoffset', days, seconds, nanos )
end

function papco_timerange::toString
  t1str= self.t1->toString()
  t2str= self.t2->toString()
  return, t1str + ' - ' + t2str
end

pro papco_timeDatum::breakApart, year, month, dom, hour, minute, second, nanos, doy=doy
  caldat, self.mjd + 2400001L, month, dom, year
  doy= self.mjd + 2400001L - julday( 1, 1, year ) + 1
  hour= self.seconds / 3600
  minute= self.seconds mod 3600 / 60
  second= self.seconds mod 60
  nanos= self.nanos
end


function papco_timeoffset::toString
  seconds= self.seconds
  hours= seconds / 3600
  seconds= seconds mod 3600
  r= ''
  if ( self.days ne 0 ) then r= r + strtrim(string( self.days ),2) + ' days '
  if ( hours ne 0 ) then r= r + strtrim(string( hours ),2) + ' hours '
  if ( seconds ne 0 ) then r= r + strtrim(string( seconds ),2) + ' sec '

  nanos= self.nanos
  millis= nanos / 1000000L
  nanos= nanos mod 1000000L
  if ( millis ne 0 ) then r= r+strtrim(string( millis ),2) + ' msec '

  if ( nanos ne 0 ) then begin
      micros= nanos / 1000L
      nanos= nanos mod 1000L
      r= r + strtrim( string( micros ),2 )
      if ( nanos ne 0 ) then begin
          r= r + '.' + strtrim( string( format='(i3.3)', nanos ),2 ) + ' usec '
      endif else begin
          r= r + ' usec '
      endelse
  endif
  return, r
end

; return <0 if mjdt1 is smaller, 0 if equal, and >0 if it's bigger
;
function papco_timeDatum::compareTo, time
  return, ( self.mjd - time.mjd ) + $
    ( self.seconds - time.seconds ) / 86400. + $
    ( self.nanos - time.nanos ) / ( 86400 *  1d9 )
end

function papco_timeDatum::lt_, time
  return, self->compareTo(time) lt 0
end

function papco_timeDatum::le_, time
  return, self->compareTo(time) le 0
end

function papco_timeDatum::gt_, time
  return, self->compareTo(time) gt 0
end

function papco_timeDatum::ge_, time
  return, self->compareTo(time) ge 0
end


function papco_timeDatum::min, time
  if ( self->lt_(time) ) then return, self else return, time
end

function papco_timeDatum::max, time
  if ( self->gt_( time ) ) then return, self else return, time
end

function papco_timeDatum::next, month=month, day=day
  if ( n_elements( day ) eq 1 ) then begin
      return, papco_new_time( mjd= self.mjd+1 )
  endif
  if ( n_elements( month ) eq 1 ) then begin
      jd=self.mjd+2400001
      caldat, jd, mm, dd, yy
      mm= mm+month
      dd= 1
      if ( mm gt 12 ) then begin
          yy = yy + (mm-1) / 12
          mm = ( ( mm-1 ) mod 12 ) + 1
      endif
      return, papco_new_time( year=yy, month=mm, dom=dd )
  endif
end

function papco_timerange::Init, time1, time2
  if ( time1->compareTo(time2) gt 0. ) then begin
      message, 'time1 > time2'
      return, 0
  endif
  self.t1= time1
  self.t2= time2
  return, 1
end

function papco_new_timerange, time1_in, time2_in
  if ( size( time1_in, /type ) eq 7 ) then begin
      time1= papco_new_time( time1_in )
      time2= papco_new_time( time2_in )
  endif else begin
      time1= time1_in
      time2= time2_in
  endelse
  return, obj_new( 'papco_timerange', time1, time2 )
end

function papco_timerange::contains, mjdt
  return, ( self.t1->compareTo( mjdt ) le 0 ) $
    and ( mjdt->compareTo( self.t2 ) lt 0 )
end

function papco_timerange::intersects, dr
  return, ( self.t2->compareTo( dr.t1 ) gt 0 ) $
    and ( self.t1->compareTo( dr.t2 ) lt 0 )
end

function papco_timerange::getStart
  return, self.t1
end

function papco_timerange::getEnd
  return, self.t2
end

function papco_timerange::include, o
  if ( obj_isa( o,'papco_timeDatum') ) then begin
      return, obj_new( 'papco_timerange', self.t1->min(o), $
                       self.t2->max(o) )
  endif else begin
      return, obj_new( 'papco_timerange', self.t1->min(o.t1), $
                       self.t2->max(o.t2) )
  endelse
end

function papco_timeDatum::newTimeRange, time
  return, obj_new( 'papco_timerange', $
                   self->min(time), $
                   self->max(time) )
end

function papco_timeDatum::getMjd
  return, self.mjd
end

function papco_timeDatum::getSsm
  return, self.seconds + self.nanos / 1d9
end

function papco_timeDatum::toMjdt
  mjdt= { mjd:self->getMjd(), $
          t:self->getSsm() }
  return, mjdt
end

function papco_timeDatum::toString, country=country
  country=1
  time=' 00:00:00'
  caldat, self.mjd + 2400001L, month, d_om, year
  hour= self.seconds / 3600
  minutes = self.seconds mod 3600 / 60
  sec= self.seconds mod 60

  timeH=strtrim(STRING(fix(hour)), 2)
  strput, time, timeH, 3-strlen(timeH)
  timeM=strtrim(STRING(fix(minutes)), 2)
  timeS=strtrim(STRING(fix(sec)), 2)
  strput, time, timeM, 6-strlen(timeM)
  strput, time, timeS, 9-strlen(timeS)
  IF NOT keyword_set(COUNTRY) then COUNTRY=0
  IF COUNTRY eq 0 then begin
      date=strtrim(string(d_om,format='(i2.2)'),  2) + '.' + $
        strtrim(string(month,format='(i2.2)'),2)+'.'+$
        strtrim(string(year),2)
  ENDIF ELSE BEGIN
      date=strtrim(string(month,format='(i2.2)'),  2) + '/' + $
        strtrim(string(d_om,format='(i2.2)'),2)+'/'+$
        strtrim(string(year),2)
  ENDELSE
  return, date+time

end

pro papco_timeDatum__define
  struct= { papco_timeDatum, $
            mjd:0L, $
            seconds:0L, $
            nanos:0L $
          }
end

pro papco_timeRange__define
  struct= { papco_timeRange, $
            t1:obj_new(), $
            t2:obj_new() $
          }
end

pro papco_timeOffset__define
  struct= { papco_timeOffset, $
            days:0L, $
            seconds: 0L, $
            nanos: 0L }
end

pro papco_time_test
  t1= papco_new_time( '1/1/2000' )
  t2= papco_new_time( '1/2/2000' )
  t= papco_new_time( year=2000, doy=1, hour=23, minute=45, second=12, milli=345, nano=13 )
  tr= papco_new_timerange( t1, t2 )
  print, t1->tostring()
  print, ( t2->tostring() )
  print, ( t->tostring() )
  print, ( t->subtract( t1 ) )->tostring()
  print, ( tr->tostring() )
  print, tr->contains( t )
  tr2= papco_new_timerange( t1, t )
  print, ( tr2->tostring() )
  print, ( tr2->include( t2 )) ->tostring()
  tr3= papco_new_timerange( t, t2 )
  print, (tr2->include( tr3 ))->tostring()
  print, tr2->intersects( tr3 )
  print, tr->intersects( tr3 )
end

pro papco_time
  papco_time_test
end



