;******************************************************************************
;******************************************************************************
;*  file-name:   papco_util.pro
;*
;*  description: this file contains procedures and functions that are regularly
;*               used by other WIDGET-programs or by the papco-project.
;* 	         Each procedure's name starts with the prefix 'papco_', to
;*               distinguish it from other programs.
;******************************************************************************
;******************************************************************************

;******************************************************************************
;* function:     papco_module_helpfile
;*
;* description:  uses the environment-variable 'papco_modules' to identify
;*               the directory containing the helpfile
;*
;* inputs:       the filename without directory
;*
;* output:	 the complete filename
;*
;* author:       A.Keese, august 1995
;*
;* changes:      -
;******************************************************************************
function papco_module_helpfile, filename

COMMON module_paths, paths

; find all help files in the MODULE_PATH/papco_interface directories
for i=0,n_elements(paths)-1 do begin
    result=findfile(paths(i)+'papco_interface/*.help')
    if i eq 0 then helpfiles=result else helpfiles=[helpfiles,result]
endfor

; see if filename is amongst these files, get full path
result=strpos(helpfiles,filename)
index=where(result ne -1,count)

if count NE 0 then begin
     fln=helpfiles(index(0))
     return,fln
endif else return, ''

END

;******************************************************************************
;* Procedure:    papco_replace_name
;*
;* description:  Goes through the whole 'path' tree replacing all
;*               occurences of one string with another. Uses a perl call
;*               ONLY works for UNIX!
;*
;* inputs:       path       -  path to start with
;*               old, new   -  original and new string.
;*
;* keywords      filename   -  change occurrence in filename too!
;*
;* output:	 none
;*
;* author:       R. Friedel, October 1997
;*
;* changes:      -
;******************************************************************************
PRO papco_replace_name, path, old, new, MASK = MASK, FILENAME = FILENAME

IF keyword_set(MASK) THEN mask = MASK ELSE mask = '*.pro'

spawn,'find '+path+' \( -type d  \) -exec ls -d {} \;', result

FOR i=0,n_elements(result)-1 DO BEGIN

    CD,result(i)
    spawn,'pwd'
    print,"perl -e 's/"+old+"/"+new+"/gi' -p -i.bak "+mask+" "
    spawn,"perl -e 's/"+old+"/"+new+"/gi' -p -i.bak "+mask+" "

    IF keyword_set(FILENAME) THEN BEGIN
        old_flns = findfile(result(i), count = n)
        IF n NE 0 THEN BEGIN
            new_flns = strarr(n)
            FOR j = 0, n-1 DO BEGIN
                p = strpos(old_flns(j), old)
                IF p NE -1 THEN BEGIN
                    new_flns(j) = strmid(old_flns(j), 0, p) + new + $
                      strmid(old_flns(j), p+strlen(old), strlen(old_flns(j)))
                    print, old_flns(j), '  ->  ', new_flns(j)
                    spawn, 'mv '+old_flns(j)+' '+new_flns(j)
                ENDIF
            ENDFOR
        ENDIF
    ENDIF

ENDFOR
CD,'~'

END

;******************************************************************************
;* procedure:    papco_create_dir, path
;*
;* description:  checks if path contains a directory, and creates it
;*               if not. Checks for each dir in path!
;*
;* inputs:       path : directory path
;*
;* output:       none
;*
;* keywords:     NOASK    don't ask for confirmation on create
;*
;* author:       R. Friedel, November 2000
;*
;* changes:      Opsys independent form J L Roeder, Nov 2003
;******************************************************************************
PRO papco_create_dir, path, NOASK = NOASK

IF NOT papco_finddir(path) THEN BEGIN
   IF NOT keyword_set(NOASK) THEN BEGIN
       r = DIALOG_MESSAGE(['Directory', path, 'does not exist.', '', $
       		'Create?'], /QUESTION, TITLE='Create Directory')
       IF strupcase(r) EQ 'NO' THEN BEGIN
             path = '' & return
       ENDIF
   ENDIF
   catch, err
   if ( err eq 0 ) then begin
       file_mkdir, path
   endif else begin
       x= widget_message( ['file_mkdir fails, check permissions'] )
   endelse
   catch, /cancel
ENDIF

END

;******************************************************************************
;* FUNCTION:     papco_finddir, path
;*
;* DESCRIPTION:  an operating system independent routine to check if a given
;*               directory path exists.
;*
;* INPUTS:       path       string which contains the path to be checked
;*
;* OUTPUT:       Function returns:
;*               1   directory pointed to in path exists
;*               0   otherwise
;*
;* KEYWORDS:     none
;*
;* HISTORY:      written Octoberr 1997, Reiner Friedel
;*               modified to use "catch" and "cd", S. Claflin, March 1998
;*****************************************************************************
FUNCTION papco_finddir, path

;IF !version.release GE 5.4 THEN return, FILE_TEST(path)  ;sm

IF path EQ '' THEN return, 0
orig_path=papco_getenv('PAPCO_HOME')

catch, error_status
if error_status ne 0 then begin
    return, 0
endif else begin
    cd, path & cd, orig_path
ENDELSE
return, 1

END

;******************************************************************************
;* FUNCTION:     papco_accessdir, path
;*
;* DESCRIPTION:  an operating system independent routine to check if a given
;*               directory path can be written to.
;*
;* INPUTS:       path       string which contains the path to be checked
;*
;* OUTPUT:       Function returns:
;*               1   directory pointed to in path can be written to
;*               0   otherwise
;*
;* KEYWORDS:     none
;*
;* HISTORY:      written May 1998, Reiner Friedel
;*****************************************************************************
function papco_accessdir, path

  openw,unit,path+'/junk', /GET_LUN, ERROR=error_status

  if error_status ne 0 then begin
    return, 0
  endif else begin
    close,unit
    free_lun,unit
    return, 1
  endelse

end

;*****************************************************************************
;* FUNCTION:     papco_dirfilter, dirlist, path_name
;*
;* DESCRIPTION:  checks a proposed list of directories and filters out
;*               non-directories
;*
;* INPUTS:       dirlist     string array containing the list of files to be
;*                           checked
;*               path_name   path of directory containing dirlist
;*
;* OUTPUT:       Function returns:      list of directories
;*
;* KEYWORDS:     none
;*
;* HISTORY:      written March 1998, S. Claflin
;*****************************************************************************
FUNCTION papco_dirfilter, dirlist, path_name

  ndirs = n_elements(dirlist)
  path_name_select = bytarr(ndirs) + 1b

  for i=0,ndirs-1 do begin
    path_name_select(i) = papco_finddir(path_name + dirlist(i))
  endfor

  dir = where(path_name_select, count)
  if (count eq 0) then begin
    print, "% papco_dirfilter:  I can't find any directories."
  endif else begin
    dir =  dirlist(dir)
  endelse

return, dir

END

;*****************************************************************************
;* PROCEDURE:    papco_file_status
;*
;* DESCRIPTION:  checks the read/write/exisit status of a file.
;*
;* INPUTS:       file        the file to be checked
;*
;* OUTPUT:       Function returns:      list of directories
;*
;* KEYWORDS:     none
;*
;* HISTORY:      written Oktober 1998, Jermey Faden
;*****************************************************************************
pro papco_file_status, file, write=canwrite, read=canread, exist=exist

   canread=0  &  canwrite=0  &  exist=0

   if file eq '' then return

   on_ioerror, g1
   openr, lun, file, /get_lun
   canread=1
   close, lun  &  free_lun, lun

g1:
   if not demo_mode() then begin ; one can't write files in demo_mode
       on_ioerror, g2
       if canread then begin
           openw, lun, file, /append, /get_lun
           canwrite=1
           close, lun  &  free_lun, lun
       endif else begin
           openw, lun, file, /delete, /get_lun
           canwrite=1
           close, lun  &  free_lun, lun
       endelse
     endif

g2:
   f= findfile( file, count=count )
   exist= fix( count eq 1 )

end

;*****************************************************************************
;* PROCEDURE:    papco_routine_info
;*
;* DESCRIPTION:  prints out routine header information
;*
;* INPUTS:       file    name of file containing routine
;*               name    name of routine
;*
;* OUTPUT:       prints out the routine header for the routine specified
;*
;* KEYWORDS:     ROOT_DIR  start search from this root directory
;*
;* HISTORY:      written August 1999, Reiner Friedel
;*****************************************************************************
pro papco_routine_info, file, name, ROOT_DIR=ROOT_DIR

  if keyword_set(ROOT_DIR) then $
    paths=expand_path('+'+ROOT_DIR, /ARRAY, COUNT=cc) $
    else  paths=!path

  message,'Searching for file '+file+'.pro ...',/cont

  for i=0, cc-1 do begin
    result=findfile(paths(i)+'/'+file+'.pro',count=c)
    if c eq 1 then begin
      message,'Found: '+result(0), /cont
      goto, next
    endif
  endfor

  message,'File '+file+'.pro not found in IDL search path (!path)',/cont
  return

  next:
  message,'Searching for routine '+name+' ...',/cont

  print,''
  START_ID=';* CALL ('+name+'):'
  len=strlen(START_ID)
  END_ID=';;'
  inst=''  &  found=0

  openr,unit,result(0),/get_lun

  while not eof(unit) do begin
    readf,unit,inst
    if strmid(inst,0,len) eq START_ID then found=1
    if found eq 1 then begin
      if strmid(inst,0,2) eq END_ID then begin
        print,''
        close,unit  &  free_lun,unit
        return
      endif
      print,'   ',strmid(inst,3,strlen(inst))
    endif
  endwhile

  message,'No header info found for routine '+name,/cont
  print,''

end

;******************************************************************************
;* FUNCTION:     papco_check_data_env, env
;*
;* DESCRIPTION:  a routine to check if a given environmental variable
;*               exists and points to a valid directory.
;*
;* INPUTS:       env      string   -> environmental variable name
;*
;* OUTPUT:       Function returns:
;*               1   environmental variable is set
;*               0   otherwise
;*
;* KEYWORDS:     PATH returns the env variable
;*               NODIR does not perform check for existence of directory
;*
;* HISTORY:      written September 1999, Reiner Friedel
;*               updated to look for $PAPCO_DLNK in paths, January
;*                 2003, Reiner Friedel
;*****************************************************************************
FUNCTION papco_check_data_env, data_env, PATH=PATH, NODIR=NODIR

COMMON get_error, get_err_no, get_err_msg

; check env. variable pointing to data directory
path=papco_getenv(data_env)
if strlen(path) eq 0 then begin
    get_err_no=1
    get_err_msg='Environmental variable '+data_env+' not set'
    message,get_err_msg, /cont
    return, 0
endif

if keyword_set(NODIR) then return,1

;see if string "$PAPCO_DLNK" is in path, if so, check for it!
test = strmid(path, 0, 11)
IF test EQ '$PAPCO_DLNK' THEN BEGIN
    part_path = strmid(path, 12, strlen(path)-12)
    dlnk_path =  papco_getenv('PAPCO_DLNK')
    IF dlnk_path NE '' THEN path = dlnk_path+part_path ELSE BEGIN
        get_err_no=3
        get_err_msg='Environmental variable PAPCO_DLNK not set'
        message, get_err_msg, /cont
        print, '  $PAPCO_DLNK is used as part of env. variable'
        print, '  '+data_env
        return, 0
    ENDELSE
ENDIF

;new PAPCO 10.6. Automatically create directories as specified by
;data_env variable. 
papco_create_dir, path

; check for existence of data directory
result=papco_finddir(path)
if result eq 0 then begin
    get_err_no=3
    get_err_msg='Env. var. '+data_env+': path '+path+' not found'
    message,get_err_msg, /cont
    return,0
endif

return,1

end

;******************************************************************************
;* FUNCTION:     papco_findfile, path, COUNT=COUNT
;*
;* DESCRIPTION:  a routine to allow findfile to expand "~" as home dir path
;*
;* INPUTS:       path (same as for idl findfile routine)
;*
;* OUTPUT:       files found (string array)
;*
;* KEYWORDS:     count (same as for idl findfile routine)
;*
;* HISTORY:      written October 1999, Reiner Friedel
;*****************************************************************************
FUNCTION papco_findfile, path, COUNT=COUNT

; findfile doesn't recognize '~' (seems to use sh instead of csh)
if (strmid(path,0,1) eq '~') then $
    pathff = papco_getenv('HOME') + strmid(path,1,strlen(path)) $
else pathff = path

result=findfile(pathff,count=count)

return,result

END

;******************************************************************************
;* FUNCTION:     papco_write_struct, filename, struct_name, struct, $
;*                                   ARRAY=ARRAY, APPEND=APPEND
;*
;* DESCRIPTION:  a routine to write an IDL structure to a text file
;*
;* INPUTS:       filename
;*
;* OUTPUT:       two element results structure:
;*                {error - 0 no error, 1 error
;*                 text  - '' or string array if keyword ARRAY set
;*
;* KEYWORDS:     ARRAY  - do not write to file, return string array
;*               APPEND - append to file_name
;*               TAGID  - only save tags that contain TAGID
;*               UNIT   -
;*
;* HISTORY:      written February 2000, Reiner Friedel
;*               modified May 2003 to write contents of Pointers
;*****************************************************************************
function papco_write_struct, file_name, struct_name, struct, $
                             TAGID=TAGID, ARRAY=ARRAY, APPEND=APPEND

forward_function papco_write_struct

if keyword_set(APPEND) then append=APPEND else append=0
if keyword_set(TAGID) then tag_id=strupcase(TAGID) else tag_id=''

var_names=tag_names(struct)

if tag_id ne '' then BEGIN
    var_names=tag_names(struct)
    result=strpos(tag_id,'*')         ;see if wildcard tag exists in var_names
    if result ne -1 then BEGIN
        tag_id=strdel(tag_id,'*')
        result=strpos(var_names,tag_id)
        index=where(result ne -1,c)
    endif else begin                 ;see if tag exisits uniquely in var_names
        c=0 & idx=-1
        for i=0,n_elements(var_names)-1 do BEGIN
            if var_names(i) eq tag_id then BEGIN
                c=c+1 & idx=i
            endif
        ENDFOR
        index=[idx]
    ENDELSE

    if c eq 0 then return, {error:1, $
      text:['No tags containing '+tag_id+' found in '+struct_name, $
            'No output written to '+file_name]  }

ENDIF ELSE BEGIN
    index=findgen(n_elements(var_names))
ENDELSE

FOR i=0,n_elements(index)-1 DO BEGIN

    full_var_name=var_names(index(i))
    var_type=size(struct.(index(i)),/STRUCTURE)

    var_type_str=varprt(var_type.TYPE) + ' ' + $
                 varprt(var_type.N_ELEMENTS) + ' ' + $
                 varprt(var_type.N_DIMENSIONS)
    for j=0,n_elements(var_type.DIMENSIONS)-1 do $
      var_type_str=var_type_str+ ' ' +varprt(var_type.DIMENSIONS(j))
    out_str_1='-> '+strformat(full_var_name,30,/left)+ ' -> '+ $
              strformat(var_type.TYPE_NAME,10,/left)+ ' -> ' + var_type_str

    if i eq 0 then out_arr=out_str_1 else out_arr=[out_arr,out_str_1]

    CASE var_type.TYPE_NAME OF
        'STRUCT': BEGIN
            FOR k = 0, var_type.N_ELEMENTS-1 DO BEGIN
                result = papco_write_struct('', var_names(index(i)), $
                                            struct.(index(i))(k), /ARRAY)
                ;write end of structure marker
                sub_arr=result.text
                sub_arr=[sub_arr,'-> END STRUCT']
                sub_arr='  '+sub_arr
                out_arr=[out_arr,sub_arr]
            ENDFOR
        END
        'POINTER': BEGIN ;assumes ONE pointer, not array of pointers
            ptr_var = *struct.(index(i))
            ptr_type = size(ptr_var,/ST)

            IF ptr_type.TYPE_NAME EQ 'STRUCT' THEN BEGIN

                ;first write the name of the structure, then its contents
                help, ptr_var, OUTPUT=help_str
                pos1 = strpos(help_str, '->') & pos2 = strpos(help_str, 'Ar')
                struct_id = strmid(help_str, pos1+3, pos2-pos1-3)
                result = papco_write_struct('', 'struct_id', $
                         {STRUCT_ID: struct_id}, /ARRAY)
                sub_arr=result.text

                ;next write out structure contents
                result = papco_write_struct('', 'usr_struct', ptr_var, /ARRAY)
                sub_arr=[sub_arr, result.text]

                ;write end of structure marker
                sub_arr=[sub_arr,'-> END STRUCT']
                sub_arr='  '+sub_arr

            ENDIF ELSE BEGIN

                result = papco_write_struct('', 'struct_id', $
                         {PTR_VAR: ptr_var}, /ARRAY)
                sub_arr=result.text

            ENDELSE

            ;write end of pointer marker
            sub_arr=[sub_arr,'-> END POINTER']
            sub_arr='  '+sub_arr
            out_arr=[out_arr,sub_arr]
        END
        ELSE: BEGIN
            FOR j = 0, var_type.N_ELEMENTS-1 DO BEGIN
                var_value=string(struct.(index(i))(j))
                out_arr=[out_arr,var_value]
            ENDFOR
        END
    ENDCASE

ENDFOR

if not keyword_set(ARRAY) then begin
    openw, u, file_name, error=err, APPEND=append, /get_lun
    for i=0,n_elements(out_arr)-1 do printf,u,out_arr(i)
    close,u  &  free_lun,u
    return,{error:0, text:''}
endif else begin
    return,{error:0, text:out_arr}
ENDELSE

END

;******************************************************************************
; Quick wrapper for print in papco. Checks verbose tag.
;*****************************************************************************
PRO papco_prt, str1,str2,str3,str4,str5,str6,str7,str8,str9, str10, str11,  $
               VERBOSE = VERBOSE

IF keyword_set(VERBOSE) THEN verbose = 1 ELSE verbose = 0

IF verbose THEN BEGIN
    n = n_params()  &  exe_str = "print"
    FOR i = 1, n DO exe_str = exe_str+", str"+varprt(i)
    print, exe_str
    result = execute(exe_str)
ENDIF

END

;******************************************************************************
;* FUNCTION:     papco_read_struct, filename, structname, struct
;*
;* DESCRIPTION:  a routine to read an IDL structure from a text file
;*
;* INPUTS:       filename - file to read structure from, string
;*               structname - a string name for the structure
;*               struct - the structure to save. Tags may be any
;*                        variable, a structure or a pointer to a
;*                        structure.
;*
;* KEYWORDS:     TAGID
;*               UNIT     read struct from this unit number / file
;*               VERBOSE  print out info
;*
;* OUTPUT:       two element results structure:
;*                {error - 0 no error, 1 error
;*                 text  - '' or string array if keyword ARRAY set
;*
;* KEYWORDS:     ARRAY - do not write to file, return string array
;*
;* HISTORY:      written February 2000, Reiner Friedel
;*               modified May 2003 to read contents of Pointers (as
;*               part of structur, Reiner Friedel
;*****************************************************************************
FUNCTION papco_read_struct, file_name, structname, struct, $
                            TAGID=TAGID, UNIT=UNIT, VERBOSE=VERBOSE

forward_function papco_read_struct

IF keyword_set(VERBOSE) THEN verbose=VERBOSE ELSE verbose=0

dummy=';'  &  level=0
var_names=tag_names(struct)

var_type={TYPE:0, N_ELEMENTS:0l, N_DIMENSIONS:0, DIMENSIONS:lonarr(8)}

IF keyword_set(UNIT) THEN BEGIN
    u=UNIT
ENDIF ELSE BEGIN
    openr, u, file_name, error=err, /get_lun
    IF (err NE 0) THEN goto, open_error
ENDELSE

ON_IOERROR, io_error

WHILE NOT eof(u) DO BEGIN
    readf, u, dummy
    IF strmid(dummy,0,1) EQ ';' THEN CONTINUE
    result=strpos(dummy,'->')
    IF result(0) NE -1 THEN BEGIN

        result=str_sep(dummy,'->') & tag=strtrim(result(1),2)

        IF keyword_set(UNIT) THEN IF (tag EQ 'END STRUCT') THEN GOTO, out
        IF keyword_set(UNIT) THEN IF (tag EQ 'END POINTER') THEN CONTINUE

        ;see if tag exisits uniquely in var_names
        c=0 & idx=-1
        FOR i=0,n_elements(var_names)-1 DO BEGIN
            IF var_names(i) EQ tag THEN BEGIN
                c=c+1 & idx=i
            ENDIF
        ENDFOR

        if c eq 0 then begin
            if verbose then message, 'Tag ' + tag + $
              ' : does not exist in structure '+ structname+', skipped' ,/cont
            CONTINUE
        ENDIF
        if c gt 1 then begin
            if verbose then message, 'Tag ' +tag + $
              ' : duplicates in struct '+ structname+', skipped',/cont
            CONTINUE
        ENDIF

        if verbose then message,'Tag '+tag+' : read into structure '+ $
          structname,/cont

        type = strtrim(result(2),2)
        reads, result(3), var_type

        ;check for nested structures, pointers
        CASE type OF
            'STRUCT': BEGIN
                result=papco_read_struct(file_name, tag,struct.(idx), UNIT=u)
                if result.error EQ 0 THEN struct.(idx)=result.struct
            END
            'POINTER':BEGIN
                ;the file contain the contents of the pointer, this has to
                ;be read and then the pointer has to be re-created
                ;pointing at the data read in.
                readf, u, dummy
                result=str_sep(dummy,'->') & tag=strtrim(result(1),2)

                IF tag EQ 'PTR_VAR' THEN $ ;pointer is unassigned
                    struct.(idx) = PTR_NEW(0) $
                ELSE BEGIN
                    ;pointer points at a structure, don't know what it
                    ;is. Read in the data, make the structure, create pointer
                    ;pointing at it, load the pointer into plotinfo.

                    ;in the POINTER block written, the first entry
                    ;is a variable STRUCT_NAME, refering to the named
                    ;structure that the following data refers to.
                    ;if the module for this is loaded, that structure
                    ;should be defined and can be referenced by name..
                    ;If not, then issue an error.
                    IF tag EQ 'STRUCT_ID' THEN BEGIN
                        readf, u, dummy
                        struct_id = strtrim(dummy, 2)
                        ;make an instance of the user defined structure
                        result=EXECUTE('usr_struct = {'+struct_id+'}')
                        IF result NE 1 THEN GOTO, struct_error
                        ;now read data into the structure
                        result = papco_read_struct(file_name, struct_id, $
                                                   usr_struct, UNIT=u)
                        ;now load a pointer to this usr_struct into struct
                        struct.(idx) = PTR_NEW(usr_struct)
                    ENDIF

                ENDELSE
            END


            ELSE: BEGIN
                FOR i=0, var_type.N_ELEMENTS-1 DO BEGIN
                    readf, u, dummy
                    struct.(idx)(i)=dummy
                ENDFOR
            END
        ENDCASE
    ENDIF
ENDWHILE

out:

if not keyword_set(UNIT)then begin
    close,u  &  free_lun,u
endif

return,{error:0, text:'', struct:struct}

open_error:
return,{error:1, text:'Error opening file '+file_name}

io_error:
return,{error:1, text:'Error reading file '+file_name}

struct_error:
return,{error:1, text:'Struct def for '+struct_id+' not found, module loaded?'}

END

;******************************************************************************
;* PROCEDURE:    papco_write_png, fName, image
;*
;* DESCRIPTION:  a wrapper to write_png
;*               outputs individual panels als png files if needed
;*
;* INPUTS:       fName, image, r_papco, g_papco, b_papco
;*
;* OUTPUT:       none, files get written
;*
;* KEYWORDS:     none
;*
;* HISTORY:      written November 2000, Reiner Friedel
;*****************************************************************************
pro papco_write_png, fName, image

  common plot_composer, widgetdata
  common papco_color, papco_color, r_papco, g_papco, b_papco

  ;img=rotate(image,7)
  img=image

  if widgetdata.default_png_format eq 0 then begin  ;output full page

    message,'Writing PNG ' + fname, /cont
    write_png, fName, img, r_papco, g_papco, b_papco

  endif else begin  ;output individual panels
    ;loop through panels selecting out area of panels
    PlotNr=widgetData.numberOfPlots
    ;--- calculate the highest plot-panel-number
    maxPlotNo=0
    FOR i=0, PlotNr-1 DO $
    maxPlotNo=maxPlotNo > (widgetData.plotInfos(i).panelPosition+ $
			   widgetData.plotInfos(i).panelHeight)

    papco_sepdir, fName, path, file
    date_str=strmid(file,0,8)  &  ext=strmid(file,strlen(file)-4,4)
    ;check if subdirectory for this date exists, create if needed
    date_path=papco_addpath(path,date_str)
    if not papco_finddir(date_path) then $
      papco_create_dir, date_path, /NOASK
    ;check is subdirectory for LEFT & RIGHT images exists, create if needed
    left_right_path=papco_addpath(path,'left_right')
    if not papco_finddir(left_right_path) then $
      papco_create_dir, left_right_path, /NOASK

    for i=0, PlotNr-1 do begin
      ;--- construct the filename description string
      descr_str=widgetData.plotInfos(i).descr_str
      if descr_str eq '' then descr_str=widgetData.plotInfos(i).panelkind
      descr_str=strsubst(descr_str,'!C','  ')
      descr_str=STRCOMPRESS(descr_str)
      descr_str=STRTRIM(descr_str,2)
      descr_str=strsubst(descr_str,' ','_')
      descr_str=strdel(descr_str,'(')
      descr_str=strdel(descr_str,')')

      ;--- calculate the panel-vector
      pos=widgetData.plotInfos(i).panelPosition
      height=widgetData.plotInfos(i).panelHeight
      panelVector=[pos, maxPlotNo, height]
      panelset,panelVector      ;sets the panel viewport, normal coords

      if widgetdata.default_png_format eq 1 then begin

      ;--- do for image of data in plot only
      file=descr_str+'_DATA'+ext  &  fln=date_path+file
      ;--- calculate device coords of panel viewport
      norm_pos=!p.position  &  dev_pos=norm_pos
      result=convert_coord(norm_pos(0),norm_pos(1),/NORMAL,/TO_DEVICE)
      dev_pos(0:1)=result(0:1)
      result=convert_coord(norm_pos(2),norm_pos(3),/NORMAL,/TO_DEVICE)
      dev_pos(2:3)=result(0:1)
      dev_pos=round(dev_pos)
      ;---write the plot png
      message,'Writing PNG ' + fln, /cont
      panel_img=img(dev_pos(0):dev_pos(2),dev_pos(1):dev_pos(3))
      write_png, fln, panel_img, r_papco, g_papco, b_papco

      ;also write full image
      file='FULL'+ext  &  fln=date_path+file
      write_png, fln, img, r_papco, g_papco, b_papco

      endif else begin

      ;--- do for time axis image
      if i eq PlotNr-1 then begin ;key off bottom panel
        file='TIME_AXIS'+ext  &  fln=date_path+file
        ;--- calculate device coords of x-axis
        norm_pos=[0,0,0.99,!p.position(1)]  &  dev_pos=norm_pos
        result=convert_coord(norm_pos(0),norm_pos(1),/NORMAL,/TO_DEVICE)
        dev_pos(0:1)=result(0:1)
        result=convert_coord(norm_pos(2),norm_pos(3),/NORMAL,/TO_DEVICE)
        dev_pos(2:3)=result(0:1)
        dev_pos=round(dev_pos)
        ;---write the plot png
        message,'Writing PNG ' + fln, /cont
        time_img=img(dev_pos(0):dev_pos(2),dev_pos(1):dev_pos(3))
        write_png, fln, time_img, r_papco, g_papco, b_papco
      endif

      ;--- left and right side annotation are constant for a given
      ;    panel and only need to be written once!
      file=descr_str+'_LEFT'+ext  &  fln=left_right_path+file
      result=findfile(fln,count=c)
      if c eq 0 then begin
        ;--- calculate device coords of panel left side (y-axis label)
        norm_pos=[0,!p.position(1),!p.position(0),!p.position(3)]
        dev_pos=norm_pos
        result=convert_coord(norm_pos(0),norm_pos(1),/NORMAL,/TO_DEVICE)
        dev_pos(0:1)=result(0:1)
        result=convert_coord(norm_pos(2),norm_pos(3),/NORMAL,/TO_DEVICE)
        dev_pos(2:3)=result(0:1)
        dev_pos=round(dev_pos)
        ;---write the left side png
        message,'Writing PNG ' + fln, /cont
        left_img=img(dev_pos(0):dev_pos(2),dev_pos(1):dev_pos(3))
        write_png, fln, left_img, r_papco, g_papco, b_papco
      endif

      file=descr_str+'_RIGHT'+ext  &  fln=left_right_path+file
      result=findfile(fln,count=c)
      if c eq 0 then begin
        ;--- calculate device coords of panel right side (colorbar)
        norm_pos=[!p.position(2),!p.position(1),0.99,!p.position(3)]
        dev_pos=norm_pos
        result=convert_coord(norm_pos(0),norm_pos(1),/NORMAL,/TO_DEVICE)
        dev_pos(0:1)=result(0:1)
        result=convert_coord(norm_pos(2),norm_pos(3),/NORMAL,/TO_DEVICE)
        dev_pos(2:3)=result(0:1)
        dev_pos=round(dev_pos)
        ;---write the right side png
        message,'Writing PNG ' + fln, /cont
        right_img=img(dev_pos(0):dev_pos(2),dev_pos(1):dev_pos(3))
        write_png, fln, right_img, r_papco, g_papco, b_papco
      endif

      endelse

    endfor

  endelse

end

;******************************************************************************
;* PROCEDURE:
;*      papco_gap_plot, x, y, nodata, _EXTRA=e
;*
;* DESCRIPTION:
;*      This procedure plots data without connecting
;*      y data points flagged with bad data flag.
;*
;* INPUTS:
;*      x,y     arrays to be plotted
;*      nodata  bad dat flag value
;*
;* KEYWORDS:
;*     same as for plot /oplot routine
;*     OPLOT     -   use oplot
;*     AVERAGE   -   average all points in a given range between gaps
;*                   into one point.
;*
;*
;* OUTPUTS:
;*      none
;*
;* CALLING SEQUENCE:
;*      papco_gap_plot, x, y, nodata, _EXTRA=e
;*
;* MODIFICATION HISTORY:
;*     written January 2001, Reiner Friedel
;******************************************************************************
PRO papco_gap_plot, x, y, nodata, _EXTRA=e, OPLOT = OPLOT, AVERAGE = AVERAGE

xray=x  &  yray=y &  count = 0
; split up plot array into chunks of valid data only, and plot those.
index=where(yray ne nodata,c)

IF c EQ 0 THEN BEGIN
    message,'No valid points to plot',/cont
    return
ENDIF ELSE BEGIN
    i=0  &  ndat=n_elements(yray)
    if e.psym gt 0 then nn=0 else nn=1

    WHILE i NE ndat-1 DO BEGIN
        ;get to beginning of good data
        while (yray(i) eq nodata) AND (i ne ndat-1) do i=(i+1) < (ndat-1)
        sidx=i
        ;get to end of good data
        while (yray(i) ne nodata) AND (i ne ndat-1) do i=(i+1) < (ndat-1)
        eidx=i-1
        ;plot if more than nn points
        if (eidx - sidx) lt nn then CONTINUE
           IF keyword_set(OPLOT) THEN BEGIN
                IF keyword_set(AVERAGE) THEN BEGIN
                   n = n_elements(xray(sidx:eidx))
                   xp = total(xray(sidx:eidx)) / n
                   yp = total(yray(sidx:eidx)) / n
                   IF count EQ 0 THEN cont = 0 ELSE cont = 1
                    plots, xp, yp, /DATA, CONTINUE = cont, _EXTRA=e
                    count = count+1
                 ENDIF ELSE BEGIN
                    oplot, xray(sidx:eidx), yray(sidx:eidx), _EXTRA=e
                 ENDELSE
           ENDIF ELSE BEGIN
                   plot, xray(sidx:eidx), yray(sidx:eidx), _EXTRA=e
           ENDELSE

    ENDWHILE
ENDELSE
END

;******************************************************************************
;* FUNCTION:     papco_find_dates_todo, all_dates, done_dates, MSG=MSG
;*
;* DESCRIPTION:  a routine to select which dates still need doing from
;                a list of done dates and all available dates.
;                Used for batch processing.
;*
;* INPUTS:       all_dates, done_dates  string arrays of dates
;*               msg                    string of message to report
;*
;* OUTPUT:       string array of dates still to be done
;*
;* KEYWORDS:     none
;*
;* HISTORY:      written August 2000, Reiner Friedel
;*****************************************************************************
function papco_find_dates_todo, all_dates, done_dates, SAME = SAME, MSG=MSG

todo_dates = strarr(n_elements(all_dates))
nn = 0

IF KEYWORD_SET(SAME) THEN found = 1 ELSE found = 0

FOR i = 0, n_elements(all_dates)-1 DO BEGIN
    index = where(done_dates EQ all_dates(i), c)
    IF c EQ found THEN BEGIN
      todo_dates(nn) = all_dates(i)  &  nn=nn+1
    ENDIF
ENDFOR

IF keyword_set(MSG) THEN msg = MSG ELSE msg = 'To do'
  message, msg + ': '+ varprt(nn) + ' days', /cont

IF nn EQ 0 THEN return, 0

todo_dates = todo_dates(0:nn-1)
return, todo_dates

END

;******************************************************************************
;* FUNCTION:     papco_keyword
;*
;* DESCRIPTION:  a routine set a keyword to a default if not set
;*
;* INPUTS:       keyword, default value
;*
;* OUTPUT:       value of keyword
;*
;* HISTORY:      written April 2002, Reiner Friedel
;*****************************************************************************
function papco_keyword,  keyword,  default

IF keyword_set(keyword) THEN return, keyword ELSE return, default

END

;******************************************************************************
;* FUNCTION:     papco_pitch_bin
;*
;* DESCRIPTION:  a routine to make pitch angle bins centered on 90 deg
;*
;* KEYWORDS:     pa_idx : returns an array of 1 degree pitch angle
;*               mapings to the index of the corresponding pitch
;*               angle bin. This is used in binning pitch angles into
;*               the chosen pitch angle ranges - much faster!
;*
;* INPUTS:       pa_bin    -  center pitch angle bin width
;*
;* OUTPUT:       array of start / end of pa bins
;*
;* HISTORY:      written July 2002, Reiner Friedel
;******************************************************************************
FUNCTION papco_pitch_bin, pa_bin,  PA_IDX = PA_IDX

pa_rng = fltarr(1, 2) & top = 90.0+pa_bin/2.0 & bot = 90.0-pa_bin/2.0
pa_rng(0, *) = [bot, top] & next_top = 90.0
WHILE next_top LT 180 DO BEGIN
    next_top = (top+pa_bin) < 180.0 & next_bot = (bot-pa_bin) > 0.0
    pa_rng = [transpose([next_bot, bot]), pa_rng, transpose([top, next_top])]
    top = next_top & bot = next_bot
ENDWHILE

idx = bytarr(180)
n_bins = n_elements(pa_rng)/2
FOR i = 0, 179 DO BEGIN
    FOR j = 0, n_bins-1 DO BEGIN
        IF (i GE pa_rng(j, 0) ) AND (i LE pa_rng(j, 1 ) ) THEN $
          idx(i) = j
    ENDFOR
ENDFOR

PA_IDX = idx

return, pa_rng

END

;******************************************************************************
;* FUNCTION:     papco_expand_path
;*
;* DESCRIPTION:  expands $PAPCO_DLNK if the path starts with it
;*
;* KEYWORDS:     none
;*
;* INPUTS:       path
;*
;* OUTPUT:       expanded path
;*
;* HISTORY:      written MAy 2003, Reiner Friedel
;******************************************************************************
FUNCTION papco_expand_path, path

;data may contain env var $PAPCO_DLNK. Expand.
test = strmid(path, 0, 11)

IF test EQ '$PAPCO_DLNK' THEN BEGIN
    part_path = strmid(path, 12, strlen(path)-12)
    IF papco_check_data_env('PAPCO_DLNK', PATH=dlnk_path) THEN $
        expanded_path = dlnk_path+part_path $
    ELSE expanded_path = path
ENDIF ELSE expanded_path = path

return, expanded_path

END


;******************************************************************************
;* FUNCTION:     papco_compare_struct
;*
;* DESCRIPTION:  checks if two structures have the same conents.
;*               structures must be of the same type!
;*
;* KEYWORDS:     none
;*
;* INPUTS:       struct_1, struct_2
;*
;* OUTPUT:       0 = the same, 1 = not the same
;*
;* HISTORY:      written May 2003, Reiner Friedel
;******************************************************************************
FUNCTION papco_compare_struct, struct_1, struct_2

n_tags = n_elements(TAG_NAMES(struct_1))

check = 0
FOR i= 0, n_tags-1 DO BEGIN
    FOR j=0, n_elements(struct_1.(i))-1 DO BEGIN
        IF struct_1.(i)(j) NE struct_2.(i)(j) THEN check = 1
    ENDFOR
ENDFOR

return, check

END

;******************************************************************************
;* FUNCTION:     papco_interpol
;*
;* DESCRIPTION:  wrapper to the IDL function interpol, but extended
;*               functionality to preserve bad data flags.
;*
;* KEYWORDS:     NODATA - bad data flag, default is -99.0
;*
;* INPUTS:       same as IDL function interpol
;*
;* OUTPUT:       interpolated array
;*
;* HISTORY:      written June 2003, Reiner Friedel
;******************************************************************************
FUNCTION papco_interpol, V, X, U, NODATA = NODATA, _EXTRA=extra_par

IF keyword_set(NODATA) THEN bad_data = NODATA ELSE bad_data = -99.0
bad_data_arr = fltarr(n_elements(V))
idx = where(V EQ bad_data, c)
IF c NE 0 THEN bad_data_arr(idx) = 1.0

new_bad_data_arr = interpol(bad_data_arr, X, U)
new_V = interpol(V, X, U, _EXTRA=extra_par)

idx = where(new_bad_data_arr NE 0, c)
IF c NE 0 THEN new_V(idx) = bad_data

return, new_V

END

;******************************************************************************
;* FUNCTION:     papco_smooth
;*
;* DESCRIPTION:  wrapper to the IDL function smooth, but extended
;*               functionality to preserve bad data flags.
;*
;* KEYWORDS:     NODATA - bad data flag, default is -99.0
;*
;* INPUTS:       same as IDL function smooth
;*
;* OUTPUT:       smoothed array
;*
;* HISTORY:      written June 2003, Reiner Friedel
;******************************************************************************
FUNCTION papco_smooth, array, width, NODATA = NODATA, _EXTRA=extra_par

IF keyword_set(NODATA) THEN bad_data = NODATA ELSE bad_data = -99.0

;loop through array looking for all the data "between" bad data
;flags, only smooth those!

ndat = n_elements(array)
idx1 = 0l & idx2 = 0l &  i = 0l

WHILE i Lt ndat-1 DO BEGIN
    ;look for first good data point
  IF i le ndat-2 then begin
    WHILE (array(i) EQ bad_data) AND (i LE ndat-2) DO i = i+1
  ENDIF
  idx1 = (i)

    ;look for last good data point
   IF i le ndat-1 then begin
      WHILE (array(i) NE bad_data) AND (i LE ndat-2) DO i = i+1
      idx2 = i-1
   ENDIF

   IF idx1 ne ndat-1 then begin
      ;now smooth the good data
     ;  print, idx1, idx2
       size_arr=n_elements(array(idx1:idx2))
   IF size_arr gt width then $
       array(idx1:idx2) = smooth(array(idx1:idx2), width, _EXTRA=extra_par) $
   else array(idx1:idx2)=array(idx1:idx2)
   ENDIF

ENDWHILE

new_array=array
return, new_array

END

;******************************************************************************
;* FUNCTION:     papco_make_Lvt
;*
;* DESCRIPTION:  splits a data v. time array into an "L plot" array.
;*
;* KEYWORDS:     NODATA - bad data flag, default is -99.0
;*               NCUTS - guess at # of L-cuts, (overestimate!)
;*
;* INPUTS:       Yst, Yen - required L-range for binning
;*               t, y, l - time, data, l array, same base
;*
;* OUTPUT:       Y_arr - L-bins, start/end
;*               T_cut - start/end time of cut,
;*               Zarr  - L-binned data
;*
;* HISTORY:      written June 2003, Reiner Friedel
;******************************************************************************
PRO papco_make_Lvt, time, y, l, yst, yen, Y_arr, T_cut, Zarr, $
                    LBIN = LBIN, N_CUTS = N_CUTS,  NODATA = NODATA

IF keyword_set(NODATA) THEN bad_data = NODATA ELSE bad_data = -99.0
IF keyword_set(N_CUTS) THEN n_cuts = N_CUTS ELSE n_cuts = 10000

;make the l-binned array
IF keyword_set(LBIN) THEN lbin = LBIN ELSE lbin = 0.1
lrange=[yst, yen]
n_bins= (yen - yst ) / lbin
y_arr = fltarr(n_bins, 2)
y_arr(*, 0) = (findgen(n_bins) * lbin) + lrange(0)
y_arr(*, 1) = y_arr(*, 0) + lbin
l_arr = y_arr(*, 0) + lbin/2

;make output arrays
zarr=fltarr(n_cuts,n_bins)  &  t_cut=dblarr(n_cuts, 2)

message, 'Making L v. time array',/cont

;limit all data down to l-range needed here!

idx = where( ((l GE yst) AND (l LE yen)), ndat)
IF ndat NE 0 THEN BEGIN
    ln = l(idx) & yn = y(idx) & tn = time(idx)
ENDIF

apo_idx=0 & per_idx=0 & n_cut=0

;go through data, look for L-apogees and perigees, bin data
FOR i= 2l, ndat-3l DO BEGIN

    ;check for L-perigee
    IF ( (ln(i-2) GE ln(i)) AND (ln(i) LE ln(i+2)) ) THEN per_idx=i

   ;check for L-apogee
    IF ( (ln(i-2) LT ln(i)) AND (ln(i) gt ln(i+2)) ) THEN apo_idx=i

    ;if a cut is found, set the indices
    IF per_idx*apo_idx NE 0 THEN BEGIN

        ;print, n_cut, per_idx, apo_idx, abs(apo_idx-per_idx)
        IF per_idx lt apo_idx THEN BEGIN
            from=per_idx & to=apo_idx & per_idx=0
        ENDIF ELSE BEGIN
            from=apo_idx & to=per_idx & apo_idx=0
        ENDELSE

        ;fold data from l_range into L-bins. Preserve bad data flags!
        data_l=ln(from:to)  &  data_c=yn(from:to)
        min=min(data_l,MAX=max)
        cut=papco_interpol(data_c, data_l, l_arr, NODATA = bad_data)

        index=where((l_arr lt min) OR (l_arr gt max),c)
        if c ne 0 then cut(index)=-99
        index=where(cut gt 1e20,c)  &  if c ne 0 then cut(index)=-99.0

        zarr(n_cut,*)=cut
        t_cut(n_cut, 0) = tn(from)
        t_cut(n_cut, 1) = tn(to)

        n_cut=n_cut+1

    ENDIF

ENDFOR

zarr=zarr(0:n_cut-1,*)
t_cut=t_cut(0:n_cut-1, *)

END

;******************************************************************************
;* FUNCTION:     papco_make_choice_names
;*
;* DESCRIPTION:  converts the "_info" tag information from a module's
;*               control structure to a list of choice names fo button widgets.
;*
;* KEYWORDS:     none
;*
;* INPUTS:       instr, _info string
;*
;* OUTPUT:       list of choice names
;*
;* HISTORY:      written June 2003, Reiner Friedel
;******************************************************************************
FUNCTION papco_make_choice_names, instr, TITLE = TITLE

IF keyword_set(TITLE) THEN BEGIN
    dummy = strsplit(instr, ';', /EXTRACT)
    IF n_elements(dummy) EQ 2 THEN BEGIN
        title = dummy(0)+':'
        choices = dummy(1)
    ENDIF
ENDIF ELSE choices = instr

dummy = strsplit(choices, ',', /EXTRACT)
outstr = dummy
FOR i = 0, n_elements(dummy)-1 DO BEGIN
    len = strlen(dummy(i))
    p = strpos(dummy(i), ':')
    outstr(i) = strmid(dummy(i), p+2, len-p-2)
ENDFOR

return, outstr

END


;******************************************************************************
PRO papco_set_module_defaults,  module, VERBOSE = VERBOSE

COMMON get_error, get_err_no, get_err_msg

r_module_config, module, config_data, msg

IF get_err_no NE 0 THEN r_module_default, module, config_data, msg

IF get_err_no EQ 0 THEN BEGIN
    FOR i = 0, n_elements(config_data)-1 DO $
      papco_setenv,/FS,config_data(i).env_var+'='+config_data(i).path
ENDIF

IF keyword_set(VERBOSE) THEN message, msg, /cont

END

;***************************************************************************************
; FUNCTION: papco_mean
;
; PURPOSE: extension of the traditional mean function to be able to
; send in multi-dimensional arrays and take mean over only a selected dimension
;
;
; CATEGORY: extended function
;
;
;
; CALLING SEQUENCE: res=papco_mean(inarr,dim,nodata=nodata)
;
;
;
; INPUTS:  
;                inarr - array of any dimension
;                dim - scalar or vector giving dimensions to take mean
;                      over (0 for 1st dimension). Must be
;                      monotonically increasing!
;              
;
; OPTIONAL INPUTS:
;
;
;
; KEYWORD PARAMETERS:
;                NODATA - a value indicating missing data in input
;                         array (inarr)
;
;
; OUTPUTS:
;                result - array of dimensions of inarr minus elements of dim
;
;
;
; SIDE EFFECTS:
;
;
; PROCEDURE:
;
; EXAMPLE:
;
;
; MODIFICATION HISTORY:
;               written by Arne Aasnes, April 2004
;               modifies, A. Aa, August, 2004 at LANL, fixed bug
;                            keeping from taking mean over several dimensions   
;************************************************************************************


function papco_mean,inarr,dim,NODATA=nodata

; first check if NODATA is given, if it is, convert to NaN which is
; treated by the IDL mean function



if keyword_set(NODATA) then begin 
    no_ind=where(inarr eq nodata,c)
    if c gt 0 then inarr(no_ind)=!values.f_nan
endif 


; if a specific (or array of) dimension is given for averaging:

if n_params() gt  1 then begin 
    insize=size(inarr,/dimension)
    indim=n_elements(insize)
;    temparr=inarr
;    tempsize=insize
;    tempdim=indim
    cc=0
    while cc lt n_elements(dim) do begin 
    
    if  dim(cc) eq 0 then begin

        if indim eq 1 then temparr=mean(inarr,/nan)
       
        if indim eq 2 then begin 
            temparr=make_array(insize(1))
            for j=0,insize(1)-1 do temparr(j)=mean(inarr(*,j),/nan)
        endif 
        if indim eq 3 then begin 
            temparr=make_array(insize(1),insize(2))
            for j=0,insize(1)-1 do for k=0,insize(2)-1 do temparr(j,k)=mean(inarr(*,j,k),/nan)
        endif
        if indim eq 4 then begin 
            temparr=make_array(insize(1),insize(2),insize(3))
            for j=0,insize(1)-1 do for k=0,insize(2)-1 do for l=0,insize(3)-1 do temparr(j,k,l)=mean(inarr(*,j,k,l),/nan)
        endif 
    endif 
    if dim(cc) eq 1 then begin 
        if indim eq 2 then begin 
            temparr=make_array(insize(0))
            for j=0,insize(0)-1 do temparr(j)=mean(inarr(j,*),/nan)
        endif 
        if indim eq 3 then begin 
            temparr=make_array(insize(0),insize(2))
            for j=0,insize(0)-1 do for k=0,insize(2)-1 do temparr(j,k)=mean(inarr(j,*,k),/nan)
        endif
        if indim eq 4 then begin 
            temparr=make_array(insize(0),insize(2),insize(3))
            for j=0,insize(0)-1 do for k=0,insize(2)-1 do for l=0,insize(3)-1 do temparr(j,k,l)=mean(inarr(j,*,k,l),/nan)
        endif 
    endif 
    if dim(cc) eq 2 then begin 
        if indim eq 3 then begin 
            temparr=make_array(insize(0),insize(1))
            for j=0,insize(0)-1 do for k=0,insize(1)-1 do temparr(j,k)=mean(inarr(j,k,*),/nan)
        endif 
        if indim eq 4 then begin 
            temparr=make_array(insize(0),insize(1),insize(3))
            for j=0,insize(0)-1 do for k=0,insize(1)-1 do for l=0,insize(3)-1 do temparr(j,k,l)=mean(inarr(j,k,*,l),/nan)
        endif 
    endif 
    if dim(cc) eq 3 then begin 
        if indim eq 4 then begin
            temparr=make_array(insize(0),insize(1),insize(3))
            for j=0,insize(0)-1 do for k=0,insize(1)-1 do for l=0,insize(2)-1 do temparr(j,k,l)=mean(inarr(j,k,l,*),/nan)
        endif 
    endif
    insize=insize(where(findgen(indim) ne dim(cc)))
    indim=indim-1
    cc=cc+1
    inarr=temparr
    dim = dim-1 ; as we took away one dimension, reduce the index of next dimension
endwhile
outarr=inarr 
 ;   for i=0,n_elements(dim)-1 do begin
;        cc=0
;        while cc lt tempdim do begin
            
;            temparr=mean(temparr(ind),/nan)
            
;            cc=cc+1
;        endwhile
;        tempdim=tempdim-1
;    endfor 

endif else outarr=mean(inarr,/nan)

return,outarr


end


;******************************************************************************
;* PROCEDURE:
;*      papco_valid_average, x, iny,outy, outystruc,nodata,PASS_AVE=PASS_AVE
;*
;* DESCRIPTION:
;*      This procedure outputs average of valid (non-nodata)
;*      across time period where data is valid
;*
;* INPUTS:
;*      x,iny     arrays to be plotted
;*      nodata  bad data flag value
;*
;* KEYWORDS:
;*      PASS_AVE, array of data of same dimensions as iny where 1's
;*        represent period of 'x' in which average must be calculated. This was
;*        made for time where one requires the average value during a satellite
;*        pass thru and L range, but data coverage was not full (ie data gaps
;*        requested so and average during hte time period requested ignoring
;*        datagaps is needed)
;*
;*
;* OUTPUTS:
;*      outy     averaged data values
;*      outystruc, structure containing average, skew etc from idl
;*      moment function , along with std_dev and min max values
;*      of averaged array
;* CALLING SEQUENCE:
;*      papco_valid_average, x, iny,outy, outystruc,nodata,PASS_AVE=PASS_AVE
;*
;* MODIFICATION HISTORY:
;*     written August 2003 by MATT 'I pity da fool' TAYLOR
;******************************************************************************
PRO papco_valid_average,x,iny,outy,outystruc,nodata,PASS_AVE=PASS_AVE

if keyword_set(PASS_AVE) then idx_pass=PASS_AVE

xray=x  &  yray=iny &  count = 0



IF keyword_set(PASS_AVE) then begin

    index=where(yray ne nodata,c)
    not_yet=0
      IF c EQ 0 THEN BEGIN
         message,'No valid points to average',/cont
         return
       ENDIF ELSE BEGIN
         i=0  &  ndat=n_elements(yray)
       WHILE i NE ndat-1 DO BEGIN
         Y_MOMENTS={ymoment:dblarr(4),STDDEV:dblarr(1),MIN:0.0,MAX:0.0}
         ;get to beginning of pass
         while (idx_pass(i) eq 0) AND (i ne ndat-1) do i=(i+1) < (ndat-1)
         sidx=i
        ;get to end of pass
         while (idx_pass(i) eq 1) AND (i ne ndat-1) do i=(i+1) < (ndat-1)
         eidx=i-1
         ; if more than two points do
          if (eidx - sidx) lt 2 then CONTINUE
               ;set to automatically average data
               ;across valid pass
               ;then write average value to all valid
               ;data points in that range
;               first index out nodata in pass
pass_data=yray(sidx:eidx)
idx_good_data=where(pass_data ne nodata,c)
        If C eq 0 then BEGIN
                 message, 'No Valid data for PASS_AVE period, writing out null values',/cont
                RESULT=nodata
                MINI=nodata
                MAXI=nodata
                STDDEV=nodata
                Y_MOMENTS.ymoment=RESULT
                Y_MOMENTS.STDDEV=STDDEV
                Y_MOMENTS.MIN=MINI
                Y_MOMENTS.MAX=MAXI
        endif else begin

                RESULT=moment(pass_data(idx_good_data),SDEV=STDDEV)
                MINI=max(pass_data(idx_good_data))
                MAXI=min(pass_data(idx_good_data))
                Y_MOMENTS.ymoment=RESULT
                Y_MOMENTS.STDDEV=STDDEV
                Y_MOMENTS.MIN=MINI
                Y_MOMENTS.MAX=MAXI
        ENDELSE
         if not_yet eq 0 then begin   ;start the array buffer
               outystruc=temporary(Y_MOMENTS)
           endif else begin                   ;otherwise concatonate arrays
              outystruc=[(outystruc),(Y_MOMENTS)]
           endelse
           yray(sidx:eidx)=RESULT(0)
            not_yet=not_yet+1
            Y_MOMENTS=0

           ENDWHILE
        outy=yray
    ENDELSE




    ENDIF ELSE BEGIN

    ; split up plot array into chunks of valid data only, and plot those.

    index=where(yray ne nodata,c)
    not_yet=0
      IF c EQ 0 THEN BEGIN
         message,'No valid points to average',/cont
         return
       ENDIF ELSE BEGIN
         i=0  &  ndat=n_elements(yray)
       WHILE i NE ndat-1 DO BEGIN
         Y_MOMENTS={ymoment:dblarr(4),STDDEV:dblarr(1),MIN:0.0,MAX:0.0}
         ;get to beginning of good data
         while (yray(i) eq nodata) AND (i ne ndat-1) do i=(i+1) < (ndat-1)
         sidx=i
        ;get to end of good data
         while (yray(i) ne nodata) AND (i ne ndat-1) do i=(i+1) < (ndat-1)
         eidx=i-1
         ; if more than two points do
          if (eidx - sidx) lt 2 then CONTINUE
               ;set to automatically average data
               ;across valid data range
               ;then write average value to all valid
               ;data points in that range
                RESULT=moment(yray(sidx:eidx),SDEV=STDDEV)
                MINI=max(yray(sidx:eidx))
                MAXI=min(yray(sidx:eidx))
                Y_MOMENTS.ymoment=RESULT
                Y_MOMENTS.STDDEV=STDDEV
                Y_MOMENTS.MIN=MINI
                Y_MOMENTS.MAX=MAXI
           if not_yet eq 0 then begin   ;start the array buffer
               outystruc=temporary(Y_MOMENTS)
           endif else begin                   ;otherwise concatonate arrays
              outystruc=[(outystruc),(Y_MOMENTS)]
           endelse
           yray(sidx:eidx)=RESULT(0)
            not_yet=not_yet+1
            Y_MOMENTS=0
       ENDWHILE
      ENDELSE
   outy=yray
   ENDELSE


END

;******************************************************************************
;* FUNCTION:     papco_2D_interplol
;*
;* DESCRIPTION:  designed for color spec plots to make nicer looking
;*               spectra plots. Takes low resolution zmat (few energy channels)
;*               and expands this to more channels to get smooth,continuous 
;*               spectra. 
;*               Further options aloow smoothing / filling in along
;*               time too. All is done preserving nodata flags! 
;*
;* KEYWORDS:     NODATA  - bad data flag, default is -99.0
;*               YCH_NUM - new number of y-channels: 0 does nothing
;*
;* INPUTS:       zmat - spec array zmat(time, ch)
;*               time - time array, start/stop time(time, 2)
;*               yarr - channel assignments yarr(ch,2)
;*
;* OUTPUT:       updated interpolated zmat, yarr, time
;*
;* HISTORY:      written Febriary 2004, Reiner Friedel
;******************************************************************************
PRO papco_2D_interplol, zmat, yarr, time, $
                    VALID_YCH = VALID_YCH, YCH_NUM = YCH_NUM, NODATA = NODATA

IF keyword_set(VALID_YCH) THEN valid_ych = VALID_YCH ELSE BEGIN
    valid_ych = intarr(n_elements(yarr)/2) & valid_ych(*) = 1
ENDELSE 

IF keyword_set(YCH_NUM) THEN ych_num = YCH_NUM ELSE ych_num = 0
IF ych_num EQ 0 THEN return

r = size(zmat, /struct)
ndat = r.DIMENSIONS(0)

;use only valid y-channel range
idx = where(valid_ych EQ 1, c)
IF c GT 2 THEN BEGIN
    z = zmat(*, idx)
    y = yarr(idx, *)
    t = time
ENDIF 

;make "new" yarr channel asignments.
y_log_mid = (alog10(y(*, 0))+alog10(y(*, 1))) / 2.0
new_y_log_mid = congrid(y_log_mid, ych_num, /INTERP, /MINUS_ONE)
new_y_mid = 10^new_y_log_mid
new_y_diff = new_y_mid(1:ych_num-1)-new_y_mid(0:ych_num-2)
ny = fltarr(ych_num, 2)
ny(1:ych_num-1, 0) =  new_y_mid(1:ych_num-1)-new_y_diff/2.0
ny(0:ych_num-2, 1) =  new_y_mid(0:ych_num-2)+new_y_diff/2.0
ny(0, 0) = y(0, 0)
ny(ych_num-1, 1) = y(n_elements(y)/2-1, 1)

;make new zmat
message, 'interpolating spectra', /info
nz = fltarr(ndat, ych_num)
FOR i = 0, ndat-1 DO BEGIN
    spec = z(i, *)
    new_spec = papco_interpol(spec, y_log_mid, new_y_log_mid, NODATA = NODATA)
    nz(i, *) = new_spec
ENDFOR

zmat = nz
yarr = ny
time = t

END 


;******************************************************************************
;* r = v(perp)/w = mv(perp) /(|q|B)
;* E - particle energy, in keV
;* B - mag field, in nT
;* alpha - particle pitch angle
;* species - 0 elec, 1 ion, 2 oxygen
;******************************************************************************
FUNCTION papco_gyroradius, E, B, alpha, species

kev = 1000.0*1.6e-19  ;Joule
mass = 9.11e-31   ;kg
energy = e*kev
magB = B*1e-9 ;Tesla
q = 1.6e-19 ;coulomb

;find velocity - non-relativistically
v = sqrt(2*energy/mass) ;m/s

;print, v/1000.0, '   km/s'

;perpendicular velocity
v_perp = v * sin(alpha/!radeg)

r = (mass*v_perp) / (q*magB)


return, r/1000.0 ;km


END 

;******************************************************************************

