;***************************************************************************
;* PROCEDURE:
;*      PRO papco_draw_slice, panelVector, currPlotInfo, orbitNo $
;*                          get_data_call, slice_type, IsRead
;* DESCRIPTION:
;*	This procedure performs the actual data read call and the plot call
;*      for the slice thumb panel.
;*
;*
;* INPUTS:
;*	panelVector	position of the plot (see papco_conventions.text, 6)
;*	currPlotInfo	structure containing all data for this plot.
;*                      (structure type: PAPCO_PLOTINFO,
;*                                             see papco_variables.text)
;*
;* OUTPUTS:
;*   	get_data_call	a string containing the data call. Used by other
;*                      routines that need to know how to get the data
;*                      in a panel, e.g. to draw spectra, distrib. functions
;*      IsRead		a boolean which is set when the the data is read the
;*                      first time. Useful if a given data read puts data
;*                      into commons which can be plotted in many ways.
;*                      When set, data does not have to be read again.
;*      slice_type      Selects which slices are supported and for which
;*                      slice routines have been written.
;*                      This is a byte logical with 8 swittches. Currently
;*                      Three are supported (this may be extended later)
;*                       1 = mouse zooms (default, always possible)
;*                       2 = mouse slices
;*                       4 = mouse calls level zero
;*                       8, 16, 32, 64, 128 = not supported yet
;*                      you can support more than one type of slice
;*                      by arythmetically adding the options, eg
;*                      slice_type=5  support zoom and level zero
;*
;* KEYWORDS:
;* 	none
;*
;* CALLING SEQUENCE:
;*	papco_draw_slice, panelVector, currPlotInfo, orbitNo, $
;*                               get_data_call, slice_type, IsRead
;* MODIFICATION HISTORY:
;*     written November 2000, Reiner Friedel
;***************************************************************************
pro papco_draw_slice, panelVector, currPlotInfo, orbitNo, $
                      get_data_call, slice_type, IsRead, MODULE=MODULE

COMMON plot_composer, widgetData
COMMON time, xut1, xut2
COMMON get_error, get_err_no, get_err_msg
COMMON coordinateSystems, plotted_x, plotted_y   
COMMON extra, extra_PlotInfo  ; passes in papco-set keywords structure

; A slice thumb plot needs to be associated with a panel from a module
; that supports the slice type that has been chosen. It is that
; panel's readroutine and plotinfo that needs to be known.

; Depending on the direction on time traces (up or down) we look for
; the CLOSEST panel that supports the same slice type. 
 
nPlots=widgetData.numberOfPlots
all_plot_pos = widgetData.plotinfos(0:nPlots-1).PANELPOSITION

idx = where(all_plot_pos EQ panelVector(0))
curr_panel_pos=idx(0)
;a=reverse(indgen(nPlots))
;curr_panel_pos = a(curr_panel_pos)

curr_panel_name=currPlotInfo.panelkind
search_panel_pos=curr_panel_pos
panel_names=widgetData.plotInfos(0:nPlots-1).panelkind
search = bytarr(nPlots)

;get the stored allowed slice types for all drawn panels.
slice_set = widgetData.plotsDrawn(0:nPlots-1).slice_type
idx = where(slice_set LT 0, c) & IF c NE 0 THEN slice_set(idx) = 0
slice_req = currPlotInfo.soptions(0)
slice_req_val = papco_slice_value(slice_req)

;select search direction in panels - up or down
IF currPlotInfo.typevector(1) EQ 1 THEN dir='above' ELSE dir='below'

FOR i = 0, nPlots-1 DO $
  IF (slice_req_val and slice_set(i)) EQ slice_req_val THEN search(i) = 1 

idx = where(search EQ 1, c) & c1 = 0 & c2 = 0
IF c NE 0 THEN BEGIN
    IF dir EQ 'above' THEN BEGIN
        idx2 = where(idx LT curr_panel_pos, c1)
        IF c1 NE 0 THEN search_panel_pos = idx(idx2(0))
    ENDIF 
    IF dir EQ 'below' THEN BEGIN
        idx3 = where(idx GT curr_panel_pos, c2)
        IF c2 NE 0 THEN search_panel_pos = idx(idx3(0))
    ENDIF 
ENDIF ELSE BEGIN
    msg = 'Found no panels that support requested slice'
    message, msg, /cont
    GOTO, get_out
ENDELSE     

IF c1+c2 EQ 0 THEN BEGIN
    msg ='Could not find a panel '+dir+ ' that supports requested slice'
    message, msg, /cont
    GOTO, get_out    
ENDIF
     
search_panel_name = widgetData.plotsDrawn(search_panel_pos).panelkind

message, 'Associating "'+curr_panel_name+'" with "'+search_panel_name+ $
    '", in panel '+ varprt(search_panel_pos), /cont

;set the slice plot call to use for thumb plots
slice_call=currPlotInfo.soptions(1)
panelNr=search_panel_pos

;read the slice data for the associated panel, check for error
papco_get_slice_data, panelNr
IF get_err_no NE 0 THEN BEGIN
    IF papco_Draw_CheckResult(currPlotInfo.panelKind, panelVector, orbitNr, $
                            currPlotInfo.typeVector, currPlotInfo.channel) $
    THEN IsRead=1 ELSE IsRead=0      
    return
ENDIF 

get_data_call=""             ; Use this if necessary
   
; set up which mouse functions to allow - only zoom, this plots slice thumbs!
slice_type=1
  
papco_p_slice, panelVector, panelNr, currPlotInfo, _EXTRA=extra_PlotInfo

return
get_out:
get_err_msg=msg  &  get_err_no=1
IF papco_Draw_CheckResult(currPlotInfo.panelKind, panelVector, orbitNr, $
                            currPlotInfo.typeVector, currPlotInfo.channel) $
THEN IsRead=1 ELSE IsRead=0  
return

END

;***************************************************************************
pro papco_p_slice, panel, panelNr, plotinfo, $
                   OVERPLOT=OVERPLOT, $
                   PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION, $
                   SUBTABLE=SUBTABLE, _EXTRA=extra_par

COMMON mjdt, mjdt_start, mjdt_end    ;common time limit in mjdt
COMMON PLOT_COMPOSER, widgetData
COMMON coordinateSystems, plotted_x, plotted_y
COMMON papco_event_data, input_header, input_data

; no data commons here - they get passed into the slice plot routine
; to be called here
 
panelset, panel              ;set the panel viewport
papco_set_subtable, subtable ;set the color space
 
traceup = plotinfo.typevector(1) 
nrow    = plotinfo.ioptions(0)
fln     =   plotinfo.filename

; get the list of times for each slice to be plotted. Correct times to start
; from zero as start time - "normalized". That should be the time
; calling convention for slices!

tai1 = utc2tai({mjd:mjdt_start.mjd, time:mjdt_start.t*1000})
tai2 = utc2tai({mjd:mjdt_end.mjd, time:mjdt_end.t*1000}) 
xut1=0  &  xut2=tai2-tai1  

CASE 1 of
    ((Plotinfo.filename NE '') AND (Plotinfo.typevector(2) EQ 0)): BEGIN 
        ;get times from file
        r_papco_event_original, fln, VERBOSE = 1
        time = input_data.T1_TAI - tai1
    END
    ((Plotinfo.typevector(2) EQ 1) AND (Plotinfo.typevector(3) EQ 0)): BEGIN
        ;number of thumbs set
        papco_slice_time_list, times_arr, TNUM = Plotinfo.ioptions(1), $
          /MJDT, /UTAI
        IF n_elements(times_arr) EQ 0 THEN GOTO, no_plot
        time = times_arr - tai1
    END
    ((Plotinfo.typevector(2) EQ 1) AND (Plotinfo.typevector(3) EQ 1)): BEGIN
        ;time interval set
        dl = Plotinfo.ioptions(2)/86400.0d
        papco_slice_time_list, times_arr, DELTA = dl, /MJDT, /UTAI
        IF n_elements(times_arr) EQ 0 THEN GOTO, no_plot
        time = times_arr - tai1
    END
ENDCASE

;set the slice plot call to use for thumb plots
slice_call=plotinfo.soptions(1)

;set plot keywords
extra_plotPar={xrange:[xut1,xut2], xtickformat:'noticks', ystyle:5}

; add keyword structure set locally and the one passed in
extra_plotPar=create_struct(extra_plotPar, extra_par)
 
; use papco routine to draw time axis. This checks for bottom plot and uses the
; user's xtickformat if it is something other than 'noticks'.
; If you use this rotuine, make sure your data plot routine uses xstyle=5 and 
; ystyle=5 (inhibit axis) so that you don't clobber those drawn by papco.
; "down" sets the axis labeling level. 
down=0
papco_draw_time_axis, panel, OVERPLOT=OVERPLOT, _extra=extra_plotPar

pold= !p  &  aspect=1.2
papco_thumb_plot, time, SLICE_CALL=slice_call, PANEL_NR=panelnr, xstyle=5, $
    THIS_PANEL=panel, NUTITLE=plotinfo.soptions(0), $
    _extra=extra_plotPar, traceup=traceup, $
    delta= delta_time, aspect=aspect, rows=nrow
!p=pold
 
; store the coordinate information into common block
plotted_x = !x  &  plotted_y = !y
   
return

no_plot:
message, 'no thumbs for timechoice', /cont
plotted_x = !x  &  plotted_y = !y

END 
