;******************************************************************************
;* FILE:
;*    $PAPCO_PATH/papco_slice_panel.pro
;* 
;*    This file belongs to the PAPCO-project. See $PAPCO_PATH/papco.pro for
;*    further information.
;*
;* DESCRIPTION: 
;*    This file contains the default PAPCO panel editor for the slice
;*    type plot for any module.
;*    
;*    Any slices listed in the module's plot_type_slice.pro header
;*    will be automatically included in this panel.
;*
;* COMMON-BLOCKS:
;*    papco_slice_panel_data	contains the window-information
;*
;* STRUCTURES:
;*    This module creates one concrete structure, that inherits from the 
;*    PAPCO_PANEL_EDITOR-structure (see $PAPCO_PATH/papco_panel.pro for a
;*    description). 
;*       {papco_slice_panel_DATA, 
;*	     ; imagine the PAPCO_PANEL_EDITOR-structure was inserted...
;*	      
;*    For a more complete description see $PAPCO_PATH/papco_variables.text
;*
;* OVERVIEW:
;*    This function creates the editor-window:
;*       FUNCTION  papco_slice_panel, plotinfo, GROUP=group, ACTION=ACTION
;*    You might try this:
;*	 print, papco_slice_panel(PAPCO_getPlotInfoStruct())
;*       which calls up the window.
;*
;* MODIFICATION HISTORY:       
;*     November 2000, written by Reiner Friedel
;******************************************************************************

;******************************************************************************
;* FUNCTION:     
;*      FUNCTION  papco_slice_get_slices
;*
;* DESCRIPTION:  
;*                   reads available slices for a module
;*
;* INPUTS:       
;*	module       name of the module for which the slices are to be found.
;*     
;* KEYWORDS:
;* 	none
;*     
;* OUTPUT:           list of available slices from header of module_slice.pro
;*      
;* MODIFICATION HISTORY:       
;*     November 2000, written by Reiner Friedel
;******************************************************************************
FUNCTION papco_slice_get_slices, module
  
paths=str_sep(!PATH,papco_get_dirsep())
result=strpos(paths,module+'/papco_interface')
index=where(result ne -1, c)
if c eq 0 then begin
    result=strpos(paths,module+'_dev/papco_interface')
    index=where(result ne -1, c)
endif
path=paths(index(c-1))
  
dummy={PAPCO_SLICE, slice_name:'', slice_call:''}
  
IF c NE 0 THEN BEGIN 
    print,''
    message,'Useable slices:',/cont
    print,''
    in_str='' & n=0
    openr,unit,path+'/'+module+'_slice.pro', /get_lun
    readf,unit,in_str  
    readf,unit,in_str 
    readf,unit,in_str  & print,in_str

    WHILE strmid(in_str,0,6) NE ';* end' DO BEGIN 
        this_slice=strtrim(strmid(in_str,2,strlen(in_str)),2)
        dummy.slice_call=this_slice
        this_slice=strmid(this_slice,0,strlen(this_slice)-strlen(module)-1)
        this_slice=strsubst(this_slice,'_',' ')
        CASE strlowcase(this_slice) OF 
            'drawslice'       : this_slice='Vertical Slice'
            'draw_horiz_slice': this_slice='Horizontal Slice'
            ELSE: this_slice=this_slice
        endcase  
      
        dummy.slice_name=this_slice
        IF n EQ 0 THEN slice_names=dummy $
          ELSE slice_names=[slice_names,dummy]
        n=n+1 & readf,unit,in_str  & print,in_str
    ENDWHILE  
    readf,unit,in_str  & print,in_str
    print,''    
    IF n_elements(slice_names) EQ 0 THEN slice_names=dummy 
ENDIF 

close,unit  &  free_lun, unit
  
return, slice_names

END   


;******************************************************************************
;* FUNCTION:     
;*      FUNCTION  papco_slice_panel, plotinfo, GROUP=group, ACTION=ACTION
;* 
;* DESCRIPTION:  
;*	Open an editor for a slice panelplot.
;*	Its window is constructed using the methods defined in the file
;*	$PAPCO_PATH/papco_panel.pro
;*	When the user presses the 'Done'- or the 'Cancel'-Button, the values 
;*	changed by the event-methods of papco_panel.pro are returned.
;*	The returned value is a structure containing those values and a flag,
;*	'canceled', which the caller may read out to check, whether the 
;*	window was closed using the 'Cancel' or the 'Done'-Button.
;*
;* INPUTS:       
;*	plotinfo	a structure of the PAPCO_PLOTINFO-type
;*			this structure is edited in the window.
;*     
;* KEYWORDS:
;* 	GROUP		a WIDGET_BASE
;*			when specified, this is used as GROUP_LEADER
;*	ACTION		a string
;*			when specified, the 'Done'-Button is labeled with it
;*		        if not specified, the 'Done'-Button is labeled 'Done'
;*      MODULE          the name of the module that called this routine.
;*     
;* OUTPUT:	 
;*      a default-structure of type PAPCO_PANEL_EDITOR_RETURN. Though this
;*	type is nowhere defined, I like to think of it as virtually existing
;*	data-type. 
;*	All the functions in $PAPCO_PATH/papco.pro that edit plots or add new
;*	ones, expect the opened editor-windows to return a structure containing
;*	at last some fields. Those fields might make up a structure, but as no
;*	program defines the structure-type PAPCO_PANEL_EDITOR_RETURN, this 
;*	poor structure is not alive but only virtual.
;*	The return-values of plot-editors have to expand this structure - I 
;*	like to think, that they inherit from that structure - and contain at
;*	least those fields:
;*   		{$
;*			canceled:1, $
;*	    		panelHeight:1, $
;*	    		channel:0, $
;*	    		typeVector:[0,0,0,0], $ 
;*	    		manualYScaling:0 $
;*	    		yScaleMin:LONG(1), $
;*	    		yScaleMax:LONG(1), $
;*	    		panelKind:'some string'}
;*
;*      Additionally, some Panel-editors may return these fields in
;*      structure:
;*               {$
;*		        color:1, $
;*			plotStyle:1}
;*			
;*      For an enhanced description read $PAPCO_PATH/papco_variables.text
;*
;* CALLING SEQUENCE:
;*	plotinfoStruct=PAPCO_getPlotInfoStruct()
;***************************************************************************
FUNCTION papco_slice_panel, plotinfo, GROUP=GROUP, ACTION=ACTION, $
                                      MODULE = MODULE

common papco_slice_panel_data, widgetData 
  
forward_function g_papco_slice_panel_names
  
;--- if GROUP is not specified, the group-leader will be the desktop
IF NOT KEYWORD_SET(GROUP) THEN GROUP=0

;--- set the label for the action-button
IF NOT KEYWORD_SET(ACTION) THEN ACTION='Done' 

;--- the routines used by papco_panel.pro define tags for a structure. Those
;    tags will be stored in the structure 'additionalWData'. When you realize 
;    the window, you have to add your own window-specific data to that  
;    structure and afterwards assign it to widgetData.
additionalWData={canceled:1}
  
; --- Get list of available slices for the calling module ---------------------
slice_info=papco_slice_get_slices(module)

;--- create the base-window ---------------------------------------------------
panel_base=papco_WIDGET_BASE(title=strsubst(plotinfo.panelkind,'_',' '), $
                             column=1, GROUP_LEADER=group, /MODAL)

;--- the first part of the window will consist in the widgets -----------------
;    that control the height of that plot
widgets=PAPCO_panel_AddHeightWidgets(panel_base, plotinfo)
additionalWData=create_struct(widgets, additionalWData)
  
if slice_info(0).slice_name eq '' then begin
    dummy=messageBox(['Sorry...', '', 'There are no useable slice types', $
                      'defined for module '+module, $
                     '', $
                     'useable slices need to be defined in', $
                     'the header of '+module+'_slice.pro'],$
                     ['O.K.'], TITLE='PAPCO Slice Info')   
    return, additionalWData  
endif  
  
; --- Select slice type for thump plot ----------------------------------------
slice_choice_base=widget_base(panel_base, ROW = 2, /ALIGN_CENTER, /FRAME)
slice_names=slice_info.slice_name
slice_base=CW_BGROUP(slice_choice_base, Slice_names, /EXCLUSIVE, $
                     LABEL_TOP='Available slices:', $
                     UVALUE='mpb_slice', $
                     SET_VALUE=plotinfo.typevector(0)) 
plotinfo.soptions(0)=slice_info(plotinfo.typevector(0)).slice_name
plotinfo.soptions(1)=slice_info(plotinfo.typevector(0)).slice_call
 
thumb_choice_base=widget_base(panel_base, COLUMN = 1, /ALIGN_CENTER, /FRAME)

appear_base = widget_base(thumb_choice_base, ROW = 1, /ALIGN_CENTER, /FRAME)
; --- Choose up or down trace direction type-----------------------------------
traceNames=[' Down Traces ',' Up Traces '] 
trace_base=CW_BGROUP(appear_base, traceNames, /EXCLUSIVE, ROW=1, $
                     LABEL_TOP='Direction of Time markers', $
                     UVALUE='mpb_trace', $
                     SET_VALUE=plotinfo.typevector(1) < 1)  
;--- Number of rows specification ---------------------------------------------
rows_base=CW_FIELD(appear_base, TITLE="Number of Thumb Rows (0=auto): ", $ 
                   /int, value=plotinfo.ioptions(0), /COLUMN, $     
                   UVALUE='rows_base', /return_ev )  


;--- Set some options for how to choose time for thumb plots. Can 
;    specify times from a file, or some interval to subdivide time
;    range of plot. 
time_base = widget_base(thumb_choice_base, ROW = 1, /FRAME, /ALIGN_CENTER)
timeNames = ['Times From File', 'Auto Time'] 
tc_base = CW_BGROUP(time_base, timeNames, /EXCLUSIVE, ROW=1, $
                    LABEL_TOP="Thumb Time Choice:", $ 
                    UVALUE='mpb_tc', $  
                    SET_VALUE=plotinfo.typevector(2) < 1) 
autoNames = ['Set # of Thumbs', 'Set interval']   
auto_base = CW_BGROUP(time_base, autoNames, /EXCLUSIVE, ROW=1, $
                     LABEL_TOP="Auto Time on plot time range:", $  
                     UVALUE='mpb_auto', $ 
                     SET_VALUE=plotinfo.typevector(3) < 1)


t_entry_base = widget_base(thumb_choice_base, ROW = 2, /FRAME, /ALIGN_CENTER)
ef_num = CW_FIELD(t_entry_base, TITLE="# of Thumbs:", $   
                  /int, value=plotinfo.ioptions(1) > 1, /COLUMN, $    
                  UVALUE='ef_num', /return_ev )
ef_int = CW_FIELD(t_entry_base, TITLE="Interval (sec):", $   
                  /int, value=plotinfo.ioptions(2) > 1, /COLUMN, $    
                  UVALUE='ef_int', /return_ev )
ef_file = CW_FIELD(t_entry_base, TITLE="Times filename:", $   
                   /string, value=plotinfo.filename, /COLUMN, XSIZE = 75, $
                   UVALUE='ef_file', /return_ev )

;--- finally, the Actionbuttons (help, cancel, done) are added to the ---------
;    editor window.
widgets=PAPCO_panel_AddActionButtons(panel_base, Action, plotinfo, /CONFIG) 
additionalWData=create_struct(widgets, additionalWData)

;--- now, create the structure that keeps the panel-window data. You
;    always need to specify the help-file, and your structure must
;    always contain the 'additionalWData' 
thisWidgetsData={helpfile:'papco_slice_panel.help', $  
                 filename:plotinfo.filename, $
                 typevector:plotinfo.typeVector, $
                 ioptions:plotinfo.ioptions, $ 
                 soptions:plotinfo.soptions, $
                 auto_base:auto_base, $
                 ef_file:ef_file, $
                 ef_num:ef_num, $
                 ef_int:ef_int, $
                 rows_base:rows_base, $ 
                 slice_info:slice_info} 
widgetData=PAPCO_PANEL_CreateWidgetData(thisWidgetsData, additionalWData)
 
;--- realize the window
papco_Panel_Realize, panel_base, widgetData  
papco_slice_panel_refresh, /ALL
papco_xmanager, 'papco_slice_panel', panel_base

;--- the returned data consists of the standard-data inherited from
;    papco_panel.pro and your new data. Stick to the tags provided in
;    plotinfo, there are ample variables for the user
returnData={typeVector:widgetData.typeVector, $ 
            ioptions:widgetData.ioptions, $ 
            soptions:widgetData.soptions}  
  
additionalRData=PAPCO_panel_ReturnData(widgetData)
return, create_struct(ReturnData, additionalRData)
   
END


;******************************************************************************
PRO papco_slice_panel_refresh, ALL = ALL,  TIMECHOICE = TIMECHOICE
                               

COMMON papco_slice_panel_data, widgetData

IF keyword_set(ALL) THEN BEGIN
    TIMECHOICE = 1
ENDIF

IF keyword_set(TIMECHOICE) THEN BEGIN
    bases = [widgetData.auto_base, $
             widgetData.ef_file, widgetData.ef_num, widgetData.ef_int]
    CASE widgetData.typevector(2) OF
        0: senst = [0, 1, 0, 0]
        1: BEGIN
            CASE widgetData.typevector(3) OF
                0: senst = [1, 0, 1, 0]
                1: senst = [1, 0, 0, 1]
            ENDCASE
        END
    ENDCASE     
    FOR i = 0, n_elements(bases)-1 DO $
      WIDGET_CONTROL, bases(i), SENSITIVE =  senst(i)
ENDIF

END


;******************************************************************************
;* PROCEDURE:     
;*      PRO papco_slice_panel_event, event
;* 
;* DESCRIPTION:  
;*	This processes events from the XMANAGER.
;*	As this example does not extend the basic functionality of plot-
;*	editors, it directly calls the default-handler, PAPCO_PANEL_Event
;*
;* INPUTS:       
;*	event		an XManager event
;*			
;* KEYWORDS:
;* 	none
;* 
;* CALLING SEQUENCE:
;*	called by XMANAGER
;******************************************************************************
PRO papco_slice_panel_event, event

COMMON papco_slice_panel_data, widgetData

WIDGET_CONTROL, GET_UVALUE=uval, event.id
     
CASE 1 OF
    uval EQ 'mpb_slice': BEGIN 
        widgetData.typevector(0) = event.value
        widgetData.soptions(0) = $
                    widgetData.slice_info(widgetData.typevector(0)).slice_name
        widgetData.soptions(1) = $
        widgetData.slice_info(widgetData.typevector(0)).slice_call
    END 
    uval EQ 'mpb_trace': BEGIN 
        widgetData.typevector(1) = event.value
    END  
    uval EQ 'rows_base': BEGIN 
        widget_control, event.id, get_value=nrow
        widgetData.ioptions(0)= nrow  
    END   
    uval EQ 'mpb_tc': BEGIN 
        widgetData.typevector(2) = event.value
        papco_slice_panel_refresh, /TIMECHOICE
    END
    uval EQ 'mpb_auto': BEGIN 
        widgetData.typevector(3) = event.value
        papco_slice_panel_refresh, /TIMECHOICE
    END    
    uval EQ 'ef_file': BEGIN
        widgetData.filename = event.value
        keep_widgetData=widgetData
        papco_edit_slice_thumb, widgetData, /MJDT
        keep_widgetData.filename=widgetData.filename
        widgetData = keep_widgetData
        WIDGET_CONTROL, widgetData.ef_file, SET_VALUE = widgetData.filename
    END
        
    uval EQ 'ef_num': widgetData.ioptions(1) = event.value
    uval EQ 'ef_int': widgetData.ioptions(2) = event.value 
    else: begin
      widget_control,  widgetData.rows_base,  get_value = nrow
      widgetData.ioptions(0) = nrow
      widget_control,  widgetData.ef_file, get_value =  fln
      widgetData.filename = fln
      widget_control,  widgetData.ef_num, get_value = num
      widgetData.ioptions(1) = num
      widget_control,  widgetData.ef_int, get_value = sec
      widgetData.ioptions(2) = sec
      widgetData=papco_panel_Event(event, widgetData)
    end  
ENDCASE 
 
END


;******************************************************************************
;* PROCEDURE:     
;*      PRO papco_edit_slice_thumb_event_event, event
;* 
;* DESCRIPTION:  
;*	This processes events from the XMANAGER.
;*	As this example does not extend the basic functionality of plot-
;*	editors, it directly calls the default-handler, papco_PANEL_Event
;*
;* INPUTS:       
;*	event		an XManager event
;*			
;* KEYWORDS:
;* 	none
;* 
;* CALLING SEQUENCE:
;*	called by XMANAGER
;*
;* MODIFICATION HISTORY:       
;*      June 1998, written by Reiner Friedel
;******************************************************************************
PRO papco_edit_slice_thumb_event, event

COMMON edit_slice_thumb_data, editwidgetData

WIDGET_CONTROL,GET_UVALUE=uval, event.id
  
;test for any of the time entries to be correct format.
if strmid(uval,0,11) eq 'ef_series_t' then begin
    this_mjdt=CONVERT_timeToSeconds(event.value(0), /T90, /MJDT)
    if this_mjdt.err ne 0 then begin
      dummy=messageBox(this_mjdt.errtext, ['Ok'], title='Wrong time format')
      return
    endif
    result=convert_secondsToTime(this_mjdt.value, /T90, /MJDT)
ENDIF
  
case uval of 
    'ef_series_t1': begin
      editwidgetData.soptions(2)=result(0)
      widget_control, editwidgetData.ef_series_t1, SET_VALUE=result(0)
      papco_edit_slice_thumb_refresh, /LIST
    end
    'ef_series_t2': begin
      editwidgetData.soptions(3)=result(0)
      widget_control, editwidgetData.ef_series_t2, SET_VALUE=result(0)
      papco_edit_slice_thumb_refresh, /LIST
    end
    'ef_series_dl': begin
      editwidgetData.soptions(4)=event.value(0)
      papco_edit_slice_thumb_refresh,/DELTA, /LIST
    end
    'ef_series_o1': begin
      editwidgetData.ioptions(3)=event.value(0)
      papco_edit_slice_thumb_refresh, /ORBIT      
    end
    'ef_series_o2': begin
      editwidgetData.ioptions(4)=event.value(0)
      papco_edit_slice_thumb_refresh, /ORBIT            
    end
    'spb_save_series': begin
      message,'Writing user file',/cont
      print,'  ',editwidgetData.PATH+editwidgetData.FILENAME
      widget_control,editwidgetData.auto_series_lst, GET_VALUE=times_arr
      openw,unit,editwidgetData.PATH+editwidgetData.FILENAME, /get_lun
      for i=0,n_elements(times_arr)-1 do printf,unit,times_arr(i)
      close,unit  &  free_lun,unit
      editwidgetData.status=1
      papco_panel_Refresh, editwidgetData,  /ADD_FILE
    end
    
    else: begin
      widgetData=papco_panel_Event(event, editwidgetData)
      if xregistered('papco_edit_slice_thumb') then $
        if editwidgetData.status then $
        widget_control,editwidgetData.spb_save_series,sensitive=1 else $
        widget_control,editwidgetData.spb_save_series,sensitive=0
    end
ENDCASE

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO papco_edit_slice_thumb_event_refresh
;* 
;* DESCRIPTION:  
;*	This processes refreshes papco_edit_slice_thumb widget
;*
;* INPUTS:       
;*	none
;*			
;* KEYWORDS:
;* 	refresh actions to be taken
;* 
;* CALLING SEQUENCE:
;*	called by XMANAGER
;*
;* MODIFICATION HISTORY:       
;*      September 2000, written by Reiner Friedel
;******************************************************************************
PRO papco_edit_slice_thumb_refresh, ALL=ALL, LIST=LIST, $
                                    DELTA=DELTA, ORBIT=ORBIT
  
COMMON edit_slice_thumb_data, editwidgetData
COMMON plot_composer, widgetData

IF keyword_set(ALL) THEN BEGIN 
    DELTA=1
    LIST=1
    ORBIT=1
END 
  
if keyword_set(DELTA) then begin
    in_str=editwidgetData.soptions(4)
    case in_str of
      'day'  : begin
        days=1  & hr=0  &  mn=0  & s=0
        dec_d=double(days)+(hr*3600.0d +mn*60.0d +double(s))/86400.0d
        dl_str=string(days,hr,mn,s,format="(i3.3,'/',i2.2,':',i2.2,':',i2.2)")
      end
      'week': begin
        days=7  & hr=0  &  mn=0  & s=0
        dec_d=double(days)+(hr*3600.0d +mn*60.0d +double(s))/86400.0d
        dl_str=string(days,hr,mn,s,format="(i3.3,'/',i2.2,':',i2.2,':',i2.2)")
      end
      'month': begin  ;special case
        dec_d=-1.0d
        dl_str='month' 
      end
      'year' : begin  ;special case
        dec_d=-2.0d
        dl_str='year'  
      end
      
      else: begin               ;valid fomat is ddd/hh:mm:ss
        dl=in_str
        ;look for ddd string - either before /, or length 3
        p=strpos(dl,'/') 
        if p eq -1 then begin
          if strlen(dl) eq 3 then days=fix(dl) else days=0
        endif else begin
          days=fix(strmid(dl,0,3))
          dl=strmid(dl,p+1,strlen(dl)-p-1)
        endelse
        ;parse time string
        t_arr=str_sep(dl,':')
        if t_arr(0) ne '' then hr=fix(t_arr(0)) else hr=0
        if n_elements(t_arr) gt 1 then mn=fix(t_arr(1)) else mn=0
        if n_elements(t_arr) gt 2 then s=fix(t_arr(2)) else s=0.
        dec_d=double(days)+(hr*3600.0d +mn*60.0d +double(s))/86400.0d
        dl_str=string(days,hr,mn,s,format="(i3.3,'/',i2.2,':',i2.2,':',i2.2)")
      endelse
    endcase
    editwidgetData.soptions(4)=dl_str
    widget_control, editwidgetData.ef_series_dl, SET_VALUE=dl_str
    editwidgetData.delta=dec_d    
endif
    
IF keyword_set(LIST) THEN BEGIN 
    IF editwidgetData.soptions(2) NE '' AND $
      editwidgetData.soptions(3) NE '' AND $
      editwidgetData.soptions(4) NE '' THEN BEGIN
        widget_control,HOURGLASS=1
        papco_slice_time_list,  times_arr
        IF n_elements(times_arr) EQ 0 THEN return
        widget_control, editwidgetData.auto_series_lst, SET_VALUE=times_arr
        widget_control, HOURGLASS = 0
    ENDIF  
ENDIF 
  
IF keyword_set(ORBIT) THEN BEGIN 
    st_orbit=editwidgetData.ioptions(3)
    en_orbit=editwidgetData.ioptions(4)
    if en_orbit gt st_orbit then begin
        widget_control,HOURGLASS=1
        orbit_arr=strarr(en_orbit-st_orbit+1)
        for i=0, n_elements(orbit_arr)-1 do $
          orbit_arr(i)=string(st_orbit+i,format="(i5.5)")
        widget_control, editwidgetData.auto_series_lst, SET_VALUE=orbit_arr
        widget_control,HOURGLASS=0      
    ENDIF  
ENDIF 
  
END

;******************************************************************************
;* PROCEDURE:     
;*      PRO papco_edit_slice_thumb
;* 
;* DESCRIPTION:  
;*      Allows interactive editing of slice event files. Makes use of a lot
;*      of features from panel editors!
;*
;* INPUTS:      
;*      plotinfo   a plotinfor structure
;* 
;* KEYWORDS:
;*      MJDT        default time range to current papco time range
;*      BATCH_TIMES set to a variable containing an array of batch times
;*
;* MODIFICATION HISTORY:       
;*      February 2003, written by Reiner Friedel
;******************************************************************************
PRO papco_edit_slice_thumb, plotinfo, MJDT = MJDT, $
                            BATCH_TIMES=BATCH_TIMES, GROUP=GROUP, $
                            NOWIDGET = NOWIDGET

  
COMMON plot_composer, widgetdata
COMMON mjdt, mjdt_start, mjdt_end     ;common time limit in mjdt
COMMON edit_slice_thumb_data, editwidgetData
  
; if GROUP is not specified, the group-leader will be the desktop
IF NOT KEYWORD_SET(GROUP) THEN GROUP=widgetdata.base
  
; set the label for the action-button
IF NOT KEYWORD_SET(ACTION) THEN ACTION='Done'

additionalWData={canceled:1}

dirpath=papco_getenv('PAPCO_HOME')+'papco_products/event_times/slice/'
;test for directory - make if it does not exist!  
if not papco_finddir(dirpath) then papco_create_dir, dirpath, /NOASK

filter_ext='event'
filter = '*.event'
title='PAPCO Select Slice Time File' 
IF plotinfo.filename NE '' THEN BEGIN
    papco_sepdir, plotinfo.filename, path, file, ext
    IF ext NE filter_ext THEN ext = filter_ext
    plotinfo.filename = file+'.'+ext
    IF path NE '' THEN dirpath = path
ENDIF 

IF keyword_set(MJDT) THEN BEGIN
    date_time_1 = convert_secondsToTime(mjdt_start, /T90, /MJDT, /COUNTRY)
    date_time_2 = convert_secondsToTime(mjdt_end, /T90, /MJDT, /COUNTRY)
ENDIF ELSE BEGIN
    date_time_1 = '' & date_time_2 = ''
ENDELSE
plotinfo.soptions(2) = date_time_1
plotinfo.soptions(3) = date_time_2

;--- create the base-window ---------------------------------------------------
panel_base=papco_WIDGET_BASE(title=title, column=1, GROUP_LEADER=group)  
  
;--- some panels might need to read/input to text files (event, text ----------
;    module). So a generic papco widget is supplied for this.
widgets=papco_panel_AddFile_Widgets(panel_base, plotinfo, $
                                    DIRPATH=dirpath, FILTER=filter)
additionalWData=create_struct(widgets, additionalWData)   
  
;--- Add some widgets that allow automatic generation of time series ----------
auto_series_base=widget_base(panel_base,column=2,/frame)
  
auto_series_info=widget_base(auto_series_base,column=1, /ALIGN_CENTER)
ef_series_t1=CW_FIELD(auto_series_info,/STRING,/RETURN_EVENTS,XSIZE=19,$
                        /COLUMN, TITLE='Series Start Date:', $
                        VALUE=plotinfo.soptions(2), $
                        UVALUE= 'ef_series_t1')
ef_series_t2=CW_FIELD(auto_series_info,/STRING,/RETURN_EVENTS,XSIZE=19,$
                        /COLUMN, TITLE='Series End Date:  ', $
                        VALUE=plotinfo.soptions(3), $
                        UVALUE= 'ef_series_t2')
ef_series_dl=CW_FIELD(auto_series_info,/STRING,/RETURN_EVENTS,XSIZE=19,$
                        /COLUMN, TITLE='Series Time Step: ', $
                        VALUE=plotinfo.soptions(4), $
                        UVALUE= 'ef_series_dl')
  
ef_series_o1=CW_FIELD(auto_series_info,/INTEGER,/RETURN_EVENTS,XSIZE=19,$
                        /COLUMN, TITLE='Series Start Orbit:  ', $
                        VALUE=plotinfo.ioptions(3), $
                        UVALUE= 'ef_series_o1')
ef_series_o2=CW_FIELD(auto_series_info,/INTEGER,/RETURN_EVENTS,XSIZE=19,$
                        /COLUMN, TITLE='Series End Orbit: ', $
                        VALUE=plotinfo.ioptions(4), $
                        UVALUE= 'ef_series_o2')
  
;--- Add file action buttons --------------------------------------------------
file_action_base=widget_base(auto_series_info,/ALIGN_BOTTOM)
spb_save_series=widget_button(file_action_base, $
                                VALUE=' Save Series to File ', $
                                UVALUE='spb_save_series')
if plotinfo.filename eq '' then widget_control,spb_save_series,sensitive=0
  
if keyword_set(BATCH_TIMES) then batch_times=BATCH_TIMES else batch_times=''
auto_series_lst=widget_text(auto_series_base,scr_xsize=290, $
                              value=batch_times, $
                              scr_ysize=350, /SCROLL)
  
;--- finally, the Actionbuttons (help, cancel, done) are added to the ---------
;    editor window.
widgets=papco_panel_AddActionButtons(panel_base, Action, plotinfo)
additionalWData=create_struct(widgets, additionalWData)
   
;--- now, create the structure that keeps the panel-window data. You ----------
;    always need to specify the help-file, and your structure must
;    always contain the 'additionalWData'.
thisWidgetsData={helpfile:'papco_edit_slice_thumb.help', $
                 soptions:plotinfo.soptions, $
                 ioptions:plotinfo.ioptions, $
                 delta:0d, $
                 ef_series_t1:ef_series_t1, $
                 ef_series_t2:ef_series_t2, $
                 ef_series_dl:ef_series_dl, $
                 ef_series_o1:ef_series_o1, $
                 ef_series_o2:ef_series_o2, $
                 spb_save_series:spb_save_series, $
                 auto_series_lst:auto_series_lst}
 
editwidgetData=papco_panel_CreateWidgetData(thisWidgetsData, additionalWData)
   
;--- realize the window -------------------------------------------------------
papco_edit_slice_thumb_refresh, /ALL
papco_panel_Refresh, editwidgetData,  /ADD_FILE  
IF keyword_set(NOWIDGET) THEN return
papco_Panel_Realize, panel_base, editwidgetData

papco_xmanager, 'papco_edit_slice_thumb', panel_base

;set return values
plotinfo.filename = editwidgetData.PATH+editwidgetData.FILENAME
plotinfo.soptions = editwidgetData.soptions
  
END 


;******************************************************************************
;* PROCEDURE:     
;*      PRO papco_slice_time_list
;******************************************************************************
PRO papco_slice_time_list,  times_arr, DELTA = DELTA, TNUM = TNUM, $
                            UTAI = UTAI,  MJDT = MJDT

COMMON mjdt, mjdt_start, mjdt_end    ;common time limit in mjdt
COMMON plot_composer, widgetdata
COMMON edit_slice_thumb_data, editwidgetData
    
IF keyword_set(MJDT) THEN BEGIN 
    st_mjdt = mjdt_start
    en_mjdt = mjdt_end
ENDIF ELSE BEGIN
    res=CONVERT_timeToSeconds(editwidgetData.soptions(2), /T90, /MJDT)
    st_mjdt=res.value
    res=CONVERT_timeToSeconds(editwidgetData.soptions(3), /T90, /MJDT)
    en_mjdt=res.value
ENDELSE

tai1 = utc2tai({mjd:st_mjdt.mjd, time:st_mjdt.t*1000})
tai2 = utc2tai({mjd:en_mjdt.mjd, time:en_mjdt.t*1000})

;DELTA (dl) is double decimal day fraction
IF keyword_set(TNUM) THEN BEGIN
    dl = -9.0
    t_step = (tai2-tai1)/TNUM
ENDIF ELSE $
    IF keyword_set(DELTA) THEN dl = DELTA ELSE dl=editwidgetData.delta

IF dl EQ 0 THEN return
tai = tai1 & c = 0

WHILE tai LT tai2 DO BEGIN
          
    CASE 1 OF 
        (dl EQ -1.0) : BEGIN        ;delta time is months
            r = TAI2UTC(tai, /EXTERNAL)
            r.month = r.month+1
            IF r.month GT 12 THEN BEGIN
                r.month = 1 & r.year = r.year+1
            ENDIF
            tai = UTC2TAI(r)
        END
        (dl EQ -2.0) : BEGIN        ; delta time is years
            r = TAI2UTC(tai, /EXTERNAL)
            r.year=r.year+1
            tai = UTC2TAI(r)
        END
        (dl EQ -9.0) : BEGIN        ; delta time is range div TNUM
            tai = tai1 + t_step/2 + t_step*c
        END 
        (dl GT 0) : BEGIN           ;delta time set normally
            r = tai2utc(tai) 
            tai = tai+dl*86400.0d
        END     
    ENDCASE 
  
    r = TAI2UTC(tai)
    t1_str=convert_secondsToTime({mjd:r.mjd, t:r.time/1000.0}, $
                          /T90, /MJDT, COUNTRY = widgetData.default_us_europe)

    IF keyword_set(UTAI) THEN $
      IF c EQ 0 THEN times_arr = tai ELSE times_arr = [times_arr, tai] $
    ELSE $
      IF c EQ 0 THEN times_arr = t1_str ELSE times_arr = [times_arr,t1_str]
    c=c+1

ENDWHILE 

END
