;******************************************************************************
;******************************************************************************
;* FILE:	 
;*    string_functions.pro
;* 
;* DESCRIPTION:  
;*    This file contains functions for processing and formatting strings.
;*    All the functions are quite simple, but they make programming easier, 
;*    source-code better to read, and they help you to avoid errors. 
;*
;*    What is better to read ?   strright(aString, 5)
;*			     or strmid(aString, strlen(aString)-5, 5)
;*    In addition, if you use 'strmid', you always have to get the
;*    indices and the lenghtes right - sometimes you have to add 1,
;*    sometimes not. This is a source for errors: 
;*    e.g.:  a=strfrom(aString, 3)
;*            =strmid(string, 3, strlen(string)-3+1)
;*
;* OVERVIEW:
;*    Those procedures are contained:
;* 
;*    FUNCTION strfrom, string, from                                           
;*                                                                             
;*    FUNCTION strright, string, length                                        
;*                                                                             
;*    FUNCTION strleft, string, length                                         
;*                                                                             
;*    FUNCTION strcenter, aString, len                                         
;*                                                                             
;*    FUNCTION strFormat, aString, len, CENTER=CENTER, LEFT=LEFT, RIGHT=RIGHT  
;*                                                                             
;*    FUNCTION isDigits, string                                                
;*    
;* MODIFICATION HISTORY:       
;*     august,september 1995, written by A.Keese 
;******************************************************************************
;******************************************************************************

;******************************************************************************
;* 
;* FUNCTION:     
;*      FUNCTION strfrom, string, from
;* 
;* DESCRIPTION:  
;*      return the right part of the string starting from position 'from'.
;* 
;* INPUTS:       
;*      string       a string
;*      from         an integer (0,...)
;*                   
;* OUTPUT:	 
;*      a string
;*
;* CALLING SEQUENCE:
;*      a=strfrom('12345', 1)  ;=> a EQ '2345'
;*      a=strfrom('12345', 5)  ;=> a EQ ''
;*      a=strfrom('12345', 0)  ;=> a EQ '12345'
;*
;* MODIFICATION HISTORY:       
;*      september 1995, written by A.Keese 
;******************************************************************************
FUNCTION strfrom, string, from
   RETURN, strmid(string, from, strlen(string)-from+1)
END

;******************************************************************************
;* FUNCTION:     
;*      strright, string, length
;* 
;* DESCRIPTION:  
;*	returns a the rightmost 'length' bytes of 'string'
;*
;* INPUTS:       
;*	string	a String
;*	length  an Integer (0,...)
;* 
;* OUTPUT:	 
;*	a String
;*
;* CALLING EXAMPLES:
;*	otter=strright('find outer otter', 5) ;=> otter EQ 'otter'
;*      empty=strright('the lazy fox does not jump', 0) ;=> empty EQ ''
;*
;* MODIFICATION HISTORY:       
;*     written august 1995, Andreas Keese
;******************************************************************************
FUNCTION strright, string, length
   RETURN, strmid(string, strlen(string)-length, length)
END

;******************************************************************************
;* FUNCTION:     
;*      strleft, string, length
;* 
;* DESCRIPTION:  
;*	returns a the leftmost 'length' bytes of 'string'
;*
;* INPUTS:       
;*	string	a String
;*	length  an Integer
;* 
;* OUTPUT:	 
;*	a String
;*
;* CALLING EXAMPLES:
;*	find=strright('find outer otter', 4)
;*      empty=strleft('the lazy fox does not jump', 0) ;=> empty EQ ''
;*
;* MODIFICATION HISTORY:       
;*     written august 1995, Andreas Keese
;*
;******************************************************************************
FUNCTION strleft, string, length
   RETURN, strmid(string, 0, length)
END


;******************************************************************************
;* FUNCTION:     
;*      FUNCTION strFormat, aString, len, CENTER=CENTER, LEFT=LEFT, $
;*                                        RIGHT=RIGHT
;* 
;* DESCRIPTION:  
;*	returns the centered, left- or right-formatted string 'aString' 
;*      expanded or reduced to length 'len' by filling with blanks or 
;*      by cutting of a part of it.
;*
;* INPUTS:       
;*	string	a String
;*	length  an Integer
;* 
;* OUTPUT:	 
;*	a String
;*
;* CALLING EXAMPLES:
;*	a1=strformat('*', 5, /LEFT)        ---> a1=='*    '
;*      a1=strformat('**, 5, /RIGHT)       ---> a1=='   **'
;*
;*	a1=strformat('*', 5, /CENTER)      ---> a1=='  *  '
;*      a1=strformat('**, 5, /CENTER)      ---> a1==' **  '
;*	a1=strformat('   aaa', 5, /CENTER) ---> a1==' aaa '
;*
;*	a1=strformat('12345', 2, /CENTER)  ---> a1=='12'
;*
;* MODIFICATION HISTORY:       
;*     written august 1995, Andreas Keese
;******************************************************************************
FUNCTION strFormat, aString, len, CENTER=CENTER, LEFT=LEFT, RIGHT=RIGHT
   str=strtrim(aString,2)
   IF strlen(str) GT len THEN return, strmid(str, 0, len)
   l=fix((len-strlen(str)) / 2)
   r=l+(len-strlen(str)) MOD 2
   dummy='                                                                '
   IF KEYWORD_SET(CENTER) THEN $
      return, strmid(dummy, 0, l) + str + strmid(dummy, 0, r)
   IF KEYWORD_SET(RIGHT) THEN $
      return, strmid(dummy, 0, l) + strmid(dummy, 0, r) + str
   return, str + strmid(dummy, 0, l) + strmid(dummy, 0, r)
END

;******************************************************************************
;* FUNCTION:     
;*      strcenter, aString, len
;* 
;* DESCRIPTION:  
;*	returns the string 'aString' expanded or cut to length 'len' by 
;*      filling with blanks or cutting of its right, so that the 'aString' 
;*      is centered in the new string.
;*
;* INPUTS:       
;*	string	a String
;*	length  an Integer
;* 
;* OUTPUT:	 
;*	a String
;*
;* CALLING EXAMPLES:
;*	a1=strcenter('*', 5)     ---> a1=='  *  '
;*      a1=strcenter('**, 5)     ---> a1==' **  '
;*	a1=strcenter('   aaa',5) ---> a1==' aaa '
;*      a1=strcenter('1234567', 3) -> at=='123'
;*
;* MODIFICATION HISTORY:       
;*     written august 1995, Andreas Keese
;******************************************************************************
FUNCTION strcenter, aString, len
   RETURN, strFormat(aString, len, /CENTER)
END


;******************************************************************************
;* FUNCTION:     
;*      FUNCTION isDigits, string
;* 
;* DESCRIPTION:  
;*      check, if a string contains digits only - blanks in 'string' are 
;*      ignored !
;*
;* INPUTS:       
;*      string - a string
;* 
;* OUTPUT:	 
;*      1 if 'string' contains digits (that is, '0','1',...,'9') only
;*      0 otherwise
;*
;* CALLING SEQUENCE:
;*      if not isDigits('123') then print, 'error in isDigits()'
;*      if not isDigits('1 2 3') then print, 'error in isDigits()'
;*      if not isDigits('123   ') then print, 'error in isDigits()'
;*
;*      if isDigits('123x') then print, 'error in isDigits()'
;*      if isDigits('x ') then print, 'error in isDigits()'
;*
;* MODIFICATION HISTORY:       
;*      september 1995, written by A.Keese 
;******************************************************************************
FUNCTION isDigits, string
     teststr=strtrim(string, 2)
     FOR i=0, strlen(teststr)-1 DO $
       IF strmid(teststr, i, 1) GT '9' OR strmid(teststr, i, 1) LT '0' THEN $
          RETURN, 1 EQ 0
     RETURN, 1 EQ 1
END

;******************************************************************************
FUNCTION strchminus,string

     outstr=string
     pp=0
     while 1 eq 1  do begin
        pp=strpos(outstr,'-',pp)
        if pp eq -1 then return,outstr
        strput,outstr,'_',pp
     endwhile
   END
  
;******************************************************************************
FUNCTION strplotdesc,string

     outstr=string
     pp=strpos(outstr,'_',0)
     outstr=strmid(outstr,pp+1,strlen(outstr))
     return,outstr
END

;******************************************************************************
FUNCTION strplottype,string

     name='papcoadd'
     ext='.pro'       
     outstr=string
     pp=strpos(outstr,name,0)
     outstr=strmid(outstr,pp,strlen(outstr)) 
     pp=strpos(outstr,'_',0)
     outstr=strmid(outstr,pp+1,strlen(outstr))
     pp=strpos(outstr,ext,0)
     outstr=strmid(outstr,0,pp)

     return,outstr
    
END

;******************************************************************************
FUNCTION strdelversion,string
        
  outstr=string
  pp=strpos(outstr,';',0)
  if pp ne -1 then outstr=strmid(outstr,0,pp)
  return,outstr
    
END

;******************************************************************************
FUNCTION strsubst,string,find,new

  outstr=string
  pp=0
  while 1 eq 1  do begin
    pp=strpos(outstr,find,pp)
    if pp eq -1 then return,outstr
    strput,outstr,new,pp
  endwhile
      
END
     
;******************************************************************************
FUNCTION strdel,string,find

  outstr=string
  pp=0
  while 1 eq 1  do begin
    pp=strpos(outstr,find,pp)
    if pp eq -1 then return,outstr
    outstr=strmid(outstr,0,pp)+strmid(outstr,pp+strlen(find),strlen(outstr))
  endwhile
      
END
  
;******************************************************************************
FUNCTION varprt,var, LEN=LEN, FIXED=FIXED
  
  if keyword_set(FIXED) then $
    out_str=strtrim(string(var, format="(g10.4)"),2) $
  else $
    out_str=strtrim(string(var),2)
  
  if keyword_set(LEN) then out_str=strleft(out_str,LEN)
  return, out_str

END
  
;******************************************************************************
PRO status, v01, v02, v03, v04, v05, v06, v07, v08, v09, v10
  
  result=n_params()
  print,result
  if result gt 10 then message,'Can only output 10 items',/cont
  for i=0, (result-1)<9  do begin
    var_name=''
    
  endfor  
  
END  

;******************************************************************************
FUNCTION str_replace, string, find, new

  outstr=string

  p = strpos(outstr,find)

  IF p EQ -1 THEN return, string

  front = strmid(outstr, 0, p)
  back = strmid(outstr, p+strlen(find), strlen(outstr))

  new_str = front+new+back
  return, new_str

END
