;******************************************************************************
;* DESCRIPTION:  
;*      General routine to calculate mag coords and values from model in the
;*      unilib library using the idl interface routine uxidl.
;*      input tai, pos can also be an array.
;* 
;* INPUTS:       
;*      tai         time in TAI (int. atomic time) format
;*      pos         structure containing position
;*      kcompstr    component format: SPH (spherical), CAR (cartesian)
;*      kinstr      coordinate system of pos: GEO, GEI, MAG, SM, GSM, GSE
;*      
;* KEYWORDS:
;*      EXTMOD      external field model index
;*                      0: none
;*                      1: Mead & Fairfield [1975]
;*                      2: Tsyganenko short [1987]
;*                      3: Tsyganenko long [1987]
;*                      4: Tsyganenko [1989c]
;*                      5: Olson & Pfitzer quiet [1977]             (default)
;*                      6: Olson & Pfitzer dynamic [1988]
;*                      7: Tsyganenko [1996]
;*                      8: Ostapenko & Maltsev [1997]
;*      INTMOD      internal field model index
;*                      0: DGRF/IGRF 1945-2000                      (default)
;*                      1: Jensen & Cain (1962)
;*                     -1: Jensen & Cain (1962) fixed to epoch 1960 
;*                         and corrected for the SAA westward drift
;*                      2: GSFC 12/66
;*                     -2: GSFC 12/66 fixed to epoch 1960 
;*                         and corrected for the SAA westward drift
;*                      3: Dipole magnetic field
;*                      4: DGRF/IGRF 45-95 use the Kluge algorithm to 
;*                         evaluate the geomagnetic field
;*      PITCH       pitch angle for which L is calculated (default 90 deg)
;*      KP          kp index value, (default=2)
;*      INT_NAME    returns the name of the internal model
;*      EXT_NAME    returns the name of the external model
;*      KOUTSTR     coordinate system of output mag vector, see kinstr
;*      HAVE_MDATE  time is already in format mdate required by unilib
;*      LSTAR       Calculate Roederer L-value, default McIlwain only
;* 
;* OUTPUTS
;*      bvec        magnetic field vector in coordinate system B_KOUTSTR
;*                  default  B_KOUTSTR is 'GSM''
;*      bequ        mag field at equator on field line
;*      Lval        L-value
;*      mlt         Magnetic local time
;*      ilat        Magnetic latitude
;*      error       error status
;*
;* CALLING SEQUENCE:
;*      stand-alone. Needs @papco_addin_unilib in PAPCO.
;*
;* MODIFICATION HISTORY:       
;*      May 2002, written by R.Friedel
;******************************************************************************
PRO get_mag_unilib, tai, pos, kcompstr, kinstr, $                   ; inputs
                    bvec, bequ, Lval, maglt, ilat, error, $         ; outputs 
                    INTMOD=INTMOD, EXTMOD=EXTMOD,  $
                    INT_NAME=INT_NAME, EXT_NAME=EXT_NAME, $
                    LSTAR = LSTAR, KOUTSTR = KOUTSTR, $
                    PITCH=PITCH, VERBOSE=VERBOSE,  HAVE_MDATE = HAVE_MDATE, $
                    DRIFTSHELL = DRIFTSHELL

COMMON get_error, get_err_no, get_err_msg

;get unilib shareable object location. Unilib is initialized (if available) 
;at papco startup time (papco_unilib_init) which also sets up
;required structures and env. variable 'PAPCO_UNILIB'.
lib = papco_getenv('PAPCO_UNILIB', /NOFS)
IF lib EQ '' THEN BEGIN
    get_error = 1 & return
ENDIF
entry = 'uxidl_'

if keyword_set(VERBOSE) then verbose=VERBOSE else verbose=0     
if keyword_set(INTMOD) then kint=long(INTMOD) else kint=0l    ;
if keyword_set(EXTMOD) then kext=long(EXTMOD) else kext=0l    
if keyword_set(PITCH) then alpha=double(PITCH) else alpha=90.0d
if keyword_set(KOUTSTR) THEN koutstr = KOUTSTR ELSE koutstr = 'GSE'

;set up verbose output for unilib kunit direct output to screen if set
;to 6
IF verbose THEN kunit = 6L ELSE kunit = 0L

;NOTE!!!! Variable INTMOD is set 0,1,2,3,4,5,6 which needs to map to
;the correct kint values needed by UNILIB!!!
kint_idx = [0l, 1l, -1l, 2l, -2l, 3l, 4l]
kint = kint_idx(intmod)

;make output arrays as needed
ndat = n_elements(tai) & bvec = fltarr(4, ndat) & bequ = fltarr(ndat)
Lval = fltarr(ndat) & maglt = fltarr(ndat) & ilat = fltarr(ndat)
Ls = fltarr(ndat)

;set up input position coordinate system
case strupcase(kinstr) OF
    'GED': kin = 0l ;geodetic (normal long/lat)
    'GEO': kin = 1l ;geographic
    'GEI': kin = 2l ;geocentric equatorial inertial 
    'MAG': kin = 3l ;geomagnetic
    'SM' : kin = 4l ;solar magnetic
    'GSM': kin = 5l ;geocentric solar magnetospheric
    'GSE': kin = 6l ;geocentric solar ecliptic
    ELSE:BEGIN
        message, 'Invalid input coordinate format: '+kinstr, /cont & return
    END
ENDCASE

;set up B field output coordinate system
case strupcase(koutstr) OF
    'GEO': kout = 1l ;geographic
    'GEI': kout = 2l ;geocentric equatorial inertial 
    'MAG': kout = 3l ;geomagnetic
    'SM' : kout = 4l ;solar magnetic
    'GSM': kout = 5l ;geocentric solar magnetospheric
    'GSE': kout = 6l ;geocentric solar ecliptic
    ELSE:BEGIN
        message, 'Invalid ouput coordinate format: '+koutstr, /cont & return
    END
ENDCASE

;set up input/output component format
CASE strupcase(kcompstr) OF
    'SPH' : kcomp = 0l ;spherical
    'CAR' : kcomp = 1l ;cartesian
    ELSE:BEGIN
        message, 'Invalid component format: '+kcompstr, /cont & return
    END
ENDCASE   

;make structure of returned error codes for all the UNILIB routines
;called here
error = {UM510:0l, UM520:0l, UT550:0l, UM530:0l, UL220:0l, UM538:0l, $
         UD310:0l, UD317:0l,UD330:0l}
tag_num = N_TAGS(error)
  
;select a internal geomagnetic field model. IGRF depends on year #
IF keyword_set(HAVE_MDATE) THEN year = double(tai(0).iyear) ELSE BEGIN
    t = TAI2UTC(tai(0), /EXTERNAL ) & year = double(t.year)
ENDELSE

lbint = string('',format="(a32)") & ifail = 0l
r = call_external(lib, entry,'UM510', kint, year, lbint, kunit, ifail)
int_name = lbint
error.UM510 = ifail

FOR i = 0, ndat-1 DO BEGIN

    ;compute modified Julian date as required
    IF keyword_set(HAVE_MDATE) THEN mdate = tai(i) ELSE BEGIN
        t = TAI2UTC(tai(i), /EXTERNAL )
        mdate = {papco_zdat} ;see papco_unilib_init for struct def
        mdate.IYEAR = t.year & mdate.IMONTH = t.month  & mdate.IDAY = t.day
        mdate.IHOUR = t.hour & mdate.IMIN   = t.minute & mdate.SECS = t.second 
    ENDELSE     

    result = call_external(lib, entry,'UT540', mdate)

    ;select an external magnetic field model
    param = dblarr(10) & lbext = string('',format="(a32)")
    r = call_external(lib, entry,'UM520', $
                      kext, mdate.amjd, param, lbext, kunit, ifail)
    ext_name = lbext & error.UM520 = ifail
    kunit = 0l
    
    ;convert to spherical coords if needed
    mpos = {papco_zgeo}  ;see papco_unilib_init for struct def
    IF kcomp EQ 0 THEN BEGIN     ;spherical input
        mpos.radius = double(pos(0, i))
        mpos.colat  = double(pos(1, i))
        mpos.elong  = double(pos(2, i))
    ENDIF ELSE BEGIN            ;cartesian input
        mposxyz = {papco_zxyz}  ;see papco_unilib_init for struct def
        mposxyz.x = pos(0, i) & mposxyz.y = pos(1, i) & mposxyz.z = pos(2, i)
        r = call_external(lib, entry,'UT546', mposxyz, mpos)  ;to spherical
    ENDELSE

    ;NOTE: ALL MAG ROUTINES IN UNLIB USE GEO COORDS AS INPUT

    ;special case - if input in in geodetic (GED, kin=0) use UM536
    IF kin EQ 0 THEN BEGIN
        mpos_geo = {papco_zgeo}  ;see papco_unilib_init for struct def
        r = call_external(lib, entry,'UM536', mpos, mpos_geo)
    ENDIF ELSE BEGIN  
        ;get cartesian transformation matrix for input coord system to GEO
        kfrom = kin & kto = 1l     ;convert --> Geographic coordinate (GEO) 
        IF kfrom NE kto THEN BEGIN 
            trans = dblarr(3, 3)
            r = call_external(lib, entry,'UT550', kfrom, kto, trans, ifail)
            mpos_geo = mpos & error.UT550 = ifail
            r = call_external(lib, entry,'UT555', mpos, mpos_geo, trans)
        ENDIF ELSE  mpos_geo = mpos
    ENDELSE

    ;evaluate the magnetic field. This also updates the Sun position
    ;and the SM coordinates (done by UM522 and UM524)
    ;output vector is in spherical vector components, units Gauss
    ; 1 T = 1e4 G -> 1 G = 10e5 nT
    mb = {papco_zvec}  ;see papco_unilib_init for struct def
    r = call_external(lib, entry,'UM530', mpos_geo, mb, ifail)       
    error.UM530 = ifail

    ;convert to coordinate system required.
    kfrom = 1L & kto = kout
    IF kfrom NE kto THEN BEGIN 
        trans = dblarr(3, 3)
        r = call_external(lib, entry,'UT550', kfrom, kto, trans, ifail)
        mb_out = mb & error.UT550 = ifail
        r = call_external(lib, entry,'UT556', mpos, mb, mb_out, trans)
    ENDIF ELSE  mb_out = mb

    ;now convert the mag field vector to the coordinates required
    ;first convert spherical vector to cartesian vector components.
    mxyz_geo = {papco_zxyz} & mxyz = mxyz_geo
    r = call_external(lib, entry, 'UT542', mpos, mb, mxyz_geo)
    r = call_external(lib, entry, 'UT542', mpos, mb_out, mxyz)
    ;convert to nT
    FOR j = 0, 2 DO mxyz.(j) = mxyz.(j)*1e5
    FOR j = 0, 2 DO mxyz_geo.(j) = mxyz_geo.(j)*1e5
    FOR j = 0, 2 DO bvec(j, i) =  mxyz.(j)
    bvec(3, i) = sqrt(mxyz.x^2+mxyz.y^2+mxyz.z^2)

    ;evaluate the (B, L) coordinates
    nfbm = 1l & fbm = 0.0d & flm = 0.0d & fkm = 0.0d & fsm = 0.0d
    fbeq = 0.0d & fs = 0.0d
    r = call_external(lib, entry,'UL220', $ 
                      mpos_geo,alpha,nfbm,fbm,flm,fkm,fsm,fbeq,fs,ifail)
    Lval(i) = flm & bequ(i) = fbeq*1e5 & error.UL220 = ifail

    ;evaluate the magnetic local time and latitude
    xmlt = 0.0d &  xilat = 0.0d
    r = call_external(lib, entry,'UM538', $
                      mpos_geo, mdate.amjd, xmlt, xilat, ifail)
    maglt(i) =  xmlt & ilat(i) = xilat & error.UM538 = ifail

    ;evaluate L* Roederer L-value by third adiab. inv. 
    IF keyword_set(LSTAR) THEN BEGIN
        IF (flm GT 0) AND (flm LE 8.5) THEN BEGIN 
            
            ;trace drift shell. number of lines to be traced should increase
            ;with L 
            fbm0 = fbm & flm0 = flm & falt = 0.0d 
            IF keyword_set(DRIFTSHELL) THEN knfl = long(DRIFTSHELL) ELSE $
               knfl = long(15*flm)
            ktyplus = 3l & altmin = 0.0d
            ;use newer trace drift shell routine UD317
            ;mlab0 = {papco_zlbl} ;see papco_unilib_init for struct def
            ;mlab0.lbmp = 255 & mlab0.llmi = 0 & mlab0.linv = 0
            ;mlab0.fbmp = fbm0 & mlab0.flmi = flm0
            ;r = call_external(lib, entry, 'UD317', $
            ;                  mlab0, falt, ktyplus, altmin, ifail)
            ;error.UD317 = ifail
            
            ;mlab0 = {linv:      0L, lbmp:         1L,     $
            ;         lkauf:     0L, llmi:         1L, lalp0:    0L,     $
            ;         lphi:      0L, ltim:         0L, finv:  0.0d0,     $
            ;         fbmp:   fbm0,  fkauf:     0.0d0, flmi:  flm0,      $
            ;         falp0:  0.0d0, fphi:      0.0d0, ftim:  0.0d0}

            ;r = call_external(lib, entry, 'RF317', $
            ;                  mlab0, falt, ktyplus, altmin, ifail)
            ;error.UD317 = ifail

            r = call_external(lib, entry,'UD310', $
                              fbm0, flm0, falt, knfl, ktyplus, ifail)
            error.UD310 = ifail
            
            ;evaluate third invariant
            phi = 0.0d & star = 0.0d
            r = call_external(lib, entry, 'UD330', $
                              phi, star, ifail)
            error.UD330 = ifail

            print, i, star,  flm, fbm0, error.UD310, error.UD330, $
                   format = "(i4.4,' ',2(f7.3),' ',f9.6,2(i8))"

            IF star GT 20 THEN ls(i) = -99 ELSE ls(i) = star
        ENDIF ELSE ls(i) = -99
    ENDIF ELSE ls(i) = -99

    IF (verbose GT 0) AND (((i/verbose) * verbose) EQ i) THEN BEGIN 
        print, ''
        print, i, tai2utc(tai(i),  /ECS),', In GEO: ', mpos, $
                format = "(i4.4,',  ',a22,a10,3(f9.2))"
        print, 'L,L*,MLT,MLAT: ', Lval(i), ls(i), maglt(i), ilat(i), $
                format = "(a14,4(f9.2))"
        print, 'Mag vec GEO:  ', mxyz_geo.(0), mxyz_geo.(1), mxyz_geo.(2), $
                sqrt(mxyz_geo.x^2+mxyz_geo.y^2+mxyz_geo.z^2), ' nT', $
                format = "(a14,4(f9.2),a3)"
        print, 'Mag vec '+koutstr+':  ', mxyz.(0), mxyz.(1), mxyz.(0), $
                sqrt(mxyz.x^2+mxyz.y^2+mxyz.z^2), ' nT', $
                format = "(a14,4(f9.2),a3)"
        tags = tag_names(error) & err_str = ''
        FOR j = 0, n_elements(tags)-1 DO IF error.(j) LT 0 THEN $
          err_str = err_str+tags(j)+':'+varprt(error.(j))+', '
        print, 'Error: ', err_str
    ENDIF 

ENDFOR

IF keyword_set(LSTAR) THEN lstar = ls

END
