;+
; NAME: 
;       CNVRTTIM
;
; PURPOSE:
; This is a general purpose routine to convert from Universal Time 
;       to binary EPOCH time in seconds. The universal time values are given in
;       normal units. This routine is used heavily in the summary plotting 
;       routines to set the time axis to even day units.
;       The input and output times may be arrays - This does *not* 
;       work with IDL 1.
;	Scalar input yields scalar output.
;	
; CATEGORY: Time conversion
;
; CALLING SEQUENCE: Cnvrttim,iyear,idoy,ihour,imin,sec,time
;
; INPUTS:               Iyear   = Integer year value 
;                       Idoy    = Integer day of the year
;                       Ihour   = Integer hour of the day
;                       Imin    = Integer minute of the hour
;                       Sec     = DOUBLE seconds of the minute
;		can be scalars or arrays
;
; OUTPUTS:        Time    = Time in seconds referenced to the EPOCH 
;                                 of January 1,1950 = 0.00 seconds.
;                                 Double precision real values;
;				  structure is the same as for the inputs.
;
; RESTRICTIONS:
;               Input time should be between 1 jan 1990 and 31 dec 1999.
;               Otherwise the output is wrong. An error message is printed, 
;               but the program is not interrupted.
;               Out of range values for idoy,imonth,ihour,imin,or sec also
;               lead to an error message.
;               Leap seconds after 1989 are not considered.
;
; MODIFICATION HISTORY:
; written 24 jul 89     S. Mazuk
; revised 1 dec 92      M. Fraenz
; revised 23 jun 92     M. Reuss - array inputs
;
;-
;
	Pro cnvrttim, iyear, idoy, ihour, imin, sec, time
;
;
; calculate the offset sum, the time sec(1990.0-1950.0) in seconds
;
;  =14610d*86400s+25Ls
	secd=24L*3600L
	sum90 =14610L*secd+25L
;
; seconds of year from 1990 til 1999 (include leap seconds here)
	sec365=secd*365L
	sec366=secd*366L
;
   secy= $
   [sec365,sec365,sec366,sec365,sec365,sec365,sec366,sec365,sec365,sec365]
;
   erstr='***error message from CNVRTTIM****'
   counts = n_elements(idoy)
   ltime = lonarr(counts)
   FOR i = 0, counts-1 DO BEGIN
; check dates
    IF (idoy(i) LT 1) OR (idoy(i) GT 366) THEN print,erstr+'DOY:',idoy(i)
    IF (iyear(i) LT 90) OR (iyear(i) GT 99) THEN print,erstr+'YEAR:',iyear(i)
    IF (ihour(i) LT 0) OR (ihour(i) GT 23) THEN print,erstr+'HOUR:',ihour(i)
    IF (imin(i) LT 0) OR (imin(i) GT 59) THEN print,erstr+'MIN:',imin(i)
    IF (sec(i) LT 0.0) OR (sec(i) GT 60.0) THEN print,erstr+'sec:',sec(i)
;
    ltime(i) = sum90
    IF iyear(i) GE 91 THEN $
        for j=91,iyear(i) do ltime(i)=ltime(i)+ secy(j-91)
   ENDFOR
   ltime=ltime+(idoy-1)*secd+ihour*3600L+imin*60L
   time=double(ltime)+sec
   IF (counts EQ 1) THEN time=time(0)
;
return
end
;
