;+
; NAME:
;	TIMCNVRT
;
; PURPOSE:
;	Convert from binary EPOCH time to Universal Time.
;	Optionally, month and day of the month are also computed.
;	
; CATEGORY:
;
; CALLING SEQUENCE:
;
; TIMCNVRT, time,iyear,idoy,ihour,imin,isec [,imonth,cmonth,moday]
;
; INPUTS:		Time	= Time in seconds referenced to the EPOCH 
;					of January 1,1950 = 0.00 seconds.
;					Double or LONG value.
;               	   only times GE sec(1990.0)
;
; OPTIONAL INPUTS:
;	
; KEYWORD PARAMETERS:
;
; OUTPUTS:		Iyear	= Integer year value 
;			idoy	= Integer day of the year
;			Ihour	= Integer hour of the day
;			Imin	= Integer minute of the hour
;			ISec	= Integer seconds of the minute
;
; OPTIONAL OUTPUTS:
;			Imonth	= Integer month of the year
;			Cmonth	= Character month of the year
;			Moday	= Integer day of the month
;
; COMMON BLOCKS:
;
; SIDE EFFECTS:
;
; RESTRICTIONS:
;
; PROCEDURE:
;	The times must be later than the beginning of 1990.
;
; EXAMPLE:
;
; MODIFICATION HISTORY:
; created 9/8/89	S. Mazuk
; revised 30 nov 92     M. Fraenz
; revised 21 jun 93	M. Reuss 
; revised 30 jun 93     M. Fraenz taking arrays
; 
;
;-
;
   PRO TIMCNVRT, time,iyear,idoy,ihour,imin,isec,imonth,cmonth,moday
;
   COMMON MAINVARS,timemin,timemax,rwbit,rw3bit,langbit
; Set up the variables
;
;
;
;
; calculate the offset sum, the time to the end of 1989 in seconds
;  =14610d*86400s+25Ls
        secd=24L*3600L
        sum90 =14610L*secd+25L
; seconds of year from 1990 til 1999 (include leap seconds here)
        sec365=secd*365L
        sec366=secd*366L
;  
   secy= $
   [sec365,sec365,sec366,sec365,sec365,sec365,sec366,sec365,sec365,sec365]
;
; calculate the iyear value
;
        ltime=long(time)
        tcounts=n_elements(time)
        iyear=INTARR(tcounts)
        idoy=INTARR(tcounts)
        ihour=INTARR(tcounts)
        imin=INTARR(tcounts)
        isec=INTARR(tcounts)
  IF (N_PARAMS(0) GE 7) THEN BEGIN;
; Set up the Month array
;
   langbit=0
   IF langbit EQ 0 THEN $
     moarray = ['   ','Jan','Feb','Mar','Apr','May','Jun', $
               'Jul','Aug','Sep','Oct','Nov','Dec'] $
   ELSE $
     moarray = ['     ','Jan  ','Febr ','Maerz','April','Mai  ','Juni ', $
              'Juli ','Aug  ','Sept ','Okt  ','Nov  ','Dez  '] 
;
   dayarray=[0,31,28,31,30,31,30,31,31,30,31,30,31]
   imonth=INTARR(tcounts)
   moday=INTARR(tcounts)
   cmonth=STRARR(tcounts)
  ENDIF
        FOR tc=0,tcounts-1 DO BEGIN
         i=0
         rtime=ltime(tc)-sum90
         while rtime GE secy(i) do begin
          rtime=rtime-secy(i)
          i=i+1
         endwhile
         iyear(tc)=90+i
         idoy(tc)=fix(rtime/secd)
         rsec=rtime-idoy(tc)*secd
         ihour(tc)=fix(rsec/3600L)
         rsec=rsec-ihour(tc)*3600L
         imin(tc)=fix(rsec/60)
         isec(tc)=rsec-imin(tc)*60
         idoy(tc)=idoy(tc)+1
;
       IF (N_PARAMS(0) GE 7) THEN BEGIN
; Now calculate the month and day information
;
	if (FIX(iyear(tc)/4)*4 eq iyear(tc) ) then dayarray(2)=29 $
        else dayarray(2)=28
;
        i=1
        rdoy=idoy(tc)-1
        while rdoy GE dayarray(i) do begin
         rdoy=rdoy-dayarray(i)
         i=i+1
        endwhile
        imonth(tc)=i
        moday(tc)=rdoy+1
        cmonth(tc)=moarray(i)
       ENDIF
      ENDFOR
      IF tcounts EQ 1 THEN BEGIN
        iyear=iyear(0)
        idoy=idoy(0)
        ihour=ihour(0)
        imin=imin(0)
        isec=isec(0)
        IF (N_PARAMS(0) GE 7) THEN BEGIN
         imonth=imonth(0)
         moday=moday(0)
         cmonth=cmonth(0)
        ENDIF
      ENDIF
	return
	end
;
