;******************************************************************************
;*  NAME:        r_ae
;*
;*  DESCRIPTION: Routine to read ae files. Adheres to
;*               papco conventions: Error common block
;*                                  Uses MJDT time
;*                                  papco read philosophy
;*
;*  INPUT:       none (uses common block time from papco)
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  KEYWORDS:    VERBOSE  -  enables print comments, save/restore comments
;*
;*  DATE:        June 1996
;*
;*  AUTOR:       R. Friedel
;*
;*  CHANGES:     February 1997. Used MJDT time now, and is a fully
;*               running example module on it's own
;*               October 1997. Rewritten to be more general and fully
;*               adhering to papco philosophy. Added remore Get feature, and
;*               the example data file is no longer part of the 
;*               module - it needs to be gotten remotely first!
;******************************************************************************
PRO r_ae, VERBOSE=VERBOSE

COMMON ae_data, input_header, input_data
COMMON mjdt, mjdt_start, mjdt_end
COMMON get_error, get_err_no, get_err_msg
   
; common get_error is used by papco to check if the data read was successful.
; it is up to the user to set this correctly and to make sure that the read- 
; routine does not crash on an I/O error - otherwise papco will crash at that
; point, too. papco draws an empty panel when get_err_no in not equal to zero
; and prints get_err_msg in that panel for user information. papco will try
; to invoce the plot routine for this data if get_err_no = 0 , otherwise an
; emplty panel with the error message get_err_msg is plotted.

get_err_no=0  &  get_err_msg='ae read succesfully'
   
; check VERBOSE keyword and set
if keyword_set(VERBOSE) then verbose=VERBOSE else verbose=0     
   
; clear data arrays for reuse
input_header=0  &  input_data=0   
   
; loop through from start to end time concatonating data files into one
; array. Here we assume that the data is stored in files per day, and
; that the time info in each file is seconds of day. PAPCO standard is
; to hand over data to the plot routines with the time running in
; seconds from the number of seconds of the startday.
time=mjdt_start  &  no_files_read=0  &  day_count=0.0
   
WHILE mjdt_end.mjd GE time.mjd DO BEGIN
    
    utc = {MJD:long(time.mjd), TIME:long(time.t*1000)}
    datestr = ANYTIM2CAL(utc, FORM = 8, /DATE)
     
    ;read one day of data
    r_ae_onefile, datestr, VERBOSE=VERBOSE
  
    ;check for an error in get_error - and then get the file remotely!
    ;comment this section out if you do not want this feature.
    ;if get_err_no ne 0 then begin      
    ;  c_url=papco_getenv('AE_FTP_ARCHIVE')
    ;  c_file=fln+'.idl'  &  c_download_dir=path
    ;  papco_get_remote_data, c_url, c_file, c_download_dir
    ;  ;check if file "arrived" and then read it
    ;  if get_err_no eq 0 then begin
    ;    r_ae_onefile, path+fln, VERBOSE=VERBOSE
    ;  endif else begin
    ;    get_err_msg='No data locally, remote get failed!C  ' + get_err_msg
    ;    return
    ;  endelse  
    ;endif   
     
    ;if a file was succesfully read, concatonate!
    if get_err_no eq 0 then BEGIN  
        if no_files_read eq 0 then begin   ;start the array buffer
            input_data_buf=temporary(input_data)
        endif else begin                   ;otherwise concatonate arrays
            input_data_buf=[temporary(input_data_buf),temporary(input_data)]
        ENDELSE       
        input_data=0                      ;clear array for reuse
        no_files_read=no_files_read+1
    ENDIF 
    day_count=day_count+1
    time.mjd=time.mjd+1                 ; next day
    ; check for any time left on last day - skip if none
    if (mjdt_end.mjd eq time.mjd) AND (mjdt_end.t eq 0) then break  
ENDWHILE          
   
; check if any files were read   
if (no_files_read ne 0) and (get_err_no ne 1) then begin
    get_err_no=0
    get_err_msg=string(no_files_read,format="(i2)")+ ' files read succesfully'
    ;clear arrays and load buffer into output array
    input_data=0  &  input_data=temporary(input_data_buf)
endif else begin
    get_err_no=1
    get_err_msg='No files found for timerange'
    return
ENDELSE 
END 

;******************************************************************************
;*  NAME:        r_ae_onefile,fln
;*
;*  DESCRIPTION: reads one file of type plot_file. Assumes data has been
;*               stored in IDL saveset using structures input_header and
;*               input_data
;*
;*  INPUT:       fln  - string of filename
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        October 1997
;*
;*  AUTOR:       R. Friedel
;******************************************************************************
pro r_ae_onefile, datestr, VERBOSE=VERBOSE
  
common ae_data, input_header, input_data
common get_error, get_err_no, get_err_msg
   
get_err_no=0  &  get_err_msg='ae onefile read succesfully'
  
; check env. variable pointing to data directory
if not (papco_check_data_env('AE_DATA', PATH=path)) then return
 
; clear data arrays for reuse
input_header=0  &  input_data=0   
   
fln = path+'ae_al_au_'+datestr+'.idl'

on_ioerror, read_original
restore, fln, VERBOSE=VERBOSE
dealloc_lun,/quiet   ;frees logical units used by save/restore
message,'ae_al_au_'+datestr+'.idl', /cont
return

read_original:       ;if the .idl file is not found, try to read the original
r_ae_onefile_original, datestr, VERBOSE=VERBOSE
      
end   
  
;******************************************************************************
;*  NAME:        r_ae_onefile_original,fln
;*
;*  DESCRIPTION: reads one file of type plot_file_original. Under the
;*               papco read philosophy, files should be read the first
;*               time in whatever format they might have - ascii,
;*               fortran binary, pascal binary, etc. The files might also
;*               be in some compressed format, so they need to be uncompressed
;*               first. 
;*               
;*               You might also have to change time formats. 
;*               A lot of files are stored per day, and have seconds of
;*               that day as the time. This is what is assumed here and
;*               is papco standard. You need to convert the time
;*               to papco standard if you intend to use the plotroutines 
;*               supplied with papco (use the routines in
;*               $papco_BASE/papco_Lib/time_convert.pro). 
;*               If you use your own plot routines this step is not needed.
;*               
;*               The first read might be slow, esp if the format is ascii.
;*               Use the IDL "save" command to make a binary copy of the
;*               file which can then be read quickly using "restore". The
;*               next time only the IDL saveset is read, which is much
;*               faster.
;*               
;*               In order to be able to concatonate the arrays easily,
;*               it makes sense to have the data in an array of time-ordered
;*               structures, which makes adding together days of data
;*               trivial using the IDL concatonation operator "[]".
;*               
;*  INPUT:       fln  - string of filename
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        October 1997
;*
;*  AUTOR:       R. Friedel
;******************************************************************************
pro r_ae_onefile_original, datestr, VERBOSE=VERBOSE
  
common ae_data, input_header, input_data
common get_error, get_err_no, get_err_msg  
   
get_err_no=0  &  get_err_msg='ae_onefile_orignal read succesfully'
   
; check env. variable pointing to data directory
if not (papco_check_data_env('AE_DATA', PATH=path)) then return

; clear data arrays for reuse
input_header=0  &  input_data=0   
   
;------------------------------------------------------------------------------
; read the original data file - THIS NEEDS TO BE REWRITTEN BY USER!!!!
  
;read ae, au, al for one day into one file and save as idl saveset. 

indices = ['AE', 'AL', 'AU']
year_str = strmid(datestr, 0, 4)
year = fix(year_str)
month = fix(strmid(datestr, 4, 2))
day = fix(strmid(datestr, 6, 2))

; define the header and data structures. Edit to suit your data!
max_no_rec=1500              ;the maximum number time records in your data
ext='.dat'                   ;file extention of your dat files
input_header={ae_header,tag1:0,tag2:0}
dummy={ae_data, tai:0.0d, ae:-99.0, al:-99.0, au:-99.0}
input_data=replicate(dummy, max_no_rec)
files_read = 0
 
FOR i =  0, 2 DO BEGIN
    dir = path+papco_get_pathsep()+year_str+papco_get_pathsep()
    fln = dir+indices(i)+'_'+datestr+ext
    ;check if file exists
    r = findfile(fln, count = c1)
    IF c1 EQ 0 THEN BEGIN
        message, indices(i)+'_'+datestr+ext+ ' not found', /info
        CONTINUE
    ENDIF 
    dmy = {dec_hour:0.0, val:0.0}
    c = 0
    openr, u, fln, /get_lun
    WHILE NOT eof(u) DO BEGIN 
        readf, u, dmy
        hr = fix(dmy.dec_hour)
        min = fix((dmy.dec_hour - hr) * 60)
        sec = fix((dmy.dec_hour - hr - min/60.0) * 3600)
        IF i EQ 0 THEN BEGIN 
            tai = utc2tai({YEAR:year, MONTH:month, DAY:day, $
                           HOUR:hr, MINUTE:min, SECOND:sec, MILLISECOND:0})
            input_data(c).tai = tai
        ENDIF 
        input_data(c).(i+1) = dmy.val
        c = c+1
    ENDWHILE
    close, u & free_lun, u
    files_read = files_read + 1
ENDFOR

IF files_read EQ 0 THEN BEGIN
    get_err_no = 2
    get_err_msg = 'No data fiels found for '+datestr
    message, get_err_msg, /info
    return
ENDIF 

input_data = input_data(0:c-1)

; end of user written section
;------------------------------------------------------------------------------
   
; save read data with correct time. Rename data to *.idl
fln = path+'ae_al_au_'+datestr+'.idl'
message, 'Saving '+fln, /info
save, filename=fln, input_header, input_data, /COMPRESS, VERBOSE=VERBOSE
dealloc_lun,/quiet              ;frees logical units used by save/restore
return
  
out:
  ; if there was an error, report it. 
  get_err_no=!ERROR
  get_err_msg='!C '+!ERR_STRING

stop
   
END 
