;***************************************************************
; is_scalar_ae.pro
;***************************************************************

; check to see if a variable is a scalar or not
;
; given  : var               an array or scalar
; return : <function>  int   1 if a scalar, 0 if not

function is_scalar_ae, var

return, (size(var))(0) eq 0

end ; is_scalar_ae

;***************************************************************
;is_leap_ae.pro
;***************************************************************

;PURPOSE  :            To determine if a given year is
;                      a leap year
;
;INPUTS   :            int year (YYYY)    the year to be checked
;
;OUTPUTS  :            int result         1 if the year is leap
;                                         0 otherwise
;
;PROCEDURE:            a year is a leap year if it is divisible by
;                      4 or 400 but not 100. (i.e. 2000 is a leap
;                      year, 1900 is not)
;
;CALLING SEQUENCE:     result = is_leap_ae(year)
;
;AUTHOR          :     Doug Rowland    University of Minnesota
;                              School of Physics and Astronomy
;
;CREATED         :     10/15/96
;
;REVISION HISTORY:     N/A

function is_leap_ae,year

if (is_scalar_ae(year)) then begin
   if ((year mod 400) eq 0) then return,1
   if ((year mod 100) eq 0) then return,0
   if ((year mod   4) eq 0) then return,1
   return,0
endif else begin
   yearvec = intarr(n_elements(year))
      for i=0L,n_elements(year)-1 do begin
         if ((year(i) mod 400) eq 0) then yearvec(i) = 1
         if ((year(i) mod 100) eq 0) then yearvec(i) = 0
         if ((year(i) mod   4) eq 0) then yearvec(i) = 1
      endfor	 
   return,yearvec
endelse
end ; is_leap_ae
;********************************************************************
; doy_to_day_ae.pro
;
; convert day of year to consecutive days for 1990 and 1991.
; 1990 starts at day 0.0 and 1991 starts at day 365.0
;
; given  : doy          the day of the year
;          year         then year 1990 or 1991
; return : <function>   the day in 1990 or 1991

function doy_to_day_ae, doy, year

if (is_scalar_ae(doy)) then begin  
   ;convert doy to 'time' day and add 365 if in 1991 day = double(doy) - 1
   ;if (year eq 1991) then day = day + 365D
   day = day + 365D * (year - 1990)
endif else begin

   npts = n_elements(year)
   day = fltarr(npts)
      index = where(year ne 1990)
      day = doy - 1
      if (index(0) ne -1) then day(index) = day(index) + 365.*(year(index)-1990)
;   for i = 0L,npts-1 do begin
;      day(i) = double(doy(i)) - 1
;      if (year(i) eq 1991) then day(i) = day(i) + 365D
;   endfor

endelse

return, day

end ; doy_to_day_ae

;********************************************************************
;***************************************************************
;month_to_doy_ae.pro
;***************************************************************

;PURPOSE  :            find day of year from month, day of month,
;                      year
;
;INPUTS   :            int month (MM)   the month
;                      int   day (DD)   the   day
;                      int  year (YYYY) the year
;
;OUTPUTS  :            int doy   (DDD)  day of year
;                                       (Jan. 1 is doy 1)
;
;PROCEDURE:            consults look up table for days per month
;                      handles leap years using is_leap.pro
;
;CALLING SEQUENCE:     doy = month_to_doy_ae(month,day,year)
;
;AUTHOR          :     Doug Rowland    University of Minnesota
;                              School of Physics and Astronomy
;
;CREATED         :     10/15/96
;
;REVISION HISTORY:     N/A

function month_to_doy_ae,month,day,year

days    = [0,31,59,90,120,151,181,212,243,273,304,334]
days_lp = [0,31,60,91,121,152,182,213,244,274,305,335]

if (is_scalar_ae(year)) then begin
   if (is_leap_ae(year) eq 1) then begin
      use_days = days_lp
   endif else use_days = days

   doy = use_days(month - 1) + day
  
endif else begin 
   npts = n_elements(year)
   doy = fltarr(npts)
   yearvec = is_leap_ae(year)
   for i=0L,npts-1 do begin
      if(yearvec(i) eq 0) then begin
	 use_days = days
      endif else use_days = days_lp
      doy(i) = use_days(month(i) - 1) + day(i)
   endfor
endelse

return, doy

end ; month_to_doy_ae
;****************************************************************************
;*  NAME:        read_ae_ascii
;*
;*  BESCHREIBUNG:Routine zum Einlesen von ae-index dateien (binary)
;*
;*  INPUT:       common block xut1,xut2
;*
;*  OUTPUT:
;*
;*  DATUM:       September 1995
;*
;*  AUTOR:       R Friedel
;*
;*  NEBENEFFEKTE:
;*
;*  AENDERUNGEN: - R. Friedel October 1995  (added error-checking and the 
;*				             get_error common block)
;*                 the common block get_error contains an error number 
;*		   (get_err_no) and a corresponging message (get_err_msg)
;*                 0 : AE file read succesfully
;*                 1 : AE environmental variable (AE_DATA) for
;*		       data path not set
;*                 2 : other errors
;;****************************************************************************
pro read_ae_ascii

common get_error,get_err_no,get_err_msg
common aeascii,ae,au,al,ao,aetimemjd
common mjdt,mjdt_start,mjdt_end

get_err_no=2
get_err_msg='AE ASCII file not read'

c_path=papco_getenv('AE_ASCII_DATA')
if strlen(c_path) eq 0 then begin
   get_err_no=1
   get_err_msg='AE env. variable (AE_ASCII_DATA) for data path not set'
   print,get_err_msg
   return
endif

;use info in common block mjdt_start,mjdt_end to construct
;the list of filenames.  Files should contain ae,al,au,ao indices
;for one month.  If necessary, read more than one file to get
;whole time period.

convert_t90_to_date,mjdt_start,syear,sday_oy,shour,sminute,ssecond,$
   smonth,scmonth,sday_om,/MJDT
convert_t90_to_date,mjdt_end,eyear,eday_oy,ehour,eminute,esecond,$
   emonth,ecmonth,eday_om,/MJDT

tempmonth = smonth
tempyear  = syear
numfiles = (eyear-syear)*12. + (emonth-smonth) + 1.
filelist  = strarr(numfiles)

if (numfiles le 0) then begin
   get_err_no = 2
   get_err_msg='AE end time before start time'
   print,get_err_msg
   return
endif

filecnt = 0

for l=0,numfiles-1 do begin
   filelist(l) = 'ae'+string(tempyear,tempmonth,format='(i4.4,i2.2)')
   tempmonth = tempmonth + 1
   if (tempmonth gt 12) then begin
     tempmonth = 1
     tempyear = tempyear+1
   endif
endfor


for filecnt = 0,numfiles-1 do begin
  r_ae_ascii_onefile,c_path+filelist(filecnt) 
  if get_err_no ne 0 then return
  
    if (filecnt eq 0) then begin
       aebuf = ae & albuf = al & aobuf = ao & aubuf = au & aetimemjdbuf = aetimemjd
    endif else begin
       aebuf = [aebuf,ae] & albuf = [albuf,al] & aobuf = [aobuf,ao] & aubuf = [aubuf,au] & aetimemjdbuf = [aetimemjdbuf,aetimemjd]
    endelse
endfor

ae = aebuf & al = albuf & ao = aobuf & au=aubuf & aetimemjd = aetimemjdbuf


END

;******************************************************************************
pro r_ae_ascii_onefile,fln

common aeascii,ae,au,al,ao,aetimemjd
common get_error,get_err_no,get_err_msg

on_ioerror,read_original
spawn,'gunzip '+fln+'.idl'
restore,fln+'.idl'
spawn,'gzip '+fln+'.idl'
get_err_no = 0
get_err_msg = 'AE ASCII idl save file '+fln+' restored'
dealloc_lun,/quiet
return

read_original:
    spawn,'gunzip '+fln  
    get_lun,unit 
    openr, unit,fln,ERROR=err
    
    if (err ne 0) then begin
      print,fln+' ; ASCII not available.'
      get_err_no = 2
      get_err_msg = 'AE ASCII not available'
      return
    endif  
    if (err eq 0) then begin
      get_err_no = 0
      get_err_msg = 'AE ASCII file read successfully'
    endif
     
    r_ae_ascii_onefile_original,unit,aetimemjd,ae,al,ao,au 
    save,aetimemjd,ae,al,ao,au,filename=fln+'.idl'
    spawn,'gzip '+fln,aespajunk
    spawn,'gzip '+fln+'.idl'
    message,'gzipped data file '+fln, /cont

END

;******************************************************************************
pro r_ae_ascii_onefile_original,unit,timemjdo,aeo,alo,aoo,auo
    
;assume maximum number of nrecs = 24 * 31 *60 = 44640, around 45000
  npoints = 45000  &  minutes = indgen(60)  &  minvals = lonarr(60) & name=''
  
;make time array. Read first line of file for this.
  point_lun,unit,0  
  readf,unit,year,month,day,hr,name,minvals,hrval, $
    format='(12X,I2,I2,I2,1X,I2,A2,11X,60I6,I6)' 
  
  thismonth=month  &  i=0l
  start_mjdt=convert_date_to_t90(day=1,month=month,year=year,/MJDT)
  add_mjdt=start_mjdt  &  add_mjdt.mjd=1  &  add_mjdt.t=0
  day_sec=findgen(1440)*60.0
  timemjdo = replicate(start_mjdt,npoints)
  convert_t90_to_Date, start_mjdt, year, day_oy, hour, minute, second, $
    thismonth, cmonth, day_om, /MJDT
    
  while thismonth eq month do begin
    print, year,thismonth,month,day_om,hour,minute,second    
    timemjdo(i:i+1439).mjd=start_mjdt.mjd
    timemjdo(i:i+1439).t=day_sec ;fill this day with seconds
    start_mjdt=addSubTime(start_mjdt,add_mjdt,/ADD)
    convert_t90_to_Date, start_mjdt, year, day_oy, hour, minute, second, $
      thismonth, cmonth, day_om, /MJDT
    i=i+1440
  endwhile
  timemjdo=timemjdo(0:i-1)
  
  aeo=lonarr(i)  &  auo=lonarr(i)  &  alo=lonarr(i)  &  aoo=lonarr(i)

  point_lun,unit,0  
  while not eof(unit) do begin 
    readf,unit,year,month,day,hr,name,minvals,hrval, $
      format='(12X,I2,I2,I2,1X,I2,A2,11X,60I6,I6)' 
    ;work out start /end points based on time
    startpoint=(day-1)*1440l+hr*60l
    endpoint=startpoint+59
    
    print,name,' ',year,month,day,hr
    
    case name of
      'AE': aeo(startpoint:endpoint) = minvals
      'AL': alo(startpoint:endpoint) = minvals
      'AO': aoo(startpoint:endpoint) = minvals
      'AU': auo(startpoint:endpoint) = minvals
     endcase    
    
  endwhile      

  close, unit  &  free_lun, unit
  
end
