;******************************************************************************
;* PROCEDURE:     
;*      p_aero_heo_line, panel, type, OUTPUT=OUTPUT, $
;*           OVERPLOT=OVERPLOT,$
;*           PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           SUBTABLE=SUBTABLE, $
;*           _EXTRA=extra_par
;* 
;* DESCRIPTION:  
;*	This procedure plots data of type data_type for inclusion in papco.  
;*      Data must have been read and placed into common blocks before this
;*      routine is called. Behavior is controlled by the vector panel
;*	which controls positioning of the plot (see papco_conventions.text
;*      in $papco). In general, the following plot conventions are used:
;*              - if the panel is at the bottom plot time axis plus any
;*                additional axis (ie ephemeris) needed.
;*              - if panel is at the top also plot a title
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a papco_colorbar.
;*              - panels all have common time
;*              - y-scaling (or z-scaling for color plots) is either
;*              - automatic or manualy set using common yscale or zscale   
;*
;* INPUTS:       
;*      panel   three element integer vector. Controls positioning of the plot
;*              (see papco_conventions.text in $papco)
;*              
;* OUTPUTS: 
;*      none
;*
;* KEYWORDS:
;*      OUTPUT  if set call the papco routine that outputs plot data
;*              to a file.
;*    OVERPLOT  if you support this you can alter the appearance of your
;*              plot if it is an overplot to an existing panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mag. field.
;*              see p_crres_mag for an example.
;*    PLOTS_ATTHISPOSITION  used to control the left side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and line style.   
;* 
;* CALLING SEQUENCE:
;*       p_aero_heo_line, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY:       
;******************************************************************************
pro p_aero_heo_line, panel, plotinfo, OUTPUT=OUTPUT, OVERPLOT=OVERPLOT, $
                     PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION, $
                     SUBTABLE=SUBTABLE, _EXTRA=extra_par

; The following common blocks are needed:
COMMON mjdt, mjdt_start, mjdt_end               ;time limit in mjdt
COMMON yscale, yscl                             ;man/auto yscaling
COMMON zscale, zscl	       	                ;man/auto zscaling
COMMON shift_label, down                        ;for x-axis label shifting
COMMON coordinateSystems, plotted_x, plotted_y  ;info on coords

; One of the common blocks should contain the plot data as returned by the
; data read procedure r_aero_heo. 
COMMON aero_heo_data, input_header, input_data
COMMON aero_heo

COMMON aero_heo_slice, time, ymat, yarr, extra_plotPar, utitle, uytitle
  
COMMON papco_color_names  
color_indx=[black, red, green, blue, magenta, cyan, $
            burgundy, olive, dark_green, teal, royal_blue, violet]

IF keyword_set(OUTPUT) THEN output = OUTPUT ELSE output = 0

;restrict data to actual time range requested - this makes for faster zooming
;input data is in TAI. For plotting, always start at zero to get maximum
;resolution (plot works in float only, large values get rounded!) 
tai_1=utc2tai({mjd:mjdt_start.mjd, time:mjdt_start.t*1000})
tai_2=utc2tai({mjd:mjdt_end.mjd, time:mjdt_end.t*1000})

index=where((input_data.tai ge tai_1) AND (input_data.tai le tai_2),c)
if c ne 0 then data=input_data(index) ELSE GOTO, noplot
xut1=0  &  xut2=tai_2-tai_1 & time = data.tai-tai_1

control = *plotinfo.USR_PTR1
sat_name = aero_heo_sats(control.SAT_ID)
nodata = -99.0

; make index of channels to be plotted - from binary number in control.product 
n_ch = n_elements(aero_heo_products) & to_plot=bytarr(n_ch)
FOR i=0, n_ch-1 DO IF (control.product and 2l^i) EQ 2l^i THEN to_plot(i)=1
pl_idx=where(to_plot eq 1, nplot)

;make a data ymat of all channels to be plotted
ndat = n_elements(data)
ymat = fltarr(ndat, nplot)
tags = tag_names(data(0))
FOR i = 0, nplot-1 DO BEGIN
    idx = where(aero_heo_products(pl_idx(i)) EQ tags)
    ymat(*, i) = data.(idx(0))
ENDFOR
idx = where(ymat le 0,c) & if c ne 0 then ymat(idx)=nodata

utitle = 'aero_heo'

CASE control.cal OF
    0: BEGIN
        uytitle = 'counts/s'
    END
    1: BEGIN
        uytitle = 'integral flux'
        ;here flux is counts div by geometric factor, giving integral flux
        FOR i = 0, nplot-1 DO $
          ymat(*, i) = (ymat(*, i) / aero_heo_G(pl_idx(i)))/aero_heo_ecal(i)
    END
    2: BEGIN
        uytitle = 'integral flux'
        ;here flux is counts div by geometric factor, giving integral flux
        FOR i = 0, nplot-1 DO ymat(*, i) = ymat(*, i) / aero_heo_G(pl_idx(i))
        message,' Diff flux not yet implemented',/info
    END    
ENDCASE

yarr = aero_heo_ech

IF output EQ 2 THEN return
panelset,panel			;sets the panel position viewport

; now do the actual data plot, data only, no axis

CASE control.graph OF
    0: BEGIN                    ;line
        rot = 0
        ;set y scaling
        IF (yscl(panel(0),0) EQ 1) THEN BEGIN 
            yst=yscl(panel(0),1)  &   yen=yscl(panel(0),2)
        ENDIF ELSE BEGIN
            papco_autorange, ymat, yst, yen, $
              log=zscl(panel(0),3), exclude=0, nodata=nodata
            yscl(panel(0),1)=yst  &  yscl(panel(0),2)=yen
        ENDELSE       

        papco_y_label, yst, yen, log=yscl(panel(0),3)

        extra_plotPar={xrange:[xut1,xut2], yrange:[yst,yen],  $
                       ylog:yscl(panel(0),3), xtickformat:'noticks'}
   
        ; add keyword structure set here with the one passed in
        extra_plotPar=create_struct(extra_plotPar, extra_par)

        ; use papco routine to draw time axis. 
        down=0
        papco_draw_time_axis, panel, OVERPLOT=OVERPLOT, _extra=extra_plotPar

        FOR i = 0, nplot-1 DO BEGIN
            extra_plotPar.color=color_indx(i mod n_elements(color_indx))
            c=i/5 & mult=i mod 5 
            name = aero_heo_products(pl_idx(i))
            IF i eq 0 then channels=[name] ELSE channels=[channels, name]
            IF (nplot NE 1) THEN  BEGIN 
                FOR j=1,c DO  name='!C'+name
                xyouts,!P.position(0)+0.01+0.135*mult, $
                       !P.position(3)-0.015, name, /normal, $
                       charsize=!p.charsize, color=extra_plotPar.color
            ENDIF   
             
            index=where(ymat(i,*) LT 0,c) & IF c ne 0 then ymat(i,index)=-99

            papco_gap_plot, time, ymat(*, i), nodata,  xstyle = 5, ystyle = 5,$
                            _extra=extra_plotPar           

        ENDFOR   

    END

    1: BEGIN                    ;spec
        

    END

    2: BEGIN                    ;L-spec
        rot = 1
        
        ;choose first channel set
        ch_idx = pl_idx(0)
        
        ;set y scaling - preset L range as 1 to 12
        IF (yscl(panel(0),0) EQ 1) THEN BEGIN 
            yst=yscl(panel(0),1)  &   yen=yscl(panel(0),2)
        ENDIF ELSE BEGIN
            yst = 1.0 & yen = 12.0
            yscl(panel(0),1)=yst  &  yscl(panel(0),2)=yen
        ENDELSE     

        papco_y_label, yst, yen, log=yscl(panel(0),3)

        papco_make_Lvt, time, ymat(*, ch_idx), data.LSHELL, yst, yen, $
                        Y_arr, T_cut, Zarr       
        stop

       ;set z-scaling
        if (zscl(panel(0),0) eq 1) then begin
            zst=zscl(panel(0),1)  &  zen=zscl(panel(0),2)
        endif else begin
            papco_autorange, Zarr, zst, zen, $
              log=zscl(panel(0),3), exclude=0, nodata = nodata
            zscl(panel(0),1)=zst  &  zscl(panel(0),2)=zen   
        ENDELSE 

        extra_plotPar={xrange:[xut1,xut2], yrange:[yst,yen], zrange:[zst,zen],$
                       ylog:0, zlog:zscl(panel(0),3), $
                       xtickformat:'noticks', $
                       ztickformat:'papco_color_bar_log_ticks', ztitle:uytitle}

        ;add keyword structure set locally and the one passed in
        extra_plotPar=create_struct(extra_plotPar, extra_par)

       ; do color plot and color bar plot. nodata specifies the "nodata" flag
       ; in your data. Here we do not plot axis and data seperate, as
       ; papco_plot_colorspec cannot be used as an overplot!
       resx=2 & dx = 4
       n_ch = n_elements(Y_arr)/2

       IF yscl(panel(0),3) THEN resy=(alog10(yen)-alog10(yst))/n_ch / 10 ELSE $
           resy=((yen)-(yst))/n_ch / 10

       papco_plot_colorspec, zarr, t_cut,y_arr, nodata=nodata, $
             resx=resx, resy=resy, $
             dx = dx, xstyle=5, ystyle=5, _extra=extra_plotPar
 
       ; store the coordinate information into common block
       plotted_x = !x  &  plotted_y = !y

       down=0
       papco_draw_time_axis, panel, OVERPLOT=OVERPLOT, _extra=extra_plotPar 
  
       papco_color_bar, _extra=extra_plotPar

       uytitle = 'L'

    END

ENDCASE   

; plot y-axis label at left of plot. Use scalable routine!  
left_side_label,panel, uytitle,  /rot90

; store the coordinate information into common block for slices
plotted_x = !x  &  plotted_y = !y

; plot info right of the panel. If you want overplots to have their own labels
; you need to add position-control for this label using the keyword
; PLOTS_ATTHISPOSITION
if not keyword_set(overplot) then begin
    right_side_label,panel,utitle, ROT = ROT, _extra=extra_Par
endif else begin
    for i=0, PLOTS_ATTHISPOSITION-1 do utitle='!C'+utitle
    right_side_label,panel,utitle, ROT = ROT, _extra=extra_Par
endelse
   
; check if data needs to be output to file. Use the optional keywords
; to add more information to the data file
;      DESCRIPTION   a string or string array describing the data
;      TIME_LABEL    a string decribing the x-axis format
;      CHANNELS      an (ch,2) array giving the channel start/end
;      Y_LABEL       a string with the y-axis label
;      Z_LABEL       a string with the z-axis label
;      RS_LABEL      a string with the righ-side label of the plot
  
if KEYWORD_SET(OUTPUT) then begin
    print,'% p_aero_heo_line: Writing plot data out to file'
    description='aero_heo template sample data - single energy channel line'
    time_label='Seconds since start of day'
    ;channels=yarr
    y_label=uytitle
    ;z_label=uztit
    rs_label=utitle
    papco_write_data, time, yray, $
                      DESCRIPTION=description, TIME_LABEL=time_label, $
                      Y_LABEL=y_label, $
                      RS_LABEL=RS_LABEL
ENDIF   

return

;jump to here if no data to plot 
noplot: 
  message,'No data to plot', /cont
  plotted_x = !x  &  plotted_y = !y
 
END  
