;******************************************************************************
;*  NAME:        r_aero_heo
;*
;*  DESCRIPTION: Routine to read aero_heo files. Adheres to
;*               papco conventions: Error common block
;*                                  Uses MJDT time
;*                                  papco read philosophy
;*
;*  INPUT:       none (uses common block time from papco)
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  KEYWORDS:    VERBOSE  -  enables print comments, save/restore comments
;*
;*  DATE:        June 1996
;*
;*  AUTOR:       R. Friedel
;*
;*  CHANGES:     February 1997. Used MJDT time now, and is a fully
;*               running example module on it's own
;*               October 1997. Rewritten to be more general and fully
;*               adhering to papco philosophy. Added remore Get feature, and
;*               the example data file is no longer part of the 
;*               module - it needs to be gotten remotely first!
;******************************************************************************
pro r_aero_heo, plotinfo, VERBOSE=VERBOSE

common aero_heo_data, input_header, input_data
common mjdt, mjdt_start, mjdt_end
common get_error, get_err_no, get_err_msg
COMMON aero_heo ;defined in aero_heo_commons.pro

; common get_error is used by papco to check if the data read was successful.
; it is up to the user to set this correctly and to make sure that the read- 
; routine does not crash on an I/O error - otherwise papco will crash at that
; point, too. papco draws an empty panel when get_err_no in not equal to zero
; and prints get_err_msg in that panel for user information. papco will try
; to invoce the plot routine for this data if get_err_no = 0 , otherwise an
; emplty panel with the error message get_err_msg is plotted.

get_err_no=0  &  get_err_msg='aero_heo read succesfully'
   
; check env. variable pointing to data directory
if not (papco_check_data_env('AERO_HEO_DATA', PATH=path)) then return
   
; check VERBOSE keyword and set
if keyword_set(VERBOSE) then verbose=VERBOSE else verbose=0     
   
; clear data arrays for reuse
input_header=0  &  input_data=0   
   
;get control variables for this module
control = *plotinfo.USR_PTR1

; loop through from start to end time concatonating data files into one
; array. HEO data in in file per orbit, several fiels per day.
; file format is fN_YYYYDDD_HHMM
; year, day of year, start time of orbit, N- HEO number

;make list of files to get
mjdt_time =  mjdt_start

path = path+'exp/'+strlowcase(aero_heo_sats(control.sat_id))

cd, path, current = cur_path
FOR i = mjdt_start.mjd-1, mjdt_end.mjd DO BEGIN
   mjdt_time.mjd = i
   IF (mjdt_time.mjd EQ mjdt_end.mjd) AND (mjdt_end.t) EQ 0 THEN break
   convert_t90_to_date, mjdt_time, yr, doy, hr, min, sec, mon, cmon, dom, $
      /mjdt
   datestr = string(yr, doy, format = "(i4.4,i3.3)")
   res = findfile('*'+datestr+'*', count = n)
   IF NOT keyword_set(files) THEN files = res ELSE files = [files, res]
ENDFOR
cd, cur_path

;loop through file to read
no_files_read=0
FOR i = 0, n_elements(files)-1 DO BEGIN
    ;read one day of data
    r_aero_heo_onefile, files(i), VERBOSE=VERBOSE
    ;if a file was succesfully read, concatonate!
    if get_err_no eq 0 then begin 
      if no_files_read eq 0 then input_data_buf=temporary(input_data) $
      else input_data_buf=[temporary(input_data_buf),temporary(input_data)]
      no_files_read=no_files_read+1
    endif
ENDFOR
   
; check if any files were read   
if (no_files_read ne 0) and (get_err_no ne 1) then begin
    get_err_no=0
    get_err_msg=string(no_files_read,format="(i2)")+ ' files read succesfully'
    ;clear arrays and load buffer into output array
    input_data=0  &  input_data=temporary(input_data_buf)
endif else begin
    get_err_no=1 & get_err_msg='No files found for timerange'
    message, get_err_msg, /cont 
    return
endelse      

END 

;******************************************************************************
;*  NAME:        r_aero_heo_onefile,fln
;*
;*  DESCRIPTION: reads one file of type plot_file. Assumes data has been
;*               stored in IDL saveset using structures input_header and
;*               input_data
;*
;*  INPUT:       fln  - string of filename
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        October 1997
;*
;*  AUTOR:       R. Friedel
;******************************************************************************
pro r_aero_heo_onefile, fln, VERBOSE=VERBOSE
  
common aero_heo_data, input_header, input_data
common get_error, get_err_no, get_err_msg
   
get_err_no=0  &  get_err_msg='aero_heo onefile read succesfully'
   
; check env. variable pointing to data directory
if not (papco_check_data_env('AERO_HEO_DATA', PATH=path)) then return

; make / set directory for IDL savesets
path = path+'savesets/'
IF NOT papco_finddir(path) THEN BEGIN
    message, 'Creating directory for IDL saveset:', /cont
    print, '  '+ path
    papco_spawn, 'mkdir '+path
ENDIF

; clear data arrays for reuse
input_header=0  &  input_data=0

on_ioerror, read_original
restore,path+fln+'.idl', VERBOSE=VERBOSE
dealloc_lun,/quiet   ;frees logical units used by save/restore
message,'saveset read '+ fln, /cont
return

read_original:       ;if the .idl file is not found, try to read the original
r_aero_heo_onefile_original, fln, VERBOSE=VERBOSE
      
END 
  
;******************************************************************************
;*  NAME:        r_aero_heo_onefile_original,fln
;*
;*  DESCRIPTION: reads one file of type plot_file_original. Under the
;*               papco read philosophy, files should be read the first
;*               time in whatever format they might have - ascii,
;*               fortran binary, pascal binary, etc. The files might also
;*               be in some compressed format, so they need to be uncompressed
;*               first. 
;*               
;*               You might also have to change time formats. 
;*               A lot of files are stored per day, and have seconds of
;*               that day as the time. This is what is assumed here and
;*               is papco standard. You need to convert the time
;*               to papco standard if you intend to use the plotroutines 
;*               supplied with papco (use the routines in
;*               $papco_BASE/papco_Lib/time_convert.pro). 
;*               If you use your own plot routines this step is not needed.
;*               
;*               The first read might be slow, esp if the format is ascii.
;*               Use the IDL "save" command to make a binary copy of the
;*               file which can then be read quickly using "restore". The
;*               next time only the IDL saveset is read, which is much
;*               faster.
;*               
;*               In order to be able to concatonate the arrays easily,
;*               it makes sense to have the data in an array of time-ordered
;*               structures, which makes adding together days of data
;*               trivial using the IDL concatonation operator "[]".
;*               
;*  INPUT:       fln  - string of filename
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        October 1997
;*
;*  AUTOR:       R. Friedel
;******************************************************************************
pro r_aero_heo_onefile_original, fln, VERBOSE=VERBOSE
  
common aero_heo_data, input_header, input_data
common get_error, get_err_no, get_err_msg  
   
get_err_no=0  &  get_err_msg='aero_heo_onefile_orignal read succesfully'
   
; check env. variable pointing to data directory
if not (papco_check_data_env('AERO_HEO_DATA', PATH=path)) then return

; clear data arrays for reuse
input_header=0  &  input_data=0   
   
;------------------------------------------------------------------------------
; read the original data file - THIS NEEDS TO BE REWRITTEN BY USER!!!!
  
; define the header and data structures

input_header={aero_heo_head,header_str:'', ndat:0, nch:0, sat_id:''}

;read header of ascii data file and make structure
header='' & on_ioerror, out  
openr,unit,path+fln,/get_lun
IF keyword_set(VERBOSE) THEN BEGIN 
    message,'Reading ascii file...(slow)...',/cont
    print,  path+fln
ENDIF

readf, unit, header
;fix some of the header strings so they can be made into TAGS
header = strsubst(header, ' Status', '_Status')
header = strsubst(header, ' Long', '_Long')
header = strsubst(header, ' Lat', '_Lat')
header = strsubst(header, ' X', '_X')
header = strsubst(header, ' Y', '_Y')
header = strsubst(header, ' Z', '_Z')
header = strtrim(strcompress(header), 2)
header_list = strsplit(header, ' ', /EXTRACT)
input_header.nch = n_elements(header_list)

;make the "create_struct" function call needed to make the structure!
initval=fltarr(input_header.nch)
call_prm="'"+header_list(0)+"',"+ string(initval(0),format="(f3.1)")
result=EXECUTE('dummy=create_struct('+call_prm+')')
FOR i = 1, input_header.nch-1 DO BEGIN
    call_prm="dummy,'"+header_list(i)+"',"+string(initval(i),format="(f3.1)")
    result=EXECUTE('dummy=create_struct('+call_prm+')')
ENDFOR

dummy2=create_struct(name='aero_heo_data', dummy, 'TAI',0.0d)

;make enough data array for one orbit's worth of data
input_data=replicate(dummy2, 6000)

;now read all data into input_data
ndat = 0 
WHILE NOT eof(unit) DO BEGIN
    readf,unit,dummy
    ;add time tag for TAI and compute TAI. HEO time is year, decimal doy
    dummy2=create_struct(name='aero_heo_data', dummy, 'TAI',0.0d)
    year = dummy.YEAR & doy = fix(dummy.DECDAY)
    utc = doy2utc(doy ,year)
    utc.time = 86400000 * (dummy.DECDAY-fix(dummy.DECDAY))
    dummy2.tai=UTC2TAI(utc)
    input_data(ndat)=dummy2
    ;print, ndat, dummy2.(0), dummy2.(1), dummy2.tai, TAI2UTC(dummy2.tai, /ECS)
    ndat = ndat+1
ENDWHILE 

input_header.ndat = ndat
input_data = input_data(0:ndat-1) 

; end of user written section
;------------------------------------------------------------------------------
   
; save read data with correct time. Rename data to *.idl
save,filename=path+'savesets/'+fln+'.idl',input_header,input_data, $
  /compress, VERBOSE=VERBOSE
dealloc_lun,/quiet              ;frees logical units used by save/restore
return

out:
  ; if there was an error, report it. 
  get_err_no=!ERROR
  get_err_msg='!C '+!ERR_STRING
   
end

;******************************************************************************
;make heo ephemeris files, calculate magnetic coordinates
PRO m_aero_heo_eph, FORCE=FORCE, INTMOD=INTMOD, EXTMOD=EXTMOD, PITCH=PITCH, $
                    VERBOSE=VERBOSE, NOREAD = NOREAD,  SAVE = SAVE, $
                    COMPARE=COMPARE

common aero_heo_data, heo_header, heo_data
common heo_eph_data, input_header, input_data
common mjdt, mjdt_start, mjdt_end
common get_error, get_err_no, get_err_msg

; check keywords and set
if keyword_set(VERBOSE) then verbose=VERBOSE else verbose=0     
if keyword_set(INTMOD) then kint=long(INTMOD) else kint=0l
if keyword_set(EXTMOD) then kext=long(EXTMOD) else kext=0l
if keyword_set(PITCH) then alpha=long(PITCH) else alpha=90.0d

;set up plot window for comparisons
if keyword_set(COMPARE) then BEGIN
    IF compare NE 2 THEN BEGIN 
        window,15,title='Compare Mag Coords', $
        YSIZE=600,XSIZE=800,xpos=60,ypos=1024-600  &  wset,15  
    ENDIF 
endif  

;make new mag field coord structure
new_struct_name='MODEL_'+varprt(kint)+'_'+varprt(kext)
new_struct=create_struct(name='UNILB', 'L',0.0,'MLT',0.0,'MLAT',0.0)  
  
;see if ephemeris file exists. if not, make it.
IF NOT keyword_set(NOREAD) THEN r_heo_eph, VERBOSE=VERBOSE

IF get_err_no NE 0 THEN BEGIN
    ;read original heo data file
    IF NOT keyword_set(NOREAD) THEN r_aero_heo, VERBOSE=VERBOSE
    IF get_err_no NE 0 THEN BEGIN
        message, 'No HEO data found', /cont
        return
    ENDIF ELSE BEGIN
        tai_1 = utc2tai({mjd:mjdt_start.mjd, time:mjdt_start.t*1000.0d}) 
        tai_2 = utc2tai({mjd:mjdt_end.mjd, time:mjdt_end.t*1000.0d}) 
        index=where((heo_data.tai ge tai_1) AND (heo_data.tai le tai_2),c)
        if c ne 0 then data=heo_data(index) ELSE BEGIN
            message, 'No HEO data found for timerange', /cont
            get_err_no = 1
            return
        ENDELSE
        ;make a structure of tags to adopt from HEO file
        dummy = {TAI:0.0d, GEO_LONG:0.0, GEO_LAT:0.0, RADDIST:0.0, $
                LSHELL:0.0, INVLAT:0.0, MLT:0.0, MLAT:0.0}
        new_tags = tag_names(dummy) & heo_tags=tag_names(data(0))
        input_data = replicate(dummy, n_elements(data))
        FOR i = 0, n_elements(new_tags)-1 DO BEGIN
            idx = where(heo_tags EQ new_tags(i), c)
            IF c EQ 1 THEN input_data.(i) = data.(idx(0))
        ENDFOR     
        input_header = heo_header
    ENDELSE 
ENDIF

;check for existing tags, work out what needs to be done    
tags=tag_names(input_data(0))
; check if required mag coords exist in file.
index=where(tags eq new_struct_name,c)
if c eq 0 then BEGIN
    need_to_do=1
    dummy2=create_struct(input_data(0),new_struct_name,new_struct)
    dummy3=replicate(dummy2,n_elements(input_data)) 
    ;fill new array with copies of existing array
    for j=0,n_elements(data)-1 do $ 
        for m=0,n_elements(tags)-1 do dummy3(j).(m)=input_data(j).(m)
    input_data=temporary(dummy3)
endif else need_to_do=0

;get the index of the new mag coords tag
tags=tag_names(input_data(0))
idx=where(tags eq new_struct_name) & new_idx=idx(0)    

;calculate new magnetic coords from UNILB library
IF need_to_DO OR keyword_set(FORCE) THEN BEGIN
    message,' calculating L, MLT, MLAT for '+new_struct_name, /cont
      

    ;setup unilib shareab;e object location
    lib_dir = '/n/toaster/u/friedel/idl/fortran/UNILIB/ver_2.06/source'
    lib_name = lib_dir+'/unilib.so'
    entry_name = 'uxidl_'

    ;initialize UNILIB library
    ;MOD by RF to UT990: set kinit to 533 to disable magnetopause checking
    kunit = 533l & kinit = 1 & iver = 0l
    result = call_external(lib_name, entry_name,'UT990', $
                            kunit, kinit, iver)
  
    ;select a internal geomagnetic field model
    t = TAI2UTC(input_data(0).TAI, /EXTERNAL )
    year = double(t.year) & lbint = string('',format="(a32)") & ifail = 0l
    result = call_external(lib_name, entry_name,'UM510', $
                           kint, year, lbint, kunit, ifail)

    FOR i = 0, n_elements(input_data)-1 DO BEGIN

        ;compute modified Julian date as required - idl roputines.
        ;NOTE: tmjd is modified Julian date: JD - 240000.5
        ;amjd is modified Julian Day based on the 1st of January 1950
        ;tmjd = TAI2UTC(input_data(i).TAI)
        ;tmjd2amjd = doy2utc(1,1950)
        ;amjd = double(tmjd.mjd-tmjd2amjd.mjd)+double(tmjd.time)/86400000.0d

        ;compute modified Julian date as required
        t    = TAI2UTC(input_data(i).TAI, /EXTERNAL )
        mdate = {iyear:long(t.year), imonth:long(t.month), iday:long(t.day), $
                 ihour:long(t.hour), imin:long(t.minute), $
                 secs:double(t.second), amjd:0.0d}
        result = call_external(lib_name, entry_name,'UT540', mdate)

        ;select an external magnetic field model
        param = dblarr(10) & lbext = string('',format="(a32)")

        result = call_external(lib_name, entry_name,'UM520', $
                               kext, mdate.amjd, param, lbext, kunit, ifail)
         
        ;convert the location from geodetic to geocentric coordinates
        Re_heo = 6378.14d
        mgde = {radius:double(input_data(i).RADDIST)*Re_heo, $
                colat:90.0d - double(input_data(i).GEO_LAT), $
                elong:double(input_data(i).GEO_LONG)}
        mgeo = {radius:0.0d, colat:0.0d, elong:0.0d}
        result = call_external(lib_name, entry_name,'UM536', $
                               mgde, mgeo)

        ;evaluate the magnetic field. This also updates 
        ;the Sun position and the SM coordinates (done by UM522 and UM524)
        mb = {dnrm:0.0d, rho:0.0d, theta:0.0d, phi:0.0d}
        result = call_external(lib_name, entry_name,'UM530', $
                               mgeo, mb, ifail)   

        ;evaluate the (B, L) coordinates
        nfbm = 1l
        fbm = 0.0d & flm = 0.0d & fkm = 0.0d & fsm = 0.0
        fbeq = 0.0d & fs = 0.0d
        result = call_external(lib_name, entry_name,'UL220', $ 
                               mgeo,alpha,nfbm,fbm,flm,fkm,fsm,fbeq,fs,ifail)

        ;evaluate the magnetic local time and latitude
        xmlt = 0.0d &  xlat = 0.0d
        result = call_external(lib_name, entry_name,'UM538', $
                               mgeo, mdate.amjd, xmlt, xlat, ifail)

        IF verbose EQ 2 THEN begin
            print, TAI2UTC(input_data(i).TAI,  /ECS), i
            print, 'DateTime,L value,MLT, MLAT: ', flm, xmlt, xlat
            print, 'Heo file values             ', $
                   double(input_data(i).LSHELL), $
                   double(input_data(i).MLT), double(input_data(i).MLAT)
        ENDIF 

        input_data(i).(new_idx).L=flm  
        input_data(i).(new_idx).mlt=xmlt 
        input_data(i).(new_idx).mlat=xlat

    ENDFOR
   
         
    IF keyword_set(SAVE) THEN BEGIN
        ; check env. variable pointing to data directory
        if not (papco_check_data_env('AERO_HEO_DATA', PATH=path)) then return
        fln = path+'eph/'+save
        save, filename = fln, input_header, input_data, $
          /compress, verbose = verbose
    ENDIF 

ENDIF ELSE BEGIN
    message,  'Mag coords exist for: '+new_struct_name, /cont
ENDELSE 

IF keyword_set(COMPARE) then begin
    !P.multi=0 & !P.position=0 & !P.region=0 & erase
    !P.multi=[0,2,2,0,0]
    plot,input_data.LSHELL, yr=[0,30], ytitle='L-value', col=1
    oplot, input_data.(new_idx).L
    !P.multi=[2,2,2,0,0]
    plot,input_data.MLT, yr=[0,24], ytitle='MLT',col=1
    oplot, input_data.(new_idx).mlt
    !P.multi=[1,2,2,0,0]
    plot,input_data.MLAT, yr=[-90,90], ytitle='Mag Lat',col=1
    oplot, input_data.(new_idx).mlat
ENDIF  

close, /all

END

;******************************************************************************
;make heo ephemeris files, calculate magnetic coordinates
PRO m_aero_heo_eph_batch, INTMOD=INTMOD, EXTMOD=EXTMOD, PITCH=PITCH, $
                          VERBOSE=VERBOSE, DATE_1=DATE_1, DATE_2=DATE_2, $
                          ADOPT_MJDT = ADOPT_MJDT, FORCE=FORCE, $
                          COMPARE = COMPARE

common mjdt, mjdt_start, mjdt_end

IF NOT keyword_set(ADOPT_MJDT) THEN BEGIN 

    IF keyword_set(DATE_1) then date_1=DATE_1 else date_1='19971227'
    day=fix(strmid(date_1,6,2)) & month=fix(strmid(date_1,4,2)) 
    year=fix(strmid(date_1,0,4))
    mjdt_1=convert_date_to_t90(day=day, month=month, year=year, /MJDT)
    mjdt_start=mjdt_1
    
    IF keyword_set(DATE_2) then date_2=DATE_2 else date_2='20010601' 
    day=fix(strmid(date_2,6,2)) & month=fix(strmid(date_2,4,2))
    year=fix(strmid(date_2,0,4))
    mjdt_2=convert_date_to_t90(day=day, month=month, year=year, /MJDT)
    mjdt_end=mjdt_2

ENDIF

keep_mjdt_start = mjdt_start & keep_mjdt_end = mjdt_end

;loop through period in step day intervals. 
step = 1
FOR i=keep_mjdt_start.mjd, keep_mjdt_end.mjd, step DO BEGIN 

    mjdt_start.mjd=i    & mjdt_start.t = 0
    mjdt_end.mjd=i+step & mjdt_end.t = 0
    
    tai = utc2tai({mjd:mjdt_start.mjd, time:mjdt_start.t}) 
    t   = TAI2UTC(tai, /EXTERNAL )
    
    save = string(t.year, t.month, t.day, $
                  format = "(i4.4,i2.2,i2.2,'_heo_f3_eph.idl')")
    print, ''
    message, 'Doing for: '+save, /cont
    
    save = 0

    m_aero_heo_eph, FORCE=FORCE, INTMOD=INTMOD, EXTMOD=EXTMOD, PITCH=PITCH, $
      VERBOSE=VERBOSE, NOREAD = NOREAD,  SAVE = SAVE, $
      COMPARE = COMPARE

    IF keyword_set(compare) THEN compare = 2

ENDFOR 

mjdt_start = keep_mjdt_start & mjdt_end = keep_mjdt_end

END 
