;******************************************************************************
;* FILE: ampte_cce_panel.pro
;*
;* DESCRIPTION: 
;*    This is a demonstration on how to implement a papco-panel-editor.
;*    It makes use of all the features provided by $papco_XX/papco_panel.pro
;*    (a collection of standard papco widgets). It constructs an editor for a
;*    very simple papco-plot, that has all the default-functionality.
;*
;* COMMON-BLOCKS:
;*    ampte_cce_panel_data   contains the window-information
;*
;* STRUCTURES:
;*    This widget inherits a PlotInfo structure as an argument. This
;*    structure is then modified by the user according to his/her needs:
;*    Either by including standard PAPCO widgets from papco_panel.pro,
;*    which extend this structure, or by modifying its tags. The modified
;*    structure is then handed back to PAPCO.
;* 
;*    The PlotInfo structure is created by a call to papco_getPlotInfoStruct
;*    and contains a series of tags which are used to describe a given panel.
;*    These tags come in three categories:
;*       1. Tags used by PAPCO internally - don't touch!          
;*       2. Tags added by including standard PAPCO widgets. You can edit 
;*          or preset these tags if you wish.
;*       3. User Tags for control of user-specified features. There
;*          are enough user tags specified to handle most needs. 
;*    The main limitation of the User Tags is that you're stuck with
;*    what has been defined in PAPCO. If you REALLY need additional tags,
;*    this entails changing the PAPCO core, and should be done by
;*    whoever maintains the core. You can do this yourself too, by editing
;*    pro papco_getPlotInfoStruct contained in $papco_XX/papco/papco.pro
;*    but your changes will then not propagate to the next PAPCO version.
;*
;*    The PAPCO 09 set of user tags is:
;*
;*       typeVector:intarr(4), $ ; user keyword (old, for compatibility)
;*       swittch:lonarr(3),     $ ; user keyword (old, for compatibility)
;*       options:fltarr(20),   $ ; user keyword (float options)
;*       ioptions:lonarr(16),  $ ; user keyword (long options)
;*       soptions:strarr(16)   $ ; user keyword (string options)
;*
;*    NB! If you need more variables to commincate between your panel 
;*    editor and your plot routine then you can use your own common block
;*
;*    For a more complete description of the other PlotInfo tags and the
;*    tags used by PAPCO supplied widget refer to
;*    $papco_doc/papco_variables.text
;*
;* OVERVIEW:
;*    This function creates the editor-window:
;*       FUNCTION  ampte_cce_panel, aPlotInfo, GROUP=group, ACTION=ACTION
;*    To test you might try this to get a stand-alone widget:
;*       a=papco_getPlotInfoStruct()   which creates the structure
;*                                     containing the plot information 
;*	 a=ampte_cce_panel(a)          which calls up the panel window.
;*    Interrogating a with help,a,/str shows the contents. 
;*
;* MODIFICATION HISTORY:       
;*     august 1995, written by A.Keese
;*     december 1995, modified by R.Friedel at Goddard
;*     april 1996, modified by R.Friedel
;*     march 1997, modified by R.Friedel
;*     August 1998, modified by R.Friedel
;******************************************************************************

;******************************************************************************
;* FUNCTION:     
;*      FUNCTION  ampte_cce_panel, aPlotInfo, GROUP=group, ACTION=ACTION
;* 
;* DESCRIPTION:  
;*	Open an editor widget for a panel, using the methods defined in the 
;*	file $papco_XX/papco_panel.pro .
;*	When the user presses the 'Done'- or the 'Cancel'-Button, the values 
;*	changed by the event-methods of papco_panel.pro are returned.
;*	The returned value is a structure containing those values and a flag,
;*	'canceled', which the caller may read out to check, whether the 
;*	window was closed using the 'Cancel' or the 'Done'-Button.
;*
;* INPUTS:       
;*      aPlotInfo       a structure of the papco_plotinfo-type
;*                      this structure is edited in the panel editor window.
;* KEYWORDS:
;*      GROUP           a WIDGET_BASE
;*	                when specified, this is used as GROUP_LEADER
;*      ACTION          a string
;*	                when specified, the 'Done'-Button is labeled with it
;*		        if not specified, the 'Done'-Button is labeled 'Done'
;*
;* OUTPUT:	 
;*      a default-structure of type papco_panel_editor_return. Though this
;*      type is nowhere defined, it is a virtually existing data-type. 
;*      All the functions in $papco_XX/papco.pro that edit plots or add new
;*      ones, expect the opened editor-windows to return a structure containing
;*      at least some fields. Typically this structure contains fields
;*      used by the papco_panel widgets, and some fields "inherited"
;*      from aplotinfo but modified by the user.
;*
;* CALLING SEQUENCE:
;*      newPlotData=papco_PANEL_TEMPLATE(aplotInfo, ACTION=, GROUP=)
;******************************************************************************
FUNCTION  ampte_cce_panel, aPlotInfo, GROUP=group, ACTION=ACTION

  COMMON ampte_cce_panel_data, widgetData
  common ampte_cce, names

;--- if GROUP is not specified, the group-leader will be the desktop
  IF NOT KEYWORD_SET(GROUP) THEN GROUP=0

;--- set the label for the action-button
  IF NOT KEYWORD_SET(ACTION) THEN ACTION='Done'

;--- the routines used by papco_panel.pro define tags for a structure. Those
;    tags will be stored in the structure 'additionalWData'. When you realize 
;    the window, you have to add your own window-specific data to that  
;    structure and afterwards assign it to widgetData.
  additionalWData={canceled:1}

;--- create the base-window ---------------------------------------------------
  panel_base=papco_widget_base(title=aPlotInfo.panelKind, column=1, /modal, $
                               GROUP_LEADER=group)

;--- All papco_panel_add routines are papco panel widgets that can be included
;    in the panel editor or not. They provide functionality that papco 
;    supports. They dynamically extend the panel editors widget structure, and
;    all the events they generate are supported by papco_panel_event, 
;    transparent to the user. 

;--- the first part of the window will consist of the widget that -------------
;    controls the height of that panel. This is needed for all panel editors.
  widgets=papco_panel_AddHeightWidgets(panel_base, aPlotInfo)
  additionalWData=create_struct(widgets, additionalWData)
   
;
;---  <<<<<<<<<<  insert your widgets below here!  >>>>>>>>>
;
  
  channels_base=WIDGET_BASE(panel_base, /COLUMN, /FRAME, /ALIGN_CENTER)
  
  channel_lable_base=widget_label(channels_base,$
                                  value='AMPTE CCE ion channels')
  
  type_base=CW_BGROUP(channels_base, names, UVALUE='type_base', $
                      /RETURN_INDEX, $
                      /EXCLUSIVE, COLUMN=5 )
  widget_control,type_base,SET_VALUE=aPlotInfo.ioptions(0)
;
;---  <<<<<<<<<<  insert your widgets above here!  >>>>>>>>>
;   
   
;--- for examples of standard buttons or sliders look at other panel editors.
;    some standard buttons which have been recycled before can be found in
;    papco_XX/papco_lib/more_panel_widgets.

;--- As the same panel is called when you edit a plot, interrogate the tags in 
;    the plotInfoStruct (aplotInfo) passed in and set your widgets accordingly.
;    In this way the widget is always reinitialized to its last state.
   
;--- The next five papco panel widgets will be added only if you want to
;    support that functionality, and if they make sense for your
;    ampte_cce. If you do not want to use them, just comment them out.

;--- some panels might need to read/input to text files (event, text ----------
;    module). So a generic papco widget is supplied for this. 
;  widgets=papco_panel_AddFile_Widgets(panel_base, aPlotInfo)
;  additionalWData=create_struct(widgets, additionalWData)   
;--- add the papco panel widget for choosing the color bar for color plots ----
;  widgets=papco_panel_Addcolorbar_Widgets(panel_base, aPlotInfo)
;  additionalWData=create_struct(widgets, additionalWData)
;--- add the  papco panel widget for choosing the plot-style for line plots ---
  widgets=papco_panel_add_PlotStyleButtons(panel_base, aPlotInfo, 0)
  additionalWData=create_struct(widgets, additionalWData)
;--- add the papco panel widget for choosing the y-axis-ranges ----------------
  widgets=papco_panel_AddYScale_Widgets(panel_base, aPlotInfo)
  additionalWData=create_struct(widgets, additionalWData)
;--- add the papco panel widgets for choosing the z-axis-ranges ---------------
  widgets=papco_panel_AddZScale_Widgets(panel_base, aPlotInfo)
  additionalWData=create_struct(widgets, additionalWData)
   
;--- finally, the Actionbuttons (help, cancel, done) are added to the ---------
;    editor window. This is needed for all panel editors. If you want to 
;    make use of the configure feature, then add the keyword CONFIG
  widgets=papco_panel_AddActionButtons(panel_base, Action, aPlotInfo, /CONFIG)
  additionalWData=create_struct(widgets, additionalWData)

;--- now, create the structure that keeps the panel-window data. You always
;    need to specify the help-file, and your structure must contain the
;    'additionalWData'. You should create the helpfile in the papco_interface
;    directory of your module (a template is in the ampte_cce example module)
;    Also add the ID's of your widgets so that you can perform widget_control
;    actions in the events routine or in a refresh routine if needed.
  thisWidgetsData={helpfile:'ampte_cce_panel.help', $
                   ioptions:aPlotInfo.ioptions}
  widgetData=papco_panel_CreateWidgetData(thisWidgetsData, additionalWData)

;--- realize the window -------------------------------------------------------
  papco_Panel_Realize, panel_base, widgetData
  papco_xmanager, 'ampte_cce_panel', panel_base

;--- the returned data consists of the standard-data inherited from
;    papco_panel.pro and your new data. Stick to the tags provided in
;    aPlotInfo, there are ample variables for the user
  returnData={ioptions:widgetData.ioptions}
  additionalRData=papco_panel_ReturnData(widgetData)

  return, create_struct(ReturnData, additionalRData)
   
END

;******************************************************************************
;* PROCEDURE:     
;*      PRO ampte_cce_panel_event, event
;* 
;* DESCRIPTION:  
;*      This processes events from the XMANAGER.
;*      As this example does not extend the basic functionality of plot-
;*      editors, it directly calls the default-handler, papco_PANEL_Event
;*
;* INPUTS:       
;*      event            an XManager event
;*                  
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      called by XMANAGER
;******************************************************************************
PRO ampte_cce_panel_event, event
  
  COMMON ampte_cce_panel_data, widgetData

  WIDGET_CONTROL, GET_UVALUE=uval, event.id
  
  case uval of
    'type_base' : begin
      index=event.value
      widgetData.ioptions(0)=index
    end
    else: widgetData=papco_panel_Event(event, widgetData)
  endcase
    
END
