;******************************************************************************
;*  NAME:        r_ampte_cce
;*
;*  DESCRIPTION: Routine to read ampte_cce files. Adheres to
;*               papco conventions: Error common block
;*                                  Uses MJDT time
;*                                  papco read philosophy
;*
;*  INPUT:       none (uses common block time from papco)
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  KEYWORDS:    VERBOSE  -  enables print comments, save/restore comments
;*               ORBIT    -  if set read data by orbit number 
;*
;*  DATE:        February 1999
;*
;*  AUTOR:       R. Friedel
;******************************************************************************
pro r_ampte_cce, ORBIT=ORBIT, VERBOSE=VERBOSE

  common ampte_cce_data, input_header, input_data
  common mjdt, mjdt_start, mjdt_end
  common get_error, get_err_no, get_err_msg
  
; common get_error is used by papco to check if the data read was successful.
; it is up to the user to set this correctly and to make sure that the read- 
; routine does not crash on an I/O error - otherwise papco will crash at that
; point, too. papco draws an empty panel when get_err_no in not equal to zero
; and prints get_err_msg in that panel for user information. papco will try
; to invoce the plot routine for this data if get_err_no = 0 , otherwise an
; emplty panel with the error message get_err_msg is plotted.

; check env. variable pointing to data directory
  if not (papco_check_data_env('AMPTE_CCE_ARCHIVE', path=PATH)) then return
  
  get_err_no=0  &  get_err_msg='ampte_cce read succesfully'
   
; check VERBOSE keyword and set
  if keyword_set(VERBOSE) then verbose=VERBOSE else verbose=0     
  verbose=1
  
; clear data arrays for reuse
  input_header=0  &  input_data=0   
   
; loop through from start to end time concatonating data files into one
; array. Here we assume that the data is stored in files per day, and
; that the time info in each file is seconds of day. papco standard is
; to hand over data to the plot routines with the time running in
; seconds from the number of seconds of the startday.
    
  time=mjdt_start  &  no_files_read=0  &  day_count=0.0
   
  while mjdt_end.mjd ge time.mjd do begin
     
    ;assemble the data file from the time range requested
    convert_t90_to_date, time, year, day_oy, hour, minute, second,$
                         month, cmonth, day_om, /mjdt   
    datestr=string(year-1980,day_oy,format='(i1.1,i3.3)')
    fln='eav2_'+datestr
        
    ;read one day of data
    r_ampte_cce_onefile, fln, VERBOSE=VERBOSE
  
    ; check for an error in get_error - and then get the file remotely!
    ; comment this section out if you do not want this feature.
    if get_err_no ne 0 then begin
      c_url=papco_getenv('AMPTE_CCE_FTP_ARCHIVE')
      c_file=fln+'.idl'
      c_download_dir=path
      papco_get_remote_data, c_url, c_file, c_download_dir
      ;check if file "arrived" and then read it
      if get_err_no eq 0 then $
        r_ampte_cce_onefile, path+fln, VERBOSE=VERBOSE $
      else begin
        get_err_msg='% r_ampte_cce: No data locally, remote get failed!C  ' $
          + get_err_msg
        return
      endelse  
    endif   
    
    ;if a file was succesfully read, concatonate!
    if get_err_no eq 0 then begin 
      if no_files_read eq 0 then begin   ;start the array buffer
        ;input_data.time=input_data.time+86400.0*day_count
        input_data_buf=temporary(input_data)
      endif else begin                   ;otherwise concatonate arrays
        ;input_data.time=input_data.time+86400*day_count
        input_data_buf=[temporary(input_data_buf),temporary(input_data)]
      endelse        
      input_data=0                     ;clear array for reuse
      no_files_read=no_files_read+1
    endif
    day_count=day_count+1
    time.mjd=time.mjd+1         ; next day
  endwhile
   
; check if any files were read   
  if (no_files_read ne 0) and (get_err_no ne 1) then begin
    get_err_no=0
    get_err_msg='% r_ampte_cce: '+string(no_files_read,format="(i2)")+ $
      ' files read succesfully'
    ;clear arrays and load buffer into output array
    input_data=0  &  input_data=input_data_buf  &  input_data_buf=0
  endif else begin
    get_err_no=1
    get_err_msg='% r_ampte_cce: no files found for timerange'
    return
  endelse      
   
; restrict data to actual time range requested
  ;ut1=mjdt_start.t
  ;ut2=long(mjdt_end.mjd-mjdt_start.mjd)*86400l+mjdt_end.t
  ;index=where((input_data.Time ge ut1) and (input_data.Time le ut2),count)
  ;if count ne 0 then begin
  ;  input_data=input_data(index)
  ;endif   

end

;******************************************************************************
;*  NAME:        r_ampte_cce_onefile,fln
;*
;*  DESCRIPTION: reads one file of type plot_file. Assumes data has been
;*               stored in IDL saveset using structures input_header and
;*               input_data
;*
;*  INPUT:       fln  - string of filename
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        October 1997
;*
;*  AUTOR:       R. Friedel
;******************************************************************************
pro r_ampte_cce_onefile, fln, VERBOSE=VERBOSE
  
  common ampte_cce_data, input_header, input_data
  common get_error, get_err_no, get_err_msg
   
  get_err_no=0
  get_err_msg='ampte_cce onefile read succesfully'
  
; check env. variable pointing to data directory
  if not (papco_check_data_env('AMPTE_CCE_DATA', path=PATH)) then return
  
; clear data arrays for reuse
  input_header=0  &  input_data=0   
   
  on_ioerror, read_original
   
  restore, path+fln+'.idl',VERBOSE=verbose
  dealloc_lun,/quiet    ;frees logical units used by save/restore
  message,'Saveset read: ', /cont
  print,'  ',path+fln+'.idl'
  return

  read_original:
  ; if the .idl save file could not be found, then try to read the original
  r_ampte_cce_onefile_original, fln, VERBOSE=VERBOSE
      
end   
  
;******************************************************************************
;*  NAME:        r_ampte_cce_onefile_original,fln
;*
;*  DESCRIPTION: reads one file of type plot_file_original. Under the
;*               papco read philosophy, files should be read the first
;*               time in whatever format they might have - ascii,
;*               fortran binary, pascal binary, etc. The files might also
;*               be in some compressed format, so they need to be uncompressed
;*               first. 
;*               
;*               You might also have to change time formats. 
;*               A lot of files are stored per day, and have seconds of
;*               that day as the time. This is what is assumed here and
;*               is papco standard. You need to convert the time
;*               to papco standard if you intend to use the plotroutines 
;*               supplied with papco (use the routines in
;*               $papco_BASE/papco_Lib/time_convert.pro). 
;*               If you use your own plot routines this step is not needed.
;*               
;*               The first read might be slow, esp if the format is ascii.
;*               Use the IDL "save" command to make a binary copy of the
;*               file which can then be read quickly using "restore". The
;*               next time only the IDL saveset is read, which is much
;*               faster.
;*               
;*               In order to be able to concatonate the arrays easily,
;*               it makes sense to have the data in an array of time-ordered
;*               structures, which makes adding together days of data
;*               trivial using the IDL concatonation operator "[]".
;*               
;*  INPUT:       fln  - string of filename
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        October 1997
;*
;*  AUTOR:       R. Friedel
;******************************************************************************
pro r_ampte_cce_onefile_original, fln, VERBOSE=VERBOSE
  
  common ampte_cce_data, input_header, input_data
  common get_error, get_err_no, get_err_msg  
   
  get_err_no=0  &  get_err_msg='ampte_cce_onefile_orignal read succesfully'
   
; clear data arrays for reuse
  input_header=0  &  input_data=0   
   
;------------------------------------------------------------------------------
; read the original data file - THIS NEEDS TO BE REWRITTEN BY USER!!!!
    
; check env. variable pointing to data directory
  if not (papco_check_data_env('AMPTE_CCE_ARCHIVE', path=PATH)) then return
  
; see if there is a file.  
  zipped=0
  result=findfile(path+fln+'*.dat',count=c)
  if c eq 0 then begin          ;see if zipped file is there
    result=findfile(path+fln+'*.gz',count=c)  
    if c eq 1 then begin
      zipped=1
      unzip_file=strmid(result(0),0, strlen(result(0))-3)
      if keyword_set(VERBOSE) then $
        message,'gunzip -c '+result(0)+' > '+unzip_file,  /cont
      spawn,'gunzip -c '+result(0)+' > '+unzip_file
    endif 
  endif 
  result=findfile(path+fln+'*.dat',count=c)    
  if c eq 0 then begin ; no files found
    get_err_no=1
    get_err_msg='Original Ampte file '+fln+'*.dat'+'not found'
    message,get_err_msg,/cont
    return
  endif
  
  papco_sepdir, result(0), dir, file
  file_day=strmid(file,6,3)
  
  message,'Reading original data: '+file, /cont

  
; define the header and data structures. Edit to suit your data!
  max_no_rec=3000              ;the maximum number time records in your data
  input_header={ampte_cce_header, $
                VERSION     : 0.9, $
                DATATYPE    : 'AMPTE/CCE ions from Kazue Takahashi', $
                NPOINTS     : 0, $
                NBANDS      : 5, $
                NSECTORS    : 2, $
                TSTART      : convert_date_to_t90(/mjdt), $
                TSTOP       : convert_date_to_t90(/mjdt), $
                CH_POSITIONS: fltarr(2, 5), $
                SECTOR_POS  : ['45<PA<135','45>PA>135'], $ 
                CAL         : 1, $
                RAT         : 1, $
                DIV         : 1, $
                TITLE       : '', $
                XTITLE      : 'UT:', $
                YTITLE      : 'Energy', $
                ZTITLE      : 'Cts', $
                PTITLE      : 'N/A'  }
                
  input_header.CH_POSITIONS(0,*)=[ 24., 34., 50., 83.,151.]              
  input_header.CH_POSITIONS(1,*)=[ 34., 50., 83.,151.,285.]    
  
  dummy={ampte_cce_data, time:convert_date_to_t90(/mjdt), $
         e0a:0.0, e0b:0.0, $
         e1a:0.0, e1b:0.0, $
         e2a:0.0, e2b:0.0, $
         e3a:0.0, e3b:0.0, $
         e4a:0.0, e4b:0.0 }
  input_data=replicate(dummy, max_no_rec)
  
  dummy_struct={yr:0,day:0,hr:0,mn:0,s:0.0, $
                e0a:0.0, e0b:0.0, $
                e1a:0.0, e1b:0.0, $
                e2a:0.0, e2b:0.0, $
                e3a:0.0, e3b:0.0, $
                e4a:0.0, e4b:0.0 }
  
; set orbit number in header
  input_header.TITLE=strmid(fln,strlen(fln)-4,strlen(fln))
  
  dummy_str='' & count=0
  ;on_ioerror, out  
  openr,unit,path+file,/get_lun
  while not eof(unit) do begin  ;read in data and store in input_data
    readf,unit,dummy_struct    
    t_mjdt=convert_date_to_t90(doy=dummy_struct.day,  year=dummy_struct.yr, $
                               hour=dummy_struct.hr, minutes=dummy_struct.mn, $
                               seconds=dummy_struct.s, /MJDT)
    convert_t90_to_date, t_mjdt, year, day_oy, hour, minute, second,$
      month, cmonth, day_om, /mjdt 
    if day_oy ne file_day then goto, next
    input_data(count).time=t_mjdt
    for i=1,10 do input_data(count).(i)=dummy_struct.(i+4)
    count=count+1
    next:
  endwhile
  close, unit & free_lun,unit
  input_data=temporary(input_data(0:count-1))
  
  input_header.TSTART=input_data(0).time
  input_header.TSTOP=input_data(count-1).time
  
  ;remove unzipped file
  if keyword_set(VERBOSE) then message,'rm ' + result(0), /cont
  spawn,'rm ' + result(0)
  
  message,'Original data read succesfully  '+result(0), /cont

; end of user written section
;------------------------------------------------------------------------------
   
  data_dir='AMPTE_CCE_DATA' &  path=papco_getenv(data_dir)
  
; save read data with correct time. Rename data to *.idl
  file=strmid(file,0,strlen(file)-4)
  save,filename=path+fln+'.idl',input_header,input_data, VERBOSE=verbose  
  dealloc_lun,/quiet              ;frees logical units used by save/restore
  if keyword_set(VERBOSE) then begin
    message, 'Saved data to IDL saveset: '+path+fln+'.idl', /cont
  endif
  
  return
  
  out:
  ; if there was an error, report it. Use the IDL error variables. Error
  ; 171 is for "file not fond" which is used to invoke the remote get.  
  get_err_no=!ERROR
  get_err_msg='% r_ampte_cce_onefile_original:!C '+!ERR_STRING
  err_string=str_sep(get_err_msg,'!C')
  for i=0,n_elements(err_string)-1 do print,err_string(i)
   
end
