;******************************************************************************
;*  NAME:        r_ampte_eph
;*
;*  DESCRIPTION: Routine to read ampte_eph files. Adheres to
;*               papco conventions: Error common block
;*                                  Uses MJDT time
;*                                  papco read philosophy
;*
;*  INPUT:       none (uses common block time from papco)
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  KEYWORDS:    VERBOSE  -  enables print comments, save/restore comments
;*               ORBIT    integer     retrun data by orbit number
;*
;*  DATE:        July 1999
;*
;*  AUTOR:       R. Friedel
;******************************************************************************
pro r_ampte_eph, VERBOSE=VERBOSE, ORBIT=ORBIT

  common ampte_eph_data, input_header, input_data
  common mjdt, mjdt_start, mjdt_end
  common get_error, get_err_no, get_err_msg
   
; common get_error is used by papco to check if the data read was successful.
; it is up to the user to set this correctly and to make sure that the read- 
; routine does not crash on an I/O error - otherwise papco will crash at that
; point, too. papco draws an empty panel when get_err_no in not equal to zero
; and prints get_err_msg in that panel for user information. papco will try
; to invoce the plot routine for this data if get_err_no = 0 , otherwise an
; emplty panel with the error message get_err_msg is plotted.
  get_err_no=0  &  get_err_msg='ampte_eph read succesfully'
   
; check VERBOSE keyword and set
  if keyword_set(VERBOSE) then verbose=VERBOSE else verbose=0    
  
; set mjdt common block from orbit number is orbit set  
  if keyword_set(ORBIT) then begin
    result=papco_setorbit(orbit, SATELLITE='AMPTE', /NOWIDGET)
  endif    
   
; clear data arrays for reuse
  input_header=0  &  input_data=0   
   
; loop through from start to end time concatonating data files into one
; array. Here we assume that the data is stored in files per day, and
; that the time info in each file is seconds of day. papco standard is
; to hand over data to the plot routines with the time running in
; seconds from the number of seconds of the startday.
  no_files_read=0  &  year_count=0.0
  
  convert_t90_to_date, mjdt_start, year_start, day_oy, hour, minute, second,$
    month, cmonth, day_om, /mjdt 
  TAI_start=UTC2TAI({YEAR:year_start, MONTH:month, DAY:day_om, HOUR:hour, $
                     MINUTE:minute, SECOND:second, MILLISECOND:0.0})
  
  convert_t90_to_date, mjdt_end, year_end, day_oy, hour, minute, second,$
    month, cmonth, day_om, /mjdt 
  TAI_end=UTC2TAI({YEAR:year_end, MONTH:month, DAY:day_om, HOUR:hour, $
                     MINUTE:minute, SECOND:second, MILLISECOND:0.0})
  
  while year_end ge year_start do begin
     
    datestr=string(year_start-1900,format='(i2.2)')
    fln='cce_pos_'+datestr
     
    ; read one day of data
    r_ampte_eph_onefile, fln, VERBOSE=VERBOSE
      
    ; check for an error in get_error - and then get the file remotely!
    ; comment this section out if you do not want this feature.
    if get_err_no ne 0 then begin         
      c_url=papco_getenv('AMPTE_EPH_FTP_ARCHIVE')
      c_file=fln+'.idl'
      c_download_dir=path
      papco_get_remote_data, c_url, c_file, c_download_dir
      ;check if file "arrived" and then read it
      if get_err_no eq 0 then $
        r_ampte_eph_onefile, path+fln, /VERBOSE $
      else begin
        get_err_msg='% r_ampte_eph: No data locally, remote get failed!C  ' $
          + get_err_msg
        return
      endelse  
    endif   
     
    ;if a file was succesfully read, concatonate!
    if get_err_no eq 0 then begin 
      if no_files_read eq 0 then begin   ;start the array buffer
        input_data_buf=temporary(input_data)
      endif else begin                   ;otherwise concatonate arrays
        input_data_buf=[temporary(input_data_buf),temporary(input_data)]
      endelse        
      input_data=0                     ;clear array for reuse
      no_files_read=no_files_read+1
    endif
    year_count=year_count+1
    year_start=year_start+1                  ; next year
  endwhile
  
; check if any files were read   
  if (no_files_read ne 0) and (get_err_no ne 1) then begin
    get_err_no=0
    get_err_msg=varprt(no_files_read)+' files read succesfully'
    message,get_err_msg,/cont
    ;clear arrays and load buffer into output array
    input_data=0  &  input_data=temporary(input_data_buf)
  endif else begin
    get_err_no=1
    get_err_msg='no files found for timerange'
    message,get_err_msg,/cont
    return
  endelse      

; restrict data to actual time range requested. Time is in TAI.
  index=where((input_data.Time ge TAI_start) and $
              (input_data.Time le TAI_end),count)
  if count ne 0 then input_data=input_data(index) $
  else begin
    get_err_no=2   
    get_err_msg='no data found for timerange'
    message,get_err_msg,/cont
  endelse  
  
end

;******************************************************************************
;*  NAME:        r_ampte_eph_onefile,fln
;*
;*  DESCRIPTION: reads one file of type plot_file. Assumes data has been
;*               stored in IDL saveset using structures input_header and
;*               input_data
;*
;*  INPUT:       fln  - string of filename
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        October 1997
;*
;*  AUTOR:       R. Friedel
;******************************************************************************
pro r_ampte_eph_onefile, fln, VERBOSE=VERBOSE
  
  common ampte_eph_data, input_header, input_data
  common get_error, get_err_no, get_err_msg
   
  get_err_no=0
  get_err_msg='ampte_eph onefile read succesfully'  
  
; check env. variable pointing to data directory
  if not (papco_check_data_env('AMPTE_EPH_DATA', path=PATH)) then return
  
; clear data arrays for reuse
  input_header=0  &  input_data=0   
  on_ioerror, read_original
   
  restore,path+fln+'.idl',verbose=verbose
  dealloc_lun,/quiet    ;frees logical units used by save/restore
  message,fln+' saveset read succesfully', /cont
  return

  read_original:
  ; if the .idl save file could not be found, then try to read the original
  r_ampte_eph_onefile_original, fln, VERBOSE=VERBOSE
      
end   

;******************************************************************************
pro s_ampte_eph_onefile, fln, VERBOSE=VERBOSE

  common ampte_eph_data, input_header, input_data
  common get_error, get_err_no, get_err_msg
   
  get_err_no=0
  get_err_msg='ampte_eph onefile saved succesfully'    
  
; check env. variable pointing to data directory
  if not (papco_check_data_env('AMPTE_EPH_DATA', path=PATH)) then return  
  
  save, filename=path+fln+'.idl',input_header, input_data, verbose=verbose
  message,fln+' saveset saved succesfully', /cont
  return  
end

;******************************************************************************
;*  NAME:        r_ampte_eph_onefile_original,fln
;*
;*  DESCRIPTION: reads one file of type plot_file_original. Under the
;*               papco read philosophy, files should be read the first
;*               time in whatever format they might have - ascii,
;*               fortran binary, pascal binary, etc. The files might also
;*               be in some compressed format, so they need to be uncompressed
;*               first. 
;*               
;*               You might also have to change time formats. 
;*               A lot of files are stored per day, and have seconds of
;*               that day as the time. This is what is assumed here and
;*               is papco standard. You need to convert the time
;*               to papco standard if you intend to use the plotroutines 
;*               supplied with papco (use the routines in
;*               $papco_BASE/papco_Lib/time_convert.pro). 
;*               If you use your own plot routines this step is not needed.
;*               
;*               The first read might be slow, esp if the format is ascii.
;*               Use the IDL "save" command to make a binary copy of the
;*               file which can then be read quickly using "restore". The
;*               next time only the IDL saveset is read, which is much
;*               faster.
;*               
;*               In order to be able to concatonate the arrays easily,
;*               it makes sense to have the data in an array of time-ordered
;*               structures, which makes adding together days of data
;*               trivial using the IDL concatonation operator "[]".
;*               
;*  INPUT:       fln  - string of filename
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        October 1997
;*
;*  AUTOR:       R. Friedel
;******************************************************************************
pro r_ampte_eph_onefile_original, fln, VERBOSE=VERBOSE
  
  common ampte_eph_data, input_header, input_data
  common get_error, get_err_no, get_err_msg  
   
  get_err_no=0
  get_err_msg='ampte_eph_onefile_orignal read succesfully'
  
; check for env. variable pointing to data directory
  data_dir='AMPTE_EPH_ARCHIVE'
  path=papco_getenv(data_dir)
  if strlen(path) eq 0 then begin
    get_err_no=1
    get_err_msg='Environmental variable '+data_dir+' not set'
    print,get_err_msg
    return
  endif
   
; check for existence of data directory
  cc=papco_finddir(path)
  if cc eq 0 then begin
    get_err_no=2
    get_err_msg='Data directory '+path+' not found'
    print,get_err_msg
    return
  endif  
  
; clear data arrays for reuse
  input_header=0  &  input_data=0   
   
;------------------------------------------------------------------------------
; read the original data file - THIS NEEDS TO BE REWRITTEN BY USER!!!!
  
  message,'reading and converting time...',/cont
  
; define the header and data structures. Edit to suit your data!
  max_no_rec=30000             ;the maximum number time records in your data
  ext='.geo'                   ;file extention of your dat files
  input_header={ampte_eph_head,descr:'',t_format:''}
  input_header.descr='AMPTE CCE ephemeris, 20 min resolution, geographic'
  input_header.t_format='Atomic International Time (TAI)'
  dummy1={year:0, day:0, hour:0, min:0}
  dummy2={radius:0.0, lat:0.0, long:0.0}
  dummy3=create_struct('time',0.0d, dummy2)
  input_data=replicate(dummy3, max_no_rec)
  
  ndat=-1
  ;on_ioerror, out  
  openr,unit,path+fln+ext,/get_lun
  while not eof(unit) do begin
    ;read in data and store in input_data
    ndat=ndat+1
    readf,unit,dummy1,dummy2
    ;convert time to TAI
    utc=string(dummy1.year+1900,dummy1.day, dummy1.hour, dummy1.min, $
               format="(i4.4,'-',i3.3,'T',i2.2,':',i2.2)")
    input_data(ndat).time=UTC2TAI(utc)    
    for i=1,3 do input_data(ndat).(i)=dummy2.(i-1)
  endwhile
  close,unit & free_lun,unit
  
  input_data=input_data(0:ndat)

; end of user written section
;------------------------------------------------------------------------------
  
; check for env. variable pointing to data directory
  data_dir='AMPTE_EPH_DATA'
  path=papco_getenv(data_dir)
  if strlen(path) eq 0 then begin
    get_err_no=1
    get_err_msg='Environmental variable '+data_dir+' not set'
    print,get_err_msg
    return
  endif
   
; check for existence of data directory
  cc=papco_finddir(path)
  if cc eq 0 then begin
    get_err_no=2
    get_err_msg='Data directory '+path+' not found'
    print,get_err_msg
    return
  endif    
  
; save read data. Rename data to *.idl
  save,filename=path+fln+'.idl',input_header,input_data,/verbose  
  dealloc_lun,/quiet              ;frees logical units used by save/restore
  return
  
  out:
  ; if there was an error, report it.
  get_err_no=!ERROR
  get_err_msg='r_ampte_eph_onefile:!C '+!ERR_STRING
   
end

;******************************************************************************
;* add magnetic coordinates to ampte ephemeris
;*
;******************************************************************************
pro m_ampte_eph, year
  
  common ampte_eph_data, input_header, ampte_data
  common mjdt, mjdt_start, mjdt_end
  common get_error, get_err_no, get_err_msg  
  
; ampte eph is in files per year - so loop in years  
  year_start=year
  
  datestr=string(year_start-1900,format='(i2.2)')
  fln='cce_pos_'+datestr
  
  ; read one year of data
  r_ampte_eph_onefile, fln
  if get_err_no ne 0 then begin
    message,'No data for year '+varprt(year),/cont
    return
  endif  
 
  input_header.DESCR=input_header.DESCR+' - mag coords added'
    
  dummy1={ampte_eph, $
         time:0.0d, Radius:0.0, Lat:0.0, Long:0.0, L:0.0, MLT:0.0, MLAT:0.0}
  
  input_data=replicate(dummy1,n_elements(ampte_data))
  input_data.time=ampte_data.time
  input_data.Radius=ampte_data.Radius
  input_data.Lat=ampte_data.Lat
  input_data.Long=ampte_data.Long  
  
  message,'Calculating Olson Pfitzer Mag coords for year '+varprt(year),/cont
  for i=0,n_elements(ampte_data)-1 do begin
    result=TAI2UTC(input_data(i).time, /external)    
    time_date={year:result.year, month:result.month, day:result.day, $
               hour:result.hour, min:result.minute, secs:result.second}
    pos={radius:input_data(i).RADIUS,$
         latitude:input_data(i).LAT,$
         longitude:input_data(i).LONG}    
    f77_unilib_get_BL, time_date, pos, fbm, flm, xmlt, xlat, $
      INTMOD=0, EXTMOD=5, PITCH=90.0
    input_data(i).L=flm
    input_data(i).mlt=xmlt
    input_data(i).mlat=xlat    
    ;print, time_date, pos, flm, xmlt, xlat
  endfor  
  
  ampte_data=input_data
  s_ampte_eph_onefile, fln
  
end  
