;******************************************************************************
;* FILE:
;*    $PAPCO_XX/PAPCO_templates/cassini_caps_line_panel.pro
;* 
;*    This file belongs to the PAPCO-project. See $PAPCO_XX/papco.pro for
;*    further information.
;*
;* DESCRIPTION: 
;*    This file contains a demonstration on how to implement a PAPCO-
;*    panel-editor. It makes use of all the features provided by 
;*    $PAPCO_XX/papco_panel.pro (a collection of standard PAPCO widgets)
;*    
;*    It constructs an editor for a very simple PAPCO-plot, that has only
;*    default-functionality.
;*
;* FILES YOU MAY WANT TO READ (under $PAPCO_XX/PAPCO_doc)
;*    papco_conventions.text - which conventions were used when developing ?
;*    papco_variables.text   - what structures and global variables
;                              are used by which file ?
;*
;* COMMON-BLOCKS:
;*    PAPCO_PANEL_TEMPLATE_DATA	contains the window-information
;*
;* STRUCTURES:
;*    This module creates one concrete structure, that inherits from the 
;*    PAPCO_PANEL_EDITOR-structure (see $PAPCO_XX/papco_panel.pro for a
;*    description). 
;*       {PAPCO_PANEL_TEMPLATE_DATA, 
;*	     ; icaps_lineine the PAPCO_PANEL_EDITOR-structure was inserted...
;*	      
;*    For a more complete description see $PAPCO_DOC/papco_variables.text
;*
;* OVERVIEW:
;*    This function creates the editor-window:
;*       FUNCTION  PAPCO_PANEL_TEMPLATE, aPlotInfo, GROUP=group, ACTION=ACTION
;*    You might try this:
;*       a=PAPCO_getPlotInfoStruct()   which creates the structure
;*                                     containing the plot information 
;*	 a=cassini_caps_line_panel(a)
;*       which calls up the window.
;*       Interrogating a with help,a,/str shows the contents. This is a good
;*       way of testing your own panel editor. 
;*
;* MODIFICATION HISTORY:       
;*     Written April 2001,  S. Bourdarie (ONERA/DESP)
;******************************************************************************

;******************************************************************************
;* FUNCTION:     
;*      FUNCTION  cassini_caps_line_panel, aPlotInfo, GROUP=group, ACTION=ACTION
;* 
;* DESCRIPTION:  
;*	Open an editor for a panelplot.
;*	Its window is constructed using the methods defined in the file
;*	$PAPCO_XX/papco_panel.pro
;*	When the user presses the 'Done'- or the 'Cancel'-Button, the values 
;*	changed by the event-methods of papco_panel.pro are returned.
;*	The returned value is a structure containing those values and a flag,
;*	'canceled', which the caller may read out to check, whether the 
;*	window was closed using the 'Cancel' or the 'Done'-Button.
;*
;* INPUTS:       
;*      aPlotInfo       a structure of the PAPCO_PLOTINFO-type
;*                      this structure is edited in the panel  editor window.
;* KEYWORDS:
;*      GROUP           a WIDGET_BASE
;*	                when specified, this is used as GROUP_LEADER
;*      ACTION          a string
;*	                when specified, the 'Done'-Button is labeled with it
;*		        if not specified, the 'Done'-Button is labeled 'Done'
;*
;* OUTPUT:	 
;*      a default-structure of type PAPCO_PANEL_EDITOR_RETURN. Though this
;*      type is nowhere defined, I like to think of it as virtually existing
;*      data-type. 
;*      All the functions in $PAPCO_XX/papco.pro that edit plots or add new
;*      ones, expect the opened editor-windows to return a structure containing
;*      at last some fields. Those fields might make up a structure, but as no
;*      program defines the structure-type PAPCO_PANEL_EDITOR_RETURN, this 
;*      poor structure is not alive but only virtual.
;*      The return-values of plot-editors have to expand this structure - I 
;*      like to think, that they inherit from that structure - and contain at
;*      least those fields:
;*   		{$
;*			canceled:1, $
;*	    		panelHeight:1, $
;*	    		typeVector:[0,0,0,0], $ 
;*                      switch:[0,0,0], $
;*	                fileName:'', $        	 	
;*	    		panelKind:'some string'}
;*
;*      The fields typeVector, switch and fileName are meant for the user to
;*      use and set in his/her panel editor.
;*          type_vector - a four element array for choosing the type of data
;*                        (eg. elec or ion, diff or integ., pitch angle, 
;*                             or sector, channel ....)
;*          switch      - a three element vector for specific plot appearance
;*                        (eg. smoothing/averaging, interpolating ...)
;*          filename    - a filename
;*      These tags can be used for ANYTHING really, these are just
;*      suggestions. You are, however, limited to these three tags, and only
;*      these tags will get stored when you use the "SAVE" function.
;*
;*      NB! If you need more variables to commlemms_linete between your
;*     panel editor and your plot routine then use your own common block
;*
;*      Additionally, some Panel-editors may return more fields, depending on
;*      which PAPCO supplied widgets were used:
;*              {$
;*	                manualYScaling:0, $
;*	                manualZScaling:0, $
;*	                yscaleMin:0.0, $
;*	                yscaleMax:0.0, $
;*		        zscaleMin:0.0, $
;*	                zscaleMax:0.0, $
;*	                yLinLog:1,$
;*	                zLinLog:1,$
;*	                channel:0, $	      
;*		        color:1, $
;*			plotStyle:1}
;*
;*      PAPCO internally uses some more tags which the user need not worry
;*      about:
;*              {$
;*	                panelPosition:0,$
;*	                overplot:0, $          	  	 	
;*	                xut1:0.0, $                       
;*                      xut2:0.0}                           
;*
;*                  
;*      For an enhanced description read 
;*                          $PAPCO_XX/PAPCO_doc/papco_variables.text
;*
;* CALLING SEQUENCE:
;*      newPlotData=PAPCO_PANEL_TEMPLATE(aplotInfo, ACTION=, GROUP=)
;******************************************************************************
FUNCTION  cassini_caps_line_panel, aPlotInfo, GROUP=group, ACTION=ACTION

   COMMON cassini_caps_line_panel_DATA, widgetData

; if GROUP is not specified, the group-leader will be the desktop
   IF NOT KEYWORD_SET(GROUP) THEN GROUP=0

; set the label for the action-button
   IF NOT KEYWORD_SET(ACTION) THEN ACTION='Done'

; the routines used by papco_panel.pro define tags for a structure. Those
; tags will be stored in the structure 'additionalWData'. When you realize the
; window, you have to add your own window-specific data to that structure 
; and afterwards assign it to widgetData.
   additionalWData={canceled:1}

;--- create the base-window ---------------------------------------------------
   panel_base=WIDGET_BASE(title=aPlotInfo.panelKind,$
       column=1, $
       GROUP_LEADER=group)

; All PAPCO_panel_Add* routines are PAPCO panel widgets that can be included
; in the panel editor or not. They provide functionality that PAPCO supports.
; They dynamically extend the panel editors widget structure, and all the 
; events they generate are supported by PAPCO_panel_Event, transparent to the
; user. 

;--- the first part of the window will consist of the widget that -------------
;    control the height of that panel. This is needed for all panel editors.
   widgets=PAPCO_panel_AddHeightWidgets(panel_base, aPlotInfo)
   additionalWData=create_struct(widgets, additionalWData)

;
; <<<<<<<<<<  insert your widgets below here!  >>>>>>>>>
;
;  widget to chose the specie (caps_line)
;
   spec_base=WIDGET_BASE(panel_base, /FRAME, COLUMN=3,/exclusive) 
   typeNames=['ELS','IMS','IBS']

   spb_type=lonarr(N_ELEMENTS(typeNames))

   FOR i=0, N_ELEMENTS(typeNames)-1 DO $
      spb_type(i)=WIDGET_BUTTON(spec_base, $
			VALUE=typeNames(i), $
			UVALUE='spb_type'+strtrim(string(i), 2))

     WIDGET_CONTROL, spb_type(aPlotInfo.ioptions(0)), SET_BUTTON=1

; --- ion species--------------------------------------------------------
   ionBase =WIDGET_BASE(panel_base, /FRAME, row=1) 
   label_base=WIDGET_LABEL(ionBase, value='Ion')
   sl_ion=WIDGET_SLIDER(ionBase, /DRAG, $
                        VALUE=aPlotInfo.ioptions(5) , $
                            UVALUE='sl_ion',$
                            MINIMUM=0, MAXIMUM=6,xsize=500)
   if aPlotInfo.ioptions(0) eq 1 then $
	 WIDGET_CONTROL, ionBase, SENSITIVE=1 $
	 else WIDGET_CONTROL, ionBase, SENSITIVE=0
;
; --- energy ------------------------------------------------------------------
   energyBase =WIDGET_BASE(panel_base, /FRAME, column=1) 
   sl_energy=WIDGET_SLIDER(energyBase, /DRAG, $
                        VALUE=aPlotInfo.ioptions(1) , $
                            UVALUE='sl_energy',$
                            MINIMUM=0, MAXIMUM=62,xsize=500)
   lbl_energy=WIDGET_LABEL(energyBase, $
			VALUE=strcenter('Energy ',50), $
			/ALIGN_CENTER)			      

; selection for the average mode

   av1_base=WIDGET_BASE(panel_base, /FRAME, COLUMN=2) 
   label_base=WIDGET_LABEL(av1_base, value='Average mode')
   
   typeNames=['none','elevation','azimut','elevation + azimut']
   Av_base = WIDGET_BASE(av1_base,  /row, /EXCLUSIVE) 

   spb_aver=lonarr(N_ELEMENTS(typeNames))

   FOR i=0,3 DO $
      spb_aver(i)=WIDGET_BUTTON(Av_base, $
			VALUE=typeNames(i), $
			UVALUE='spb_aver'+strtrim(string(i), 2))

   WIDGET_CONTROL, spb_aver(aPlotInfo.ioptions(2)), SET_BUTTON=1

;
; elevation selection

   ele_Base=WIDGET_BASE(panel_base, ROW=2, /FRAME)
   label_base=WIDGET_LABEL(ele_Base, value='Elevation min')
   sl_elemin=WIDGET_SLIDER(ele_Base, $
			/DRAG, $
		        VALUE=aPlotInfo.typevector(0) < 7 > 0, $
                        UVALUE='sl_elemin',$
                        MINIMUM=0, MAXIMUM=7,xsize=400)

   label_base=WIDGET_LABEL(ele_Base, value='Elevation max')
   sl_elemax=WIDGET_SLIDER(ele_Base, $
			/DRAG, $
		        VALUE=aPlotInfo.typevector(1) < 7 > 0, $
                        UVALUE='sl_elemax',$
                        MINIMUM=0, MAXIMUM=7,xsize=400)



; azimut selection

   azi_Base=WIDGET_BASE(panel_base, ROW=2, /FRAME)
   label_base=WIDGET_LABEL(azi_Base, value='Azimut min')
   sl_azimin=WIDGET_SLIDER(azi_Base, $
			/DRAG, $
		        VALUE=aPlotInfo.typevector(2) < 15 > 0, $
                        UVALUE='sl_azimin',$
                        MINIMUM=0, MAXIMUM=15,xsize=400)

   label_base=WIDGET_LABEL(azi_Base, value='Azimut max')
   sl_azimax=WIDGET_SLIDER(azi_Base, $
			/DRAG, $
		        VALUE=aPlotInfo.typevector(3) < 15 > 0, $
                        UVALUE='sl_azimax',$
                        MINIMUM=0, MAXIMUM=15,xsize=400)
;;
;;
; <<<<<<<<<<  insert your widgets above here!  >>>>>>>>>
;   
   
; for examples of standard buttons or sliders look at other panel editors.
; some standard buttons which have been recycled before can be found in
; $PAPCO_MODULES/panels&slice/more_panel_widgets

; As the same panel is called when you edit a plot, interrogate the tags in the
; plotInfoStruct (aplotInfo) passed in and set your widgets accordingly. 
; In this way the widget is always reinitialized to its last state.

; The next four PAPCO panel widgets will be added only if you want to
; support that functionality, and if they make sense for your cassini_caps_line.

;--- add the  PAPCO panel widget for choosing the plot-style for line plots ---
   widgets=PAPCO_panel_add_PlotStyleButtons(panel_base, aPlotInfo, 0)
   additionalWData=create_struct(widgets, additionalWData)
;--- add the PAPCO panel widget for choosing the y-axis-ranges ----------------
   widgets=PAPCO_PANEL_AddYScale_Widgets(panel_base, aPlotInfo)
   additionalWData=create_struct(widgets, additionalWData)
   
;--- finally, the Actionbuttons (help, cancel, done) are added to the ---------
;    editor window. This is needed for all panel editors.
;    if you want to make use of the configure feature, then add the
;    keyword CONFIG
   widgets=PAPCO_panel_AddActionButtons(panel_base, Action, aPlotInfo, /CONFIG)
   additionalWData=create_struct(widgets, additionalWData)

;--- now, create the structure that keeps the panel-window data. You
;    always need to specify the help-file, and your structure must
;    always contain the 'additionalWData'. You should also actually create 
;    that helpfile in the papco_interface directory of your module (a template
;    for this help file exist in the cassini_caps_line example module) and describe 
;    your data and your choices in your widget. Put in here also the ID's of 
;    the widgets you create so that you can perform widget_control actions on
;    them them in the events routine or in a refresh routine if needed.
   thisWidgetsData={helpfile:'cassini_caps_line_panel.help', $
                    typevector:aPlotInfo.typevector, $
                    ioptions:aPlotInfo.ioptions, $
                    spb_type:spb_type,$
		    lbl_energy:lbl_energy,$
		    ionBase:ionBase,$
		    sl_elemax:sl_elemax,$
		    sl_elemin:sl_elemin,$
		    sl_azimax:sl_azimax,$
		    sl_azimin:sl_azimin}
   widgetData=PAPCO_PANEL_CreateWidgetData(thisWidgetsData, additionalWData)

;--- realize the window
   PAPCO_Panel_Realize, panel_base, widgetData
   cassini_caps_line_panel_refresh,/G_ENERGY,/SL_ENERGY


; As this is a function that will return all changes made by the
; user, the xmanager has to be called with the keyword MODAL. If we
; wouldn't specify /MODAL, no changes would be reported but the 
; data contained in aPlotInfo would be returned.
   xmanager, 'cassini_caps_line_panel', panel_base, /MODAL

; the returned data consists of the standard-data inherited from
; PAPCO_PANEL.PRO and your new data - the typeVector-tag and switch-tag must
; always be included, but you can add more. However, if possible stick to the 
; tags provided in aPlotInfo, it makes life easier. 
   returnData={typevector:widgetData.typevector, $
                 ioptions:WidgetData.ioptions}
   additionalRData=PAPCO_panel_ReturnData(widgetData)

   return, create_struct(ReturnData, additionalRData)
END

;******************************************************************************
;* PROCEDURE:     
;*      PRO cassini_caps_line_panel_refresh, LIST=LIST
;* 
;* DESCRIPTION:  
;*      update the display of the editor-widgets
;* 
;* INPUTS:       
;*      none
;* 
;* KEYWORDS:
;*      only the fields corresponding to the set keyword are updated
;*
;* CALLING SEQUENCE:
;*      have a look at PRO salammbo_panel_event
;*
;* MODIFICATION HISTORY:       
;*      November 2000, modified by S. BOURDARIE 
;*      April 2001, modified by S. BOURDARIE 
;******************************************************************************
PRO cassini_caps_line_panel_refresh,G_ENERGY = G_ENERGY, SL_ENERGY = SL_ENERGY,$
                   ELS=ELS, IBS=IBS, IMS=IMS

   COMMON cassini_caps_line_panel_DATA, widgetData
   COMMON cassini_caps_line_energy, energy


   SPEC=0
   if keyword_set(G_ENERGY) then begin
     case widgetData.ioptions(0) of
       0: cassini_caps_definition,/ELS
       1: cassini_caps_definition,/IMS
       2: cassini_caps_definition,/IBS
     endcase
   endif

   if keyword_set(SL_ENERGY) then begin
     WIDGET_CONTROL, widgetData.lbl_energy, $
           SET_value='Energy : '+$
	   string(energy(widgetData.ioptions(1)),format='(f9.2)')+' eV'
   endif

   if keyword_set(ELS) then begin
      ELmax=7
      AZmax=15
      SPEC=1
   endif
   if keyword_set(IBS) then begin
      ELmax=2
      AZmax=15
      SPEC=1
   endif
   if keyword_set(IMS) then begin
      ELmax=7
      AZmax=7
      SPEC=1
   endif
   if SPEC then begin
      WIDGET_CONTROL, widgetData.sl_elemax, SET_SLIDER_MIN=0, $
        			 SET_SLIDER_MAX=ELmax
      WIDGET_CONTROL, widgetData.sl_elemin, SET_SLIDER_MIN=0, $
        			 SET_SLIDER_MAX=ELmax
      WIDGET_CONTROL, widgetData.sl_azimax, SET_SLIDER_MIN=0, $
        			 SET_SLIDER_MAX=AZmax
      WIDGET_CONTROL, widgetData.sl_azimin, SET_SLIDER_MIN=0, $
        			 SET_SLIDER_MAX=AZmax

      widgetData.typevector(0)=widgetData.typevector(0)<ELmax
      widgetData.typevector(1)=widgetData.typevector(1)<ELmax
      widgetData.typevector(2)=widgetData.typevector(2)<AZmax
      widgetData.typevector(3)=widgetData.typevector(3)<AZmax
   endif
END


;******************************************************************************
;* PROCEDURE:     
;*      PRO cassini_caps_line_panel_event, event
;* 
;* DESCRIPTION:  
;*      This processes events from the XMANAGER.
;*      As this example does not extend the basic functionality of plot-
;*      editors, it directly calls the default-handler, PAPCO_PANEL_Event
;*
;* INPUTS:       
;*      event            an XManager event
;*                  
;* KEYWORDS:
;*       none
;* 
;* CALLING SEQUENCE:
;*      called by XMANAGER
;* MODIFICATION HISTORY:       
;*     Written April 2001,  S. Bourdarie (ONERA/DESP)
;******************************************************************************
PRO cassini_caps_line_panel_event, event

   COMMON cassini_caps_line_panel_DATA, widgetData

   WIDGET_CONTROL,GET_UVALUE=uval, event.id

; in this editor, we only need to process the channel slider events. events
; other than those from your own widgets need to be passed on to 
; 'PAPCO_panel_Event', which handles the events generated
; by the PAPCO-provided panel widgets. Your own added panel widgets events
; need to be processed here:  you should check 'event' for your events and call
; 'PAPCO_panel_Event', if event was not generated by your own panel widgets.
; The best way to learn is to look at some of the other module's panel editors.


   if strleft(uval, 8) EQ 'spb_type' THEN BEGIN
      choice=fix(strfrom(uval, 8)) 
      uval=strleft(uval, 8)
   endif
   if strleft(uval, 8) EQ 'spb_aver' THEN BEGIN
      av_choice=fix(strfrom(uval, 8)) 
      uval=strleft(uval, 8)
   endif



   CASE uval OF
     'spb_type' : BEGIN
         WidgetData.ioptions(0)=choice
	 cassini_caps_line_panel_refresh,/G_ENERGY,/SL_ENERGY
	 if choice eq 1 then $
	 WIDGET_CONTROL, widgetData.ionBase, SENSITIVE=1 $
	 else WIDGET_CONTROL, widgetData.ionBase, SENSITIVE=0
	 if choice eq 0 then cassini_caps_spectrum_panel_refresh,/ELS
	 if choice eq 1 then cassini_caps_spectrum_panel_refresh,/IMS
	 if choice eq 2 then cassini_caps_spectrum_panel_refresh,/IBS
     END
     'sl_ion' : WidgetData.ioptions(5) = event.value
     'sl_energy' : BEGIN
          widgetData.ioptions(1)=event.value
	  cassini_caps_line_panel_refresh,/SL_ENERGY	       
     END 
     'spb_aver' : WidgetData.ioptions(2) = av_choice
     'sl_elemin': widgetData.typevector(0)=event.value
     'sl_elemax': widgetData.typevector(1)=event.value
     'sl_azimin': widgetData.typevector(2)=event.value
     'sl_azimax': widgetData.typevector(3)=event.value
     else :$
      widgetData=papco_panel_Event(event, widgetData)
   endcase


END

