;******************************************************************************
;* file papcoadd_cassini_caps
;*      papcoadd_cassini_caps_line
;*      papcoadd_cassini_caps_spectrum
;*      papcoadd_cassini_caps_angle
;* end
;*
;* this files contains all the collected routines which have to be added in
;* for CASSINI mimi lemms. 
;*
;* THIS IS A WORKING EXAMPLE MODULE WHICH CAN BE INCLUDED INTO PAPCO AS IS
;* (it doesn't plot very exiting data though!)
;* THIS TEMPLATE IS INTENDED AS THE STARTING POINT FOR DEVELOPING OWN
;* MODULES. SIMPLY COPY THE WHOLE cassini_caps DIRECTORY TREE AND RENAME
;* EVERY OCCURRENCE OF "cassini_caps" WITH YOUR MODULE NAME, IN ALL THE FILES
;* AND FILENAMES UNDER THE cassini_caps MODULE DIRECTORY TREE.
;*
;* PAPCO is written such that if the files starting with "papcoadd" exist 
;* in the PAPCO_MODULES directory, then the routines contained therein
;* are utilized to add that plot type into the PAPCO package. 
;* This ensures easy extendibility of PAPCO; also makes PAPCO "configurable" 
;* by adding or leaving out specific plot types.
;*
;* ALL that is needed to include the module is to include the path to the
;* module directory tree in the IDL search path in the PAPCO startup file, 
;* eg assuming all modules are under the directory "PAPCO_BASE":
;*
;* !path = !path +':'+expand_path('+'+PAPCO_BASE+'/PAPCO_modules/cassini_caps')
;*
;* If the plot type is in itself a sub-menu containing more cassini_caps, then
;* these must be listed at the top of this file, as it is read by PAPCO.
;* the entries for each sub-plot-type must come after the line 
;* "file papcoadd_cassini_caps" and before the "end" line, one entry per line.
;* Each of the routines here must then be duplicated for each sub-plot-type
;* used: see $PAPCO_MODULES/papcoadd_crres_plots for an example.
;*
;* Apart from editing this template the user has to write a panel 
;* editor widget for his cassini_caps. Use the file 
;* $PAPCO_XX/PAPCO_modules/cassini_caps/PAPCO_interface/cassini_caps_panel.pro
;* as the template.
;*
;* If slice functions are to be supported a slice program needs to be written
;* too. Use the template file
;*    $PAPCO_XX/PAPCO_modules/cassini_mimi/papco_interface/cassini_mimi_slice.pro.
;*
;* Finally, a data read and data plot routine are required as well. The data
;* read routine will be specific for each cassini_mimi. An example for a standard
;* line plot is given in the file 
;*    $PAPCO_XX/PAPCO_modules/cassini_mimi/plot_read/p_data_type_line.pro
;* and for a standard color spectrogram in 
;*    $PAPCO_XX/PAPCO_modules/cassini_mimi/plot_read/p_data_type_spec.pro. 
;* These templates can also be used as a basis for more complex plots.
;*
;*
;* The Way these add-in routines work:
;* ===================================
;*
;* At PAPCO compile the routine find_modules.pro is used to search the IDL
;* search path, and all the init_module.pro batch files found in the module 
;* sub-directories \papco_interface are written to the papco_add_init.pro
;* batch file and then executed: In this way the user can specify all the 
;* routines that need to be compiled for his module at startup.
;* 
;* At the start-up of PAPCO itself, all module sub-directories 
;* \papco_interface are searched for all files beginning with "papcoadd_". 
;* From the names of these files the cassini_mimi name is stripped off and used 
;* to construct all the procedure names contained in this file. The first 
;* routine, "addplottype_cassini_mimi", is used to add the button for this 
;* cassini_mimi to the PAPCO main window. Since "addplottype_cassini_mimi" simply 
;* returns widget ID of the button, the button itself can contain further 
;* sub-menus for plot sub-types (e.g. the button for SWE can contain a menu of
;* buttons for a range of SWE plots, ie moments, spectra, etc).
;*
;* To inform PAPCO of all these sub-plot types (and to avoid having to
;* write a separate file for each sub-plot type) all the types have to
;* be listed in the header of the file papcoadd_cassini_mimi.pro. These entries
;* have to finish off with the line ";* end" - see papcoadd_crres_plots.pro
;* for an example. All cassini_mimi names are stored in the string array
;* cassini_mimis and are placed in a common block:
;*    COMMON ADDINS, cassini_mimis
;*
;* If there are no sub-plot types, then the cassini_mimi as used in the file name
;* for papcoadd_cassini_mimi.pro is used to construct the calls to the procedures
;* in that file. The convention for naming these procedures is
;* "procedure_name_cassini_mimi" , so that PAPCO can always construct the right
;* procedure call when needed. 
;*
;* When there are sub-plot types, their names get used to construct the
;* required calls. There are 6 procedures that need to be called for each
;* cassini_mimi, so that a set of these 6 procedures has to be written for each
;* sub-plot type. In the same way, you need to write a panel editor for each
;* plot type (sometimes several plot sub-types may share the same editor,
;* e.g. multiple line plots or color spectra, see papcoadd_crres_plots).
;* 
;* In PAPCO, whenever an action for a cassini_mimi is taken, the information
;* stored in the structure widgetData is queried. In this structure there
;* is an array of PlotInfo structures, which contain all the relevant 
;* information for a chosen panel. The field plotInfo.panelkind is loaded
;* with the name of the cassini_mimi. This is set by the user in this file in
;* the routine addpanel_cassini_mimi ( "panelKind='cassini_mimi'" ). THIS MUST
;* BE SET CORRECTLY AS ALL THE ROUTINE NAMES USED IN THIS ADD-IN FILE ARE
;* CONSTRUCTED USING THIS INFORMATION!
;*
;* Whenever PAPCO has to undertake some action for a
;* cassini_mimi, the field plotInfo.panelkind is compared to the entries in the
;* (COMMON ADDINS) cassini_mimiS array. If a match is found, the required
;* procedure name is constructed and called If not, an error occurs.
;*
;*    These procedures are contained:
;*
;*    PRO addplottype_cassini_caps,plottypes_base,pb_plottype
;*        ->  adds the button for the new plot type to the main window.
;*
;*    PRO plottable_cassini_caps, plottable
;*        -> Sets a variable to indicate plottable without orbit.
;*
;*    PRO descriptor_cassini_caps, plotInfo, descriptStr
;*        -> Constructs the string used to label plots in the panels window
;*
;*    PRO addpanel_cassini_caps
;*        -> Calls the panel editor for this plottype and adds the plot to the
;*            panel window.
;*
;*    PRO editor_cassini_caps, plotNumber, newPlotData
;*        -> This procedure calls the panel editor when called from 
;*           PAPCO_EditPlot (papco_paneledit.pro).
;*
;*    PRO  draw_cassini_mimi, panelVector, currPlotInfo, get_data_call, IsRead
;*        -> This procedure performs the actual data read call and the plot 
;*           call for this plottype
;******************************************************************************

;******************************************************************************
;* PROCEDURE:     
;*      PRO addplottype_cassini_caps,plottypes_base,buttonGeometry,pb_plottype
;* 
;* DESCRIPTION:  
;*      This procedure adds a button for this cassini_mimi to the plottypes_base.
;*      The UVALUE declared here is critical, since it is used to construct
;*      the procedure names of all procedures contained in this add-in
;*      package, so that they can be automatically called when needed.
;*      The button itself can contain access to a sub-menu for further
;*      cassini_mimis.
;*
;* INPUTS:       
;*      plottypes_base  the widget ID of the parent widget to contain the
;*                      plottype buttons.
;*
;* OUTPUTS: 
;*      pb_plottype     the widget ID of the button created here.
;*      buttonGeometry  size of the button. Only used in simple
;*                      WIDGET_BUTTON to make them the same size as the
;*                      compound CW_PDMENU, pulldown for sub-types.
;*
;* KEYWORDS:
;*       none
;* 
;* CALLING SEQUENCE:
;*      addplottype_cassini_mimi,plottypes_base,buttonGeometry,pb_plottype
;*
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;*     modified April 2001, S. Bourdarie for cassini caps
;******************************************************************************
pro addplottype_cassini_caps,plottypes_base,buttonGeometry,pdm_cassini_caps


  MenuDesc = [ $
        { CW_PDMENU_S,   3, strcenter('CASS CAPS',10) }, $ 
        { CW_PDMENU_S, 0, ' configure CASSINI CAPS ' }, $
        { CW_PDMENU_S, 0, ' plot LINE' }, $
        { CW_PDMENU_S, 0, ' plot SPECTRUM ' }, $
        { CW_PDMENU_S, 0, ' plot ANGLE' } ]
  pdm_cassini_caps = CW_PDMENU(plottypes_base, MenuDesc, $
                  UVALUE='pdm_cassini_caps')

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO plottable_cassini_caps, plottable
;*
;* DESCRIPTION:  
;*      Sets a variable to indicate plottable without orbit.
;*
;* INPUTS:       
;*      none
;*
;* OUTPUTS: 
;*      plottable        a boolean 0 if plottable only by CRRES orbit
;*                                1 if plottable by time also.
;*
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      plottable_cassini_mimi, plottable
;*
;* MODIFICATION HISTORY:       
;*      written November 1995, Reiner Friedel
;*     modified April 2001, S. Bourdarie for cassini caps
;******************************************************************************
pro plottable_cassini_caps_line, plottable

   plottable=1     ; 1 -> plotable without orbit info, by time 
                   ; 0 -> plotable only by orbit number not by time 
END

pro plottable_cassini_caps_spectrum, plottable

   plottable=1     ; 1 -> plotable without orbit info, by time 
                   ; 0 -> plotable only by orbit number not by time 
END

pro plottable_cassini_caps_angle, plottable

   plottable=1     ; 1 -> plotable without orbit info, by time 
                   ; 0 -> plotable only by orbit number not by time 
END

pro canget_cassini_caps_line, canget

   canget=  1      ; 1 -> plot routine can return just data 
                   ; 0 -> feature disabled
END

pro canget_cassini_caps_spectrum, canget

   canget=  1      ; 1 -> plot routine can return just data 
                   ; 0 -> feature disabled
END

pro canget_cassini_caps_angle, canget

   canget=  1      ; 1 -> plot routine can return just data 
                   ; 0 -> feature disabled
END
;******************************************************************************
;* PROCEDURE:     
;*      PRO descriptor_cassini_caps, plotInfo, descriptStr
;*
;* DESCRIPTION:  
;*      Constructs the string used to label plots in the panel window.
;*      The plotinfo structure passed in here is the panel's plotinfo
;*      as returned by the panel editor for this cassini_mimi. You can use
;*      the information in that structure to construct some informative
;*      label for the panel which then appears in the Panel Compose list.
;*
;* INPUTS:       
;*      plotInfo        structure containing all data for this plot.
;*                      (structure type: PAPCO_PLOTINFO, 
;*                                             see papco_variables.text) 
;*
;* OUTPUTS: 
;*      descriptStr     a string containing the label
;*
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      descriptor_cassini_mimi, plotInfo, descriptStr
;*
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;*     modified April 2001, S. Bourdarie for cassini caps
;******************************************************************************
pro descriptor_cassini_caps_line, plotInfo, descriptStr

   COMMON cassini_caps_line_energy, energy

     case PlotInfo.ioptions(0) of
       0: cassini_caps_definition,/ELS
       1: cassini_caps_definition,/IMS
       2: cassini_caps_definition,/IBS
     endcase

   typeNames=['ELS','IMS','IBS']


   AvNames=['not av.','elev. av.','azim. av.','elev.+azim. av.']


   descriptStr=descriptStr+typeNames(PlotInfo.ioptions(0))+' En.' +$
     strtrim(string(energy(PlotInfo.ioptions(1))),2)+' eV ' +$
     AvNames(PlotInfo.ioptions(2))+'['+ $
     strtrim(string(PlotInfo.typevector(0)),2)+'-'+$
     strtrim(string(PlotInfo.typevector(1)),2)+']['+$
     strtrim(string(PlotInfo.typevector(2)),2)+'-'+$
     strtrim(string(PlotInfo.typevector(3)),2)+']'
     
END

pro descriptor_cassini_caps_spectrum, plotInfo, descriptStr

   typeNames=['ELS','IMS','IBS']


   AvNames=['not av.','elev. av.','azim. av.','elev.+azim. av.']



   descriptStr=descriptStr+typeNames(PlotInfo.ioptions(0))+' '+$
     AvNames(PlotInfo.ioptions(2))
   
   if PlotInfo.ioptions(2) eq 1 then $
     descriptStr=descriptStr+'['+strtrim(string(PlotInfo.typevector(0)),2)+','+$
             strtrim(string(PlotInfo.typevector(1)),2)+'] Azi.'+strtrim(string(PlotInfo.typevector(2)),2)
   if PlotInfo.ioptions(2) eq 2 then $
      descriptStr=descriptStr+'['+strtrim(string(PlotInfo.typevector(2)),2)+','+$
             strtrim(string(PlotInfo.typevector(3)),2)+'] Ele.'+strtrim(string(PlotInfo.typevector(0)),2)
   if PlotInfo.ioptions(2) eq 3 then $
      descriptStr=descriptStr+'['+strtrim(string(PlotInfo.typevector(0)),2)+','+$
             strtrim(string(PlotInfo.typevector(1)),2)+']'+$
	     '['+strtrim(string(PlotInfo.typevector(2)),2)+','+$
             strtrim(string(PlotInfo.typevector(3)),2)+']'

END

pro descriptor_cassini_caps_angle, plotInfo, descriptStr

   COMMON cassini_caps_line_energy, energy

     case PlotInfo.ioptions(0) of
       0: cassini_caps_definition,/ELS
       1: cassini_caps_definition,/IMS
       2: cassini_caps_definition,/IBS
     endcase

   typeNames=['ELS','IMS','IBS']


   AvNames=['not av.','elev. av.','azim. av.']


   descriptStr=descriptStr+typeNames(PlotInfo.ioptions(0))+' En.' +$
     strtrim(string(energy(PlotInfo.ioptions(1))),2)+' eV ' +$
     AvNames(PlotInfo.ioptions(2))
  

   if PlotInfo.ioptions(2) eq 0 then begin
      if PlotInfo.ioptions(3) eq 0 then $ ;elevation vs time
         descriptStr=descriptStr+' Azi.'+strtrim(string(PlotInfo.typevector(2)),2)
      if PlotInfo.ioptions(3) eq 1 then $ ;azimut vs time
         descriptStr=descriptStr+' Ele.'+strtrim(string(PlotInfo.typevector(0)),2)
   endif
   if PlotInfo.ioptions(2) eq 1 then $
     descriptStr=descriptStr+'['+strtrim(string(PlotInfo.typevector(0)),2)+','+$
             strtrim(string(PlotInfo.typevector(1)),2)+']'
   if PlotInfo.ioptions(2) eq 2 then $
      descriptStr=descriptStr+'['+strtrim(string(PlotInfo.typevector(2)),2)+','+$
             strtrim(string(PlotInfo.typevector(3)),2)+']'

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO addpanel_cassini_caps
;*
;* DESCRIPTION:  
;*      Calls the panel editor for this plottype and adds the plot to the
;*      panel window.
;*      Using an instance of the PAPCO_PLOTINFO structure called "defaults"
;*      the default settings of the panel editor can be controlled. The
;*      desired default values for the plotinfo tags need to be edited by
;*      the user. This plotinfo structure will be passed to the panel editor
;*      When it is first called for the cassini_mimi.
;*      "panelEditor" must be set to the name of the panel editor routine.
;*
;* INPUTS:       
;*      event           event value of papco main window. This is
;*                      only used if there are plot sub_types, 
;*                      for example see papcoadd_crres_plots
;* OUTPUTS: 
;*      none
;*
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      addpanel_cassini_mimi
;*
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;*     modified April 2001, S. Bourdarie for cassini caps
;***************************************************************************
pro addpanel_cassini_caps,event

   common get_error, get_err_no, get_err_msg

; The FIRST time you use this module you will have to configure it,
; i.e. set the data paths etc. Once you have done this PAPCO will
; write a plot_type.config file into your papco_user_XX/papco_modules_defaults
; directory. 
; So here we check if this plot_type.config file exists, and if not,
; we disallow the execution of any other option apart from "configure
; plot_type". In this way the user is forced to configure the module
; before using it the first time!

   panelKind='cassini_caps'
   r_module_config, panelKind, config_data, msg
   if (get_err_no ne 0) and (event.value ne 1) then begin
     message,'You need to configure the module first', /cont
     papco_module_configure,panelKind
     return
   endif     

   CASE Event.Value OF 
      1: BEGIN
         papco_module_configure, panelKind
         return
      END
      2: BEGIN
         panelKind='cassini_caps_line'
         panelEditor=panelkind+'_panel'
         defaults=papco_getPlotInfoStruct()  ; returns a basic plotinfo structure
         defaults.typeVector=[0,7,0,15]       ; set some tags to desired defaults
         defaults.channel=0                  ; channel tag is used for the color table
         defaults.ioptions(0)=0
	 defaults.ioptions(2)=3
     END
      3: BEGIN
         panelKind='cassini_caps_spectrum'
         panelEditor=panelkind+'_panel'
         defaults=papco_getPlotInfoStruct()  ; returns a basic plotinfo structure
         defaults.typeVector=[0,7,0,15]       ; set some tags to desired defaults
         defaults.channel=0                  ; channel tag is used for the color table
	 defaults.ioptions(2)=3
      END
      4: BEGIN
         panelKind='cassini_caps_angle'
         panelEditor=panelkind+'_panel'
         defaults=papco_getPlotInfoStruct()  ; returns a basic plotinfo structure
         defaults.typeVector=[0,7,0,15]       ; set some tags to desired defaults
         defaults.channel=0                  ; channel tag is used for the color table
         defaults.ylinlog=0
	 defaults.ioptions(2)=2
      END
   ENDCASE
   papco_Add_Plot, panelEditor, panelKind, initialPlotInfo=defaults

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO editor_cassini_caps, plotNumber, newPlotData
;* 
;* DESCRIPTION:  
;*      This procedure calls the panel editor when called from 
;*      PAPCO_EditPlot (papco_paneledit.pro). Provided here to allow
;*      several cassini_mimis to share a common panel editor. Normally this
;*      procedure does not need to be edited beyond the required name
;*      change for cassini_mimi.
;*
;* INPUTS:       
;*      plotNumber      anInteger. 0<=plotNumber<=widgetData.numberOfPlots
;*                      the index of the plot in widgetData.plotInfos.
;*            
;* OUTPUTS: 
;*      newPlotData     the new structure returned by the panel editor
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      editor_cassini_mimi, plotNumber, newPlotData
;*
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;*     modified April 2001, S. Bourdarie for cassini caps
;******************************************************************************
PRO editor_cassini_caps_line, plotNumber, newPlotData

  COMMON PLOT_COMPOSER, widgetData

   plotInfo=widgetData.plotInfos(plotNumber)
   newPlotData=cassini_caps_line_panel(plotInfo, $
        ACTION='Update '+plotInfo.panelKind, $
      GROUP=widgetData.main_base)

END

PRO editor_cassini_caps_spectrum, plotNumber, newPlotData

  COMMON PLOT_COMPOSER, widgetData

   plotInfo=widgetData.plotInfos(plotNumber)
   newPlotData=cassini_caps_spectrum_panel(plotInfo, $
        ACTION='Update '+plotInfo.panelKind, $
      GROUP=widgetData.main_base)

END

PRO editor_cassini_caps_angle, plotNumber, newPlotData

  COMMON PLOT_COMPOSER, widgetData

   plotInfo=widgetData.plotInfos(plotNumber)
   newPlotData=cassini_caps_angle_panel(plotInfo, $
        ACTION='Update '+plotInfo.panelKind, $
      GROUP=widgetData.main_base)

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO draw_cassini_caps, panelVector, currPlotInfo, orbitNo $
;*                          get_data_call, slice_type, IsRead
;* DESCRIPTION:  
;*      This procedure performs the actual data read call and the plot call
;*      for this plottype. The user need to pass to his plot routine all the
;*      parameters needed, typically these parameters are stored in the 
;*      currplotinfo structure or constructed from them. The currplotinfo
;*      structure tags normally used for passing information from the panel
;*      editor to the plot routine are:
;*          type_vector - a four element array for choosing the type of data
;*                        (eg. elec or ion, diff or integ., pitch angle, 
;*                             or sector, channel ....)
;*          switch      - a three element vector for specific plot appearance
;*                        (eg. smoothing/averaging, interpolating ...)
;*          filename    - a filename
;*      
;*      These tags have grown out of traditional usage and can be used for
;*      anything the user wants to use them for. There are plenty variables
;*      for most applications: Adding a new tag to the plotinfo structure
;*      is not user-definable but need to be done at the PAPCo core level.
;* 
;*      The panelVector MUST be passed to your plotroutine as it is used
;*      to set the panel position.
;*      The _EXTRA=extra_PlotInfo should be passed if you intend to make use
;*      of PAPCO functionality such as overplotting, linestyle choice and
;*      color bar choice.
;*
;* INPUTS:       
;*      panelVector    position of the plot (see papco_conventions.text, 6)
;*      currPlotInfo   structure containing all data for this plot.
;*                     (structure PAPCO_PLOTINFO, see papco_variables.text) 
;*
;* OUTPUTS: 
;*      get_data_call  a string containing the data call. Used by other
;*                     routines that need to know how to get the data
;*                     in a panel, e.g. to draw spectra, distribution functions
;*      IsRead         a boolean which is set when the the data is read the
;*                     first time. Useful if a given data-read puts data
;*                     into commons which can be plotted in many ways.
;*                     When set, data does not have to be read again.
;*      slice_type      Selects which slices are supported and for which
;*                      slice routines have been written.
;*                      This is a byte logical with 8 switches. Currently
;*                      Three are supported (this may be extended later)
;*                        1 = mouse zooms (utility slice, always possible)
;*                        2 = mouse slices
;*                        4 = mouse calls level zero
;*                        8 = mouse views slice (polar)  
;*                       16 = mouse views ephemeris
;*                       32 = mouse writes paneldata (utility slice)
;*                       64, 128 = not supported yet
;*                      you can support more than one type of slice
;*                      by arithmetically adding the options, eg
;*                      slice_type=5  support zoom and level zero
;*
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      draw_cassini_mimi, panelVector, currPlotInfo, orbitNo, $
;*                                 get_data_call, slice_type, IsRead
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;*     modified April 2001, S. Bourdarie for cassini caps
;******************************************************************************
pro draw_cassini_caps_line, panelVector, currPlotInfo, orbitNo, $
                                 get_data_call, slice_type, IsRead, $
                     OUTPUT=OUTPUT

   COMMON extra, extra_PlotInfo  ; passes in PAPCO-set keywords structure

   get_data_call="r_cassini_caps"   ; data read call used by slices

; set parameter for plot routine. In this example, channel 1 -10 chooses the 
; channel number for a line plot, channel=0 chooses a spectrum color plot.
   channel = currPlotInfo.ioptions
; set up which mouse functions to allow
   slice_type=1+8+16


   IF NOT IsRead THEN BEGIN      ; check if data has already been read
      r_cassini_caps,currPlotInfo.ioptions(0)
      ; check the read for success.
      IF PAPCO_Draw_CheckResult(currPlotInfo.panelKind, $
                                panelVector, orbitNr, $
                                currPlotInfo.typeVector, $
                                currPlotInfo.channel) $
      THEN IsRead=1 
   ENDIF


  IF IsRead THEN BEGIN
      if currPlotInfo.ioptions(0) ne 1 then $
        p_cassini_caps_line, panelVector,currPlotInfo.typeVector, channel, OUTPUT=OUTPUT, _EXTRA=extra_PlotInfo $
      else $
        p_cassini_caps_line2, panelVector,currPlotInfo.typeVector, channel, OUTPUT=OUTPUT, _EXTRA=extra_PlotInfo
      IsRead=0
   ENDIF

; Note: The structure extra_PlotInfo passed to the plot routine contains the
;       following tags: 
;                  overPlot               - overplot switch
;                  plots_AtThisPosition   - no. of overplots here
;                  subtable               - index of PAPCO colortable 
;                  PSYM                   - linestyle keyword for plot
;                  COLOR                  - linecolor keyword for plot
;               
;       They are set by PAPCO and your plot routine can use them if you
;       want to support that functionality.

END


pro draw_cassini_caps_spectrum, panelVector, currPlotInfo, orbitNo, $
                                 get_data_call, slice_type, IsRead, $
                     OUTPUT=OUTPUT

   COMMON extra, extra_PlotInfo  ; passes in PAPCO-set keywords structure

   get_data_call="r_cassini_caps"   ; data read call used by slices

; set parameter for plot routine. In this example, channel 1 -10 chooses the 
; channel number for a line plot, channel=0 chooses a spectrum color plot.
   channel = currPlotInfo.ioptions
; set up which mouse functions to allow
   slice_type=1+8+16


   IF NOT IsRead THEN BEGIN      ; check if data has already been read
      r_cassini_caps,currPlotInfo.ioptions(0)
      ; check the read for success.
      IF PAPCO_Draw_CheckResult(currPlotInfo.panelKind, $
                                panelVector, orbitNr, $
                                currPlotInfo.typeVector, $
                                currPlotInfo.channel) $
      THEN IsRead=1 
   ENDIF


  IF IsRead THEN BEGIN
      if currPlotInfo.ioptions(0) ne 1 then $
        p_cassini_caps_spectrum, panelVector,currPlotInfo.typeVector, channel,	OUTPUT=OUTPUT, _EXTRA=extra_PlotInfo $
      else $
        p_cassini_caps_spectrum2, panelVector,currPlotInfo.typeVector, channel, OUTPUT=OUTPUT, _EXTRA=extra_PlotInfo
      IsRead=0
   ENDIF

; Note: The structure extra_PlotInfo passed to the plot routine contains the
;       following tags: 
;                  overPlot               - overplot switch
;                  plots_AtThisPosition   - no. of overplots here
;                  subtable               - index of PAPCO colortable 
;                  PSYM                   - linestyle keyword for plot
;                  COLOR                  - linecolor keyword for plot
;               
;       They are set by PAPCO and your plot routine can use them if you
;       want to support that functionality.

END

pro draw_cassini_caps_angle, panelVector, currPlotInfo, orbitNo, $
                                 get_data_call, slice_type, IsRead, $
                     OUTPUT=OUTPUT

   COMMON extra, extra_PlotInfo  ; passes in PAPCO-set keywords structure

   get_data_call="r_cassini_caps"   ; data read call used by slices

; set parameter for plot routine. In this example, channel 1 -10 chooses the 
; channel number for a line plot, channel=0 chooses a spectrum color plot.
   channel = currPlotInfo.ioptions
; set up which mouse functions to allow
   slice_type=1+8+16


   IF NOT IsRead THEN BEGIN      ; check if data has already been read
      r_cassini_caps,currPlotInfo.ioptions(0)
      ; check the read for success.
      IF PAPCO_Draw_CheckResult(currPlotInfo.panelKind, $
                                panelVector, orbitNr, $
                                currPlotInfo.typeVector, $
                                currPlotInfo.channel) $
      THEN IsRead=1 
   ENDIF


  IF IsRead THEN BEGIN
      if currPlotInfo.ioptions(0) ne 1 then $
        p_cassini_caps_angle, panelVector,currPlotInfo.typeVector, channel, OUTPUT=OUTPUT, _EXTRA=extra_PlotInfo $
      else $
        p_cassini_caps_angle2, panelVector,currPlotInfo.typeVector, channel, OUTPUT=OUTPUT, _EXTRA=extra_PlotInfo
      IsRead=0
   ENDIF

; Note: The structure extra_PlotInfo passed to the plot routine contains the
;       following tags: 
;                  overPlot               - overplot switch
;                  plots_AtThisPosition   - no. of overplots here
;                  subtable               - index of PAPCO colortable 
;                  PSYM                   - linestyle keyword for plot
;                  COLOR                  - linecolor keyword for plot
;               
;       They are set by PAPCO and your plot routine can use them if you
;       want to support that functionality.

END
