;******************************************************************************
;* file p_cassini_caps_angle.pro
;*
;******************************************************************************
;******************************************************************************
;* PROCEDURE:     
;*      p_cassini_caps_angle, panel, type, $
;*           OVERPLOT=OVERPLOT,$
;*           PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           SUBTABLE=SUBTABLE, $
;*           _EXTRA=extra_par
;* 
;* DESCRIPTION:  
;*	This procedure plots data of type data_type for inclusion in papco.  
;*      Data must have been read and placed into common blocks before this
;*      routine is called. Behaviour is controlled by the vector panel
;*	which controls positioning of the plot (see papco_conventions.text
;*      in $papco). In general, the following plot conventions are used:
;*      	- if the panel is at the bottom plot time axis plus any
;*                additional axis (ie ephemeris) needed.
;*              - if panel is at the top also plot a title
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a papco_colorbar
;*              - panels all have common time
;*              - y-scaling (or z-scaling for color plots) is either
;*              - automatic or manually set using common yscale or zscale    
;*
;* INPUTS:       
;*	panel	three element integer vector. Controls positioning of the plot
;*              (see papco_conventions.text in $papco)
;*      type    optional parameter for options. Normally is a four element
;*              integer vector. This is normally equivalent to the field
;*              typeVector of the papco structure papco_PLOTINFO which is set
;*              by the panel editor. Additional fields that can be used if
;*              needed are papco_PLOTINFO.switch (three element vector) and 
;*              papco_PLOTINFO.channel (integer). If your plot routine needs
;*              more input fields than these you have to extend the definition
;*              of the structure papco_PLOTINFO
;*              (procedure papco_getPlotInfoStruct in $papco/papco.pro). Try to
;*              avoid this!
;*              
;* OUTPUTS: 
;*   	none
;*
;* KEYWORDS:
;*      OVERPLOT  if you support this you can alter the appearance of your
;*              plot if it is an overplot to an exisiting panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mag. field
;*              (see p_crres_mag for an example).
;*      PLOTS_ATTHISPOSITION  used to control the left side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and line style.   
;* 
;* CALLING SEQUENCE:
;*       p_map, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY:       
;*     Written April 2001,  S. Bourdarie (ONERA/DESP)
;******************************************************************************
pro p_cassini_caps_angle, panel,type, channel,OUTPUT=OUTPUT, $
                      OVERPLOT=OVERPLOT, $
                      PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION, $
                      SUBTABLE=SUBTABLE, $
                      _EXTRA=extra_par

; It doesn't make much sense to use a color plot as an overplot, as it will
; hide what's below. Line styles are also not useful. So the keywords 
; OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,_EXTRA=extra_par
; are not used at all here, but are in the call for consistency...

; NOTE: I use a little trick in passing in the keywords to to this
; routine. The calling routine's call is simply:
;         p_map_line, panelVector, currPlotInfo.typeVector(0), $ 
;                           _EXTRA=extra_PlotInfo
; extra_PlotInfo contains the tags OVERPLOT, PLOTS_ATTHISPOSITION,
; SUBTABLE and then some tags which are meant to be passed to the plot
; call. By putting in the explicit keywords in the procedure definition they
; get split out of the structure extra_PlotInfo that was passed in, and only
; those tags whichare meant for the plot routine remain in extra_par.

; Comments here should include an explanation of the plot options used
; in type, switch. Replace type in the call with whatever variables
; you need, or add more

; Comments here should include an explanation of the plot options used
; in type, switch. Replace type in the call with whatever variables
; you need, or add more

; The following common blocks are needed:

   common mjdt, mjdt_start, mjdt_end    ;common time limit in mjdt
   common yscale, yscl			;man/auto yscaling
   common zscale, zscl			;man/auto zscaling
   common shift_label, down             ;common for x-axis label shifting
   common coordinateSystems, plotted_x, plotted_y  
                                        ;info on coords used by mouse

; One of the common blocks should contain the plot data as returned by the
; data read procedure r_cassini_mimi_lemms. 

   common cassini_caps_data, input_header, input_data
   COMMON cassini_caps_line_energy, energy

   papco_Set_SubTable, SUBTABLE         ;sets the papco color table index
   panelset,panel			;sets the panel position viewport

; option to automatically calculate ytick numbers in order to avoid to
; many ticks if panels get too narrow. Maximum is set to 10. You may
; leave this out and allow plot to use the built-in defaults.

   ytickno=50*(!p.position(3)-!p.position(1))
   if ytickno gt 10 then ytickno=10

; since there is a common time axis plots look better if they all use the same
; number of xticks. In papco, this number is 6 by default. This is set by papco
; using the IDL graphics system variable !X.TICKS.
; You can overide this for your own plot by setting the varbale
; below. However, for plotting panels together, it makes sense to stick to the
; default. If you do set this variable, it needs to be included in the 
; extra_plotPar_common definition below.

;   xtickno=6

; NB!!! NB!!! NB!!! NB!!! NB!!! NB!!! NB!!! NB!!! 
; Avoid setting the global IDL graphics system variables in your plot routine,
; as they may affect other modules' plot routines. The IDL graphics system 
; variables are set by papco to default values that gurantee some uniformity
; of plot appearance and should not be tampered with - unless you don't
; care if your plot "looks good" when combined with the other modules.
; NB!!! NB!!! NB!!! NB!!! NB!!! NB!!! NB!!! NB!!! 

; for the color plot routine, you need to set up arrays containing the upper  
; and lower limits of each channel. Here is an example for CRRES MEB energy 
; channels. There must be as many entries as there are channels in the data
; array! 

  case channel(0) of
    0: cassini_caps_definition,/ELS
    1: cassini_caps_definition,/IMS
    2: cassini_caps_definition,/IBS
 endcase

  if channel(3) eq 0 then  begin ; plot elevation vs time
     uytitle='elevation'
     if channel(0) eq 0 then begin  ; ELS
        yminarr=findgen(8)
        ymaxarr=findgen(8)+1.
     endif else begin   ;IBS
        yminarr=findgen(3)
        ymaxarr=findgen(3)+1.
     endelse  
  endif else begin   ; plot azimut vs time
     uytitle='azimut'
     yminarr=findgen(16)
     ymaxarr=findgen(16)+1.
  endelse   
          
  n_channels=n_elements(yminarr)
  yarr=fltarr(n_channels,2)   
  yarr(*,0)= yminarr 
  yarr(*,1)=ymaxarr

; for color plots, the y-axis normally refers to a channel. You have to 
; re-scale the data array to the range required before passing it to the
; color plot routine, also the ymin and ymax arrays.

; the plot-calls normally use the arrays time for the time axis and yarr (ymat 
; for color plots) for the y or z axis.
; depending on your plot options different data may be selected and different
; axis-labeling may be needed. This is normally done using a case statement.

; in this example we have a single test array, so no case statement is needed.

; set the begin end end times. papco convention is to use running secconds
; starting with the no of seconds since the begiing of the start day.
   time=input_data.time

; restrict data to actual time range requested - this makes for faster zooming
   xut1=mjdt_start.t                       
   xut2=long(mjdt_end.mjd-mjdt_start.mjd)*86400l+mjdt_end.t

   index=where((time ge xut1) and (time le xut2),count)
   if count ne 0 then begin
      input_data=input_data(index)
      time=time(index)
   endif

   speNames=['ELS','IMS','IBS']

   uztit='[counts]'
   utitle='CASSINI-CAPS '+ speNames(channel(0))+'!c' 
   
   if channel(2) eq 0 then begin
      if channel(3) eq 0 then $ ;elevation vs time
         utitle=utitle+'not av. Azi.'+strtrim(string(type(2)),2)+$
	 ' '+strtrim(string(energy(channel(1))),2)+' eV'
      if channel(3) eq 1 then $ ;azimut vs time
         utitle=utitle+'not av. Ele.'+strtrim(string(type(0)),2)+$
	 '!c'+strtrim(string(energy(channel(1))),2)+' eV'
   endif
   if channel(2) eq 1 then $
     utitle=utitle+'Ele. av.['+strtrim(string(type(0)),2)+','+$
             strtrim(string(type(1)),2)+']!c'+$
	     strtrim(string(energy(channel(1))),2)+' eV'
   if channel(2) eq 2 then $
      utitle=utitle+'Azi. av.['+strtrim(string(type(2)),2)+','+$
             strtrim(string(type(3)),2)+']!c'+$
	     strtrim(string(energy(channel(1))),2)+' eV'

   nodat=65535
   yray=fltarr(n_elements(time),n_elements(yminarr))

   if channel(3) eq 0 then begin
      for i=0,n_elements(yminarr)-1 do begin
         if channel(2) eq 0 then begin  ;no average
            yray(*,i)=input_data_buf(*).data(i,channel(1),type(2))
         endif

      
         if channel(2) eq 2 then  begin  ;azimut average
	    for tp=0,n_elements(time)-1 do begin
              indix=where(input_data(tp).data(i,channel(1),type(2):type(3)) ne nodat,count)     
              if count ne 0 then $
                 yray(tp,i)=total(input_data(tp).data(i,channel(1),indix))/float(count)$
	      else yray(tp,i)=nodat
           endfor
         endif
      endfor
   endif else begin
      for i=0,n_elements(yminarr)-1 do begin
         if channel(2) eq 0 then begin  ;no average
            yray(*,i)=input_data_buf(*).data(type(0),channel(1),i)
         endif

      
         if channel(2) eq 1 then begin  ;elevation average
	    for tp=0,n_elements(time)-1 do begin
              indix=where(input_data(tp).data(type(0):type(1),channel(1),i) ne nodat,count)     
              if count ne 0 then $
                 yray(tp,i)=total(input_data(tp).data(indix,channel(1),i))/float(count)$
	      else yray(tp,i)=nodat
           endfor
         endif
      endfor
   endelse


   yst=yminarr(0)
   yen=ymaxarr(n_elements(ymaxarr)-1)


; the variables used above have the following meaning:
;       uztit   : the z-axis (papco_colorbar) label
; 	uytitle : the y-axis label
;	utitle  : the right-of plot label defining the data



; the procedure autorange is optional and sets the zlimit automatically for
; the data, attempting to use the full dynamic range of the color table.
; you may need to supply your own procedure for your own data. 
; (in $papco/PLOT_prog/Common)

; yscl or zscale is an array of dimensions (*,4) and contains info
; on the scaling to be used. The first dimension identifies the panel, and the
; next dimension has three entries:  
;         yscl(*,0)  is  0 for "use automatic scaling", 1 for "use manual"
;         yscl(*,1)  is then ymin (or zmin)
;         yscl(*,2)  is then ymax (or zmax)
;         yscl(*,3)  is 0 for lin, 1 for log axis
; when scaling is set to automatic, the automatic y limits are loaded in.
; for color plots, the z-limits are used for the color (z-axis) scaling
   
   if (zscl(panel(0),0) eq 1) then begin
      zmin=zscl(panel(0),1)
      zmax=zscl(panel(0),2)
   endif else begin
;      autorange,panel(0),time,yray,zmax,zmin,1
      papco_autorange,yray,zmin,zmax,LOG=yscl(panel(0),3),nodat=nodat,exclude=5.,/verbose
      zscl(panel(0),1)=zmin
      zscl(panel(0),2)=zmax
   endelse
   if zscl(panel(0),3) eq 1 and ABS(floor(ALOG10(zmax))-floor(Alog10(zmin))) lt 1 then zmin=zmin/10.

   if (yscl(panel(0),0) eq 1) then begin
      yst=yscl(panel(0),1)
      yen=yscl(panel(0),2)
   endif else begin
      yscl(panel(0),1)=yst
      yscl(panel(0),2)=yen
   endelse

; uae auto y-labeling routine to get "good" y-lables. This sets the global IDL
; graphics system variables !Y.MINOR, !Y.TICKV, !Y.TICKS - so if you want to 
; use this feature, avoid setting these or their keywords in your plot call.
  papco_y_label, yst, yen, log=yscl(panel(0),3)    


; set up extra plot keywords, common ones first. 
; yrange, xrange  are always needed. 
; the rest you can set if you want to override papco defaults.
; See $papco_BASE/papco/papco_draw for the definition of the papco default
; graphic sytem variable settings.
  extra_plotPar_common={yrange:[yst,yen], xrange:[xut1,xut2], zrange:[zmin,zmax], $
                 ylog:yscl(panel(0),3), zlog:zscl(panel(0),3), $
                 ztitle:uztit, ystyle:1, $
                 xtickformat:'noticks',ztickformat:'papco_color_bar_log_ticks'}


; add keyword structure set here with the one passed in

   extra_plotPar=create_struct(extra_plotPar_common, extra_par)


; use papco routine to draw time axis. This checks for bottom plot and uses the
; user's xtickformat if it is something other than 'noticks'.
; If you use this rotuine, make sure your data plot routine uses xstyle=5 and 
; ystyle=5 (inhibit axis) so that you don't clobber those drawn by papco.  
; "down" sets the axis labeling level. 
  down=0
  papco_draw_time_axis, panel, OVERPLOT=OVERPLOT, _extra=extra_plotPar


; do color plot and color bar plot. nodata specifies the "nodata" flag
; in your data. Here we do not plot axis and data seperate, as
; papco_plot_colorspec cannot be used as an overplot!
  
  ;check the routine papco_plot_colorspec in papco_XX\papco_lib\color_plot
  ;for all the possible options of papco_plot_colorspec.
  ;Hint: you might want to use the resx, resy keywords, which set the
  ;resolution to plot add. Without these, the routine will try to use
  ;device resolution, which for postscript may result in HUGE files!
  ;resx, resy are in data coordinates. If the axis is logarithmic, 
  ;resx, resy need to be too.
  resx=(time(n_elements(time)-1)-time(0))/n_elements(time)/2.
  if yen lt yst then begin
     index=where(yminarr ge yen and yminarr le yst,n_channels)
  endif else begin
     index=where(yminarr ge yst and yminarr le yen,n_channels)
  endelse
  n_channels=n_channels+1
  
  if (yscl(panel(0),3) eq 0) then begin
    resy=ABS((yen-yst)/n_channels/2.)
  endif else begin
    resy=ABS((ALOG(yen)-ALOG(yst))/n_channels/2.)
  endelse
  print,resx,resy
  papco_plot_colorspec,yray,time,yarr, nodata=nodat, resx=resx, resy=resy, $
    xstyle=5, ystyle=5, _extra=extra_plotPar
  papco_color_bar, _extra=extra_plotPar
                    
; plot y-axis label at left of plot. Use scalable routine!  
  left_side_label,panel,uytitle,/rot90  
  
; store the coordinate information into common block
  plotted_x = !x  &  plotted_y = !y

; Since here papco_colorbars are plotted, labels are rotated by 90
; degrees and then plotted. No allowance is made for overplots, as
; color plots cannot be an overplot.
  right_side_label,panel,utitle,/rot90

end 





