;******************************************************************************
;* file p_cassini_caps_line2.pro
;*
;******************************************************************************
;******************************************************************************
;* PROCEDURE:     
;*      p_cassini_caps_line2, panel, type, $
;*           OVERPLOT=OVERPLOT,$
;*           PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           SUBTABLE=SUBTABLE, $
;*           _EXTRA=extra_par
;* 
;* DESCRIPTION:  
;*	This procedure plots data of type data_type for inclusion in PAPCO.  
;*      Data must have been read and placed into common blocks before this
;*      routine is called. Behavior is controlled by the vector panel
;*	which controls positioning of the plot (see papco_conventions.text
;*      in $PAPCO). In general, the following plot conventions are used:
;*              - if the panel is at the bottom plot time axis plus any
;*                additional axis (ie ephemeris) needed.
;*              - if panel is at the top also plot a title
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a colorbar.
;*              - panels all have common time
;*              - y-scaling (or z-scaling for color plots) is either
;*              - automatic or manualy set using common yscale or zscale   
;*
;* INPUTS:       
;*      panel   three element integer vector. Controls positioning of the plot
;*              (see papco_conventions.text in $PAPCO)
;*      type    optional parameter for options. Normally is a four element
;*              integer vector. This is normally equivalent to the field
;*              typeVector of the PAPCO structure PAPCO_PLOTINFO which is set
;*              by the panel editor. Additional fields that can be used if
;*              needed are PAPCO_PLOTINFO.switch (three element vector) and 
;*              PAPCO_PLOTINFO.channel (integer). If your plot routine needs
;*              more input fields than these you have to extend the definition
;*              of the structure PAPCO_PLOTINFO
;*              (procedure PAPCO_getPlotInfoStruct in $PAPCO/papco.pro). Try to
;*              avoid this!
;*              
;* OUTPUTS: 
;*      none
;*
;* KEYWORDS:
;*      OVERPLOT  if you support this you can alter the appearance of your
;*              plot if it is an overplot to an existing panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie caps. field.
;*              see p_crres_caps for an example.
;*      PLOTS_ATTHISPOSITION  used to control the left side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and line style.   
;* 
;* CALLING SEQUENCE:
;*       p_cassini_caps_line, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY:       
;*     Written April 2001,  S. Bourdarie (ONERA/DESP)
;******************************************************************************
pro p_cassini_caps_line2, panel,type, channel,OUTPUT=OUTPUT, $
                      OVERPLOT=OVERPLOT, $
                      PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION, $
                      SUBTABLE=SUBTABLE, $
                      _EXTRA=extra_par

; NOTE: I use a little trick in passing in the keywords to to this
; routine. The calling routine's call is simply:
;         p_cassini_caps_line, panelVector, currPlotInfo.typeVector(0), $ 
;                           _EXTRA=extra_PlotInfo
; extra_PlotInfo contains the tags OVERPLOT, PLOTS_ATTHISPOSITION,
; SUBTABLE and then some tags which are meant to be passed to the plot
; call. By putting in the explicit keywords in the procedure definition they
; get split out of the structure extra_PlotInfo that was passed in, and only
; those tags which are meant for the plot routine remain in extra_par.

; Comments here should include an explanation of the plot options used
; in type, switch. Replace type in the call with whatever variables
; you need, or add more

; The following common blocks are needed:

   common mjdt, mjdt_start, mjdt_end    ;common time limit in mjdt
   common yscale, yscl			;man/auto yscaling
   common shift_label, down             ;common for x-axis label shifting
   common coordinateSystems, plotted_x, plotted_y  
                                        ;info on coords used by mouse
COMMON papco_color_names, black, red, green, yellow, blue, capsenta, cyan, $
  white, burgundy, olive, dark_green, teal, $
  royal_blue, violet, dark_grey, grey

; One of the common blocks should contain the plot data as returned by the
; data read procedure r_cassini_caps. 

   common cassini_caps_data, input_header, input_data
   common time, xut1,xut2

   panelset,panel			;sets the panel position viewport

; make color array
  couleur_arr = [black, red, green, yellow, blue, capsenta, cyan, $
  white, burgundy, olive, dark_green, teal, $
  royal_blue, violet, dark_grey, grey]

; option to automatically calculate ytick numbers in order to avoid to
; many ticks if panels get too narrow. Maximum is set to 10. You may
; leave this out and allow plot to use the built-in defaults.

   ytickno=50*(!p.position(3)-!p.position(1))
   if ytickno gt 10 then ytickno=10

; since there is a common time axis plots look better if they all use the same
; number of xticks. In PAPCO, this number is 6 by default. This is set by PAPCO
; using the IDL graphics system variable !X.TICKS.
; You can override this for your own plot by setting the variable
; below. However, for plotting panels together, it makes sense to stick to the
; default. If you do set this variable, it needs to be included in the 
; extra_plotPar_common definition below.

;   xtickno=6

; NB!!! NB!!! NB!!! NB!!! NB!!! NB!!! NB!!! NB!!! 
; Avoid setting the global IDL graphics system variables in your plot routine,
; as they may affect other modules' plot routines. The IDL graphics system 
; variables are set by PAPCO to default values that guarantee some uniformity
; of plot appearance and should not be tampered with - unless you don't
; care if your plot "looks good" when combined with the other modules.
; NB!!! NB!!! NB!!! NB!!! NB!!! NB!!! NB!!! NB!!! 

; the plot-calls normally use the arrays time for the time axis and yarr (ymat 
; for color plots) for the y or z axis.
; depending on your plot options, different data may be selected and different
; axis-labeling may be needed. This is normally done using a case statement.
; In this example the data array consist of flux values in ten channels 
; at different energies, typical of particle spectrometer data. Here I use the
; channel information passed in the plotinfo structure from the panel editor
; to select the channel to be drawn.

; set the begin end end times. PAPCo convention is to use running seconds
; starting with the no of seconds since the beginning of the start day.
   time=input_data.time
   xut1=mjdt_start.t                       
   xut2=long(mjdt_end.mjd-mjdt_start.mjd)*86400l+mjdt_end.t

;time between two points (needed not to plot missing data)

   index=where((time ge xut1) and (time le xut2),count)
   if count ne 0 then begin
      time=time(index)
      input_data=input_data(index)
   endif

  if type(0) gt type(1) then begin
     burp=type(0)
     type(0) = type(1)
     type(1) = burp
  endif
  if type(2) gt type(3) then begin
     burp=type(2)
     type(2) = type(3)
     type(3) = burp
  endif

   speNames=['ELS','IMS','IBS']
   utitle1='CASSINI-CAPS '+ speNames(channel(0))+strtrim(string(channel(5)),2)+'!c'

   if channel(2) eq 0 then begin
     utitle1=utitle1+'not av.'
     tmin = 100.
   endif
   if channel(2) eq 1 then begin
     utitle1=utitle1+'ele. av.['+strtrim(string(type(0)),2)+','+$
             strtrim(string(type(1)),2)+']'
     tmin=25.*60.
   endif
   if channel(2) eq 2 then begin
      utitle1=utitle1+'azi. av. ss['+strtrim(string(type(2)),2)+','+$
             strtrim(string(type(3)),2)+']'
      tmin=100.
   endif
   if channel(2) eq 3 then begin
      utitle1=utitle1+'ele.+azi. av. ss['+strtrim(string(type(0)),2)+','+$
             strtrim(string(type(1)),2)+']'+$
	     '['+strtrim(string(type(2)),2)+','+$
             strtrim(string(type(3)),2)+']'
      tmin=25.*60.
   endif


   nodat=65535

   if channel(2) eq 0 then ncurv=type(1)-type(0)+1+type(3)-type(2)+1
   if channel(2) eq 1 then ncurv=type(3)-type(2)+1
   if channel(2) eq 2 then ncurv=type(1)-type(0)+1
   if channel(2) eq 3 then ncurv=1
   yray = fltarr(count,ncurv)
   utitle=strarr(ncurv)
   
   ind=-1
   if channel(2) eq 0 then begin  ;no average
      for i=type(0),type(1) do begin
         for j=type(2),type(3) do begin
            ind = ind+1
            yray(*,ind)=input_data(*).data(channel(5),i,channel(1),j)
            utitle(ind)='EL'+strtrim(string(i),2)+' AZ'+strtrim(string(j),2)
         endfor
      endfor
   endif
   
   if channel(2) eq 1 then begin  ;elevation average
      for j=type(2),type(3) do begin
         ind = ind+1
	 for tp=0,n_elements(time)-1 do begin
           indix=where(input_data(tp).data(channel(5),*,channel(1),j) ne nodat,count)     
           if count ne 0 then $
              yray(tp,ind)=total(input_data(tp).data(channel(5),indix,channel(1),j))/float(count)$
	   else yray(tp,ind)=nodat
        endfor
        utitle(ind)='AZ'+strtrim(string(j),2)
      endfor
   endif

   if channel(2) eq 2 then  begin;azimut average
      for i=type(0),type(1) do begin
         ind = ind+1
         for tp=0,n_elements(time)-1 do begin
           indix=where(input_data(tp).data(channel(5),i,channel(1),*) ne nodat,count)     
           if count ne 0 then $
              yray(tp,ind)=total(input_data(tp).data(channel(5),i,channel(1),indix))/float(count)$
	   else yray(tp,ind)=nodat
         endfor
         utitle(ind)='EL'+strtrim(string(i),2)
      endfor
   endif

   if channel(2) eq 3 then  begin;elevation+azimut average
      for tp=0,n_elements(time)-1 do begin
         count=0
         for i=type(0),type(1) do begin
            for j=type(2),type(3) do begin
              if input_data(tp).data(channel(5),i,channel(1),j) ne nodat then begin 
	        count=count+1
                yray(tp,0)=yray(tp,0)+input_data(tp).data(channel(5),i,channel(1),j)
	      endif
	    endfor
         endfor
         if count ne 0 then yray(tp,0)=yray(tp,0)/float(count) else $
         yray(tp,0)=nodat
      endfor
      utitle(0)=''
   endif


   uytitle='[counts]'

   ind = -1
   for i=0,ncurv-1 do begin
      ind = ind+1
      yvect = yray(*,ind)
      index=where(yvect ne nodat, count)
      if count ne 0 then begin
          yvect=yvect(index)
          t=time(index)
      endif else t=time
      if yscl(panel(0),3) eq 0 then $
         ylim4,t,yvect,ymin,ymax $
      else ylim_io,t,yvect,ymin,ymax
      if (ind eq 0) then begin
         yst = ymin
         yen = ymax
      endif else begin
         if ymin lt yst then yst = ymin
         if ymax gt yen then yen = ymax
      endelse
   endfor

; the variables used above have the following meaning:
; 	uytitle : the y-axis label
;	utitle  : the right-of plot label defining the data
;
; the procedure YLIM3 is optional and sets the ylimit automatically for the
; data leaving a 10% gap at the top and bottom limits.
; this routine is very simple and does not handle spikes or bad data, you may
; need to supply your own procedure for your own data. 
; (in $PAPCO/PAPCO_Lib/ylim.pro)

; yscl of common block yscale is an array of dimensions (*,4) and contains info
; on the yscaling to be used. The first dimension identifies the panel, and the
; next dimension has three entries:  
;         yscl(*,0)  is  0 for "use automatic scaling", 1 for "use manual"
;         yscl(*,1)  is then ymin (or zmin)
;         yscl(*,2)  is then ymax (or zmax)
;         yscl(*,3)  is 0 for "linear" and 1 for "log"  scale
; when scaling is set to automatic, the automatic y-limits are loaded in.
   
   if (yscl(panel(0),0) eq 1) then begin
      yst=yscl(panel(0),1)
      yen=yscl(panel(0),2)
   endif else begin
      yscl(panel(0),1)=yst
      yscl(panel(0),2)=yen
   endelse

; uae auto y-labeling routine to get "good" y-lables. This sets the global IDL
; graphics system variables !Y.MINOR, !Y.TICKV, !Y.TICKS - so if you want to 
; use this feature, avoid setting these or their keywords in your plot call.
  papco_y_label, yst, yen, log=yscl(panel(0),3)    


; set up extra plot keywords, common ones first. 
; yrange, xrange  are always needed. 
; the rest you can set if you want to override PAPCO defaults.
; See $PAPCO_BASE/PAPCO/papco_draw for the definition of the PAPCO default
; graphic system variable settings.

; set up extra plot keywords, common ones first. 
; yrange, xrange  are always needed. 
; the rest you can set if you want to override papco defaults.
; See $papco_BASE/papco/papco_draw for the definition of the papco default
; graphic sytem variable settings.
  extra_plotPar_common={yrange:[yst,yen], xrange:[xut1,xut2], $
                 ylog:yscl(panel(0),3), $
                 ystyle:1, xtickformat:'noticks'}

; add keyword structure set here with the one passed in

   extra_plotPar=create_struct(extra_plotPar_common, extra_par)


; use papco routine to draw time axis. This checks for bottom plot and uses the
; user's xtickformat if it is something other than 'noticks'.
; If you use this rotuine, make sure your data plot routine uses xstyle=5 and 
; ystyle=5 (inhibit axis) so that you don't clobber those drawn by papco.  
; "down" sets the axis labeling level. 
  down=0
  papco_draw_time_axis, panel, OVERPLOT=OVERPLOT, _extra=extra_plotPar     


   down=0
   color1_plotpar=extra_plotPar
   color1_plotpar.color=1

   if not keyword_set(overplot) then begin
          plot,time,yray, _extra=color1_plotPar, xstyle=1, ystyle=1, $
           /NODATA
    endif

;create new structure to use PLOTS with IDL 4.

    extra_gapplotPar = create_struct({PSYM:0},{COLOR:0L},{THICK:1},{LINESTYLE:0})
    extra_gapplotPar.PSYM = extra_plotPar.PSYM
    extra_gapplotPar.COLOR = extra_plotPar.COLOR
    extra_gapplotPar.THICK = extra_plotPar.THICK
    extra_gapplotPar.LINESTYLE = extra_plotPar.LINESTYLE

; now do the actual data plot, data only, no axis
    couleur_start = extra_plotPar.color
    couleur_index = where(couleur_start eq couleur_arr)
    if couleur_index(0) eq -1 then couleur_index(0) = 0
    couleur = extra_plotPar.color
    ind = -1
    for i=0,ncurv-1 do begin
       ind = ind + 1
       yvect = yray(*,ind)
       index=where(yvect ne nodat, count)
       if count ne 0 then begin
          yvect=yvect(index)
          t=time(index)
       endif
       extra_gapplotPar.color = couleur
       cassini_caps_gap_oplot,tmin,t,yvect, _extra=extra_gapplotPar
       couleur = couleur_arr(couleur_index(0)+ind+1<15)
    endfor

; plot y-axis label at left of plot. Use scalable routine!  
  left_side_label,panel,uytitle,/rot90  

; store the coordinate information into common block for slices

   plotted_x = !x
   plotted_y = !y


; plot info right of the panel. If you want overplots to have their own labels
; you need to add position-control for this label using the keyword
; PLOTS_ATTHISPOSITION

   extra_Par.color = couleur_start
   couleur = extra_Par.color
   if not keyword_set(overplot) then begin
     extra_Par.color = 1
     right_side_label,panel,utitle1,_extra=extra_Par
     utitle2 = '!C!C'
     ind = -1
     for i=0,ncurv-1 do begin
        ind = ind+1
        extra_Par.color = couleur
        utitleS = utitle2 + '!C' + utitle(ind)
        right_side_label,panel,utitleS,_extra=extra_Par
        utitle2 = utitle2 + '!C'
        couleur = couleur_arr(couleur_index(0)+ind+1<15)
     endfor
   endif else begin
     for i=0, PLOTS_ATTHISPOSITION-1 do utitle='!C'+utitle1
     right_side_label,panel,utitle,_extra=extra_Par
   endelse


; check if data needs to be output to file. Use the optional keywords
; to add more information to the data file
;      DESCRIPTION   a string or string array describing the data
;      TIME_LABEL    a string decribing the x-axis format
;      CHANNELS      an (ch,2) array giving the channel start/end
;      Y_LABEL       a string with the y-axis label
;      Z_LABEL       a string with the z-axis label
;      RS_LABEL      a string with the righ-side label of the plot
;      NODATA        contains the no data flag value.
  
  if KEYWORD_SET(OUTPUT) then begin
    print,'% p_cassini_caps_line: Writing plot data out to file'
    print,'% NOT IMPLEMENTED !!'
  endif
   



end 


;******************************************************************************
;* PROCEDURE:     
;*      ylim_io,time,yray,yst,yen
;* 
;* DESCRIPTION:  
;*      This procedure search for min and max in a log scale in array yray
;*
;* INPUTS:  
;*      time     
;*      yray     array to be plotted
;*              
;* OUTPUTS: 
;*      yst,yen   min and max
;* 
;* CALLING SEQUENCE:
;*      ylim_io,time,yray,yst,yen
;*
;* MODIFICATION HISTORY:       
;*     written September 2000, Sebastien Bourdarie, at ONERA/DESP
;******************************************************************************
pro ylim_io,time,yray,yst,yen

common time, xut1,xut2

index=where((time ge xut1) and (time le xut2) and (yray gt 0.),count)

if count gt 2 then begin
  yen=max(yray(index),min=yst)
  yst = 10.^floor(ALOG10(yst))
  yen = 10.^ceil(ALOG10(yen))
endif else begin
  yen=10.
  yst=1.
endelse

end


;******************************************************************************
;* PROCEDURE:     
;*      cassini_caps_gap_oplot, x, y, _EXTRA=e
;* 
;* DESCRIPTION:  
;*      This procedure plots data without connecting data point if they are
;*      further than 5 min. apart.
;*
;* INPUTS: 
;*      step    time step between two measurements      
;*      x,y     arrays to be plotted
;*              
;* OUTPUTS: 
;*      none
;* 
;* CALLING SEQUENCE:
;*      cassini_caps_gap_oplot, x, y, _EXTRA=e
;*
;* MODIFICATION HISTORY:       
;*     written September 2000, Sebastien Bourdarie, at ONERA/DESP
;******************************************************************************
pro cassini_caps_gap_oplot,step,x,y,_EXTRA=e

; work out jumps in y. If a jump is larger than step, don't plot

   ndat=n_elements(x)
   if (ndat eq 0) then return
   for i=0l,ndat-2l do begin
      int=abs(x(i+1)-x(i))
      if int lt step then begin
        plots,x(i),y(i) 
        plots,x(i+1),y(i+1),/continue,noclip=0 ,_EXTRA=e
      endif
    endfor

end


