;******************************************************************************
;*  NAME:        r_cassini_caps
;*
;*  DESCRIPTION: Routine to read cassini_caps files. Adheres to
;*               papco conventions: Error common block
;*                                  Uses MJDT time
;*                                  papco read philosophy
;*
;*  INPUT:       none (uses common block time from papco)
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  KEYWORDS:    VERBOSE  -  enables print comments, save/restore comments
;*
;* MODIFICATION HISTORY:       
;*     Written April 2001,  S. Bourdarie (ONERA/DESP)
;******************************************************************************
pro r_cassini_caps,dataType, VERBOSE=VERBOSE

   common cassini_caps_data, input_header, input_data
   common mjdt, mjdt_start, mjdt_end
   common get_error, get_err_no, get_err_msg

; common get_error is used by papco to check if the data read was successful.
; it is up to the user to set this correctly and to make sure that the read- 
; routine does not crash on an I/O error - otherwise papco will crash at that
; point, too. papco draws an empty panel when get_err_no in not equal to zero
; and prints get_err_msg in that panel for user information. papco will try
; to invoce the plot routine for this data if get_err_no = 0 , otherwise an
; emplty panel with the error message get_err_msg is plotted.

   get_err_no=0
   get_err_msg='cassini_caps read succesfully'
   
   
; check for env. variable pointing to data directory
   data_dir='CASSINI_CAPS_DATA'
   path=papco_getenv(data_dir)
   if strlen(path) eq 0 then begin
     get_err_no=1
     get_err_msg='Environmental variable '+data_dir+' not set'
     print,get_err_msg
     return
   endif
   
; check for existence of data directory
   cc=papco_finddir(path)
   if cc eq 0 then begin
     get_err_no=2
     get_err_msg='Data directory '+path+' not found'
     print,get_err_msg
     return
   endif
   
; check VERBOSE keyword and set
   if keyword_set(VERBOSE) then verbose=VERBOSE else verbose=0     
   
; clear data arrays for reuse
   input_header=0  &  input_data=0   
   
; loop through from start to end time concatonating data files into one
; array. Here we assume that the data is stored in files per day, and
; that the time info in each file is seconds of day. papco standard is
; to hand over data to the plot routines with the time running in
; seconds from the number of seconds of the startday.
   
   time=mjdt_start  &  no_files_read=0  &  day_count=0.0
   filebase=['ELS','IMS','IBS']   
   while mjdt_end.mjd ge time.mjd do begin
     if mjdt_end.mjd eq time.mjd then hour_end = mjdt_end.t else hour_end=86400.
     while time.t le hour_end do begin
     
        ; assemble the data file from the time range requested
        convert_t90_to_date, time, year, day_oy, hour, minute, second,$
                          month, cmonth, day_om, /mjdt   
        fln = '/'+filebase(dataType)+'/'+filebase(dataType)+'_'
        datestr=string(year,day_oy,hour,format='(i4.4,i3.3,i2.2)')
        fln=fln+datestr
        found=findfile(path+fln+'*.*',count=nfs)
        if nfs ne 0 then begin
          ; read one hour of data
	  resul = str_sep(found(0),'.')
          r_cassini_caps_onefile, resul(0), /VERBOSE
     
          ; check for "OPENR: Error opening file" in error string - and then get 
          ; the file remotely!
          ; comment this section out if you do not want this feature.
;     res=strpos(!err_string,'OPENR: Error opening file')
;     if res ne -1 then begin
;       c_url=papco_getenv('cassini_caps_FTP_ARCHIVE')
;       c_file=fln+'.idl'
;       c_download_dir=path
;       papco_get_remote_data, c_url, c_file, c_download_dir
;       ;check if file "arrived" and then read it
;       if get_err_no eq 0 then r_cassini_caps_onefile, path+fln, /VERBOSE
;     endif   
     
          ;if a file was succesfully read, concatonate!
          if get_err_no eq 0 then begin 
           if no_files_read eq 0 then begin   ;start the array buffer
              input_data_buf=temporary(input_data)
              input_header_buf = input_header
            endif else begin                   ;otherwise concatonate arrays
              input_data_buf=[temporary(input_data_buf),temporary(input_data)]
            endelse        
            input_data=0                     ;clear array for reuse
            no_files_read=no_files_read+1
          endif
        endif
	time.t = time.t+3600.
      endwhile
      time.t = 0.
      time.mjd=time.mjd+1
   endwhile         
   
; check if any files were read   
   if (no_files_read ne 0) and (get_err_no ne 1) then begin
     get_err_no=0
     get_err_msg='% r_cassini_caps: '+string(no_files_read,format="(i2)")+ $
       ' files read succesfully'
     ;clear arrays and load buffer into output array
     input_data=0  &  input_data=input_data_buf  &  input_data_buf=0
     input_header = input_header_buf
;converting time to PAPCO plot standard
     input_data(*).time=(input_data(*).mjdt_time.mjd-mjdt_start.mjd) $
                     *86400+input_data(*).mjdt_time.t

   endif else begin
     get_err_no=1
     get_err_msg='% r_cassini_caps: no files found for timerange'
     return
   endelse      
   
; reduce array to time interval required. Convert PAPCO time to heo time and
; select.
   ut1=mjdt_start.t
   ut2=long(mjdt_end.mjd-mjdt_start.mjd)*86400l+mjdt_end.t
   index=where((input_data.Time ge ut1) and (input_data.Time le ut2),count)
   if count ne 0 then begin
      input_data=input_data(index)
   endif else begin
     get_err_no=1
     get_err_msg='% r_cassini_caps: no data found for timerange'
     return
   endelse      


end

;******************************************************************************
;*  NAME:        r_cassini_caps_onefile,fln
;*
;*  DESCRIPTION: reads one file of type plot_file. Assumes data has been
;*               stored in IDL saveset using structures input_header and
;*               input_data
;*
;*  INPUT:       fln  - string of filename
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;* MODIFICATION HISTORY:       
;*     Written April 2001,  S. Bourdarie (ONERA/DESP)
;******************************************************************************
pro r_cassini_caps_onefile, fln, VERBOSE=VERBOSE
  
   common cassini_caps_data, input_header, input_data
   common get_error, get_err_no, get_err_msg
   
   get_err_no=0
   get_err_msg='cassini_caps onefile read succesfully'
   
; clear data arrays for reuse
   input_header=0  &  input_data=0   
   
   on_ioerror, read_original
   
   restore,fln+'.idl',/verbose
   dealloc_lun,/quiet    ;frees logical units used by save/restore
   print,'% r_cassini_caps_onefile: idl saveset read succesfully'
   return

   read_original:
   ; if the .idl save file could not be found, then try to read the original
   r_cassini_caps_onefile_original, fln, VERBOSE=VERBOSE
      
end   
  
;******************************************************************************
;*  NAME:        r_cassini_caps_onefile_original,fln
;*
;*  DESCRIPTION: reads one file of type plot_file_original. Under the
;*               papco read philosophy, files should be read the first
;*               time in whatever format they might have - ascii,
;*               fortran binary, pascal binary, etc. The files might also
;*               be in some compressed format, so they need to be uncompressed
;*               first. 
;*               
;*               You might also have to change time formats. 
;*               A lot of files are stored per day, and have seconds of
;*               that day as the time. This is what is assumed here and
;*               is papco standard. You need to convert the time
;*               to papco standard if you intend to use the plotroutines 
;*               supplied with papco (use the routines in
;*               $papco_BASE/papco_Lib/time_convert.pro). 
;*               If you use your own plot routines this step is not needed.
;*               
;*               The first read might be slow, esp if the format is ascii.
;*               Use the IDL "save" command to make a binary copy of the
;*               file which can then be read quickly using "restore". The
;*               next time only the IDL saveset is read, which is much
;*               faster.
;*               
;*               In order to be able to concatonate the arrays easily,
;*               it makes sense to have the data in an array of time-ordered
;*               structures, which makes adding together days of data
;*               trivial using the IDL concatonation operator "[]".
;*               
;*  INPUT:       fln  - string of filename
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;* MODIFICATION HISTORY:       
;*     Written April 2001,  S. Bourdarie (ONERA/DESP)
;******************************************************************************
pro r_cassini_caps_onefile_original, fln, VERBOSE=VERBOSE
  
   common cassini_caps_data, input_header, input_data
   common get_error, get_err_no, get_err_msg  
   
   get_err_no=0
   get_err_msg='cassini_caps_onefile_orignal read succesfully'
   
; clear data arrays for reuse
   input_header=0  &  input_data=0   
   
;------------------------------------------------------------------------------
; read the original data file - THIS NEEDS TO BE REWRITTEN BU USER!!!!
  
; define the header and data structures. Edit to suit your data!
   ext='.cdf'                   ;file extention of your dat files

   file_name = findfile(fln+ext, count=nofiles)
   if nofiles eq 0 then begin
	get_err_no=2
	get_err_msg='File '+fln+ext+' does not exist'
	print, get_err_msg
	return
   endif
   get_err_no=0

; search the species to read (ELS, IMS or IBS)
  res = strpos(fln,'/ELS/')
  if res ne -1 then test=0
  res = strpos(fln,'/IMS/')
  if res ne -1 then test=1
  res = strpos(fln,'/IBS/')
  if res ne -1 then test=2
  


;define structure to store the data and info

   time={MJDT, MJD:0l, T:0.0d}
   result=rstrpos(fln,'/')
   year = fix(strmid(fln,result+5,4))
   doy  = fix(strmid(fln,result+9,3))

   if test eq 0 then begin ;ELS

       dummy={data : lonarr(8,63,16), $
                              mjdt_time    : time     , $
                              time         : 0.0d0}
   endif
   if test eq 1 then begin   ;IMS
       dummy={data : lonarr(7,8,63,8), $
                              mjdt_time    : time     , $
                              time         : 0.0d0}
   endif
   if test eq 2 then begin   ;IBS
       dummy={data : lonarr(3,255,16), $
                              mjdt_time    : time     , $
                              time         : 0.0d0}
   endif

    id=cdf_open(fln+ext)
    info=cdf_inquire(id)
    if info.maxrec eq 0 then begin
	get_err_no=2
	get_err_msg='File '+fln+ext+' empty'
	print, get_err_msg
	return
   endif


    varName = strarr(info.nzvars)
    for i=0,info.nzvars-1 do begin
       res=cdf_varinq(id,i,/zvar)
       varName(i) = res.name
    endfor
    index = where(strpos(varName,'Data') ne -1,count)
    if count ne 0 then varName = varName(index) else begin
	get_err_no=2
	get_err_msg='No zVariable in file '+fln+ext
	print, get_err_msg
	return
    endelse
        
    input_data=replicate(dummy,info.maxrec)
    input_data(*).data(*,*,*)=65535


; read time

    sec1 = julday(1,1,1970)
    sec2 = julday(1,1,year) + doy-1
    sec = (sec2-sec1)*86400
    cdf_varget, id, 'Time_sec', buffer, rec_count=info.maxrec
    buffer = buffer-sec
    for i=0,info.maxrec-1 do begin
       dummy.mjdt_time=convert_date_to_t90(doy=doy,  year=year, $
                                           seconds=buffer(i), /MJDT)
       input_data(i) = dummy
    endfor
    
; read telemetry mode
    tm = strarr(info.maxrec)
    cdf_varget, id, 'Telemetry', buffer, rec_count=info.maxrec
    for i=0,info.maxrec-1  do tm(i)=strtrim(string(buffer(*,0,i)),2)

; read data
    for i=0,n_elements(varName)-1 do begin
; if ELS
        if test eq 0 and varName(i) ne 'Data61' $
	             and varName(i) ne 'Data61_1' $
		     and varName(i) ne 'Data81' $
		     and varName(i) ne 'Dataa1' $
		     and varName(i) ne 'Datac1' $
		     and varName(i) ne 'Data01' $
		     and varName(i) ne 'Data6c' $
		     and varName(i) ne 'Dataac' then begin $
	   print,varName(i),' not taken into account in caps_data'
	   goto,endvar
	endif
        if test eq 1 and varName(i) ne 'Data64' $
		     and varName(i) ne 'Data84' $
		     and varName(i) ne 'Dataa4' $
		     and varName(i) ne 'Datac4' $
		     and varName(i) ne 'Data04' then begin $
	   print,varName(i),' not taken into account in caps_data'
	   goto,endvar
	endif
        if test eq 2 and varName(i) ne 'Data62' $
		     and varName(i) ne 'Data82' $
;		     and varName(i) ne 'Dataa2' $
		     and varName(i) ne 'Dataa2_1' $
		     and varName(i) ne 'Data02' then begin $
	   print,varName(i),' not taken into account in caps_data'
	   goto,endvar
	endif
        cdf_varget, id, varName(i), buffer, rec_count=info.maxrec
	if test eq 0 then begin
	  if varName(i) eq 'Data01' then begin ;8EL * 63E/Q * 16AZ
	     index=where(tm eq '16 kbps',count)
	     for j=0,count-1 do input_data(index(j)).data=buffer(*,*,*,index(j))
	  endif
	  if varName(i) eq 'Datac1' then begin ;8EL * 32E/Q
	     index=where(tm eq '250 bps',count)
	     for j=0,count-1 do begin
	         for k=0,30 do for kk=0,1 do input_data(index(j)).data(*,2*k+kk,0)=buffer(*,k,index(j))
                 input_data(index(j)).data(*,62,0)=buffer(*,31,index(j))
		 for k=1,14 do input_data(index(j)).data(*,*,k)=input_data(index(j)).data(*,*,0)
             endfor
	  endif
	  if varName(i) eq 'Dataa1' then begin ;8EL * 32E/Q
	     index=where(tm eq '500 bps_sw',count)
	     for j=0,count-1 do begin
	         for k=0,30 do for kk=0,1 do input_data(index(j)).data(*,2*k+kk,0)=buffer(*,k,index(j))
                 input_data(index(j)).data(*,62,0)=buffer(*,31,index(j))
		 for k=1,14 do input_data(index(j)).data(*,*,k)=input_data(index(j)).data(*,*,0)
             endfor
	  endif
	  if varName(i) eq 'Data81' then begin ;8EL * 63E/Q * 3AZ
	     index=where(tm eq '1 kbps',count)
	     for j=0,count-1 do begin
	        for k=0,4 do input_data(index(j)).data(*,*,k)=buffer(*,*,0,index(j))/5
	        for k=5,9 do input_data(index(j)).data(*,*,k)=buffer(*,*,1,index(j))/5
	        for k=10,15 do input_data(index(j)).data(*,*,k)=buffer(*,*,2,index(j))/6
	     endfor
	  endif
	  if varName(i) eq 'Data61_1' then begin ;8EL * 63E/Q
	     index=where(tm eq '2 kbps_sw',count)
	     for j=0,count-1 do begin
	        input_data(index(j)).data(*,*,0)=buffer(*,*,index(j))
                for k=1,15 do input_data(index(j)).data(*,*,k)=input_data(index(j)).data(*,*,0)
             endfor
	  endif
	  if varName(i) eq 'Data61' then begin ;8EL * 8E/Q * 16AZ
	     index=where(tm eq '2 kbps_sw',count)
	     for j=0,count-1 do begin
	        for k=0,6 do begin
		   for kk=0,7 do input_data(index(j)).data(*,8*k+kk,*)=buffer(*,k,*,index(j))
                endfor		   
                for kk=0,6 do input_data(index(j)).data(*,56+kk,*)=buffer(*,7,*,index(j))
	     endfor
	  endif
	  if varName(i) eq 'Data6c' then begin ;8EL * 63E/Q * 4AZ
	     index=where(tm eq '2 kbps_mag',count)
	     for j=0,count-1 do begin
	        for k=0,3 do begin
		   for kk=0,3 do input_data(index(j)).data(*,*,4*k+kk)=buffer(*,*,k,index(j))
                endfor		   
	     endfor
	  endif
	  if varName(i) eq 'Dataac' then begin ;8EL * 63E/Q * 1AZ
	     index=where(tm eq '500 bps_mag',count)
	     for j=0,count-1 do begin
	        input_data(index(j)).data(*,*,0)=buffer(*,*,index(j))
                for k=1,15 do input_data(index(j)).data(*,*,k)=input_data(index(j)).data(*,*,0)
             endfor
	  endif
        endif

;if IMS	
	if test eq 1 then begin
	  if varName(i) eq 'Data04' then begin ;7ION * 8EL * 63E/Q * 8AZ
	     index=where(tm eq '16 kbps',count)
	     for j=0,count-1 do input_data(index(j)).data(*,*,*,*)=buffer(*,*,*,*,index(j))
	  endif
	  if varName(i) eq 'Datac4' then begin ;2ION * 4EL * 32E/Q
	     index=where(tm eq '250 bps',count)
	     for j=0,count-1 do begin
	        for k=0,30 do for kk=0,1 do input_data(index(j)).data(0:1,0,2*k+kk,0)=buffer(*,0,k,index(j))
		input_data(index(j)).data(0:1,0,62,0)=buffer(*,0,31,index(j))
		for k=0,31 do for kk=0,1 do input_data(index(j)).data(0:1,1,2*k+kk,0)=buffer(*,0,k,index(j))
		input_data(index(j)).data(0:1,1,62,0)=buffer(*,0,31,index(j))
	        for k=0,31 do for kk=0,1 do input_data(index(j)).data(0:1,2,2*k+kk,0)=buffer(*,1,k,index(j))
		input_data(index(j)).data(0:1,2,62,0)=buffer(*,1,31,index(j))
		for k=0,31 do for kk=0,1 do input_data(index(j)).data(0:1,3,2*k+kk,0)=buffer(*,1,k,index(j))
		input_data(index(j)).data(0:1,3,62,0)=buffer(*,1,31,index(j))
	        for k=0,31 do for kk=0,1 do input_data(index(j)).data(0:1,4,2*k+kk,0)=buffer(*,2,k,index(j))
		input_data(index(j)).data(0:1,4,62,0)=buffer(*,2,31,index(j))
		for k=0,31 do for kk=0,1 do input_data(index(j)).data(0:1,5,2*k+kk,0)=buffer(*,2,k,index(j))
		input_data(index(j)).data(0:1,5,62,0)=buffer(*,2,31,index(j))
	        for k=0,31 do for kk=0,1 do input_data(index(j)).data(0:1,6,2*k+kk,0)=buffer(*,3,k,index(j))
		input_data(index(j)).data(0:1,6,62,0)=buffer(*,3,31,index(j))
		for k=0,31 do for kk=0,1 do input_data(index(j)).data(0:1,7,2*k+kk,0)=buffer(*,3,k,index(j))
		input_data(index(j)).data(0:1,7,62,0)=buffer(*,3,31,index(j))
		for k=1,7 do input_data(index(j)).data(0:1,*,*,k) = input_data(index(j)).data(0:1,*,*,0)
             endfor
	  endif
	  if varName(i) eq 'Dataa4' then begin ;2ION * 4EL * 32E/Q
	     index=where(tm eq '500 bps_sw/mag',count)
	     for j=0,count-1 do begin
	        for k=0,30 do for kk=0,1 do input_data(index(j)).data(0:1,0,2*k+kk,0)=buffer(*,0,k,index(j))
		input_data(index(j)).data(0:1,0,62,0)=buffer(*,0,31,index(j))
		for k=0,31 do for kk=0,1 do input_data(index(j)).data(0:1,1,2*k+kk,0)=buffer(*,0,k,index(j))
		input_data(index(j)).data(0:1,1,62,0)=buffer(*,0,31,index(j))
	        for k=0,31 do for kk=0,1 do input_data(index(j)).data(0:1,2,2*k+kk,0)=buffer(*,1,k,index(j))
		input_data(index(j)).data(0:1,2,62,0)=buffer(*,1,31,index(j))
		for k=0,31 do for kk=0,1 do input_data(index(j)).data(0:1,3,2*k+kk,0)=buffer(*,1,k,index(j))
		input_data(index(j)).data(0:1,3,62,0)=buffer(*,1,31,index(j))
	        for k=0,31 do for kk=0,1 do input_data(index(j)).data(0:1,4,2*k+kk,0)=buffer(*,2,k,index(j))
		input_data(index(j)).data(0:1,4,62,0)=buffer(*,2,31,index(j))
		for k=0,31 do for kk=0,1 do input_data(index(j)).data(0:1,5,2*k+kk,0)=buffer(*,2,k,index(j))
		input_data(index(j)).data(0:1,5,62,0)=buffer(*,2,31,index(j))
	        for k=0,31 do for kk=0,1 do input_data(index(j)).data(0:1,6,2*k+kk,0)=buffer(*,3,k,index(j))
		input_data(index(j)).data(0:1,6,62,0)=buffer(*,3,31,index(j))
		for k=0,31 do for kk=0,1 do input_data(index(j)).data(0:1,7,2*k+kk,0)=buffer(*,3,k,index(j))
		input_data(index(j)).data(0:1,7,62,0)=buffer(*,3,31,index(j))
		for k=1,7 do input_data(index(j)).data(0:1,*,*,k) = input_data(index(j)).data(0:1,*,*,0)
             endfor
	  endif
	  if varName(i) eq 'Data84' then begin ;2ION * 4EL * 32E/Q
	     index=where(tm eq '1 kbps',count)
	     for j=0,count-1 do begin
	        for k=0,30 do for kk=0,1 do input_data(index(j)).data(0:1,0,2*k+kk,0)=buffer(*,0,k,index(j))
		input_data(index(j)).data(0:1,0,62,0)=buffer(*,0,31,index(j))
		for k=0,30 do for kk=0,1 do input_data(index(j)).data(0:1,1,2*k+kk,0)=buffer(*,0,k,index(j))
		input_data(index(j)).data(0:1,1,62,0)=buffer(*,0,31,index(j))
	        for k=0,30 do for kk=0,1 do input_data(index(j)).data(0:1,2,2*k+kk,0)=buffer(*,1,k,index(j))
		input_data(index(j)).data(0:1,2,62,0)=buffer(*,1,31,index(j))
		for k=0,30 do for kk=0,1 do input_data(index(j)).data(0:1,3,2*k+kk,0)=buffer(*,1,k,index(j))
		input_data(index(j)).data(0:1,3,62,0)=buffer(*,1,31,index(j))
	        for k=0,30 do for kk=0,1 do input_data(index(j)).data(0:1,4,2*k+kk,0)=buffer(*,2,k,index(j))
		input_data(index(j)).data(0:1,4,62,0)=buffer(*,2,31,index(j))
		for k=0,30 do for kk=0,1 do input_data(index(j)).data(0:1,5,2*k+kk,0)=buffer(*,2,k,index(j))
		input_data(index(j)).data(0:1,5,62,0)=buffer(*,2,31,index(j))
	        for k=0,30 do for kk=0,1 do input_data(index(j)).data(0:1,6,2*k+kk,0)=buffer(*,3,k,index(j))
		input_data(index(j)).data(0:1,6,62,0)=buffer(*,3,31,index(j))
		for k=0,30 do for kk=0,1 do input_data(index(j)).data(0:1,7,2*k+kk,0)=buffer(*,3,k,index(j))
		input_data(index(j)).data(0:1,7,62,0)=buffer(*,3,31,index(j))
		for k=1,7 do input_data(index(j)).data(0:1,*,*,k) = input_data(index(j)).data(0:1,*,*,0)
             endfor
	  endif
	  if varName(i) eq 'Data64' then begin ;6ION * 4EL * 63E/Q * 2AZ
	     index=where(tm eq '2 kbps_sw/mag',count)
	     for j=0,count-1 do begin
	        for k=0,3 do for kk=0,1 do input_data(index(j)).data(0:5,2*k+kk,*,0)=buffer(*,k,*,0,index(j))
	        for k=0,3 do for kk=0,1 do input_data(index(j)).data(0:5,2*k+kk,*,1)=buffer(*,k,*,0,index(j))
	        for k=0,3 do for kk=0,1 do input_data(index(j)).data(0:5,2*k+kk,*,2)=buffer(*,k,*,0,index(j))
	        for k=0,3 do for kk=0,1 do input_data(index(j)).data(0:5,2*k+kk,*,3)=buffer(*,k,*,0,index(j))
	        for k=0,3 do for kk=0,1 do input_data(index(j)).data(0:5,2*k+kk,*,4)=buffer(*,k,*,1,index(j))
	        for k=0,3 do for kk=0,1 do input_data(index(j)).data(0:5,2*k+kk,*,5)=buffer(*,k,*,1,index(j))
	        for k=0,3 do for kk=0,1 do input_data(index(j)).data(0:5,2*k+kk,*,6)=buffer(*,k,*,1,index(j))
	        for k=0,3 do for kk=0,1 do input_data(index(j)).data(0:5,2*k+kk,*,7)=buffer(*,k,*,1,index(j))
             endfor
	  endif
        endif
	
;if IBS	
	if test eq 2 then begin
	  if varName(i) eq 'Data02' then begin ;3EL * 255E/Q * 16AZ
	     index=where(tm eq '16 kbps',count)
	     for j=0,count-1 do input_data(index(j)).data(*,*,*)=buffer(*,*,*,index(j))
	  endif
	  if varName(i) eq 'Dataa2_1' then begin ;128E/Q
	     index=where(tm eq '500 bps_sw',count)
	     for j=0,count-1 do begin
	        for k=0,126 do for kk=0,1 do input_data(index(j)).data(0,2*k+kk,0)=buffer(k,index(j))
	        input_data(index(j)).data(0,254,0)=buffer(127,index(j))
		for k=1,15 do begin
		    input_data(index(j)).data(0,254,k)=input_data(index(j)).data(0,254,0)
		    input_data(index(j)).data(1,254,k)=input_data(index(j)).data(0,254,0)
		    input_data(index(j)).data(2,254,k)=input_data(index(j)).data(0,254,0)
		endfor
	     endfor
	  endif
;	  if varName(i) eq 'Dataa2' then begin ;1E/Q
;	     index=where(tm eq '500 bps_sw',count)
;	     for j=0,info.maxrec-1 do input_data(j).data(0,0,0)=buffer(*,j)
;	  endif
	  if varName(i) eq 'Data82' then begin ;3EL * 255E/Q
	     index=where(tm eq '1 kbps',count)
	     for j=0,count-1 do begin
	        input_data(index(j)).data(*,*,0)=buffer(*,*,index(j))
	        for k=1,14 do input_data(index(j)).data(*,*,k)=input_data(index(j)).data(*,*,0)
	     endfor
	  endif
	  if varName(i) eq 'Data62' then begin ;3EL * 255E/Q
	     index=where(tm eq '2 kbps_sw',count)
	     for j=0,count-1 do begin
	        input_data(index(j)).data(*,*,0)=buffer(*,*,index(j))
	        for k=1,14 do input_data(index(j)).data(*,*,k)=input_data(index(j)).data(*,*,0)
	     endfor
	  endif
        endif
	endvar:
    endfor
    cdf_close, id
        


; end of user written section
;------------------------------------------------------------------------------
   
   
; save read data with correct time. Rename data to *.idl
  save,filename=fln+'.idl',input_header,input_data,/verbose  
  dealloc_lun,/quiet              ;frees logical units used by save/restore
  return
  
  out:
  ; if there was an error, report it. Use the IDL error variables. Error
  ; 171 is for "file not fond" which is used to invoke the remote get.  
  get_err_no=!ERROR
  get_err_msg='r_cassini_caps_onefile:!C '+!ERR_STRING
   
end
