;+
; FUNCTION:  cluster_ts_eff_corr
;
; DESCRIPTION: 
;
;     to calculate CODIF TOF and efficiencies with corrected H+ or O+ 
;	polynomial coefficients 
;
; INPUT: 
;      energy:  Array of energy
;      pac:     Post acceleration voltage
;      species: Species for which the efficiency is calculated 
;      sens:    The sensitivity (HS = 1, LS = 0)
;      ang:     the number of angle bins
;      packets: number of data packets
;      INCRATES: flag (0 or 1) if set, uses sum(PF's) in efficiency calculation
;      EFF_FILE: efficiency file to use for H+ or O+
; NOTES:
;	Angle Maps for ESIC are as follows:
;
;
;				88 ANGLE PRODUCTS
;	1:		0		1		2		3
;	2:     	4	5	6	7	8	9	10	11   
;	3: 12  13  14  15  16  17   18 19   20 21  22  23   24  25   26 27    
;	4: 28  29  30  31  32  33   34 35   36 37  38  39   40  41   42 43
;	5: 44  45  46  47  48  49   50 51   52 53  54  55   56  57   58 59
;	6: 60  61  62  63  64  65   66 67   68 69  70  71   72  73   74 75
;	7:  	76 	77 	78 	79 	80 	81 	82 	83
;	8:		84		85		86		87
;
;				24 ANGLE PRODUCTS
;
;	1:		0		1
;	2:		2		3
;	3:	4	5	6	7
;	4:	8	9	10	11
;	5:	12	13	14	15
;	6:	16	17	18	19
;	7:		20		21
;	8:		22		23
;
;
; RETURN: 
;      eff:  Efficiency 
;
; CREATED BY: Dimple P. Patel	
;	     University of New Hampshire
;	     Space Science Group
;	     dpatel@teams.unh.edu
; DATE: 8/11/97    
; LAST MODIFICATION: 11/2/99
; VERSION: 3.31
; MODIFICATION HISTORY:
;	4/16/98 - Added new angle product (32A) for RPA products
;	4/17/98 - Added two new angle products (96A) and (128A) for RPA
;	7/10/98 - Modified eqs_ts_eff to make a version which uses
;                 separate curves for each pixel on the HS side. (LMK)
;	10/23/98 - changed array sizes to handle full data set.  Was only
;		calculating efficiencies for one packet but now calculates
;		efficiencies for all packets.  RETURN - [E,A,packets]
;	12/18/98 - for 12 angle pixels was totalling all numbers
;		at the end instead of specific rows		
;       6/2/99 - added SUM(PF's) to eff calculation for H+ and O+ only
;       7/12/99 - ask user for pixel adjustment file (use 0218 or find
;                 one closest to the date looked at)
;       8/18/99 - took previous change one step further, user is
;                 allowed to choose which eff. file to use
;       8/19/99 - removed sections prompting user for eff file name -
;                 this is taken care of when plot settings are choosen
;                 added keyword 'EFF_FILE'
;       9/24/99 - flight adjusted corrections now default for He+,
;                 pre-flight adjustments are in 'preflight_He_eff.dat'
;       11/2/99 - added time adjustement factor
;                 (Eff_time_adjustment.txt')
;               - added another correction to sumPF correction
;               - made an adjustment to proton eff curves
;	3/01/00 - Added flight adjusted efficiencies for He++, based
;		  on the He+ flight efficiencies.  Got rid of code that
;		  had special case for He++.   Got rid of check for KEYWORD incrates.
;                 Now just checks if incrates = 'yes'.  This should cause a
;                 crash if incrates has not been set at all (LMK)
;       3/09/00 - close efficiency file (CM)

FUNCTION cluster_ts_eff_corr, energy, pac, species, sens, ang, $
                              packets, INCRATES = incrates, $
                              EFF_FILE = eff_file, TIME = time, $
                              OEBINS = oebins, EBINS=ebins, $
                              ANODE_EFFIC = anode_effic, $
                              ABSOL_EFFIC = absol_effic
  
  angle_to_pixel_88 =    [0, 0, 0, 0, $
                          1, 1, 1, 1, 1, 1, 1, 1, $			
                          2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, $
                          3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, $
                          4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, $
                          5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, $
                          6, 6, 6, 6, 6, 6, 6, 6, $			
                          7, 7, 7, 7]
  
  angle_to_pixel_24 =   [0, 0, $
                         1, 1, $
                         2, 2, 2, 2, $
                         3, 3, 3, 3, $
                         4, 4, 4, 4, $
                         5, 5, 5, 5, $
                         6, 6, $
                         7, 7]
  
  IF ang EQ 88 THEN $
    anode = [[0,4,12,28,44,60,76,84],[3,11,27,43,59,75,83,87]]
  IF ang EQ 24 THEN $
    anode = [[0,2,4,8,12,16,20,22],[1,3,7,11,15,19,21,23]]
  
  sf = [1, 2, 1, 1]             ;species factor for energy: H+, He++, He+, O+
  
  mp0 = fltarr(2)
  mp1 = fltarr(2)
  mp2 = fltarr(2)
  mp3 = fltarr(2)
  
  
;---------------------------------------------------------------------------
; Pixel adjustment coefficients are 8 for the HS and 6 for the LS.
; Because of this pixel 0 and pixel 7 are set to one for the LS case.
; For this, the array is initialized full with ones.
;---------------------------------------------------------------------------
  
  pix_adjust = fltarr(8,2)+1
  
;---------------------------------------------------------------------------
  
  edim = dimen1(energy)
  
  print,'EDIM: ', edim
  
;---------------------------------------------------------------------------
; Since the two cases (HS & LS) will be done seperatly the tof_eff
; array will have 8 pixels instead of 14 in the EQS software
;---------------------------------------------------------------------------
  
  tof_eff = fltarr(edim, 8)
  
;---------------------------------------------------------------------------
  
  
;---------------------------------------------------------------------------  
; Load the efficiency and pixel adjustment coefficients
; They are species and sensitivity dependent
;---------------------------------------------------------------------------  
  
  CASE species OF
    
    0:BEGIN
        ; High Sensitivity
      mp0(1) = anode_effic(2,0)
      mp1(1) = anode_effic(2,1)
      mp2(1) = anode_effic(2,2)
      mp3(1) = anode_effic(2,3)
      pix_adjust(*,1) = anode_effic(2,4:11)
      ; Low Sensitivity
      mp0(0) = anode_effic(2,12)
      mp1(0) = anode_effic(2,13)
      mp2(0) = anode_effic(2,14)
      mp3(0) = anode_effic(2,15)
      pix_adjust(1:6,0) = anode_effic(2,16:21)
    END
    
    1:BEGIN
        ; High Sensitivity
      mp0(1) = anode_effic(3,0)
      mp1(1) = anode_effic(3,1)
      mp2(1) = anode_effic(3,2)
      mp3(1) = anode_effic(3,3)
      pix_adjust(*,1) = anode_effic(3,4:11)
      ; Low Sensitivity
      mp0(0) = anode_effic(3,12)
      mp1(0) = anode_effic(3,13)
      mp2(0) = anode_effic(3,14)
      mp3(0) = anode_effic(3,15)
      pix_adjust(1:6,0) = anode_effic(3,16:21)
    END
    
    2:BEGIN
        ; High Sensitivity
      mp0(1) = anode_effic(4,0)
      mp1(1) = anode_effic(4,1)
      mp2(1) = anode_effic(4,2)
      mp3(1) = anode_effic(4,3)
      pix_adjust(*,1) = anode_effic(4,4:11)
      ; Low Sensitivity
      mp0(0) = anode_effic(4,12)
      mp1(0) = anode_effic(4,13)
      mp2(0) = anode_effic(4,14)
      mp3(0) = anode_effic(4,15)
      pix_adjust(1:6,0) = anode_effic(4,16:21)
    END
    
    3:BEGIN
        ; High Sensitivity
      mp0(1) = anode_effic(5,0)
      mp1(1) = anode_effic(5,1)
      mp2(1) = anode_effic(5,2)
      mp3(1) = anode_effic(5,3)
      pix_adjust(*,1) = anode_effic(5,4:11)
      ; Low Sensitivity
      mp0(0) = anode_effic(5,12)
      mp1(0) = anode_effic(5,13)
      mp2(0) = anode_effic(5,14)
      mp3(0) = anode_effic(5,15)
      pix_adjust(1:6,0) = anode_effic(5,16:21)
    END
    
  ENDCASE

;---------------------------------------------------------------------------
; Resize the tof_eff array by adding the packets dimension. Now, each
; packet contains a replicate of the original tof_eff array.
;---------------------------------------------------------------------------
  
  tof_eff = REBIN(tof_eff, edim, 8, packets)
  
  IF N_ELEMENTS(pac) GT 1 THEN $
    pac = REFORM(pac)
  
  consta = 1
  
  SEVSFR = consta
  
  timeAdjFactor = absol_effic(0)
  ;timeAdjFactor = 1
  
;CM   IF species EQ 0 THEN BEGIN
;CM     IF sens(0) EQ 1 THEN sp = 0 ELSE sp = 1			
;CM  ENDIF ELSE sp = species + 1
  FOR npack = 0, packets-1 DO BEGIN
    FOR i = 0, edim - 1 DO BEGIN
      tof_energy = REFORM(energy(i, 0, *) + pac(*))/1000.
      FOR pix = 0,7 DO BEGIN
        tof_eff(i,pix,npack) = (mp0(sens(npack))+$
                            mp1(sens(npack))*sf(species)*tof_energy +$
                            mp2(sens(npack))*(sf(species)*tof_energy)^2 + $
                            mp3(sens(npack))*(sf(species)*tof_energy)^3)* $
                            SEVSFR(0)*timeAdjFactor
        tof_eff(i,pix,npack) = tof_eff(i,pix,npack) / $
          pix_adjust(pix,(sens(npack)))
      ENDFOR
    ENDFOR
  ENDFOR

  CASE ang of 
    88: RETURN, tof_eff(*, angle_to_pixel_88, *)
    24: RETURN, tof_eff(*, angle_to_pixel_24, *)
  ENDCASE
  
END

