;+
;PROCEDURE:	convert_cluster_units
;PURPOSE:	to convert units of data from the Equator-S
;		instruments.
;INPUT:		
;	data:	A 3d structure such as those generated by
;	get_eqs_ion_sample,etc.
;	units:	A string telling the procedure which units to convert to, such
;		as ncounts,rate,nrate,eflux,flux
;       eff_routine: name of efficiency routine to use
;
;KEYWORDS:
;	INCRATES: passed along to efficiency routine, if set the rates
;	(p35) are used in the efficiency calculation
;       EFF_FILE: passed along to efficiency routine, hold name of
;       efficiency fileto use for H+/O+
;
; REVISION HISTORY:
;
;   Made from convert_tms_units.pro
;   Last modification:     Dimple P. Patel    University of New Hampshire
;	
;		'COUNTS': unit_name = 'COUNTS'
;		'NCOUNTS': unit_name = 'NCOUNTS (1/bin)'
;		'RATE': unit_name = 'RATE (1/s)'
;		'NRATE': unit_name = 'NRATE (1/s-bin)'
;		'EFLUX': unit_name = 'EFLUX (eV/cm^2-s-sr-eV)'
;		'DIFF FLUX': unit_name = 'DIFF FLUX (1/cm^2-s-sr-(eV/e))'
;		'DIST FUNC': unit_name = 'DIST FUNC (s^3/cm^3-(km^3)'	
;	
;	6/18/98 - I have put the entire calculation in a loop to facilitate
;		checking the PAC HV and the sensitivity for the efficiencies
;		and the geometric factor, respectively.
;	7/9/98 - Changed efficiency calculation to use user choosen
;		routine.  This is now stored in the data structure.
;	8/24/98 - set data.eff = eff after eff are re-calculated
;	8/26/98 - changed units of DIFF FLUX from 1/cm^2-s-sr-eV to 
;		1/cm^2-s-sr-(eV/e)
;	10/9/98 - Added efficiency calculation and keep loop inside
;		this routine.  The scale for each packet is separatly
;		calculated and put into an array.  This is then
;		multiplied by the data in one step.
;	2/1/99 - can handle unit conversion for himass
;	2/4/99 - changed calculation of NCOUNTS and RATE ::
;		was - ncounts = 1/geom -> now - ncounts = 1/(sf*geom)
;		was - rate = 1/dt*sf ->	  rate - 1/dt 
;       6/25/99 - make sure all arrays muliplied are of same dimension
;       8/12/99 - remove variable 'all' and add data*scale to loop
;                 rather than waiting until after the scale factors
;                 have been calculated for all packets
;       8/19/99 - added keyword EFF_FILE
;
;-

FUNCTION convert_cluster_units, data, units, eff_routine, $
                                INCRATES = incrates, $
                                EFF_FILE = eff_file, $
                                PACKETS = packets
  
;---------------------------------------------------------------------------  
; Identify the specie that corresponds to a particular 3d product  
;---------------------------------------------------------------------------  
  
  species = -1
  IF data.data_product EQ 12 OR $
    data.data_product EQ 13 OR $
    data.data_product EQ 14 THEN species = 0 ; H+
  
  IF data.data_product EQ 15 OR $
    data.data_product EQ 16 THEN species = 1 ; He+++
  
  IF data.data_product EQ 46 OR $
    data.data_product EQ 47 THEN species = 2 ; He+
  
  IF data.data_product EQ 48 OR $
    data.data_product EQ 49 THEN species = 3 ; O+
  
  IF data.data_product EQ 17 OR $
    data.data_product EQ 18 THEN species = 4 ; hm
    
;---------------------------------------------------------------------------  
  
  n_e = data.nenergy            ; number of energies   
  nbins = data.nbins            ; number of bins 
  energy = DOUBLE(data.energy(*,*,0))  ; in eV    
  geof= data.geom_factor         ; geometric factor of smallest bin
  ;mass = DOUBLE(data.mass)      ; scaler
  
; all = FLTARR(n_e,nbins,data.packets)
;  IF data.data_name EQ 'hm' THEN BEGIN
; 	all = DBLARR(n_e,nbins,32,data.packets)
;    energy = REBIN(energy, n_e, nbins, 32)
;  ENDIF
  
; -------------------------------------------------------------------------  
; Calculate CODIF TOF and efficiencies  
; -------------------------------------------------------------------------  

  IF species(0) NE -1 THEN BEGIN
    PRINT, 'CALCULATING EFFICIENCIES'
    IF eff_routine EQ 'cluster_ts_eff_corr' THEN $
      seff = CALL_FUNCTION(eff_routine,$
                           REBIN(DOUBLE(energy),n_e, nbins, packets) ,$
                           data.pac, species(0), $
                           data.sensitivity, nbins, packets, $
                           ANODE_EFFIC=data.anode_effic, $
                           ABSOL_EFFIC=data.absol_effic $
                           )
                           ;OEBINS = data.oebins, EBINS=data.ebins)
  ENDIF
  
  PRINT, 'CONVERTING UNITS FROM ',data.units_name,' TO ',units

; -------------------------------------------------------------------------  
; Define the spin fraction for the 88 and 24 angle products.
; (it can be placed in a common file)
; -------------------------------------------------------------------------  
  IF data.nbins EQ 88 THEN BEGIN
    sf_d = [0.25, 0.25, 0.25, 0.25, $
            0.125, 0.125, 0.125, 0.125, 0.125, 0.125, 0.125, 0.125, $
            0.0625, 0.0625, 0.0625, 0.0625, 0.0625, 0.0625, 0.0625, $
            0.0625, 0.0625, 0.0625, 0.0625, 0.0625, 0.0625, 0.0625, $
            0.0625, 0.0625, 0.0625, 0.0625, 0.0625, 0.0625, 0.0625, $
            0.0625, 0.0625, 0.0625, 0.0625, 0.0625, 0.0625, 0.0625, $
            0.0625, 0.0625, 0.0625, 0.0625, 0.0625, 0.0625, 0.0625, $
            0.0625, 0.0625, 0.0625, 0.0625, 0.0625, 0.0625, 0.0625, $
            0.0625, 0.0625, 0.0625, 0.0625, 0.0625, 0.0625, 0.0625, $
            0.0625, 0.0625, 0.0625, 0.0625, 0.0625, 0.0625, 0.0625, $
            0.0625, 0.0625, 0.0625, 0.0625, 0.0625, 0.0625, 0.0625, $
            0.0625, 0.125, 0.125, 0.125, 0.125, 0.125, 0.125, 0.125,$
            0.125, 0.25, 0.25, 0.25, 0.25] 
  ENDIF ELSE BEGIN
    IF data.nbins EQ 24 THEN BEGIN
      sf_d = [0.5, 0.5, $
              0.5, 0.5, $
              0.25, 0.25, 0.25, 0.25, $
              0.25, 0.25, 0.25, 0.25, $
              0.25, 0.25, 0.25, 0.25, $
              0.25, 0.25, 0.25, 0.25, $
              0.5, 0.5, $
              0.5, 0.5]
    ENDIF ELSE BEGIN
      print, 'SPIN_FRACT array no defined'
    ENDELSE
  ENDELSE
  sf = fltarr(n_e,nbins)
  FOR i = 0, n_e-1 DO sf(i,*)=sf_d
; -------------------------------------------------------------------------  
  
  scale_all_packets=dblarr(n_e,nbins,packets)
  FOR i = 0, packets - 1 DO BEGIN
    
    dt = data.integ_t(i)           ; CM
    print, 'dt = ', dt
;    dt = data.integ_t(i)
    eff = REFORM(seff(*,*,i))
    gf = geof(i)
    
    CASE strupCASE(units) of
      'COUNTS' :  scale = 1.
      'NCOUNTS':  scale = 1. / (sf)
      'RATE'   :  scale = 1. / (dt)
      'NRATE'  :  scale = 1. / (dt * sf)
      'EFLUX'  :  scale = 1. / (dt * sf * eff * (gf))
      'DIFF FLUX' :  scale = 1. / $
        ((dt * sf * eff * (gf)) * energy)
      'DIST FUNC' :  scale = 1. / ($
                                    (dt * sf * eff * (gf)) * energy^2 * $
                                    2./mass/mass*DOUBLE(1e5) )
      ELSE: BEGIN
        PRINT,'Undefined units: ',units
      END
    ENDCASE
    scale_all_packets(*,*,i)=double(scale)
  ENDFOR
    
;---------------------------------------------------------------------------
; For LS (Low Senitivity) pixels 0 and 7 are blocked wich means that
; certain angles have to be set to zero
;---------------------------------------------------------------------------
  
  low_sens=where(data.sensitivity EQ 0, ct_ls)
  IF ct_ls GT 1 THEN BEGIN
  
    angle_to_pixel_88 =    [0, 0, 0, 0, $
                            1, 1, 1, 1, 1, 1, 1, 1, $			
                            2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, $
                            3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, $
                            4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, $
                            5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, $
                            6, 6, 6, 6, 6, 6, 6, 6, $			
                            7, 7, 7, 7]
    
    angle_to_pixel_24 =   [0, 0, $
                           1, 1, $
                           2, 2, 2, 2, $
                           3, 3, 3, 3, $
                           4, 4, 4, 4, $
                           5, 5, 5, 5, $
                           6, 6, $
                           7, 7]
    
    CASE nbins OF
      
      88: angle_array = angle_to_pixel_88
      24: angle_array = angle_to_pixel_24
      
    ENDCASE
    
    
    blocked_pix_index = where(angle_array EQ 0 OR angle_array EQ 7)
    scale_all_packets(*,blocked_pix_index,low_sens) = 0.
    
  ENDIF
;---------------------------------------------------------------------------    
  data.data=data.data * scale_all_packets ; CM
  data.units_name = units
  
;all = 0
  n_e = 0
  nbins = 0
  energy = 0
  geom = 0
  gf = 0
  sf = 0
  mass = 0
  mass = 0
  dt = 0
  scale = 0
  pac = 0
  sens = 0
  eff = 0
  seff = 0

  RETURN,data
  
END

