;+
;FUNCTION:      get_cis_cod_3d, prod_num, sat, time=time
;INPUT:
;       prod_num: Product number
;	sat: Satellite number (1-4)
;	time: If set, return time array of data values (index ignored)
;
;PURPOSE:   Retrieve CIS/CODIF 3D data.
;
;CREATED BY:    Peter Schroeder
;LAST MODIFICATION:  August 2000
;
;-
function get_cis_cod_3d, prod_num, sat, time=time

  case prod_num of
    12: enang = [16,88]
    13: enang = [31,88]
    14: enang = [31,24]
    15: enang = [16,88]
    16: enang = [31,88]
    17: enang = [16,88]
    18: enang = [31,88]
    46: enang = [16,88]
    47: enang = [16,88]
    48: enang = [31,88]
    49: enang = [31,88]
  endcase
  
  nenergy = enang[0]
  nangles = enang[1]
  
  options = [long(sat),0l,long(prod_num),0l]
  
  if keyword_set(time) then return,get_cis_data_times(options)
  
  basic_data = get_cis_all_data(options)
  ;if basic_data eq 0 then return
  ndatapoints = n_elements(basic_data.data)
  
  if not keyword_set(basic_data) then return, 0
  
  dtime = cis_tplot_time(basic_data.header.time_in_ms)
  op_mode = basic_data.header.op_mode
  sensitivity = basic_data.header.sensitivity
  
  get_cis_cod_angles,nangles,theta,phi,dtheta,dphi
  
;---------------------------------------------------------------------------
; CM Replicate angle definitions for all energy bins creating 2D arrays  
;---------------------------------------------------------------------------
  th     = REPLICATE(1, 1, nangles) * theta
  theta  = REBIN(th, nenergy, nangles)
  ph     = REPLICATE(1, 1, nangles) * phi
  phi    = REBIN(ph, nenergy, nangles)
  dth    = REPLICATE(1, 1, nangles) * dtheta
  dtheta = REBIN(dth, nenergy, nangles)
  dph    = REPLICATE(1, 1, nangles) * dphi
  dphi   = REBIN(dph, nenergy, nangles)
  
;---------------------------------------------------------------------------
  energy_arr = fltarr(nenergy,nangles,ndatapoints)
  denergy_arr = energy_arr
  geom_factor = dblarr(ndatapoints)
  delta_t = dblarr(ndatapoints)
  tlm_rate = basic_data.header.tlm_rate
  pspin = basic_data.header.pspin
  for j = 0,ndatapoints-1 do begin
    geom_factor[j]=get_cis_cod_geom_factor(dtime[j], sat, sensitivity[j])
    get_cis_cod_energies,dtime[j],sat,nenergy,op_mode[j],sensitivity[j],$
      energy,denergy
    
    ;------------------------------------------------------------------------
    ; Check compressed/uncompressed mode and product numbers
    ;------------------------------------------------------------------------
    new_prod_num = prod_num
    CASE prod_num OF
      
      12: IF basic_data.data(j).k1 NE 255 THEN new_prod_num = 39
      13: IF basic_data.data(j).k1 NE 255 THEN new_prod_num = 40
      14: IF basic_data.data(j).k1 NE 255 THEN new_prod_num = 41
      15: IF basic_data.data(j).k1 NE 255 THEN new_prod_num = 42
      16: IF basic_data.data(j).k1 NE 255 THEN new_prod_num = 43
      17: IF basic_data.data(j).k1 NE 255 THEN new_prod_num = 44
      18: IF basic_data.data(j).k1 NE 255 THEN new_prod_num = 45
      46: IF basic_data.data(j).k1 EQ 255 THEN new_prod_num = 32
      47: IF basic_data.data(j).k1 EQ 255 THEN new_prod_num = 33
      48: IF basic_data.data(j).k1 EQ 255 THEN new_prod_num = 34
      49: IF basic_data.data(j).k1 EQ 255 THEN new_prod_num = 35
      
    ENDCASE
    ;------------------------------------------------------------------------
    num_spins = get_cis_cod_accum_spins(dtime[j], sat, new_prod_num,$
                                        tlm_rate[j], op_mode[j])
    delta_t[j] = num_spins*pspin[j]/1d3
    for i = 0,nangles-1 do begin
      energy_arr[*,i,j] = energy
      denergy_arr[*,i,j] = denergy
    endfor
  endfor
  if nenergy eq 31 then integ_t = delta_t / 32. ; CM
  if nenergy eq 16 then integ_t = delta_t /16. ; CM
  
;---------------------------------------------------------------------------  
; CM get calibration structures
;---------------------------------------------------------------------------  
  cal_struct1 = get_cis_cal(dtime[0], sat, 'CIS1')
  cal_struct2 = get_cis_cal(dtime[0], sat, 'EFF')
;---------------------------------------------------------------------------  
  
;---------------------------------------------------------------------------  
; CM From the product number deduce the specie and from there the mass
;---------------------------------------------------------------------------  
  
  species = -1
  IF prod_num EQ 12 OR $
    prod_num EQ 13 OR $
    prod_num EQ 14 THEN species = 0 ; H+
  
  IF prod_num EQ 15 OR $
    prod_num EQ 16 THEN species = 1 ; He+++
  
  IF prod_num EQ 46 OR $
    prod_num EQ 47 THEN species = 2 ; He+
  
  IF prod_num EQ 48 OR $
    prod_num EQ 49 THEN species = 3 ; O+
  
  IF prod_num EQ 17 OR $
    prod_num EQ 18 THEN species = 4 ; hm
  
  
  mass = [0.0104389, 0.0208778, 0.0417556, 0.167022]
                                ; mass/charge units: eV/((km/s)^2)/e
  
;---------------------------------------------------------------------------  
retdata = { $
	project_name: 'CLUSTER CODIF', $
	data_name: 'Product '+strcompress(prod_num,/rem), $
	data_product: prod_num, $
	species: species, $
	units_name: 'Counts', $
	units_procedure: 'convert_cod_units', $
	valid: 1, $
	time: dtime, $
	end_time: dtime+delta_t, $
	delta_t: delta_t, $
        integ_t: integ_t, $ ;CM
	geom_factor: geom_factor, $
	nenergy: nenergy, $
	nbins: nangles, $
	bins: replicate(1b, nenergy, nangles), $
	energy: energy_arr, $
	denergy: denergy_arr, $
	theta: theta, $
	phi: phi, $
	dtheta: dtheta, $
	dphi: dphi, $
	k1: basic_data.data.k1, $
	k2: basic_data.data.k2, $
	data: float(basic_data.data.data), $
	mass: mass(species), $ ;CM
        scale: fltarr(nenergy,nangles), $ ; CM
        pac:fix(cal_struct1.post_accel_volt), $ ; CM
        anode_effic:cal_struct1.anode_effic_coeff_table, $ ; CM
        absol_effic:cal_struct2.absolute_efficiencies, $ ; CM
        sensitivity:sensitivity $ ;CM
}
	
return, retdata
end
