;+
;PROCEDURE:	get_spec_cis
;PURPOSE:
;  Creates "TPLOT" variable by summing 3D data over selected angle bins.
;
;INPUT:		sat: Satellite number
;		inst: Instrument (0: CODIF, 1: HIA)
;		prod: Product number
;
;KEYWORDS:	bins: a keyword telling which bins to sum over
;		gap_time: time gap big enough to signify a data gap (def 200)
;		no_data: returns 1 if no_data else returns 0
;		units:	convert to these units if included
;               NAME:  New name of the Data Quantity
;               BKG:  A 3d data structure containing the background counts.
;               FLOOR:  Sets the minimum value of any data point to sqrt(bkg).
;               MISSING: value for bad data.
;		KEV: If set, energy array and energy labels are put in keV.
;		     The default is to put energy in eV.
;
;CREATED BY:	Peter Schroeder
;-

PRO get_spec_cis, sat, inst, prod,  $
                  mid_times, data, energy, $
                  bins=bins, $       
                  gap_time=gap_time, $ 
                  no_data=no_data, $
                  units = units,  $
                  name  = name, $
                  bkg = bkg, $
                  missing = missing, $
                  trange=trange, $
                  floor = floor, $
                  kev = kev
  
ex_start = systime(1)

;----------------------------------------------------------------------------
; Choose reading routine
;----------------------------------------------------------------------------
CASE inst OF
	0:	routine = 'get_cis_cod_data'
	1:	routine = 'get_cis_hia_data'
ELSE: BEGIN
	print, 'Instrument number must be 0 (CODIF) or 1 (HIA)'
	RETURN
END
ENDCASE
;----------------------------------------------------------------------------

;----------------------------------------------------------------------------
; Read data - Convert units
;----------------------------------------------------------------------------
dat = call_function(routine,prod,sat)
packets=n_elements(dat.time)
IF NOT keyword_set(units) THEN units = 'Counts'
IF units NE 'Counts' THEN $
  dat = convert_cluster_units(dat,units,'cluster_ts_eff_corr', $
                             packets=packets)
;----------------------------------------------------------------------------

;----------------------------------------------------------------------------
; Substract background
;----------------------------------------------------------------------------
IF keyword_set(bkg) THEN   dat = sub3d(dat,bkg)
;----------------------------------------------------------------------------

n_e = dat.nenergy
nbins = dat.nbins
IF keyword_set(bins) EQ 0 THEN str_element,dat,'bins',bins

IF n_elements(bins) EQ (nbins*n_e) THEN BEGIN
	bins = total(bins,1) GT 0
	binsindx = where(bins EQ 0,binscnt)
	IF binscnt EQ 0 THEN bins = 0
ENDIF


times = dat.time
IF n_elements(times) EQ 1 AND times(0) EQ 0 THEN BEGIN
        print,'No valid data in timerange.'
        store_data,ytitle,dat=0
        RETURN
ENDIF

firstvalid = -1
IF dat.valid EQ 1 THEN firstvalid = 0
i = 0
WHILE firstvalid EQ -1 DO BEGIN
	IF i EQ n_elements(times) THEN BEGIN
		print,'No valid data in timerange.'
		store_data,ytitle,dat=0
		RETURN
	ENDIF
	dat=call_function(routine,prod,sat,index=i)
	IF dat.valid EQ 1 THEN firstvalid = i
	i = i + 1
ENDWHILE

nenergy = dat.nenergy

max = n_elements(times)

data   = fltarr(max,nenergy)
energy   = fltarr(max,nenergy)

count = dat.nbins
IF NOT keyword_set(bins) THEN ind=indgen(dat.nbins) ELSE ind=where(bins,count)
IF count NE dat.nbins THEN ytitle = ytitle+'_'+strtrim(count,2)
IF keyword_set(bkg) THEN ytitle = ytitle+'b'
IF keyword_set(name) EQ 0 THEN name=ytitle ELSE ytitle = name
ytitle = ytitle+' ('+units+')'

IF NOT keyword_set(units) THEN units = 'Counts'
IF units EQ 'Counts' THEN norm = 1 ELSE norm = count

IF NOT keyword_set(missing) THEN missing = !values.f_nan

last_time = (dat.time(0)+dat.end_time(0))/2.
last_bin= (dat.end_time(0)-dat.time(0))/2.

default_gap_time=120.
IF NOT KEYWORD_SET(gap_time) THEN gap_time = default_gap_time

;time=dblarr(max)
;last_time = (dat.time(0)+dat.end_time(0))/2. 
;last_bin = (dat.end_time(0)-dat.time(0))/2.

;FOR n = 0, packets-1 DO BEGIN
;  m = WHERE(time EQ 0)
;  m = m(0)
;  replace = 0
;  IF ABS(((dat.time(n)+dat.end_time(n)))/2.-last_time) GE gap_time THEN BEGIN
;    time(m) = last_time + last_bin
;    data(m,*)=missing
;    time(m+1)=((dat.time(n)+dat.end_time(n))/2.)- $
;      (dat.end_time(n)-dat.time(n))
;    data(m+1,*)=missing
;  ENDIF
;ENDFOR


IF dat.valid NE 0 THEN BEGIN
  nfindx = where(finite(dat.data) NE 1,cnt) ; check for bad values i.e.
  IF cnt NE 0 THEN dat.data(nfindx) = 0. ; Inf or NaN
  mid_times = times
  data = transpose(total( dat.data(*,ind,*), 2)/norm)
  energy = transpose(total( dat.energy(*,ind,*), 2)/count)
  mid_times = dat.time+(dat.end_time-dat.time)/2d
ENDIF ELSE BEGIN
  data=missing
  energy=missing
ENDELSE


IF keyword_set(kev) THEN BEGIN
	energy = energy/1e3
	nrgs = reform(energy(firstvalid,*))
	labels = strtrim( round(nrgs), 2)+' keV'
ENDIF ELSE BEGIN
	nrgs = reform(energy(firstvalid,*))
	labels = strtrim( round(nrgs) ,2)+' eV'
ENDELSE

delta = energy - shift(energy,1,0)
w = where(delta,c)
IF c EQ 0 THEN energy = nrgs
energy = nrgs ; CM
;message,/info,'Not running smoothspikes'
;smoothspikes,times

ex_time = systime(1) - ex_start
message,string(ex_time)+' seconds execution time.',/cont,/info

RETURN

END
