;******************************************************************************
;*  NAME:        r_cluster_codif
;*
;*  DESCRIPTION: Routine to read cluster_codif files. Adheres to
;*               papco conventions: Error common block
;*                                  Uses MJDT time
;*                                  papco read philosophy
;*
;*  INPUT:       none (uses common block time from papco)
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  KEYWORDS:    VERBOSE  -  enables print comments, save/restore comments
;*
;*  DATE:        June 1996
;*
;*  AUTOR:       R. Friedel
;*
;*  CHANGES:     February 1997. Used MJDT time now, and is a fully
;*               running example module on it's own
;*               October 1997. Rewritten to be more general and fully
;*               adhering to papco philosophy. Added remore Get feature, and
;*               the example data file is no longer part of the 
;*               module - it needs to be gotten remotely first!
;******************************************************************************
pro r_cluster_codif, type, VERBOSE=VERBOSE

  common cluster_codif_data, cluster_codif_header, cluster_codif_data, $
    cluster_codif_time
  common mjdt, mjdt_start, mjdt_end
  common get_error, get_err_no, get_err_msg
   
; common get_error is used by papco to check if the data read was successful.
; it is up to the user to set this correctly and to make sure that the read- 
; routine does not crash on an I/O error - otherwise papco will crash at that
; point, too. papco draws an empty panel when get_err_no in not equal to zero
; and prints get_err_msg in that panel for user information. papco will try
; to invoce the plot routine for this data if get_err_no = 0 , otherwise an
; emplty panel with the error message get_err_msg is plotted.

  get_err_no=0  &  get_err_msg='cluster_codif read succesfully'
   
; check env. variable pointing to data directory
  if not (papco_check_data_env('CLUSTER_CODIF_DATA', path=PATH)) then return
   
; check VERBOSE keyword and set
  if keyword_set(VERBOSE) then verbose=VERBOSE else verbose=0     
   
; clear data arrays for reuse
  cluster_codif_header=0  &  cluster_codif_data=0   
   
; loop through from start to end time concatonating data files into one
; array. Here we assume that the data is stored in files per day, and
; that the time info in each file is seconds of day. PAPCO standard is
; to hand over data to the plot routines with the time running in
; seconds from the number of seconds of the startday.
  time=mjdt_start  &  no_files_read=0  &  day_count=0.0
   
  while mjdt_end.mjd ge time.mjd do begin
     
    ;assemble the data file from the time range requested
    convert_t90_to_date, mjdt_start, year, day_oy, hour, minute, second,$
                         month, cmonth, day_om, /mjdt   
    datestr=string(year-1900,month,day_om,format='(i2.2,i2.2,i2.2)')
    datestr=string(year,'-',month,'-',day_om,format='(i4.4,a1,i2.2,a1,i2.2)')
    datestr=datestr+'/00:00:00' ; DATE
    sat=type(2)+1 ; S/C
    case type(0) of ; PRODUCT
      0: prod=12
      1: prod=13
      2: prod=14
      3: prod=15
      4: prod=16
      5: prod=17
      6: prod=18
      7: prod=46
      8: prod=47
      9: prod=48
     10: prod=49
    endcase
    inst=0 ; CIS INSTRUMENT(CODIF/HIA)
    case type(1) of ; UNITS
      0: begin
        units_name='Counts'
        name='Counts'
      end
      1: begin
        units_name='DIFF FLUX'
        name='DIFF_FLUX'
      end
      2: begin
        units_name='EFLUX'
        name='EN_FLUX'
      end
    endcase
    
    inst_str=['COD','HIA']
    fln='C'+strcompress(string(sat),/remove_all) + '_' + $
      inst_str(inst) + $
      '_' + strmid(datestr,0,4)+strmid(datestr,5,2)+strmid(datestr,8,2) + $
      '_P' + strcompress(string(prod),/remove_all)
    
    ;read one day of data
    r_cluster_codif_onefile, sat, inst, prod, datestr, $
      units_name, name, path, fln, VERBOSE=VERBOSE
    
    goto, jump
    ;check for an error in get_error - and then get the file remotely!
    ;comment this section out if you do not want this feature.
    if get_err_no ne 0 then begin      
      c_url=papco_getenv('CLUSTER_CODIF_FTP_ARCHIVE')
      c_file=fln+'.idl'  &  c_download_dir=path
      papco_get_remote_data, c_url, c_file, c_download_dir
      ;check if file "arrived" and then read it
      if get_err_no eq 0 then begin
        r_cluster_codif_onefile, path+fln, VERBOSE=VERBOSE
      endif else begin
        get_err_msg='No data locally, remote get failed!C  ' + get_err_msg
        return
      endelse  
    endif   
     
    ;if a file was succesfully read, concatonate!
    if get_err_no eq 0 then begin 
      if no_files_read eq 0 then begin   ;start the array buffer
        cluster_codif_data.time=cluster_codif_data.time+86400.0*day_count
        cluster_codif_data_buf=temporary(cluster_codif_data)
      endif else begin                   ;otherwise concatonate arrays
        cluster_codif_data.time=cluster_codif_data.time+86400*day_count
        cluster_codif_data_buf=[temporary(cluster_codif_data_buf),temporary(cluster_codif_data)]
      endelse        
      cluster_codif_data=0                      ;clear array for reuse
      no_files_read=no_files_read+1
    endif
    day_count=day_count+1
    time.mjd=time.mjd+1                 ; next day
  endwhile         
   
; check if any files were read   
  if (no_files_read ne 0) and (get_err_no ne 1) then begin
    get_err_no=0
    get_err_msg=string(no_files_read,format="(i2)")+ ' files read succesfully'
    ;clear arrays and load buffer into output array
    cluster_codif_data=0  &  cluster_codif_data=temporary(cluster_codif_data_buf)
  endif else begin
    get_err_no=1
    get_err_msg='No files found for timerange'
    return
  endelse      
   
; restrict data to actual time range requested
  ut1=mjdt_start.t
  ut2=long(mjdt_end.mjd-mjdt_start.mjd)*86400l+mjdt_end.t
  index=where((cluster_codif_data.Time ge ut1) and (cluster_codif_data.Time le ut2),count)
  if count ne 0 then cluster_codif_data=cluster_codif_data(index)
jump:
end

;******************************************************************************
;*  NAME:        r_cluster_codif_onefile,fln
;*
;*  DESCRIPTION: reads one file of type plot_file. Assumes data has been
;*               stored in IDL saveset using structures cluster_codif_header and
;*               cluster_codif_data
;*
;*  INPUT:       fln  - string of filename
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        October 1997
;*
;*  AUTOR:       R. Friedel
;******************************************************************************
pro r_cluster_codif_onefile, sat,inst,prod,datestr, $
                             units_name, name, path, fln, VERBOSE=VERBOSE
  
  common cluster_codif_data, cluster_codif_header, cluster_codif_data, $
    cluster_codif_time
  common get_error, get_err_no, get_err_msg
   
  get_err_no=0  &  get_err_msg='cluster_codif onefile read succesfully'
   
; clear data arrays for reuse
  cluster_codif_header=0  &  cluster_codif_data=0   
  IF units_name EQ 'Counts' THEN fln=''
  on_ioerror, read_original
  restore,path+fln+'.idl', VERBOSE=VERBOSE
  dealloc_lun,/quiet   ;frees logical units used by save/restore
  message,'idl saveset read succesfully', /cont
  return

  read_original:       ;if the .idl file is not found, try to read the original
  r_cluster_codif_onefile_original, sat,inst,prod,datestr, $
    units_name, name, path, fln, VERBOSE=VERBOSE
      
end   
  
;******************************************************************************
;*  NAME:        r_cluster_codif_onefile_original,fln
;*
;*  DESCRIPTION: reads one file of type plot_file_original. Under the
;*               papco read philosophy, files should be read the first
;*               time in whatever format they might have - ascii,
;*               fortran binary, pascal binary, etc. The files might also
;*               be in some compressed format, so they need to be uncompressed
;*               first. 
;*               
;*               You might also have to change time formats. 
;*               A lot of files are stored per day, and have seconds of
;*               that day as the time. This is what is assumed here and
;*               is papco standard. You need to convert the time
;*               to papco standard if you intend to use the plotroutines 
;*               supplied with papco (use the routines in
;*               $papco_BASE/papco_Lib/time_convert.pro). 
;*               If you use your own plot routines this step is not needed.
;*               
;*               The first read might be slow, esp if the format is ascii.
;*               Use the IDL "save" command to make a binary copy of the
;*               file which can then be read quickly using "restore". The
;*               next time only the IDL saveset is read, which is much
;*               faster.
;*               
;*               In order to be able to concatonate the arrays easily,
;*               it makes sense to have the data in an array of time-ordered
;*               structures, which makes adding together days of data
;*               trivial using the IDL concatonation operator "[]".
;*               
;*  INPUT:       fln  - string of filename
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        October 1997
;*
;*  AUTOR:       R. Friedel
;******************************************************************************
pro r_cluster_codif_onefile_original,sat,inst,prod,datestr, $
                                     units_name, name, path, fln, $
                                     VERBOSE=VERBOSE
  
  common cluster_codif_data, cluster_codif_header, cluster_codif_data, $
    cluster_codif_time
  common get_error, get_err_no, get_err_msg  
   
  get_err_no=0  &  get_err_msg='cluster_codif_onefile_orignal read succesfully'
   
; clear data arrays for reuse
  cluster_codif_header=0  &  cluster_codif_data=0   

  spawn, 'source ~/mypapco/cluster_codif/plot_read/l1toidl/l1toidl.sh'
  spawn, 'cd ~/mypapco/cluster_codif/plot_read/l1toidl/idl'
  
  load_cis_data, sat, inst, prod, datestr
  get_spec_cis, sat, inst, prod, $
    cluster_codif_time, cluster_codif_data, cluster_codif_header, $
    units=units_name, name=name

; save read data with correct time. Rename data to *.idl
  save,filename=path+fln+'.idl',cluster_codif_header,cluster_codif_data, $
    cluster_codif_time, VERBOSE=VERBOSE
  dealloc_lun,/quiet              ;frees logical units used by save/restore
  return
  
  out:
  ; if there was an error, report it. 
  get_err_no=!ERROR
  get_err_msg='!C '+!ERR_STRING
   
end
