;******************************************************************************
;* PROCEDURE:     
;*      p_cluster_eph_line, panel, type, OUTPUT=OUTPUT, $
;*           OVERPLOT=OVERPLOT,$
;*           PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           SUBTABLE=SUBTABLE, $
;*           _EXTRA=extra_par
;* 
;* DESCRIPTION:  
;*	This procedure plots data of type data_type for inclusion in papco.  
;*      Data must have been read and placed into common blocks before this
;*      routine is called. Behavior is controlled by the vector panel
;*	which controls positioning of the plot (see papco_conventions.text
;*      in $papco). In general, the following plot conventions are used:
;*              - if the panel is at the bottom plot time axis plus any
;*                additional axis (ie ephemeris) needed.
;*              - if panel is at the top also plot a title
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a papco_colorbar.
;*              - panels all have common time
;*              - y-scaling (or z-scaling for color plots) is either
;*              - automatic or manualy set using common yscale or zscale   
;*
;* INPUTS:       
;*      panel   three element integer vector. Controls positioning of the plot
;*              (see papco_conventions.text in $papco)
;*      type    optional parameter for options. Normally is a four element
;*              integer vector. This is normally equivalent to the field
;*              typeVector of the papco structure papco_PLOTINFO which is set
;*              by the panel editor. Additional fields that can be used if
;*              needed are papco_plotinfo.ioptions (20 element vector) and 
;*              papco_PLOTINFO.channel (integer). If your plot routine needs
;*              more input fields than these you have to extend the definition
;*              of the structure papco_PLOTINFO
;*              (procedure papco_getPlotInfoStruct in $papco/papco.pro). Try to
;*              avoid this!
;*              
;* OUTPUTS: 
;*      none
;*
;* KEYWORDS:
;*      OUTPUT  if set call the papco routine that outputs plot data
;*              to a file.
;*    OVERPLOT  if you support this you can alter the appearance of your
;*              plot if it is an overplot to an existing panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mag. field.
;*              see p_crres_mag for an example.
;*    PLOTS_ATTHISPOSITION  used to control the left side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and line style.   
;* 
;* CALLING SEQUENCE:
;*       p_cluster_eph_line, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY:       
;*     written December 1995, Reiner Friedel, at Goddard
;*     added OUTPUT support, July 1998, , Reiner Friedel
;******************************************************************************
pro p_cluster_eph_line, panel, plotinfo, OUTPUT=OUTPUT, $
                        OVERPLOT=OVERPLOT, $
                        PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION, $
                        SUBTABLE=SUBTABLE, $
                        _EXTRA=extra_par

; The following common blocks are needed:
COMMON mjdt, mjdt_start, mjdt_end               ;time limit in mjdt
COMMON yscale, yscl                             ;man/auto yscaling
COMMON shift_label, down                        ;for x-axis label shifting
COMMON coordinateSystems, plotted_x, plotted_y  ;info on coords

; One of the common blocks should contain the plot data as returned by the
; data read procedure r_cluster_eph. 
COMMON cluster_eph_data, input_header, input_data
COMMON cluster_eph_tick, time, yray, tick_plotinfo
COMMON cluster_eph

IF keyword_set(OUTPUT) THEN output = output ELSE output = 0

SC_ID = plotinfo.typevector(0)
coord = plotinfo.typevector(1)
  
; set the begin end end times. CLUSTER cdf time is in cdf epoch. Plot
; always starting at zero time = xut1 for reasonable numbers.
  convert_t90_to_date, mjdt_start, yr,day_oy,hr,min,sec,mo,cmo,day_om, /mjdt
  cdf_epoch,xut1,yr,mo,day_om,hr,min,sec,0.0,/COMPUTE_EPOCH
  convert_t90_to_date, mjdt_end, yr,day_oy,hr,min,sec,mo,cmo,day_om, /mjdt
  cdf_epoch,xut2,yr,mo,day_om,hr,min,sec,0.0,/COMPUTE_EPOCH

  utitle=coord_names(coord)
  tags = TAG_NAMES(input_data(0))

  CASE 1 of
    (coord LE 2): BEGIN ;X Y Z (GSE)
      pos = strpos(tags, 'SC_R_XYZ_GSE') & idx = where(pos NE -1, c)
      yray = input_data.(idx(0))(coord); centroid position
      IF SC_ID LE 3 THEN BEGIN ;spacecraft position
        t = 'SC_DR'+varprt(SC_ID+1)+'_XYZ_GSE'
        pos = strpos(tags, t) & idx = where(pos NE -1, c)
        offset = input_data.(idx(0))(coord) & yray = yray+offset
      ENDIF
      yray = yray / 6371.0
      uytitle='R!DE!N' & nodata = -1e-31
    END
    (coord EQ 3): BEGIN ;radius
      pos = strpos(tags, 'SC_R_XYZ_GSE') & idx = where(pos NE -1, c)
      vec = input_data.(idx(0))
      yray = sqrt( vec(0,*)^2 + vec(1,*)^2 + vec(2,*)^2 )   ;centroid radius  
      IF SC_ID LE 3 THEN BEGIN ;do spacecraft position
        t = 'SC_DR'+varprt(SC_ID+1)+'_XYZ_GSE'
        pos = strpos(tags, t) & idx = where(pos NE -1, c)
        vec = vec+input_data.(idx(0))
        yray = sqrt( vec(0,*)^2 + vec(1,*)^2 + vec(2,*)^2 ) ;spacecraft radius
      ENDIF
      yray = transpose(yray)
      yray = yray / 6371.0
      uytitle='R!DE!N' & nodata = -1e-31
    END
    (coord EQ 4): BEGIN ;mlat
      t = 'C'+varprt(SC_ID+1)+'_INVAR_LAT'
      pos = strpos(tags, t) & idx = where(pos NE -1, c)
      yray = input_data.(idx(0))
      uytitle='deg' & nodata = -1e31
    END
    (coord EQ 5): BEGIN ;mlt
      t = 'C'+varprt(SC_ID+1)+'_MAG_LOCAL_TIME'
      pos = strpos(tags, t) & idx = where(pos NE -1, c)
      yray = input_data.(idx(0)) 
      uytitle='hours' & nodata = -1
    END
    (coord EQ 6): BEGIN ;l value
      t = 'C'+varprt(SC_ID+1)+'_L_VALUE'
      pos = strpos(tags, t) & idx = where(pos NE -1, c)
      yray = input_data.(idx(0))
      uytitle='L value' & nodata = -1
    END 
    (coord EQ 7): BEGIN ;B mag predict
      t = 'C'+varprt(SC_ID+1)+'_PRED_B_MAG'
      pos = strpos(tags, t) & idx = where(pos NE -1, c)
      yray = input_data.(idx(0)) 
      uytitle='nT' & nodata = -1e31
    END
    (coord EQ 8): BEGIN ;SC CONFIG QG
      t = 'SC_CONFIG_QG'
      pos = strpos(tags, t) & idx = where(pos NE -1, c)
      yray = input_data.(idx(0)) 
stop
      uytitle='' & nodata = -1e31
    END
    (coord EQ 9): BEGIN ;SC CONFIG QR
      t = 'SC_CONFIG_QR'
      pos = strpos(tags, t) & idx = where(pos NE -1, c)
      yray = input_data.(idx(0)) 
      uytitle='nT' & nodata = -1e31
    END
    (coord EQ 10): BEGIN ; min sep
      t = 'SC_DR_MIN'
      pos = strpos(tags, t) & idx = where(pos NE -1, c)
      yray = input_data.(idx(0)) 
      uytitle='km' & nodata = -1e31
    END
    (coord EQ 11): BEGIN ; max sep
      t = 'SC_DR_MAX'
      pos = strpos(tags, t) & idx = where(pos NE -1, c)
      yray = input_data.(idx(0)) 
      uytitle='km' & nodata = -1e31
    END

    ELSE: BEGIN
      message, 'No valid data coordinate', /cont
      plotted_x = !x  &  plotted_y = !y & return      
    END

  ENDCASE

; interpolate data if time period is below some limits.
  period = 3600.0*4d*1000.0 ;in ms
  dp = (xut2-xut1) / 1000.0 ; data period selected in seconds
  
  CASE 1 OF
      (dp GT 7200.0): BEGIN
          time = input_data.epoch ;original 5 min resolution          
          GOTO, no_interpolate & END
      (dp GT 1800.0) AND (dp LE 7200.0): BEGIN 
          period = 7200 & new_res = 60.0 & END
      (dp GT 300.0) AND (dp LE 1800.0): BEGIN 
          period = 1500 & new_res = 20.0 & END
      (dp GT 60.0) AND (dp LE 300.0): BEGIN 
          period = 1500 & new_res = 5.0 & END
      (dp GT 4.0) AND (dp LE 60.0): BEGIN 
          period = 1500 & new_res = 0.25 & END
      (dp LE 0.25): BEGIN 
          period = 1500 & new_res = 0.01 & END
  ENDCASE
  
  ;first limit data down to +/- 2 half perid from center time 
  ;but a lesast 25 minutes to get in 5 data points!
  n_1 = (xut2+xut1)/2.0 - (period*1000.0/2.0)
  n_2 = (xut2+xut1)/2.0 + (period*1000.0/2.0)
  index=where((input_data.epoch ge n_1) and (input_data.epoch le n_2),cc)
  IF cc NE 0 THEN BEGIN
      time = input_data(index).epoch  ;original 5 min resolution
      yray = yray(index)
      diff = (time(n_elements(time)-1)-time(0))/1000.0
      ndat = diff/new_res+1
      new_time = dblarr(ndat) 
      FOR i = 0, ndat-1 DO new_time(i) =  time(0)+i*new_res*1000.0
      new_yray = interpol(yray, time, new_time)
      time = new_time
      yray = new_yray
  ENDIF    

  no_interpolate:

; restrict data to actual time range requested - this makes for faster zooming
  index=where((time ge xut1) and (time le xut2),c)
  IF c NE 0 then BEGIN & time = time(index) & yray = yray(index) & end 
  xut2=xut2-xut1  &  time=time-xut1  &  xut1=0
  ;cdf epoch is in milliseconds. Need to divide by to get seconds. 
  time=time/1000.0 & xut1=xut1/1000.0 & xut2=xut2/1000.0

  IF output EQ 2 THEN return

  panelset,panel			;sets the panel position viewport

  ;check for valid data
  index = where(yray NE 1e-31, c)
  IF c lt 2 THEN BEGIN
    message, 'No valid data to plot', /cont
    plotted_x = !x  &  plotted_y = !y & return
  ENDIF
     
  ylim2,time,yray,yst,yen,  NODATA = NODATA
  utitle=sc_names(SC_ID)+'!C'+utitle  

; the variables used above have the following meaning:
; 	uytitle : the y-axis label
;	utitle  : the right-of plot label defining the data
;

; yscl of common block yscale is an array of dimensions (*,4) and contains info
; on the yscaling to be used. The first dimension identifies the panel, and the
; next dimension has three entries:  
;         yscl(*,0)  is  0 for "use automatic scaling", 1 for "use manual"
;         yscl(*,1)  is then ymin (or zmin)
;         yscl(*,2)  is then ymax (or zmax)
;         yscl(*,3)  is 0 for "linear" and 1 for "log"  scale
; when scaling is set to automatic, the automatic y-limits are loaded in.  
  if (yscl(panel(0),0) eq 1) then begin
    yst=yscl(panel(0),1)  &  yen=yscl(panel(0),2)
  endif else begin
    yscl(panel(0),1)=yst  &  yscl(panel(0),2)=yen
  endelse
  
; use auto y-labeling routine to get "good" y-lables. This sets the
; global IDL graphics system variables !Y.MINOR, !Y.TICKV, !Y.TICKS -
; so if you want to use this feature, avoid setting these or their
; corresponding keywords in your plot.
  papco_y_label, yst, yen, log=yscl(panel(0),3)
  
; set up extra plot keywords: yrange, xrange, zrange  are always needed. 
; xtickformat is used to set the format of the x-axis ticks, by
; default  no ticks ar plotted. 
  extra_plotPar={xrange:[xut1,xut2], yrange:[yst,yen],  $
                 ylog:yscl(panel(0),3), $
                 ytitle:uytitle, xtickformat:'noticks'}
   
; add keyword structure set here with the one passed in
  extra_plotPar=create_struct(extra_plotPar, extra_par)
   
; use papco routine to draw time axis. This checks for bottom plot
; and uses the user's xtickformat if it is something other than
; 'noticks'.
; If you use this rotuine, make sure your data plot routine uses
; xstyle=5 and ystyle=5 (inhibit axis) so that you don't clobber the
; axes drawn by papco.  
; "down" sets the axis labeling level. 
  down=0
  papco_draw_time_axis, panel, OVERPLOT=OVERPLOT, _extra=extra_plotPar    
  
; You might have to add your own xtickformat for time axis
; labeling. If you stick to the papco convention of always having your plot 
; time array having consecutive seconds from midnight of the start day, then 
; you can use the pre-defined xtickformat functions defined in ticks.pro in
; papco_Lib/axis_label.
; You can at this point also define a custom ytickformat, or leave it out for
; IDL default plot behaviour.

; now do the actual data plot, data only, no axis
  plot, [0, 1], [0, 1], _extra=extra_plotPar, ystyle = 5, xstyle = 5, /nodata
  papco_gap_plot, time, yray, nodata, xstyle=5, ystyle=5, _extra=extra_plotPar

; store the coordinate information into common block for slices
  plotted_x = !x  &  plotted_y = !y

; plot extra x-axis labels (eg. ephemeris) and a x-axis label if required.
; plot in color=1. Example here adds an axis with date information.
; The routine x_side_label puts x-axis label to the right of the axis.
; If you stick to the papco convention of always having your plot 
; time array having consecutive seconds from midnight of the start day, then 
; you can use the pre-defined xtickformat functions defined in ticks.pro in
; papco_Lib/axis_label.
; Or use any other function you define yourself.
; down sets the axis labeling level
  if (panel(0) eq 0) and not keyword_set(OVERPLOT) then begin  
    down=down+1    
  endif

; plot info right of the panel. If you want overplots to have their own labels
; you need to add position-control for this label using the keyword
; PLOTS_ATTHISPOSITION
  if not keyword_set(overplot) then begin
    right_side_label,panel,utitle,_extra=extra_Par
  endif else begin
    for i=0, PLOTS_ATTHISPOSITION-1 do utitle='!C!C'+utitle
    right_side_label,panel,utitle,_extra=extra_Par
  endelse
   
; check if data needs to be output to file. Use the optional keywords
; to add more information to the data file
;      DESCRIPTION   a string or string array describing the data
;      TIME_LABEL    a string decribing the x-axis format
;      CHANNELS      an (ch,2) array giving the channel start/end
;      Y_LABEL       a string with the y-axis label
;      Z_LABEL       a string with the z-axis label
;      RS_LABEL      a string with the righ-side label of the plot
  
  if KEYWORD_SET(OUTPUT) then begin
    message,'Writing plot data out to file',  /cont
    description='cluster_eph data - single eph coordinate'
    time_label='Seconds since start of day'
    ;channels=yarr
    y_label=uytitle
    ;z_label=uztit
    rs_label=utitle
    papco_write_data, time, yray, $
                      DESCRIPTION=description, TIME_LABEL=time_label, $
                      Y_LABEL=y_label, $
                      RS_LABEL=RS_LABEL
  endif  

end 
