;******************************************************************************
;* PROCEDURE:     
;*      p_cluster_eph_ticks, panel, type, OUTPUT=OUTPUT, $
;*           OVERPLOT=OVERPLOT,$
;*           PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           SUBTABLE=SUBTABLE, $
;*           _EXTRA=extra_par
;* 
;* DESCRIPTION:  
;*	This procedure plots data of type data_type for inclusion in papco.  
;*      Data must have been read and placed into common blocks before this
;*      routine is called. Behavior is controlled by the vector panel
;*	which controls positioning of the plot (see papco_conventions.text
;*      in $papco). In general, the following plot conventions are used:
;*              - if the panel is at the bottom plot time axis plus any
;*                additional axis (ie ephemeris) needed.
;*              - if panel is at the top also plot a title
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a papco_colorbar.
;*              - panels all have common time
;*              - y-scaling (or z-scaling for color plots) is either
;*              - automatic or manualy set using common yscale or zscale   
;*
;* INPUTS:       
;*      panel   three element integer vector. Controls positioning of the plot
;*              (see papco_conventions.text in $papco)
;*      type    optional parameter for options. Normally is a four element
;*              integer vector. This is normally equivalent to the field
;*              typeVector of the papco structure papco_PLOTINFO which is set
;*              by the panel editor. Additional fields that can be used if
;*              needed are papco_plotinfo.ioptions (20 element vector) and 
;*              papco_PLOTINFO.channel (integer). If your plot routine needs
;*              more input fields than these you have to extend the definition
;*              of the structure papco_PLOTINFO
;*              (procedure papco_getPlotInfoStruct in $papco/papco.pro). Try to
;*              avoid this!
;*              
;* OUTPUTS: 
;*      none
;*
;* KEYWORDS:
;*      OUTPUT  if set call the papco routine that outputs plot data
;*              to a file.
;*    OVERPLOT  if you support this you can alter the appearance of your
;*              plot if it is an overplot to an existing panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mag. field.
;*              see p_crres_mag for an example.
;*    PLOTS_ATTHISPOSITION  used to control the left side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and line style.   
;* 
;* CALLING SEQUENCE:
;*       p_cluster_eph_ticks, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY:       
;*     written December 1995, Reiner Friedel, at Goddard
;*     added OUTPUT support, July 1998, , Reiner Friedel
;******************************************************************************
pro p_cluster_eph_ticks, panel, plotinfo, OUTPUT=OUTPUT, $
                         OVERPLOT=OVERPLOT, $
                         PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION, $
                         SUBTABLE=SUBTABLE, _EXTRA=extra_par

COMMON plot_composer, widgetData

; The following common blocks are needed:
  common mjdt, mjdt_start, mjdt_end               ;time limit in mjdt
  common shift_label, down                        ;for x-axis label shifting
  common coordinateSystems, plotted_x, plotted_y  ;info on coords

  common cluster_eph_tick, time, yray, tick_plotinfo
  common cluster_eph

  tick_plotinfo = plotinfo
  SC_ID = plotinfo.typevector(0)
  coord = plotinfo.typevector(1)
  
  panelset,panel			;sets the panel position viewport

; set the begin end end times. CLUSTER cdf time is in cdf epoch. Plot
; always starting at zero time = xut1 for reasonable numbers.
  convert_t90_to_date, mjdt_start, yr,day_oy,hr,min,sec,mo,cmo,day_om, /mjdt
  cdf_epoch,xut1,yr,mo,day_om,hr,min,sec,0.0,/COMPUTE_EPOCH
  convert_t90_to_date, mjdt_end, yr,day_oy,hr,min,sec,mo,cmo,day_om, /mjdt
  cdf_epoch,xut2,yr,mo,day_om,hr,min,sec,0.0,/COMPUTE_EPOCH

  tunits = strarr(n_elements(coord_names))
  tunits(*) =  ''
  ;utitle=sc_names(SC_ID)+'!C'+utitle  
  extra_plotPar={xrange:[xut1,xut2], yrange:[0, 1], ystyle:5, xstyle:9, $
                 xtickformat:'noticks'}
   
; add keyword structure set here with the one passed in
  extra_plotPar=create_struct(extra_plotPar, extra_par)
   
; use papco routine to draw time axis. This checks for bottom plot
; and uses the user's xtickformat if it is something other than 'noticks'.
; "down" sets the axis labeling level. 
  down=0 & p=[0,panel(1),panel(2)]
  papco_draw_time_axis, p, OVERPLOT=0, _extra=extra_plotPar  

; store the coordinate information into common block for slices
  plotted_x = !x  &  plotted_y = !y

; plot extra x-axis labels - down sets the axis labeling level
  extra_plotPar.xtickformat='cluster_ticks'
  lbl=''
  IF SC_ID EQ 4 THEN SC_ID_lbl = 'Cc ' ELSE SC_ID_lbl='C'+varprt(SC_ID+1)+' ' 
  for i=0,down-1 do lbl=lbl+'!C'
  for i=0ll,n_elements(coord_names)-1 do BEGIN
    if (PlotInfo.typeVector(1) and 2l^i) eq 2l^i THEN BEGIN 
      tick_plotinfo.typevector(1) = i
      papco_draw_time_axis,p, OVERPLOT=0, /FORCE, _extra=extra_plotPar        
      if widgetData.default_right_label eq 0 then $
        lbl=lbl+SC_ID_lbl+strsubst(coord_names(i), '_', ' ')+' '+ $
      tunits(i) +'!C' else $
      lbl=lbl+SC_ID_lbl+strsubst(coord_names(i), '_', ' ')+'!C'
      down=down+1
    endif  
  endfor

; The routine x_side_label puts x-axis label to the right of the axis.
  x_side_label,panel, lbl

end 

;-----------------------------------------------------------------------------
; ONE function for ALL cluster eph ticks.
function cluster_ticks, axis, index, t

  common mjdt, mjdt_start, mjdt_end
  common cluster_eph_tick, time, yray, tick_plotinfo
  common shift_label, down             ;common for x-axis label shifting
  
  down_str=''  &  for i=0,down-1 do down_str=down_str+'!C'
  ;get data by call ing p_cluster_eph_line to get data only.
  IF index EQ 0 THEN p_cluster_eph_line, panel, tick_plotinfo, OUTPUT = 2

  ;eph has time starting at zero, papco_draw_time_axis at startsec of
  ;day. Need to add in. 
  starttime= double(mjdt_start.t)
  tt = time + starttime

  difft=abs(tt-t) & result=min(difft,minindex) & tick=yray(minindex)

  ;check if data is availble "near" time t
  ;print, t, result 

  CASE tick_plotinfo.typevector(1) of
    0: out_str=string(tick,format="(f6.2)") ;x GSE in R_E
    1: out_str=string(tick,format="(f6.2)") ;x GSE in R_E
    2: out_str=string(tick,format="(f6.2)") ;y GSE in R_E
    3: out_str=string(tick,format="(f5.2)") ;radius in R_E
    4: out_str=string(tick,format="(f6.2)") ;invariant latitude
    5: out_str=string(tick,format="(f5.1)") ;mlt
    6: out_str=string(tick,format="(f6.1)") ;l-val
    7: out_str=string(tick,format="(f6.2)") ;B mod
;    8: out_str=string(tick,format="(f8.2)") ;QG
;    9: out_str=string(tick,format="(f8.2)") ;QR
;   10: out_str=string(tick,format="(f8.2)") ;sep min
;   11: out_str=string(tick,format="(f8.2)") ;sep max
    else: out_str=string(tick)
  endcase
  
  return,down_str+out_str

END
