;******************************************************************************
;*  NAME:        r_cluster_eph
;*
;*  DESCRIPTION: Routine to read cluster_eph files. Adheres to
;*               papco conventions: Error common block
;*                                  Uses MJDT time
;*                                  papco read philosophy
;*
;*  INPUT:       none (uses common block time from papco)
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  KEYWORDS:    VERBOSE  -  enables print comments, save/restore comments
;*
;*  DATE:        March 2001
;*
;*  AUTOR:       R. Friedel
;*
;*  CHANGES:     
;******************************************************************************
pro r_cluster_eph, VERBOSE=VERBOSE

COMMON cluster_eph_data, input_header, input_data
COMMON mjdt, mjdt_start, mjdt_end
COMMON get_error, get_err_no, get_err_msg

get_err_no=0  &  get_err_msg='cluster_eph read succesfully'
      
; check VERBOSE keyword and set
  if keyword_set(VERBOSE) then verbose=VERBOSE else verbose=0     
   
; clear data arrays for reuse
  input_header=0  &  input_data=0   
   
; loop through from start to end time concatonating data files into one
; array. Here we assume that the data is stored in files per month.
  time=mjdt_start  &  no_files_read=0  &  month_count=0.0

; make list of months of data to get
  FOR i = mjdt_start.mjd, mjdt_end.mjd DO BEGIN
    time.mjd = i
    convert_t90_to_date, time, year, day_oy, hour, minute, second,$
                       month, cmonth, day_om, /mjdt       
    month_str=string(year-2000,month,format='(i2.2,i2.2)')
    IF month_count EQ 0 THEN BEGIN
       months = [month_str] 
       month_count = month_count+1
    ENDIF ELSE BEGIN
       IF month_str NE months(month_count-1) THEN BEGIN
         months = [months, month_str]   
         month_count = month_count+1
       ENDIF
    ENDELSE
  ENDFOR

  FOR i = 0, month_count-1 DO begin
 
    datestr = months(i)
     
    ;read one day of data
    r_cluster_eph_onefile, datestr, VERBOSE=VERBOSE
  
    ;check for an error in get_error - and then get the file remotely!
    ;comment this section out if you do not want this feature.
    ;if get_err_no ne 0 then begin      
      ;c_url=papco_getenv('CLUSTER_EPH_FTP_ARCHIVE')
      ;c_file=fln+'.idl'  &  c_download_dir=path
      ;papco_get_remote_data, c_url, c_file, c_download_dir
      ;check if file "arrived" and then read it
      ;if get_err_no eq 0 then begin
        ;r_cluster_eph_onefile, path+fln, VERBOSE=VERBOSE
      ;endif else begin
        ;get_err_msg='No data locally, remote get failed!C  ' + get_err_msg
        ;return
      ;endelse  
    ;endif   
     
    ;if a file was succesfully read, concatonate!
    if get_err_no eq 0 then begin 
      if no_files_read eq 0 then begin   ;start the array buffer
        input_data_buf=temporary(input_data)
      endif else begin                   ;otherwise concatonate arrays
        input_data_buf=[(input_data_buf),(input_data)]
        ;input_data_buf=[temporary(input_data_buf),temporary(input_data)]
      endelse        
      input_data=0                      ;clear array for reuse
      no_files_read=no_files_read+1
    endif

  endfor

; check if any files were read   
  if (no_files_read ne 0) then begin
    get_err_no=0
    get_err_msg=string(no_files_read,format="(i2)")+ ' files read succesfully'
    message, get_err_msg, /cont
    ;clear arrays and load buffer into output array
    input_data=0  &  input_data=temporary(input_data_buf)
  endif else begin
    get_err_no=1
    get_err_msg='No files found for timerange'
    return
  endelse      

end

;******************************************************************************
;*  NAME:        r_cluster_eph_onefile,fln
;*
;*  DESCRIPTION: reads one file of type plot_file. Assumes data has been
;*               stored in IDL saveset using structures input_header and
;*               input_data
;*
;*  INPUT:       fln  - string of filename
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        October 1997
;*
;*  AUTOR:       R. Friedel
;******************************************************************************
pro r_cluster_eph_onefile, fln, VERBOSE=VERBOSE
  
  common cluster_eph_data, input_header, input_data
  common get_error, get_err_no, get_err_msg
  COMMON cluster_eph_filename, filename 
 
  get_err_no=0  &  get_err_msg='cluster_eph onefile read succesfully'
   
; check env. variable pointing to data directory
  if not (papco_check_data_env('CLUSTER_EPH_SETS', PATH=path)) then return   

; clear data arrays for reuse
  input_header=0  &  input_data=0
   
  on_ioerror, read_original
  filename = path+fln+'_cluster_eph.idl'
  restore,filename, VERBOSE=VERBOSE
  dealloc_lun,/quiet   ;frees logical units used by save/restore
  message,'idl saveset read succesfully: ' + filename, /cont
  return

  read_original:       ;if the .idl file is not found, try to read the original
  r_cluster_eph_onefile_cdf, fln, VERBOSE=VERBOSE

  ;r_cluster_eph_onefile_original, fln, VERBOSE=VERBOSE
      
end   

;******************************************************************************
PRO s_cluster_eph_onefile, fln, VERBOSE=VERBOSE
  
  common cluster_eph_data, input_header, input_data
  common get_error, get_err_no, get_err_msg
  COMMON cluster_eph_filename, filename 

  get_err_no=0  &  get_err_msg='cluster_eph onefile read succesfully'
   
; check env. variable pointing to data directory
  if not (papco_check_data_env('CLUSTER_EPH_SETS', PATH=path)) then return   

  IF keyword_set(fln) THEN filename = path+fln+'_cluster_eph.idl'
  
  IF not keyword_set(filename) THEN BEGIN
      get_err_no=6  &  get_err_msg='cannot save, no filename specified'
      message, get_err_msg, /cont
      return
  END 

  save, file = filename, input_header, input_data, /COMPRESS, VERBOSE=VERBOSE

end
  
;******************************************************************************
;*  NAME:        r_cluster_eph_onefile_original,fln
;*
;*  DESCRIPTION: reads one file of type plot_file_original. Under the
;*               papco read philosophy, files should be read the first
;*               time in whatever format they might have - ascii,
;*               fortran binary, pascal binary, etc. The files might also
;*               be in some compressed format, so they need to be uncompressed
;*               first. 
;*               
;*               You might also have to change time formats. 
;*               A lot of files are stored per day, and have seconds of
;*               that day as the time. This is what is assumed here and
;*               is papco standard. You need to convert the time
;*               to papco standard if you intend to use the plotroutines 
;*               supplied with papco (use the routines in
;*               $papco_BASE/papco_Lib/time_convert.pro). 
;*               If you use your own plot routines this step is not needed.
;*               
;*               The first read might be slow, esp if the format is ascii.
;*               Use the IDL "save" command to make a binary copy of the
;*               file which can then be read quickly using "restore". The
;*               next time only the IDL saveset is read, which is much
;*               faster.
;*               
;*               In order to be able to concatonate the arrays easily,
;*               it makes sense to have the data in an array of time-ordered
;*               structures, which makes adding together days of data
;*               trivial using the IDL concatonation operator "[]".
;*               
;*  INPUT:       fln  - string of filename
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        October 1997
;*
;*  AUTOR:       R. Friedel
;******************************************************************************
pro r_cluster_eph_onefile_original, datestr, VERBOSE=VERBOSE,  $
                                    INTMOD=INTMOD, EXTMOD=EXTMOD, PITCH=PITCH

  common cluster_eph_data, input_header, input_data
  common get_error, get_err_no, get_err_msg  
   
  get_err_no=0  &  get_err_msg='cluster_eph_onefile_orignal read succesfully'
   
; check env. variable pointing to data directory
  if not (papco_check_data_env('CLUSTER_EPH_DATA', PATH=path)) then return

; check keywords and set
  if keyword_set(VERBOSE) then verbose=VERBOSE else verbose=0     
  if keyword_set(INTMOD)  then intmod=INTMOD   else intmod=0
  if keyword_set(EXTMOD)  then extmod=EXTMOD   else extmod=5
  if keyword_set(PITCH)   then pitch=PITCH     else pitch=90  

; clear data arrays for reuse
  input_header=0  &  input_data=0   
   
;------------------------------------------------------------------------------
; read the original data file - THIS NEEDS TO BE REWRITTEN BY USER!!!!
  
; define the header and data structures. Edit to suit your data!
  max_no_rec=144              ;the maximum number time records 1 every 10 min
  input_header={cluster_eph_header,tag1:0,tag2:0}

  new_struct_name='MODEL_'+varprt(intmod)+'_'+varprt(extmod)
  new_struct=create_struct(name='UNILB', 'L',0.0,'MLT',0.0,'MLAT',0.0)    

  eph = {x_gse:0.0d,  y_gse:0.0d,  z_gse:0.0d}
  eph = create_struct(name='cluster_eph', eph,new_struct_name,new_struct)

  dummy={cluster_eph_data, tai:0.0d, $
         clust1:eph, clust2:eph, clust3:eph, clust4:eph, $
         revnum:0.0d}
   
  input_data=replicate(dummy, max_no_rec)

;set up the f77 external library routine to be used  
  lib_dir = '/n/toaster/u/friedel/idl/fortran/cluster/'
  lib_name = lib_dir+'r_cluster_orbit.so'
  entry_name = 'r_cluster_orbit_'

;set date
  JEAR = 2000l+long(strmid(datestr, 0, 2))
  MONTH =      long(strmid(datestr, 2, 2))
  KDAY =       long(strmid(datestr, 4, 2))
  
;initialize variables for fortran call
  IERROR = 0l &  NSAT = 0l &  REVNUM = 0.0d  &  X = dblarr(3)
  
;loop through all 4 Cluster spacecraft
  FOR SC = 1, 4 DO BEGIN
    dir = 'cluster'+varprt(SC)+'/aux_'+varprt(SC)+'/'
 
  ;find how many continuation files there are for this day
    files = findfile(path+dir+datestr+'*ba*',  count = c)
    IF c EQ 0 THEN BEGIN
      get_err_no = 1  &  msg = 'No files found' 
      message, msg, /cont
      return
    ENDIF
 
  ;loop through all files
    FOR ff = 0, c-1 DO BEGIN
      
      FLN = files(ff)  &  print, FLN
  
    ;call fortran routine to evaluate position of spacecraft.
      FOR j = 0l, max_no_rec-1 DO begin

      ;set time
        JHR = long((j * 10l) / 60)
        MI  = (j * 10l) MOD 60
        SEC = 0.0d

      ;calculate correct time in TAI (only do once)
        IF (sc EQ 1) AND (ff EQ 0) THEN $
          input_data(j).tai=UTC2TAI({YEAR:JEAR, MONTH:MONTH, DAY:KDAY, $
                        HOUR:JHR, MINUTE:MI, SECOND:SEC, MILLISECOND:0})

      ;call external fortran routine
        result = call_external(lib_name, $         ;The shareable object file
                 entry_name, $                     ;The entry point
                 FLN, $                            ;filename, string, input   
                 JEAR,MONTH,KDAY,JHR,MI,SEC,  $    ;time, input
                 X, $                              ;pos, output
                 NSAT, REVNUM, IERROR, $           ;more output
                 /D_VALUE)                                 
        
        IF IERROR EQ 0 THEN BEGIN
          print, JHR, MI, IERROR, NSAT, REVNUM
        ;fold into inout data structure
          input_data(j).(NSAT).x_gse = X(0)
          input_data(j).(NSAT).y_gse = X(1)
          input_data(j).(NSAT).z_gse = X(2)
          input_data(j).revnum = REVNUM
        ENDIF
      
      ENDFOR
    ENDFOR
  ENDFOR

; end of user written section
;------------------------------------------------------------------------------
  
; check env. variable pointing to cdf data directory
  if not (papco_check_data_env('CLUSTER_EPH_DATA', PATH=path)) then return   

; save read data with correct time. Rename data to *.idl
  save,filename=path+datestr+'_cluster_eph.idl', $
       input_header,input_data, VERBOSE=VERBOSE 
  dealloc_lun,/quiet              ;frees logical units used by save/restore
  return
  
  out:
  ; if there was an error, report it. 
  get_err_no=!ERROR
  get_err_msg='!C '+!ERR_STRING
   
end

;******************************************************************************
PRO r_cluster_eph_onefile_cdf, fln, VERBOSE=VERBOSE

  common cluster_eph_data, input_header, input_data
  common get_error, get_err_no, get_err_msg  
   
  get_err_no=0  &  get_err_msg='cluster_eph_onefile_orignal read succesfully'
   
; check env. variable pointing to data directory
  if not (papco_check_data_env('CLUSTER_EPH_CDF', PATH=path)) then return

; clear data arrays for reuse
  input_header=0  &  input_data=0   
  
  input_header={clust_eph_header, NPOINTS :0l}


;------------------------------------------------------------------------------
; read the original data file - THIS NEEDS TO BE REWRITTEN BY USER!!!!

  ;open cdf file for that month.
  root = 'cl_jp_pgp_20' ;xyz coords for spacecraft
  fname = root+fln+'*'
  files = findfile(path+fname, count = c)
  IF c EQ 0 then begin
    root = 'CL_JP_PGP_20' ;xyz coords for spacecraft
    fname = root+fln+'*'
    files = findfile(path+fname, count = c)
  ENDIF
  
  IF c NE 0 THEN begin
    message,'Opening to make data structure',  /cont  
    print, '  '+  files(0)
    id=CDF_OPEN(files(0))
    ;First establish if time variable exists in file, then get it
    zvars=1  &  papco_cdf_cracker, id, ZVARIABLES=zvars, VERBOSE=VERBOSE
    pos=strpos(zvars(*,0),'Epoch')  &  index=where(pos ne -1,c)
    if c eq 0 then BEGIN
      get_err_no=2  &  get_err_msg='No time epoch found in cdf'
      message,get_err_msg, /cont  &  return
    endif else begin   ;read time array
      varname=zvars(index(0),0)  &  var_attrib=1  ;get variable's attributes
      papco_cdf_cracker, id, GET_VAR_ATTRIB=var_attrib, $
        VARNAME=varname, VERBOSE=VERBOSE     
      cdf_control, id, variable=varname, /zvariable, get_var_info=var_info
      var_inq=cdf_varinq(id, varname, /zvariable)
      cdf_varget,id,varname, epoch, rec_count=var_info.MAXREC 
      epoch=transpose(epoch)  
      input_header.NPOINTS = var_info.MAXREC
    endelse
    
    ;first make a list of the zvars to get
    index1 = [3, 4, 8, 9, 10, 11, 24, 25, 26, 27]
    varnames1 = zvars(index1, 0)
    
    ;now make the structure that is going to hold all data. 
    dummy = {epoch:epoch(0)}
    FOR i = 0, n_elements(varnames1)-1 DO BEGIN
      cdf_varget, id, varnames1(i), data, rec_count=1
      struct_name = str_sep(varnames1(i), '__')
      dummy = create_struct(dummy, struct_name(0), data)
    ENDFOR
    n_tags = n_elements(index1)
    cdf_close, id

    ;now add into structure mag coord data for each spacecraft.
    FOR i = 1, 4 DO BEGIN
      root = 'c'+varprt(i)+'_'+'jp_pmp_20'
      fname = root+fln+'*'
      files = findfile(path+fname, count = c)
      IF c eq 0 then begin
        root = 'C'+varprt(i)+'_'+'JP_PMP_20'
        fname = root+fln+'*'
        files = findfile(path+fname, count = c)
      endif
      IF c NE 0 THEN begin
        message,'Opening to make data structure',  /cont  
        print, '  '+  files(0) 
        id=CDF_OPEN(files(0))
        ;First establish if time variable exists in file, then get it
        zvars=1  &  papco_cdf_cracker, id, ZVARIABLES=zvars, VERBOSE = VERBOSE
        index = [3, 4, 5, 6]
        varnames = zvars(index, 0)
        FOR j = 0, n_elements(varnames)-1 DO BEGIN
          cdf_varget, id, varnames(j), data, rec_count=1
          struct_name = str_sep(varnames(j), '__')
          dummy = create_struct(dummy, 'c'+varprt(i)+'_'+struct_name(0), data)
        ENDFOR
        n_tags = n_tags+n_elements(index)
        cdf_close, id
      ENDIF
    ENDFOR

    ;give the data structure a name
    dummy= create_struct(NAME='CLUSTER_EPH_'+varprt(n_tags), dummy)

    ;now read in the actual data
    input_data = replicate(dummy, n_elements(epoch) )
    input_data.epoch = epoch  & tag_n = 1
    root = 'cl_jp_pgp_20' ;xyz coords for spacecraft
    ;open cdf file for that month.
    fname = root+fln+'*'
    files = findfile(path+fname, count = c)
    IF c EQ 0 then begin
       root = 'CL_JP_PGP_20' ;xyz coords for spacecraft
       fname = root+fln+'*'
       files = findfile(path+fname, count = c)
    ENDIF
    IF c NE 0 THEN begin
      message,'Reading from ',/cont  
      print, '  '+  files(0) 
      id=CDF_OPEN(files(0))  
      FOR i =  0, n_elements(varnames1)-1 DO BEGIN
        cdf_varget,id,varnames1(i), junk, rec_count=var_info.MAXREC 
        result = size(junk, /DIM)
        IF result(0) EQ 1 THEN junk = transpose(junk)
        input_data.(tag_n) = junk
        tag_n = tag_n+1
      ENDFOR
      cdf_close, id
    ENDIF
    
    FOR i = 1, 4 DO BEGIN
      root = 'c'+varprt(i)+'_'+'jp_pmp_20'
      fname = root+fln+'*'
      files = findfile(path+fname, count = c)
      IF c eq 0 then begin
         root = 'C'+varprt(i)+'_'+'JP_PMP_20'
         fname = root+fln+'*'
         files = findfile(path+fname, count = c)
      ENDIF
      IF c NE 0 THEN begin
        message,'Reading ',/cont
        print, '  '+  files(0) 
        id=CDF_OPEN(files(0))
        zvars=1  &  papco_cdf_cracker, id, ZVARIABLES=zvars, VERBOSE = VERBOSE
        index = [3, 4, 5, 6]
        varnames = zvars(index, 0)
        FOR j =  0, n_elements(varnames)-1 DO BEGIN
          cdf_varget,id,varnames(j), junk, rec_count=var_info.MAXREC 
          result = size(junk, /DIM)
          IF result(0) EQ 1 THEN junk = transpose(junk)
          input_data.(tag_n) = junk
          tag_n = tag_n+1
        ENDFOR
        cdf_close, id
      ENDIF        
    ENDFOR

  ENDIF ELSE BEGIN ;no data files found
    get_err_no=3     
    msg = 'No ephemeris cdf files found for '+ fln
    get_err_msg='!C '+msg
    message, msg, /cont
    return
  ENDELSE

; end of user written section
;------------------------------------------------------------------------------

; check env. variable pointing to data directory
  if not (papco_check_data_env('CLUSTER_EPH_SETS', PATH=path)) then return

; save read data with correct time. Rename data to *.idl
  message, 'Saving data as IDL savset', /cont
  fname = path+fln+'_cluster_eph.idl'
  print, '  ', fname
  save,filename=fname, input_header,input_data, /COMPRESS, VERBOSE=VERBOSE
  dealloc_lun,/quiet              ;frees logical units used by save/restore
  return

  out:
  ; if there was an error, report it. 
  get_err_no=!ERROR
  get_err_msg='!C '+!ERR_STRING
   
end


;******************************************************************************
;* reads cluster eph files and "makes" files needed for ccr eph display
;******************************************************************************
PRO m_cluster_ccr_ephemeris, startday, endday

COMMON cluster_eph_data, input_header, input_data
COMMON mjdt, mjdt_start, mjdt_end
COMMON get_error, get_err_no, get_err_msg

papco, /set_only

path = '/n/avatar/belly3/www/htdocs/ccr/SummaryPlots/JsEphemerisCluster/'

result = convert_timeToSeconds(startday, /t90, /mjdt)
t1 = result.value
result = convert_timeToSeconds(endday, /t90, /mjdt)
t2 = result.value
mjdt_start = t1 & mjdt_end = mjdt_start

FOR j = t1.mjd, t2.mjd DO BEGIN
    mjdt_start.mjd = j
    mjdt_end.mjd = j+1
    convert_t90_to_date, mjdt_start, year, day_oy, hour, minute, second,$
                       month, cmonth, day_om, /mjdt  
    
    fln = string(year, month, day_om, $
                 format = "('JsEphemerisCluster',i4.4,i2.2,i2.2,'.js')")
    date_str = string(year, month, day_om, format = "(i4.4,i2.2,i2.2)")
    message, fln, /cont

    ;read data and extract one day of 1440 values, one per minute
    r_cluster_eph
    
    ;set the begin end end times. CLUSTER cdf time is in cdf epoch. Plot
    ;always starting at zero time = xut1 for reasonable numbers.
    convert_t90_to_date, mjdt_start, yr,day_oy,hr,min,sec,mo,cmo,day_om, /mjdt
    cdf_epoch,xut1,yr,mo,day_om,hr,min,sec,0.0,/COMPUTE_EPOCH
    convert_t90_to_date, mjdt_end, yr,day_oy,hr,min,sec,mo,cmo,day_om, /mjdt
    cdf_epoch,xut2,yr,mo,day_om,hr,min,sec,0.0,/COMPUTE_EPOCH

    ;restrict data to actual time range requested 
    index=where((input_data.epoch ge xut1) and (input_data.epoch lt xut2),cc)
    if cc ne 0 then data=input_data(index)
    ndat=n_elements(data)
    IF ndat NE 288 THEN $
      FOR i = ndat, 288 DO data = [data, data(ndat-1)]

    time = data.epoch & xut2=xut2-xut1  &  time=time-xut1  &  xut1=0
    ;cdf epoch is in milliseconds. Need to divide by to get seconds. 
    time=time/1000.0 & xut1=xut1/1000.0 & xut2=xut2/1000.0

    ;position of centroid
    ccX = data.SC_R_XYZ_GSE(0) / 6371.0
    ccY = data.SC_R_XYZ_GSE(1) / 6371.0
    ccZ = data.SC_R_XYZ_GSE(2) / 6371.0

    ;mag coords of centroid
    ccL = data.CC_MODEL_0_5.L
    bad_data=data.CC_MODEL_0_5.L
    index = where(bad_data GT 0, c)
    IF c NE 0 THEN BEGIN
       bad_data(index) = 0
       bad_data = bad_data
       index = where(bad_data NE 0, c)
       IF c NE 0 THEN ccL(index) = -1
    endif  
    index = where(ccL GT 999.99, c)
    IF c NE 0 THEN ccL(index) = -1 
    
    ccMLT = data.CC_MODEL_0_5.MLT
    ccMLAT = data.CC_MODEL_0_5.MLAT

    ;open and write to file
    openw, u, path+fln,  /get_lun
    
    printf, u, 'ccX_'+date_str+' = new Array('
    FOR i = 0, 260, 20 DO BEGIN
      printf,u, ccX(i:i+19), format = "(20(f6.2,','))"
    ENDFOR
    printf,u, ccX(i:i+7), format = "(7(f6.2,','),f6.2,');')"
    
    printf, u, 'ccY_'+date_str+' = new Array('
    FOR i = 0, 260, 20 DO BEGIN
      printf,u, ccY(i:i+19), format = "(20(f6.2,','))"
    ENDFOR
    printf,u, ccY(i:i+7), format = "(7(f6.2,','),f6.2,');')"
    
    printf, u, 'ccZ_'+date_str+' = new Array('
    FOR i = 0, 260, 20 DO BEGIN
      printf,u, ccZ(i:i+19), format = "(20(f6.2,','))"
    ENDFOR
    printf,u, ccZ(i:i+7), format = "(7(f6.2,','),f6.2,');')"
    
    printf, u, 'ccL_'+date_str+' = new Array('
    FOR i = 0, 260, 20 DO BEGIN
      printf,u, ccL(i:i+19), format = "(20(f6.2,','))"
    ENDFOR
    printf,u, ccL(i:i+7), format = "(7(f6.2,','),f6.2,');')"
    
    printf, u, 'ccMLT_'+date_str+' = new Array('
    FOR i = 0, 260, 20 DO BEGIN
      printf,u, ccMLT(i:i+19), format = "(20(f5.2,','))"
    ENDFOR
    printf,u, ccMLT(i:i+7), format = "(7(f5.2,','),f5.2,');')"

    printf, u, 'ccMLAT_'+date_str+' = new Array('
    FOR i = 0, 260, 20 DO BEGIN
      printf,u, ccMLAT(i:i+19), format = "(20(f6.2,','))"
    ENDFOR
    printf,u, ccMLAT(i:i+7), format = "(7(f6.2,','),f6.2,');')"
    close, u & free_lun, u

ENDFOR

END

;------------------------------------------------------------------------------
; Take exisiting Cluster Ephemeris files and add new coordinates to
; them, such as GSM, Mag coords according to various models
; MJDT = use common block for time
PRO m_cluster_eph, SAT = SAT, VERBOSE=VERBOSE, FORCE=FORCE, $
                   DATE_1=DATE_1, DATE_2=DATE_2, $
                   INTMOD=INTMOD, EXTMOD=EXTMOD, $
                   MJDT = MJDT, NO_ORIG = NO_ORIG
  
COMMON cluster_eph_data, cluster_eph_header, cluster_eph_data
COMMON mjdt, mjdt_start, mjdt_end
COMMON get_error, get_err_no, get_err_msg

papco, /set_only
    
; check keywords and set defaults
IF keyword_set(SAT) THEN sat = string(sat) ELSE sat = 'CC'
IF keyword_set(VERBOSE) then verbose=VERBOSE else verbose=0     
IF keyword_set(INTMOD) then intmod=INTMOD else intmod=0
IF keyword_set(EXTMOD) then extmod=EXTMOD else extmod=5
IF keyword_set(FORCE) then force = FORCE ELSE force = 0
PITCH=90
  
new_struct_name=sat+'_'+'MODEL_'+varprt(intmod)+'_'+varprt(extmod)
new_struct=create_struct(name='UNILB', 'L',0.0,'MLT',0.0,'MLAT',0.0)  
    
IF NOT keyword_set(MJDT) THEN BEGIN 
    ;set dates to process. default is Cluster period to present
    if keyword_set(DATE_1) then date_1=DATE_1 else date_1='20010201'
    day=fix(strmid(date_1,6,2)) & month=fix(strmid(date_1,4,2)) 
    year=fix(strmid(date_1,0,4))
    mjdt_1=convert_date_to_t90(day=day, month=month, year=year, /MJDT)
    mjdt_start=mjdt_1
    if keyword_set(DATE_2) then BEGIN 
        day=fix(strmid(date_2,6,2)) & month=fix(strmid(date_2,4,2))
        year=fix(strmid(date_2,0,4))
    endiF ELSE BEGIN
        today = systime(/julian)   
        caldat, today, month, day, year
    ENDELSE 
    mjdt_2=convert_date_to_t90(day=day, month=month, year=year, /MJDT)
    mjdt_end=mjdt_2
ENDIF 
keep_mjdt_start = mjdt_start & keep_mjdt_end = mjdt_end
  
;loop through period in 10 day intervals (files are per month)
for i=keep_mjdt_start.mjd, keep_mjdt_end.mjd, 28 do begin
    
    mjdt_start.mjd=i & mjdt_start.t = 0
    mjdt_end.mjd=i+1 & mjdt_end.t = 0
    ;convert and print date
    convert_t90_to_Date, mjdt_start, year, day_oy, hour, minute, second, $
      month, cmonth, day_om, /MJDT
    print,''
    message,'doing for ' + string(year, month, day_om, $
                                  format = "(i4.4,i2.2,i2.2)"), /cont

    ;read ephemeris file
    r_cluster_eph, VERBOSE=VERBOSE

    IF (get_err_no eq 0) then begin 
 
      ndat=n_elements(cluster_eph_data) & need_to_do=0
  
      ;check for existing tags, work out what needs to be done
      tags=tag_names(cluster_eph_data(0))

      ; check if required mag coords exist in file.
      index=where(tags eq new_struct_name,c)
      if c eq 0 then begin
         need_to_do=1
         message, 'Adding new tag ' +new_struct_name, /cont
         dummy2=create_struct( $
                name='CLUSTER_EPH_'+varprt(n_elements(tags)+1),$
                cluster_eph_data(0),new_struct_name,new_struct)
         dummy3=replicate(dummy2,ndat)
         for j=0,ndat-1 do for m=0,n_elements(tags)-1 do $
           dummy3(j).(m)=cluster_eph_data(j).(m) 
         input_data=temporary(dummy3)
      endif    
      if keyword_set(FORCE) then need_to_do=1
    
      if not need_to_do then begin
        message,'Mag coords exist',/cont & continue
      endif  
    
      ;get the index of the new mag coords tag
      tags=tag_names(input_data(0)) 
      index=where(tags eq new_struct_name) & new_idx=index(0)    

      ;calculate L-value for satellite
      message,'Calculating mag coords...(slow)',/cont

      FOR k=0,ndat-1 DO BEGIN
      
        ;remember time from cluster is in cdf epoch!!!
        CDF_EPOCH, cluster_eph_data(k).epoch, $
          Year1, Month1, Day1, Hour1, Minute1, Second1, /BREAKDOWN_EPOCH
      
        time_date={iyear:long(year1), imonth:long(month1), iday:long(day1), $
                   ihour:long(hour1), imin:long(minute1), $
                   secs:double(second1), amjd:0.0d }

        ;cluster position is in GSE coords, km
        kcompstr = 'CAR' & kinstr = 'GSE' 
        ;set pos to centroid (default)
        pos = {coorda:input_data(k).SC_R_XYZ_GSE(0), $
               coordb:input_data(k).SC_R_XYZ_GSE(1), $
               coordc:input_data(k).SC_R_XYZ_GSE(2) }

        CASE sat OF
            'CC': BEGIN & END
            'C1': BEGIN
                pos.coorda = pos.coorda + input_data(k).SC_DR1_XYZ_GSE(0)
                pos.coordb = pos.coordb + input_data(k).SC_DR1_XYZ_GSE(1)
                pos.coordc = pos.coordc + input_data(k).SC_DR1_XYZ_GSE(2)
            END
            'C2': BEGIN
                pos.coorda = pos.coorda + input_data(k).SC_DR2_XYZ_GSE(0)
                pos.coordb = pos.coordb + input_data(k).SC_DR2_XYZ_GSE(1)
                pos.coordc = pos.coordc + input_data(k).SC_DR2_XYZ_GSE(2)
                
            END
            'C3': BEGIN
                pos.coorda = pos.coorda + input_data(k).SC_DR3_XYZ_GSE(0)
                pos.coordb = pos.coordb + input_data(k).SC_DR3_XYZ_GSE(1)
                pos.coordc = pos.coordc + input_data(k).SC_DR3_XYZ_GSE(2)
            END
            'C4': BEGIN
                pos.coorda = pos.coorda + input_data(k).SC_DR4_XYZ_GSE(0)
                pos.coordb = pos.coordb + input_data(k).SC_DR4_XYZ_GSE(1)
                pos.coordc = pos.coordc + input_data(k).SC_DR4_XYZ_GSE(2)
            END
        ENDCASE

        npos = fltarr(3)
        npos(0, 0) = pos.coorda
        npos(1, 0) = pos.coordb
        npos(2, 0) = pos.coordc

        ;calculate mag coords (from UNILIB) 
        get_mag_unilib, time_date, npos, kcompstr, kinstr, $
          bvec, Lval, mlt, ilat, error, $
          INTMOD=INTMOD, EXTMOD=EXTMOD, INT_NAME=INT_NAME, EXT_NAME=EXT_NAME,$ 
          /HAVE_MDATE
       
        input_data(k).(new_idx).L=Lval
        input_data(k).(new_idx).MLT=mlt
        input_data(k).(new_idx).MLAT=ilat

      endfor         

      ;set up input_header structure
      new_tag_name=INT_NAME+'_'+EXT_NAME    
      input_header=create_struct($
            name='CLUSTER_EPH_HEAD_'+varprt(n_elements(tags)), $
            cluster_eph_header, new_struct_name,new_tag_name)

      ;save ephemeris file
      cluster_eph_header =  temporary(input_header)
      cluster_eph_data =  temporary(input_data)
      s_cluster_eph_onefile

    endif

ENDFOR   
 
mjdt_start = keep_mjdt_start & mjdt_end = keep_mjdt_end
    
END   

;------------------------------------------------------------------------------
