;******************************************************************************
;* file p_cluster_peace_line.pro
;*
;* this file is a template for a plot routine which can be used by papco.
;* it includes all the periphery needed, the user simply has to add in
;* the actual plot statement for his/her kind of data.
;*
;* This is a working example for a pre-defined example data array.
;******************************************************************************
;******************************************************************************
;* PROCEDURE:     
;*      p_cluster_peace_line, panel, type, OUTPUT=OUTPUT, $
;*           OVERPLOT=OVERPLOT,$
;*           PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           SUBTABLE=SUBTABLE, $
;*           _EXTRA=extra_par
;* 
;* DESCRIPTION:  
;*	This procedure plots data of type data_type for inclusion in papco.  
;*      Data must have been read and placed into common blocks before this
;*      routine is called. Behavior is controlled by the vector panel
;*	which controls positioning of the plot (see papco_conventions.text
;*      in $papco). In general, the following plot conventions are used:
;*              - if the panel is at the bottom plot time axis plus any
;*                additional axis (ie ephemeris) needed.
;*              - if panel is at the top also plot a title
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a papco_colorbar.
;*              - panels all have common time
;*              - y-scaling (or z-scaling for color plots) is either
;*              - automatic or manualy set using common yscale or zscale   
;*
;* INPUTS:       
;*      panel   three element integer vector. Controls positioning of the plot
;*              (see papco_conventions.text in $papco)
;*      type    optional parameter for options. Normally is a four element
;*              integer vector. This is normally equivalent to the field
;*              typeVector of the papco structure papco_PLOTINFO which is set
;*              by the panel editor. Additional fields that can be used if
;*              needed are papco_plotinfo.ioptions (20 element vector) and 
;*              papco_PLOTINFO.channel (integer). If your plot routine needs
;*              more input fields than these you have to extend the definition
;*              of the structure papco_PLOTINFO
;*              (procedure papco_getPlotInfoStruct in $papco/papco.pro). Try to
;*              avoid this!
;*              
;* OUTPUTS: 
;*      none
;*
;* KEYWORDS:
;*      OUTPUT  if set call the papco routine that outputs plot data
;*              to a file.
;*    OVERPLOT  if you support this you can alter the appearance of your
;*              plot if it is an overplot to an existing panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mag. field.
;*              see p_crres_mag for an example.
;*    PLOTS_ATTHISPOSITION  used to control the left side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and line style.   
;* 
;* CALLING SEQUENCE:
;*       p_cluster_peace_line, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY:       
;*     written December 1995, Reiner Friedel, at Goddard
;*     added OUTPUT support, July 1998, , Reiner Friedel
;******************************************************************************
pro p_cluster_peace_line, panel, plotinfo, OUTPUT=OUTPUT, $
                      OVERPLOT=OVERPLOT, $
                      PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION, $
                      SUBTABLE=SUBTABLE, $
                      _EXTRA=extra_par

; The following common blocks are needed:
COMMON mjdt, mjdt_start, mjdt_end               ;time limit in mjdt
COMMON yscale, yscl                             ;man/auto yscaling
COMMON shift_label, down                        ;for x-axis label shifting
COMMON coordinateSystems, plotted_x, plotted_y  ;info on coords
COMMON papco_color_names

colors = [red, blue, dark_green]

; One of the common blocks should contain the plot data as returned by the
; data read procedure r_cluster_peace. 
COMMON cluster_peace_data, input_data
  
panelset,panel			;sets the panel position viewport

;time is in CDF Epoch. Need to convert PAPCO start/end to that EPOCH.
pos = strpos(tag_names(input_data), 'EPOCH')
idx = where(pos NE -1)
epoch_time = input_data.(idx(0)).dat

;get data array
cdfvar_name = strupcase(plotinfo.soptions(3))
pos = strpos(tag_names(input_data),cdfvar_name)
idx = where(pos NE -1)
data = input_data.(idx(0)).dat

;get dimension of data
data_info = size(data, /struct)

convert_t90_to_Date, mjdt_start, year,doy,hr,min,sec,mon,cmon,dom, /mjdt 
CDF_EPOCH, epoch_1, Year, mon, dom, hr, min, sec, /COMPUTE_EPOCH
convert_t90_to_Date, mjdt_end, year,doy,hr,min,sec,mon,cmon,dom, /mjdt 
CDF_EPOCH, epoch_2, Year, mon, dom, hr, min, sec, /COMPUTE_EPOCH

; restrict data in time  
index=where((epoch_time GE epoch_1) AND (epoch_time LE epoch_2 ),c)
if c ne 0 then BEGIN
    IF data_info.N_DIMENSIONS EQ 1 THEN data = data(index) ELSE $
      data = data(*, index) 
    epoch_time = epoch_time(index) 
ENDIF ELSE GOTO, noplot
xut1=0  &  xut2=epoch_2-epoch_1  & time=epoch_time-epoch_1

nodata = input_data.(idx(0)).FILLVAL
uytitle = input_data.(idx(0)).LABLAXIS+' ('+input_data.(idx(0)).UNITS+')'
utitle = 'papco cdf!C'+plotinfo.soptions(1)+'!C'+cdfvar_name
yst = input_data.(idx(0)).SCALEMIN(0)
yen = input_data.(idx(0)).SCALEMAX(0)

; the variables used above have the following meaning:
; 	uytitle : the y-axis label
;	utitle  : the right-of plot label defining the data
;
; yscl of common block yscale is an array of dimensions (*,4) and contains info
; on the yscaling to be used. The first dimension identifies the panel, and the
; next dimension has three entries:  
;         yscl(*,0)  is  0 for "use automatic scaling", 1 for "use manual"
;         yscl(*,1)  is then ymin (or zmin)
;         yscl(*,2)  is then ymax (or zmax)
;         yscl(*,3)  is 0 for "linear" and 1 for "log"  scale
; when scaling is set to automatic, the automatic y-limits are loaded in.  
if (yscl(panel(0),0) eq 1) then begin
    yst=yscl(panel(0),1)  &  yen=yscl(panel(0),2)
endif else begin
    yscl(panel(0),1) = yst & yscl(panel(0),2) = yen
endelse
  
; use auto y-labeling routine to get "good" y-lables. This sets the
; global IDL graphics system variables !Y.MINOR, !Y.TICKV, !Y.TICKS -
; so if you want to use this feature, avoid setting these or their
; corresponding keywords in your plot.
papco_y_label, yst, yen, log=yscl(panel(0),3)

extra_plotPar={xrange:[xut1,xut2], yrange:[yst,yen],  $
               ylog:yscl(panel(0),3), xtickformat:'noticks'}
   
; add keyword structure set here with the one passed in
extra_plotPar=create_struct(extra_plotPar, extra_par)
   
; use papco routine to draw time axis. This checks for bottom plot
; and uses the user's xtickformat if it is something other than
; 'noticks'.
; If you use this rotuine, make sure your data plot routine uses
; xstyle=5 and ystyle=5 (inhibit axis) so that you don't clobber the
; axes drawn by papco.  
; "down" sets the axis labeling level. 
down=0
papco_draw_time_axis, panel, OVERPLOT=OVERPLOT, _extra=extra_plotPar    
  
; You might have to add your own xtickformat for time axis
; labeling. If you stick to the papco convention of always having your plot 
; time array having consecutive seconds from midnight of the start day, then 
; you can use the pre-defined xtickformat functions defined in ticks.pro in
; papco_Lib/axis_label.
; You can at this point also define a custom ytickformat, or leave it out for
; IDL default plot behaviour.

; now do the actual data plot, data only, no axis

;case of single line plot
IF data_info.N_DIMENSIONS EQ 1 THEN BEGIN

    papco_gap_plot, time, data, nodata, $
              xstyle=5, ystyle=5, _extra=extra_plotPar

ENDIF ELSE BEGIN     

    coords = plotinfo.ioptions(1) & set = bytarr(3)
    FOR i = 0, 2 DO IF (coords and 2ll^i) eq 2ll^i THEN set(i)=1 else set(i)=0
    idx2 = where(set EQ 1, n_ch) 
    IF n_ch EQ 0 THEN GOTO, noplot
    in_label = input_data.(idx(0)).LABL_PTR_1(idx2)
    for i = 0, n_ch-1 DO BEGIN
        yray = data(idx2(i), *)
        IF n_ch NE 1 then extra_plotPar.color = colors(i)
        papco_gap_plot, time, yray, nodata, $
              xstyle=5, ystyle=5, _extra=extra_plotPar
    ENDFOR 
    ; plot channel info inside of panel.
    papco_in_side_label, panel, in_label, NCOL = colors, _extra=extra_plotPar
 
ENDELSE

; plot y-axis label at left of plot. Use scalable routine!  
left_side_label, panel, uytitle,/rot90

; store the coordinate information into common block for slices
plotted_x = !x  &  plotted_y = !y

; plot info right of the panel. If you want overplots to have their own labels
; you need to add position-control for this label using the keyword
; PLOTS_ATTHISPOSITION

if not keyword_set(overplot) then begin
    right_side_label,panel,utitle,_extra=extra_Par
endif else begin
    for i=0, PLOTS_ATTHISPOSITION-1 do utitle='!C'+utitle
    right_side_label,panel,utitle,_extra=extra_Par
endelse
   
; check if data needs to be output to file. Use the optional keywords
; to add more information to the data file
;      DESCRIPTION   a string or string array describing the data
;      TIME_LABEL    a string decribing the x-axis format
;      CHANNELS      an (ch,2) array giving the channel start/end
;      Y_LABEL       a string with the y-axis label
;      Z_LABEL       a string with the z-axis label
;      RS_LABEL      a string with the righ-side label of the plot
  
  if KEYWORD_SET(OUTPUT) then begin
    print,'% p_cluster_peace_line: Writing plot data out to file'
    description='cluster_peace template sample data - single energy channel line'
    time_label='Seconds since start of day'
    ;channels=yarr
    y_label=uytitle
    ;z_label=uztit
    rs_label=utitle
    papco_write_data, time, yray, $
                      DESCRIPTION=description, TIME_LABEL=time_label, $
                      Y_LABEL=y_label, $
                      RS_LABEL=RS_LABEL
  endif  

return

noplot: 
message,'No data to plot',/cont
plotted_x = !x  &  plotted_y = !y  

end 
