;******************************************************************************
;* PROCEDURE:     
;*      p_cluster_peace_spec, panel, plotinfo, OUTPUT=OUTPUT, $
;*           OVERPLOT=OVERPLOT,$
;*           PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           SUBTABLE=SUBTABLE, $
;*           _EXTRA=extra_par
;* 
;* DESCRIPTION:  
;*	This procedure plots data of type data_type for inclusion in papco.  
;*      Data must have been read and placed into common blocks before this
;*      routine is called. In general, the following plot conventions are used:
;*      	- if the panel is at the bottom plot time axis plus any
;*                additional axis (ie ephemeris) needed.
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a papco_colorbar
;*              - panels all have common time
;*              - y-scaling (and/or z-scaling for color plots) is either
;*              - automatic or manually set using common yscale or zscale    
;*
;* INPUTS:       
;*	panel	three element integer vector. Controls positioning of the plot
;*              (see papco_conventions.text in $papco)
;*      type    optional parameter for options. Normally is a four element
;*              integer vector. This is normally equivalent to the field
;*              typeVector of the papco structure PAPCD_PLOTINFO which is set
;*              by the panel editor. Additional fields that can be used if
;*              needed are papco_plotinfo.ioptions (20 element vector) and 
;*              PAPCO_PLOTINFO.channel (integer). If your plot routine needs
;*              more input fields than these you have to make use of
;*              an extra common block.
;*              
;* OUTPUTS: 
;*   	none
;*
;* KEYWORDS:
;*      OUTPUT  if set call the papco routine that outputs plot data
;*              to a file.
;*    OVERPLOT  if you support this you can alter the appearance of your
;*              plot if it is an overplot to an exisiting panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mag. field
;*              (see p_crres_mag for an example).
;* PLOTS_ATTHISPOSITION  used to control the left side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and line style.   
;* 
;* CALLING SEQUENCE:
;*       p_cluster_peace, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY:       
;*     written april 1996, Reiner Friedel
;*     added OUTPUT support, July 1998, , Reiner Friedel
;******************************************************************************
pro p_cluster_peace_spec, panel, plotinfo, OUTPUT=OUTPUT, $
                      OVERPLOT=OVERPLOT, $
                      PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
                      SUBTABLE=SUBTABLE, _EXTRA=extra_par  

; The following common blocks are needed for PAPCO:
COMMON mjdt, mjdt_start, mjdt_end     ;common time limit in mjdt
COMMON yscale, yscl			;man/auto yscaling
COMMON zscale, zscl			;man/auto zscaling
COMMON shift_label, down              ;common for x-axis label shifting
COMMON coordinateSystems, plotted_x, plotted_y  
                                        ;info on coords used by mouse

; One of the common blocks should contain the plot data as returned by the
; data read procedure r_cluster_peace. 
COMMON cluster_peace_data, input_data
COMMON cluster_peace_slice, time, zmat, ch_str, $
                            yarr, pa_rng, n_ek, n_pa, ch_idx, utitle 
COMMON cluster_peace

IF keyword_set(OUTPUT) THEN output = OUTPUT ELSE output = 0

var_names = tag_names(input_data)

;set channels for energy
n_ek=n_elements(input_data.CENTER_SCAN.DAT)
yarr=fltarr(n_ek,2)
yarr(*,0)=input_data.CENTER_SCAN.dat-input_data.BAND_LOW_SCAN.dat
yarr(*,1)=input_data.CENTER_SCAN.dat+input_data.BAND_HIGH_SCAN.dat

;set channels for pitch angle
pa_bin = 15
pa_rng = papco_pitch_bin(pa_bin)
n_pa = n_elements(pa_rng)/2

;get time array
pos = strpos(var_names, 'EPOCH') & idx = where(pos NE -1)
epoch_time = input_data.(idx(0)).dat
ndat = n_elements(epoch_time)

;get data array
CASE strupcase(plotinfo.soptions(1)) OF
    'FULL_HEEA':BEGIN 
        message, 'Assembling full HEEA pitch/spec data...', /cont
        pos = strpos(var_names, 'HEEA') & idx = where(pos NE -1, n_pa_bins)
        data = fltarr(ndat, n_ek, n_pa_bins)
        FOR j = 0, ndat-1 DO $
          FOR i=0,n_pa_bins-1 DO $
            data(j,*,i) = transpose(input_data.(idx(i)).dat(*,j))
    END
    'FULL_LEEA':BEGIN 
        message, 'Assembling full LEEA pitch/spec data...', /cont
        pos = strpos(var_names, 'LEEA') & idx = where(pos NE -1, n_pa_bins)
        data = fltarr(ndat, n_ek, n_pa_bins)
        FOR j = 0, ndat-1 DO $
          FOR i=0,n_pa_bins-1 DO $
            data(j,*,i) = transpose(input_data.(idx(i)).dat(*,j))
    END 
ENDCASE
nodata = input_data.(idx(0)).FILLVAL

;time is in CDF Epoch. Need to convert PAPCO start/end to that EPOCH.
convert_t90_to_Date, mjdt_start, year,doy,hr,min,sec,mon,cmon,dom, /mjdt 
CDF_EPOCH, epoch_1, Year, mon, dom, hr, min, sec, /COMPUTE_EPOCH
convert_t90_to_Date, mjdt_end, year,doy,hr,min,sec,mon,cmon,dom, /mjdt 
CDF_EPOCH, epoch_2, Year, mon, dom, hr, min, sec, /COMPUTE_EPOCH

; restrict data in time  
index=where((epoch_time GE epoch_1) AND (epoch_time LE epoch_2 ),c)
if c ne 0 then BEGIN
    data = data(index, *, *) & epoch_time = epoch_time(index)
ENDIF ELSE GOTO, noplot
xut1=0  &  xut2=(epoch_2-epoch_1)/1000.0  & time=(epoch_time-epoch_1)/1000.0
ndat = n_elements(time) & time = [[time],[time+4.0]]

;test which channels have ANY data
message, 'Finding valid energy channels...', /cont
ch_idx = bytarr(n_ek)
FOR i = 0, n_ek-1 DO begin
    test = where(data(*, i, *) NE nodata , c)
    IF c EQ 0 THEN ch_idx(i) = 0 ELSE ch_idx(i) = 1
ENDFOR
ch_idx = where(ch_idx EQ 1)
n_ek = n_elements(ch_idx)
data = data(*,ch_idx, *)
yarr = yarr(ch_idx, *)

ch_idx = where(yarr(*, 0) NE yarr(*, 1), c)
n_ek = n_elements(ch_idx)
data = data(*,ch_idx, *)
yarr = yarr(ch_idx, *)

;get variables from plotinfo into more descriptive variable names
;look for ek channels nearest to what is set by panel editor - 
;actual ek channel assignements can vary
sc_id = varprt(plotinfo.ioptions(0)+1)
cdf_type = plotinfo.typevector(0)
pa = plotinfo.ioptions(6)-1 > 1 < 13
pa2 = plotinfo.ioptions(8)-1 > 1 < 13
data_type = plotinfo.ioptions(5)
smooth = plotinfo.ioptions(10)
CASE cdf_type OF
    0: eks = ek_arr(*, 0:1)
    1: eks = ek_arr(*, 2:3)
ENDCASE
ek1_idx = (plotinfo.ioptions(7)-1) >  0 < 93
indx = where(yarr(*, 0) GE eks(ek1_idx, 0), c)
IF c NE 0 THEN ek = indx(0) ELSE ek = 0
ek2_idx = (plotinfo.ioptions(9)-1) >  0 < 93
indx = where(yarr(*, 1) LE eks(ek2_idx, 1), c)
IF c NE 0 THEN ek2 = indx(c-1) ELSE ek2 = n_ek-1

IF data_type EQ 2 THEN ek_type = 1 ELSE ek_type = 0
uztit=input_data.(idx(0)).LABLAXIS+' ('+input_data.(idx(0)).UNITS+')'
utitle='CLUSTER '+sc_id+'!CPEACE '+plotinfo.soptions(1)
nodata = input_data.(idx(0)).FILLVAL

;choose type of data to plot
message, 'Doing '+ data_names(data_type)+ '...', /cont
CASE data_type OF
    0:BEGIN                     ;Spin Averaged Spectra
        uytitle='Energy (eV)'
        ch_str='Spin Averaged'
        ;make a spin averaged spectrum. Average over pitch angles
        zmat=fltarr(ndat, n_ek)
        spin_spec = fltarr(n_ek)
        FOR i = 0, ndat-1 DO BEGIN
            spec_mat = transpose(data(i, *, *))
            spin_spec(*) = 0
            FOR j = 0, n_ek-1 DO BEGIN
                idx = where(spec_mat(*, j) NE nodata, c)
                IF c NE 0 THEN spin_spec(j) = total(spec_mat(idx, j))/c
            ENDFOR 
            zmat(i, *) = spin_spec
        ENDFOR 
        yst=yarr(0,0) & yen=yarr(n_ek-1,1)
        cluster_peace_spec_smooth, ndat, n_ek, zmat, yarr, smooth
        IF smooth GE 2 THEN ch_str = ch_str+', Smooth: '+varprt(smooth)
    END
    1:BEGIN                     ;All Pitch Angles @ one Energy
        uytitle='Pitch Angle (deg)'
        zmat=fltarr(ndat, n_pa)
        int_pitch = fltarr(n_pa)
        ch_str='E ' + varprt(yarr(ek,0)) + '-' + varprt(yarr(ek,1)) + ' eV' 
        FOR i = 0, ndat-1 DO zmat(i,*)=data(i, ek, *)
        ; make yarr for channels, pitch angles here
        yarr=pa_rng
        yst=yarr(0,0) & yen=yarr(n_pa-1,1)                
    END     
    2:BEGIN                     ;All Energies @ one Pitch Angle
        uytitle='Energy (eV)'
        ch_str='PA ' + varprt(pa_rng(pa,0)) + '-' + varprt(pa_rng(pa,1))
        zmat=data(*, *, pa)
        yst=yarr(0,0) & yen=yarr(n_ek-1,1)
    END 
    3:BEGIN                     ;All Pitch Angles for an Energy Range
        uytitle='Pitch Angle (deg)'
        ;make an range channel here. Sum over energy range.
        ;need to get energy ranges required
        ek_idx = [ek, ek2] & ek_idx = ek_idx(sort(ek_idx))
        ek_idx = ek_idx(0)+findgen(ek_idx(1)-ek_idx(0)+1)
        zmat=fltarr(ndat, n_pa)
        int_pitch = fltarr(n_pa)
        FOR i = 0, ndat-1 DO BEGIN
            spec_mat = transpose(data(i, ek_idx, *))
            int_pitch(*) = 0
            FOR j = 0, n_pa-1 DO BEGIN
                idx = where(spec_mat(j, *) NE nodata, c)
                IF c NE 0 THEN int_pitch(j) = total(spec_mat(j,idx)) $
                   ELSE int_pitch(j) = nodata
            ENDFOR
            zmat(i, *) = int_pitch
        ENDFOR     
        ch_str='E ' + varprt(yarr(ek,0)) + '-' + varprt(yarr(ek2, 1)) + ' eV'
        ; make yarr for channels, pitch angles here
        yarr=pa_rng
        yst=yarr(0,0) & yen=yarr(n_pa-1,1) 
    END 
    4:BEGIN                     ;All Energies for a Pitch Angle Range
        uytitle='Energy (eV)'
        ;make a pitch angle range. Sum over pitch angles. 
        ;need to get pitch angle ranges required
        pa_idx = [pa, pa2] & pa_idx = pa_idx(sort(pa_idx))
        pa_idx = pa_idx(0)+findgen(pa_idx(1)-pa_idx(0)+1)
        zmat=fltarr(ndat, n_ek)
        spec = fltarr(n_ek)
        FOR i = 0, ndat-1 DO BEGIN
            spec_mat = transpose(data(i, *, pa_idx))
            spec(*) = 0
            FOR j = 0, n_ek-1 DO BEGIN
                idx = where(spec_mat(*, j) NE nodata, c)
                IF c NE 0 THEN spec(j) = total(spec_mat(idx, j))/c
            ENDFOR 
            zmat(i, *) = spec
        ENDFOR 
        yst=yarr(0,0) & yen=yarr(n_ek-1,1)     
        ch_str='PA ' + varprt(pa_rng(pa,0)) + '-' + varprt(pa_rng(pa2,1)) 
        cluster_peace_spec_smooth, ndat, n_ek, zmat, yarr, smooth
        IF smooth GE 2 THEN ch_str = ch_str+', Smooth: '+varprt(smooth)  
    END
    5:BEGIN                     ;Pitch Angle Range @ an Energy Range (line)
        uytitle=uztit
        ;first get an array made of summed over energy range data.
        ;for this, recursively call this routine!
        plotinfo.ioptions(5) = 3
        p_cluster_peace_spec, panel, plotinfo, OUTPUT=2
        plotinfo.ioptions(5) = 5

        ;now, sum over pitch angle range aswell.
        
        ;make a pitch angle range. Sum over pitch angles. 
        ;need to get pitch angle ranges required
        pa_idx = [pa, pa2] & pa_idx = pa_idx(sort(pa_idx))
        pa_idx = pa_idx(0)+findgen(pa_idx(1)-pa_idx(0)+1)
        ymat = fltarr(ndat)
        FOR i = 0, ndat-1 DO BEGIN
            pitch = zmat(i, pa_idx)
            idx = where(pitch NE nodata, c)
            IF c NE 0 THEN ymat(i) = total(pitch(idx))/c $
              ELSE ymat(i) = 0
        ENDFOR
        IF smooth GE 2 THEN BEGIN
            zmat = smooth(ymat, smooth) 
            sm_str = '!CSmooth: '+varprt(smooth)
        ENDIF ELSE BEGIN
            zmat = ymat
            sm_str = ''
        ENDELSE 
        yst = 1 & yen = 2
        ch_str='PA ' + varprt(pa_rng(pa,0)) + '-' + varprt(pa_rng(pa2,1))  
        ch_str = ch_str + sm_str
    END 
ENDCASE

utitle = utitle+'!C'+ch_str

IF output EQ 2 THEN return ;if data  only is needed.

papco_Set_SubTable, SUBTABLE          ;sets the papco color table index
panelset,panel		              ;sets the panel position viewport

; the variables used above have the following meaning:
;       uztit   : the z-axis (papco_colorbar) label
; 	uytitle : the y-axis label
;	utitle  : the right-of plot label defining the data
;
; yscl or zscale is an array of dimensions (*,4) and contains info
; on the scaling to be used. The first dimension identifies the panel, and the
; next dimension has three entries:  
;         yscl(*,0)  is  0 for "use automatic scaling", 1 for "use manual"
;         yscl(*,1)  is then ymin (or zmin)
;         yscl(*,2)  is then ymax (or zmax)
;         yscl(*,3)  is 0 for lin, 1 for log axis
; when scaling is set to automatic, the automatic y limits are loaded in.
; for color plots, the z-limits are used for the color (z-axis) scaling
if (zscl(panel(0),0) eq 1) then begin
    zmin=zscl(panel(0),1)  &  zmax=zscl(panel(0),2)
endif else begin
    papco_autorange,zmat,zmin,zmax, $
      log=zscl(panel(0),3), exclude=0, nodata=nodata
    zscl(panel(0),1)=zmin  &  zscl(panel(0),2)=zmax
endelse

if (yscl(panel(0),0) eq 1) then begin 
    yst=yscl(panel(0),1)  &  yen=yscl(panel(0),2)
endif else begin
    yscl(panel(0),1)=yst  &  yscl(panel(0),2)=yen
endelse

; Overriding values by setting keywords is possible and allowed but discouraged
; since then your plot might no longer have the "common" look used in
; PAPCO. The widgetData structure is PAPCO's main internal data structure.
extra_plotPar={xrange:[xut1,xut2], yrange:[yst,yen], zrange:[zmin,zmax], $
               ylog:yscl(panel(0),3), zlog:zscl(panel(0),3), $
               ztitle:uztit, $
               xtickformat:'noticks',ztickformat:'papco_color_bar_log_ticks'}

IF data_type EQ 5 THEN BEGIN    ;reset some values for lineplot 
    extra_plotPar={xrange:[xut1,xut2], yrange:[zmin,zmax], $
                   ylog:yscl(panel(0),3), xtickformat:'noticks'}
ENDIF
  
; add keyword structure set locally and the one passed in
extra_plotPar=create_struct(extra_plotPar, extra_par)

; use auto y-labeling routine to get "good" y-lables. This sets the
; global IDL graphics system variables !Y.MINOR, !Y.TICKV, !Y.TICKS -
; so if you want to use this feature, avoid setting these or their
; corresponding keywords in your plot.
papco_y_label, yst, yen, log=yscl(panel(0),3)
  
; use papco routine to draw time axis. This checks for bottom plot and uses the
; user's xtickformat if it is something other than 'noticks'.
; If you use this rotuine, make sure your data plot routine uses xstyle=5 and 
; ystyle=5 (inhibit axis) so that you don't clobber those drawn by papco.  
; "down" sets the axis labeling level. 
down=0
papco_draw_time_axis, panel, OVERPLOT=OVERPLOT, _extra=extra_plotPar    
   
IF data_type EQ 5 THEN BEGIN ;dop lineplot

    plot, time, zmat, xstyle = 5, ystyle = 5, _extra=extra_plotPar
    IF keyword_set(OVERPLOT) THEN $
      utitle = '!C!C!C!C!C'+utitle
    right_side_label, panel, utitle, _extra=extra_plotPar
    
ENDIF ELSE BEGIN ;do spec plot

    ; do color plot and color bar plot. nodata specifies the "nodata" flag
    ; in your data. Here we do not plot axis and data seperate, as
    ; papco_plot_colorspec cannot be used as an overplot!
    ;resx, resy are in data coordinates. If the axis is logarithmic, 
    ;resx, resy need to be too.
    resx=4  &  resy=0.01 
    papco_plot_colorspec,zmat,time,yarr, nodata=nodata, resx=resx, resy=resy, $
        xstyle=5, ystyle=5, _extra=extra_plotPar
    papco_color_bar, _extra=extra_plotPar

    right_side_label,panel,utitle,/rot90

ENDELSE 

; store the coordinate information into common block
plotted_x = !x  &  plotted_y = !y

; plot y-axis label at left of plot. Use scalable routine!  
left_side_label, panel, uytitle, /rot90
   
; check if data needs to be output to file. Use the optional keywords
; to add more information to the data file
;      DESCRIPTION   a string or string array describing the data
;      TIME_LABEL    a string decribing the x-axis format
;      CHANNELS      an (ch,2) array giving the channel start/end
;      Y_LABEL       a string with the y-axis label
;      Z_LABEL       a string with the z-axis label
;      RS_LABEL      a string with the righ-side label of the plot
;      NODATA        contains the no data flag value.
if KEYWORD_SET(OUTPUT) then begin
    message, 'Writing plot data out to file', /cont
    description='cluster_peace template sample data - Energy spectra'
    time_label='Seconds since start of day'
    channels=yarr
    y_label=uytitle
    z_label=uztit
    rs_label=utitle
    nodata=0
    papco_write_data, time, zmat, $
                      DESCRIPTION=description, TIME_LABEL=time_label, $
                      CHANNELS=channels, Y_LABEL=y_label, Z_LABEL=z_label, $
                      RS_LABEL=rs_label, NODATA=nodata
ENDIF
 
return

noplot: 
message,'No data to plot',/cont
plotted_x = !x  &  plotted_y = !y
  
end 

;prog to interpolate to new energy channels. # of channels is set my
;smooth - new # = old # * smooth -1
PRO cluster_peace_spec_smooth,  ndat, n_ek, zmat, yarr, smooth

IF smooth LT 2 THEN return

IF smooth GT 1 THEN BEGIN 
    message, 'interpolating spectra...', /cont
    E1 = yarr(0, 0) & E2 = yarr(n_ek-1, 1)
    new_n_ek = n_ek*smooth-1
    logekstep =(alog10(e2)- alog10(e1))/new_n_ek
    y = fltarr(new_n_ek, 2)
    FOR i = 0, new_n_ek-1 DO BEGIN
        y(i, 0) = 10^(alog10(e1)+i*logekstep)
        y(i, 1) = 10^(alog10(e1)+(i+1)*logekstep)
    ENDFOR 
    
    old_log_ek_mid = (alog10(yarr(*, 0))+alog10(yarr(*, 1)))/2.0
    new_log_ek_mid = (alog10(y(*, 0))+alog10(y(*, 1)))/2.0

    new_zmat = fltarr(ndat, new_n_ek)
    test = fltarr(n_ek)
    ;test for zero's in spectra and don't interpolate into them
    FOR i = 0, ndat-1 DO BEGIN
        old_spec = transpose(zmat(i, *))
        test(*) = 1
        idx = where(old_spec EQ 0, c)
        IF c NE 0 THEN test(idx) = 0
        new_spec = interpol(old_spec, old_log_ek_mid, new_log_ek_mid)
        new_test = interpol(test, old_log_ek_mid, new_log_ek_mid)
        new_test = fix(new_test)
        new_zmat(i, *) = transpose(new_spec*new_test)
    ENDFOR 
ENDIF 

zmat = temporary(new_zmat)
yarr = temporary(y)

END
