;******************************************************************************
;* Sets up some common blocks for CLUSTER RAPID
;******************************************************************************
PRO cluster_rapid_commons
  
COMMON cluster_rapid, if_names, sci_names, lanl_names, $
                      ccr_plot_types, coord_names, expert_users, $
                      select_names, select_range, select_frmts, $
                      average_names, average_range, average_frmts, $
                      cluster_rapid_control

;setup array of user names that are expert users
expert_users = ['friedel', 'mggtt', 'projects']

;setup RAL if data product names
if_names=['IES_ESPCT', 'IES_PEDESTAL', 'IES_CLEAN', 'IES_CLEANBM', $
          'IES_CLEANBM_PA', 'IES_CLEANBM2', $
          'IES_CLEANBM2_PA', 'IES_EPAD2', 'IES_EPAD_16', 'IES_EPAD_16_PA']

;setup MPAe Science data product names
sci_names = ['HSPCT', 'ISPCT He', 'ISPCT CNO', $
             'I3DD H', 'I3DD He', 'I3DD CNO', 'E3DD', $
             'ESPCT', 'PED', 'DE', $ 
             'SGL1', 'SGL2', 'SGL3', 'MDATA', 'MPOLAR']

;setup pitch angle product names from LANL
lanl_names = ['IES_CLEANBM', 'IES_EPAD_16', 'I3DD H', 'I3DD He', 'I3DD CNO']

;setup CCR Plot types
ccr_plot_types = ['1', '2', '3', '4', 'Alle', 'Allp', 'AlleBM', $
                  'AllePitch', 'AlleEPADPitch']

;setup coordinates for Gradients
coord_names = ['X', 'Y', 'Z']

;enter variables for filtering the data
select_names = ['B_X', 'B_TOT']
select_range = [[-15.0, 15.0, 1], [0, 15, 1]]  ;low, high, res
select_frmts = ['(f4.1)', '(f4.1)']

;enter variables for averaging/selecting the data
average_names = ['DETECTOR', 'SECTOR', 'ENERGY','PITCH']
average_range = [[0, 8, 1], [0, 15, 1], [0, 17, 1], [0, 180, 2]]
average_frmts = ['(i1)', '(i2)', '(i2)', '(i3)']

;finally, set up a structure for all the variables needed by the
;cluster rapid module to control the behaviour of the module. This
;replaces the old use of "ioptions, options and soptions" tags of the
;plotinfo structure by a user defined structure which contains mre
;descriptive and verbose tags for easier use.

cluster_rapid_control = { $
  sc_id:0,      sc_id_info:'Spacecraft #; 0: 1, 1: 2, 2: 3, 3: 4', $
  data_src:0,   data_src_info:'Data Source; 0: RAL IF, 1: MPAe Sci, '+$
                              '2: LANL Pitch', $
  pa_src:0,     pa_src_info:'PA Mag source; 0: FGM, 1: MDATA', $
  pa_bin:10,    pa_bin_info:'Pitch angle bin size', $
  pa_smpl:0,    pa_smpl_info:'0-plot pitch bin, 1-plot pitch samples', $
  pa_norm:0,    pa_norminfo:'0: do not normalize PA distrib, 1: normalize', $
  reload:0,     reload_info:'Input; 0: reload data, 1: adopt from memory', $
  make:0,       make_info:'Make; 0: Allow, 1: Inhibit, 2: Force',$
  smooth:0,     smooth_info:'Smooth line plots/spectra; smooth range', $
  mask:0,       mask_info:'Contaminated data mask; 0: off, 1: on', $
  product:0,    product_info:'Product of data source. Index into either '+ $
                             'if_names, sci_names, lanl_names', $
  pl_type:0,    pl_type_info:'Plot type; 0: Ek @ Det/Sec, 1: Roll @ Ek/Det, '+$
                             '2: LUT, '+ $
                             '3: Ek @ PA, 4: PA @ Ek, 5: Mu Range', $
  spec:1,       spec_info:'Plot Style; 0: line plot, 1: color plot', $
  epad_ek:0,    epad_ek_info:'0-do not, 1-do resample EPAD energies', $
  mus:'',       mus_info:'String of comma seperated mu values (keV/nT)', $
  prod_4sc:0,   prod_4sc_info:'Four SC products; 0: none, 1: St.Dev., '+ $
                              '2: Tot. Var., 3: Sat. Intercal, '+ $
                              '4: Gradiometer, 5: MFE curv., 6: Gyro Rad.', $
  grad_xyz:0,   grad_xyz_info:'Gradiometer xyz components, bit toggle', $
  grad_prj:0,   grad_prj_info:'Gradiometer projections; 0-xz, 1-yz, 2-xy', $
  av_3dd:0,     av_3dd_info:'3DD Slice +/- # of spins to average', $
  pa_ctr_3dd:0, pa_ctr_3dd_info:'3DD Slice PA contour overlay 0-off,1-0n', $
  pa_lbl_3dd:0, pa_lbl_3dd_info:'3DD Slice PA label overlay 0-off,1-0n', $
  mu1:0,        mu1_info:'Low mu Ek for mu range plot; 0, 1,..', $
  mu2:0,        mu2_info:'High mu Ek for mu range plot; 0, 1,..', $
  ch:0,         ch_info:'For Line plots - 16 bit line select toggle', $
  filter:0,     filter_info:'Index into select_names / variable to use', $
  average:0,    average_info:'Index into average_names / variable to use', $
  seg_av:0,     seg_av_info:'0: plot all, 1: average',  $
  cal:0,        cal_info:'Data untits; 0: Cts, 1: Cts/s, '+ $
                         '2: Flux,  3: D.Flux, '+ $
                         '4: PSD, 5: PSD@mu, 6: DET Cal' $
  }
  
;now get the papco panel tags for the various select variables.
FOR i = 0, n_elements(select_names)-1 DO $
  cluster_rapid_control = CREATE_STRUCT(cluster_rapid_control, $
                 papco_panel_AddVar_Sel_Widgets(aBase, select_names(i), $
                                                select, st, en, v1, v2, vr, $
                                                select_frmts(i), /STRUCT_ONLY))

;get the papco_panel tags for the average variables.
FOR i = 0, n_elements(average_names)-1 DO $
  cluster_rapid_control = CREATE_STRUCT(cluster_rapid_control, $
               papco_panel_AddVar_Ave_Widgets(aBase, average_names(i), $
                                              st, en, v1, v2, vr, $
                                              average_frmts(i), /STRUCT_ONLY))
;set some defaults for the average variables
cluster_rapid_control.PITCH_AVERAGE_FROM_VAL = 80
cluster_rapid_control.PITCH_AVERAGE_TO_VAL = 100
cluster_rapid_control.ENERGY_AVERAGE_FROM_VAL = 2
cluster_rapid_control.ENERGY_AVERAGE_TO_VAL = 2
cluster_rapid_control.SECTOR_AVERAGE_FROM_VAL = 0
cluster_rapid_control.SECTOR_AVERAGE_TO_VAL = 15
cluster_rapid_control.DETECTOR_AVERAGE_FROM_VAL = 0
cluster_rapid_control.DETECTOR_AVERAGE_to_VAL = 8

;finally name the control structure. This is required for load/save operations!
cluster_rapid_control = CREATE_STRUCT(name = 'cluster_rapid_control_o', $
                                              cluster_rapid_control)

END 


;******************************************************************************
;**simple cluge to set up Jim's paths
pro path_rapid, path_zongdata, path_cals

common get_error, get_err_no, get_err_msg

r_module_config,'cluster_rapid', config_data, msg
IF get_err_no EQ 0 THEN r_module_default,'cluster_rapid', config_data, msg

index=where(config_data.ENV_VAR eq 'CLUSTER_RAPID_BINARY',c)
path_zongdata=config_data(index(0)).PATH+'/'

; check env. variable pointing to modules directory
if not (papco_check_data_env('PAPCO_MODULES', PATH=path)) then return
path_cals = path+'cluster_rapid/plot_read/'

end

;******************************************************************************
; File: DECOMPRESS.PRO  Decompresses counts from RAPID data
; Revision: 10/17/00    J L Roeder
;******************************************************************************
; Routines:
;       rd_rapid_comp   Reads compression table from file in cal directory
;       decompress              Decompresses RAPID counts
;******************************************************************************
pro rd_rapid_comp

;       Reads compression lookup table from text file

; Inputs: none

; Outputs: none

common decomp_lookup, Lut      ; Lookup table for count compression, 256 values

;       Get path to calibration files
path_rapid, path_zongdata, path_cals

fname = 'compress.txt'

fn = path_cals + fname
errno = -1
while errno ne 0 do begin

        openr, unit, fn, /GET_LUN, ERROR=errno
        if errno ne 0 then begin

                fn = pickfile(TITLE='Find compression table file '+fname, $
                   PATH=path_cals, GET_PATH=pth)
                path_cals = pth
                path_rapid, path_zongdata, path_cals

        endif

endwhile

;       Read and print header string in file
hdr1 = ' '
readf, unit, hdr1
print,'Reading: '+hdr1

;       Read energies
values = fltarr( 2, 256)
readf,unit, values
Lut = reform( values(1, *))

free_lun, unit

end

;******************************************************************************
pro rapid_decomp, counts

;       Decompress counts from lookup table in common block
;       Set 255B compressed counts to -1L.
;       Note output replaces input.

; Input/Output:
;       counts          Byte array of counts

common decomp_lookup, Lut

;       First time compute table
if n_elements( Lut) eq 0L then rd_rapid_comp

;       Decompress counts using lookup
cnts = counts
cnts = Lut( cnts)

;       Fill lost data with -1's
jlost = where (counts eq 255B)
if jlost(0) ne -1L then cnts( jlost) = -1L

;       Transfer result to output array
counts = cnts

end


;******************************************************************************
;Bit of code to be used to read in cluster rapid, cluster mag and
;cluster eph data in papco environment given time set by papco.

PRO get_data_for_3D_slice

COMMON mjdt, mjdt_start, mjdt_end     ;common time limit in mjdt
COMMON get_error, get_err_no, get_err_msg 
COMMON cluster_rapid ;see cluster_rapid_commons.pro for contents


;-----------------------------------------------------------------------------
;read ephemeris data
COMMON cluster_eph_data, eph_header, eph_data 
r_cluster_eph
IF get_err_no NE 0 THEN return ;error in reading eph data


;-----------------------------------------------------------------------------
;read cluster mag field data. Uses the papco cdf module to read the data.

sc_str = '1' ;2,3,4
info = papco_getplotinfostruct()
data_name = 'cluster_fgm'
cdfvar_name ='B_xyz_gse__C'+sc_str+'_PP_FGM'

r = papco_check_data_env('PAPCO_CDF_DATA', PATH=path)
IF r EQ 0 THEN BEGIN
    get_err_no = 1
    get_err_msg = 'Cannot get '+data_name+' data, PUM CDAcdf not loaded'
    return
ENDIF 

info.pa_ctr_3dd = data_name
info.pa_lbl_3dd = path+data_name+'/2000/c'+sc_str+'_pp_fgm_20001101_v01.cdf'
info.soptions(3) = cdfvar_name 

COMMON papco_cdf_data, cluster_mfi_data
r_papco_cdf, info, /verb
IF get_err_no NE 0 THEN return

;get time array
pos = strpos(tag_names(cluster_mfi_data), 'EPOCH')
idx = where(pos NE -1)
epoch_time = cluster_mfi_data.(idx(0)).dat

;get data array
pos = strpos(tag_names(cluster_mfi_data),strupcase(cdfvar_name))
idx = where(pos NE -1)
mag_xyz = transpose(cluster_mfi_data.(idx(0)).dat)
nodata = cluster_mfi_data.(idx(0)).FILLVAL

;convert data's cdf epoch time to tai. Tai is sec from some reference,
;cdf epoch is millisec from some reference
tai1=utc2tai({mjd:mjdt_start.mjd, time:mjdt_start.t*1000})

r1 = TAI2UTC(tai1,/ext) 
CDF_EPOCH, ep_1, r1.year, r1.month, r1.day, $
  r1.hour, r1.minute, r1.second, /COMPUTE_EPOCH

diff = tai1 - ep_1/1000.0
fgm_tai = epoch_time/1000.0 + diff


;-----------------------------------------------------------------------------
;read cluster rapid burst mode data.
cluster_rapid_control.sc_id = 0 ;1,2,3
cluster_rapid_control.data_src = 0
cluster_rapid_control.make = 1
cluster_rapid_control.product = 17
cluster_rapid_control.units = 3

;make required plotinfo struct for rapid
plotinfo = papco_getplotinfostruct()
fill_cluster_rapid_plotinfo, plotinfo, cluster_rapid_control

COMMON cluster_rapid_slice_1, time, zmat, data, yarr, utitle, uytitle, uztit
COMMON cluster_rapid_data, rapid_header, rapid_data
panel = [0, 1, 1] & orbitNo = 0 & IsRead = 0
output = 2

draw_cluster_rapid, panel, PlotInfo, orbitNo,get_data_call,slice_type,IsRead, $
                           OUTPUT = output


stop


END
