;******************************************************************************
;* FUNCTION:     
;*      FUNCTION  cluster_rapid_panel, aPlotInfo, GROUP=group, ACTION=ACTION
;* 
;* DESCRIPTION:  
;*	Open an editor widget for a panel, using the methods defined in the 
;*	file $papco_XX/papco_panel.pro .
;*	When the user presses the 'Done'- or the 'Cancel'-Button, the values 
;*	changed by the event-methods of papco_panel.pro are returned.
;*	The returned value is a structure containing those values and a flag,
;*	'canceled', which the caller may read out to check, whether the 
;*	window was closed using the 'Cancel' or the 'Done'-Button.
;*
;* INPUTS:       
;*      aPlotInfo       a structure of the papco_plotinfo-type
;*                      this structure is edited in the panel editor window.
;* KEYWORDS:
;*      GROUP           a WIDGET_BASE
;*	                when specified, this is used as GROUP_LEADER
;*      ACTION          a string
;*	                when specified, the 'Done'-Button is labeled with it
;*		        if not specified, the 'Done'-Button is labeled 'Done'
;*      SIZE_ONLY       returns size info of widget only. Used to determine
;*                      scrolling
;* OUTPUT:	 
;*      a default-structure of type papco_panel_editor_return. Though this
;*      type is nowhere defined, it is a virtually existing data-type. 
;*      All the functions in $papco_XX/papco.pro that edit plots or add new
;*      ones, expect the opened editor-windows to return a structure containing
;*      at least some fields. Typically this structure contains fields
;*      used by the papco_panel widgets, and some fields "inherited"
;*      from aplotinfo but modified by the user.
;*
;* CALLING SEQUENCE:
;*      newPlotData=papco_PANEL_TEMPLATE(aplotInfo, ACTION=, GROUP=)
;******************************************************************************
FUNCTION cluster_rapid_panel, plotinfo, GROUP=GROUP, ACTION=ACTION, $
                              SIZE_ONLY = SIZE_ONLY, SCROLL = scroll, $
                              X_SCROLL_SIZE = xsize, Y_SCROLL_SIZE = ysize 

COMMON cluster_rapid_panel_data, widgetData
COMMON cluster_rapid            ;see cluster_rapid_commons.pro for contents

;check if user is "expert user" by checking PAPCO_PATH for that user name. 
current = getenv('PAPCO_PATH') & expert = 0
FOR i = 0, n_elements(expert_users)-1 DO BEGIN
    pos = strpos(current, expert_users(i))
    IF pos NE -1 THEN expert = 1
ENDFOR

;--- if GROUP is not specified, the group-leader will be the desktop
IF NOT KEYWORD_SET(GROUP) THEN GROUP=0

;--- set the label for the action-button
IF NOT KEYWORD_SET(ACTION) THEN ACTION='Done'

;--- the routines used by papco_panel.pro define tags for a structure. Those
;    tags will be stored in the structure 'additionalWData'. When you realize 
;    the window, you have to add your own window-specific data to that  
;    structure and afterwards assign it to widgetData.
additionalWData={canceled:1}

;***NEW PAPCO 10.5*** 
;All the user's widget info is now stored in a structure,
;and the only thing stored in the plotinfostruct is the pointer to
;that structure. This allows unlimited freedom in variables for the
;module. The structure papco_cdf_control is defined in lanl_geo_eph_commons.pro

;test if the plotinfo pointer tag points to an 'INT' (means not
;assigned to a structure). If so, load in the control structure.
r = size(*plotinfo.USR_PTR1, /ST)
IF r.TYPE_NAME EQ 'INT' THEN plotinfo.USR_PTR1 = ptr_new(cluster_rapid_control)
control = *plotinfo.USR_PTR1
;control now contains the user defined control structure for this module.

;--- create the base-window ---------------------------------------------------
;    since this can be a large widget, make a left and right side.
title = plotinfo.panelKind + ', ' + papco_module_version('cluster_rapid')
panel_base=papco_WIDGET_BASE(title=title, column=1, GROUP_LEADER=group, $
                             /MODAL, /BASE_ALIGN_CENTER)

panel_top =    widget_base(panel_base, COLUMN = 1)
panel_middle = widget_base(panel_base, ROW = 1, SCROLL = scroll, $
                           X_SCROLL_SIZE = xsize, Y_SCROLL_SIZE = ysize)
panel_left   = widget_base(panel_middle, COLUMN=1, /base_align_center)
panel_right  = widget_base(panel_middle, COLUMN=1, /base_align_center)
panel_bottom = widget_base(panel_base, ROW = 1)

;--- All papco_panel_add routines are papco panel widgets that can be included
;    in the panel editor or not. They provide functionality that papco 
;    supports. They dynamically extend the panel editors widget structure, and
;    all the events they generate are supported by papco_panel_event, 
;    transparent to the user. 

;--- the first part of the window will consist of the widget that -------------
;    controls the height of that panel. This is needed for all panel editors.
widgets=papco_panel_AddHeightWidgets(panel_top, plotinfo)
additionalWData=create_struct(widgets, additionalWData)
   
;
;---  <<<<<<<<<<  insert your widgets below here!  >>>>>>>>>
;
title = 'none'
;--- some choices for data input ---------------------------------------------
input_base=widget_base(panel_left, COLUMN = 2, /FRAME)

names = papco_make_choice_names(control.sc_id_info, TITLE = title)
sc_id_base=CW_BGROUP(input_base, names, ROW=1, /EXCLUSIVE, $
                     UVALUE='spb_sc_id', LABEL_LEFT = title, $
                     SET_VALUE=control.sc_id) 

names = papco_make_choice_names(control.data_src_info, TITLE = title)
data_base=papco_CW_BGROUP(input_base, names, ROW = 1, /EXCLUSIVE, $
                     LABEL_LEFT = title, UVALUE='spb_data_src', $
                     BUTTON_UVALUE = data_uvalues, $
                     SET_VALUE=control.data_src) 
 
names = papco_make_choice_names(control.reload_info, TITLE = title)
reload_base=CW_BGROUP(input_base, names, ROW=1, /EXCLUSIVE, $
                     LABEL_LEFT = title, UVALUE='spb_reload', $
                     SET_VALUE=control.reload)

names = papco_make_choice_names(control.pa_src_info, TITLE = title)
pa_src_base=CW_BGROUP(input_base, names, ROW=1, /EXCLUSIVE, $
                     LABEL_LEFT = title, UVALUE='spb_pa_src', $
                     SET_VALUE=control.pa_src)

IF expert THEN BEGIN 
names = papco_make_choice_names(control.make_info, TITLE = title)
make_base = CW_BGROUP(input_base, names, ROW=1, /EXCLUSIVE, $
                     LABEL_LEFT = title, UVALUE='spb_make', $
                     SET_VALUE=control.make)
ENDIF

;--- smoothing of data --------------------------------------------------------
ef_smooth=CW_FIELD(input_base, ROW=1, /INTEGER, /RETURN_EVENTS, $
                   TITLE = 'Smooth:', XSIZE = 3, $
                   UVALUE='ef_smooth', VALUE=control.smooth)

;--- pitch angle bin size ----------------------------------------------------
ef_pa_bin=CW_FIELD(input_base, ROW=1, /INTEGER, /RETURN_EVENTS, $
                   TITLE = 'PA_Bin:', XSIZE = 3, $
                   UVALUE='ef_pa_bin', VALUE=control.pa_bin)

;--- mask contaminated data ---------------------------------------------------
names = papco_make_choice_names(control.mask_info, TITLE = title)
mask_base = CW_BGROUP(input_base, names, ROW=1, /EXCLUSIVE, $
                     LABEL_LEFT = title, UVALUE='spb_mask', $
                     SET_VALUE=control.mask)

;--- make a toggle base for data type selection, RAL, SCI etc -----------------
prod_out_base=WIDGET_BASE(panel_left, column=1, /base_align_center) 
prod_top_base=WIDGET_BASE(prod_out_base, /base_align_center) 

if_base = papco_CW_BGROUP(prod_top_base, if_names, COLUMN=6, $
  /EXCLUSIVE, /FRAME, LABEL_TOP='RAL IF Data Choice:', $
  UVALUE='spb_if', BUTTON_IDS = if_ids, BUTTON_UVALUE = if_uvalues, $
  SET_VALUE = control.product > 0 < (n_elements(if_names)-1)) 

sci_base =  papco_CW_BGROUP(prod_top_base, sci_names, COLUMN=8, $
  /EXCLUSIVE, /FRAME, LABEL_TOP='MPAe Sci Data Choice:', $
  UVALUE='spb_sci', BUTTON_IDS = sci_ids, BUTTON_UVALUE = sci_uvalues, $
  SET_VALUE = control.product > 0 < (n_elements(sci_names)-1)) 

have = ['HSPCT', 'ISPCT He', 'ISPCT CNO', 'I3DD H', 'I3DD He', 'I3DD CNO']
FOR i = 0,  n_elements(sci_names)-1 DO BEGIN
    pos = strpos(have, sci_names(i)) & index = where(pos NE -1,  c) 
    IF c LT 1 THEN widget_control, sci_ids(i),  sensitive = 0
ENDFOR

lanl_base =  papco_CW_BGROUP(prod_top_base, lanl_names, COLUMN=7, $
  /EXCLUSIVE, /FRAME, LABEL_TOP='LANL Pitch Data Choice:', $
  UVALUE='spb_lanl', BUTTON_IDS = lanl_ids, BUTTON_UVALUE = lanl_uvalues, $
  SET_VALUE=control.product > 0 < (n_elements(lanl_names)-1))

have = ['IES_CLEANBM', 'IES_EPAD_16']
FOR i = 0,  n_elements(lanl_names)-1 DO BEGIN
    pos = strpos(have, lanl_names(i)) & index = where(pos NE -1,  c) 
    IF c LT 1 THEN widget_control, lanl_ids(i),  sensitive = 0
ENDFOR

;--- add choices for plot types -----------------------------------------------
plot_base = WIDGET_BASE(panel_left, ROW=1, /base_align_center) 

;types of plots
names = papco_make_choice_names(control.pl_type_info, TITLE = title)
pl_type_base = papco_CW_BGROUP(plot_base, names, $
                        COLUMN=1, /EXCLUSIVE, /FRAME, $
                        LABEL_TOP=title, BUTTON_IDS = pl_type_ids, $
                        UVALUE='spb_plot', SET_VALUE=control.pl_type) 

;other choices
other_choice_base=WIDGET_BASE(plot_base,Column=1)

ps_base = WIDGET_BASE(other_choice_base, ROW = 1)
;how to plot, line or color 3D
names = papco_make_choice_names(control.spec_info, TITLE = title)
spec_base = papco_CW_BGROUP(ps_base, names, $
                        COLUMN=1, /EXCLUSIVE, /FRAME, $
                        LABEL_LEFT=title, BUTTON_IDS = spec_ids, $
                        UVALUE='spb_spec', SET_VALUE=control.spec) 
spb_av_lines = WIDGET_BUTTON(ps_base, UVALUE='spb_av_lines', $
                        VALUE='Av. lines')
WIDGET_CONTROL, spb_av_lines, SET_BUTTON = 0

toggle_base=WIDGET_BASE(other_choice_base, /FRAME, /NONEXCLUSIVE, COLUMN=2)
spb_pa_smpl=WIDGET_BUTTON(toggle_base, UVALUE='spb_pa_smpl', $
                              VALUE='PA samples/bin')
WIDGET_CONTROL, spb_pa_smpl, SET_BUTTON=control.pa_smpl
spb_epad_interp=WIDGET_BUTTON(toggle_base, UVALUE='spb_epad_interp', $
                              VALUE='CLEAN EPAD')
WIDGET_CONTROL, spb_epad_interp, SET_BUTTON=control.epad_ek
spb_pa_norm = WIDGET_BUTTON(toggle_base, UVALUE='spb_pa_norm', $
                              VALUE='PA normalize')
WIDGET_CONTROL, spb_pa_norm, SET_BUTTON=control.pa_norm


;--- sets up choices for PSD plots --------------------------------------------
fld_mu = CW_FIELD(other_choice_base, /COLUMN,  XSIZE=21, $
                  TITLE = 'Enter mu1, mu2, ... (MeV/G):', $
                  UVALUE='fld_mu', VALUE = control.mus)

;type of 4 spacecraft plots 
names = papco_make_choice_names(control.prod_4sc_info, TITLE = title)
sc4_names=['None', 'St. Dev', 'Total var.', 'Sat. Intercal', 'Gradiometer']
sc4_base = papco_CW_BGROUP(plot_base, names, $
                        COLUMN=1, /EXCLUSIVE, /FRAME, $
                        LABEL_TOP = title, $
                        UVALUE='spb_sc4', BUTTON_IDS = sc4_ids, $
                        SET_VALUE=control.prod_4sc)
widget_control,sc4_ids(2), sensitive = 0

;--- buttons for 4sc products,ie gradiometer coordinates,mu values etc --------

;select basic coordinates
n_ch = n_elements(coord_names) & coord_set = bytarr(n_ch)
FOR i=0,n_ch-1 DO BEGIN
    if (control.grad_xyz and 2l^i) eq 2l^i then coord_set(i)=1 $
    else coord_set(i)=0
ENDFOR

outer_grad_base = WIDGET_BASE(plot_base, /COLUMN)

coord_base=papco_CW_BGROUP(outer_grad_base, coord_names, ROW=1, $
                           /NONEXCLUSIVE, /FRAME, $
                           LABEL_TOP='Gradient comp (GSE)', $
                           UVALUE='spb_coord', BUTTON_IDS = coord_ids, $
                           SET_VALUE=coord_set)

;create new buttons for 2d arrow plots
plot2d_options = ['X-Z', 'Y-Z', 'X-Y']
grad_base = papco_CW_BGROUP(outer_grad_base, plot2d_options, $
                            /COLUMN, /EXCLUSIVE, /FRAME, $
                            LABEL_TOP='GRADIOMETER:', $
                            UVALUE='spb_vec', BUTTON_IDS = vec_ids, $
                            SET_VALUE=control.grad_prj)

;--- Make a toggle base for a range of additional choices...-------------------

;--- Add some choices that affect rapid slices only ---------------------------
slice_choice_base = WIDGET_BASE(panel_left, ROW=1, /base_align_center)

;choices for USER_3DD_Slice_cluster_rapid (3dd_slice_widget)
Slice_base_3dd = WIDGET_BASE(slice_choice_base, ROW=1, $
                             /FRAME, /BASE_ALIGN_CENTER)

labl_base = WIDGET_LABEL(Slice_base_3dd, VALUE = 'USER_3DD_Slice:')

ef_av_3dd = CW_FIELD(Slice_base_3dd, /INTEGER, /RETURN_EVENTS, XSIZE = 4, $
                     TITLE = 'Frame av. (+/-)', $
                     VALUE = fix(control.av_3dd), UVALUE='ef_av_3dd')

;-- set up some toggle switch options for this slice --------------------------
u3dd_toggle_base=WIDGET_BASE(Slice_Base_3dd, COLUMN=1, /NONEXCLUSIVE)
spb_pitch_3dd = WIDGET_BUTTON(u3dd_toggle_base, UVALUE='spb_pitch_3dd', $
                              VALUE='Overlay PA Contour')
WIDGET_CONTROL, spb_pitch_3dd, SET_BUTTON = fix(control.pa_ctr_3dd)

;-- set up some toggle switch options for this slice --------------------------
spb_label_3dd = WIDGET_BUTTON(u3dd_toggle_base, UVALUE='spb_label_3dd', $
                              VALUE='Label cells with PA')
WIDGET_CONTROL, spb_label_3dd, SET_BUTTON = fix(control.pa_lbl_3dd)


;--- button to choose "average" variable --------------------------------------
average_out_base=WIDGET_BASE(panel_left, /FRAME, COLUMN=1) 
average_ctr_base = WIDGET_BASE(average_out_base, ROW = 1)

names = 'select '+average_names
names = [names, 'select CH']
average_base=CW_BGROUP(average_ctr_base, names, ROW=1, /EXCLUSIVE, $
                      UVALUE='spb_average', SET_VALUE=control.average)

;--- add label to show currently chosen range of variables -------------------
av_lbl_base = widget_label(average_out_base, $
                           value = strformat('Not set yet', 65, /center))


;--- add sliders for all variables in "average_names" ------------------------
;    Multiple map bases.
avsl_out_base = WIDGET_BASE(average_out_base, COLUMN=1, /align_center)
avsl_base = WIDGET_BASE(avsl_out_base, /ALIGN_CENTER)

;--- make select widgets for all variables in "average_names" -----------------
;    use papco_panel compound widget for this!
FOR i = 0, n_elements(average_names)-1 DO BEGIN

   r = execute('FROM_VAL = control.'+average_names(i)+'_AVERAGE_FROM_VAL')
   r = execute('TO_VAL = control.'+average_names(i)+'_AVERAGE_TO_VAL')

   FROM_VAL = FROM_VAL > average_range(0, i) < average_range(1, i)
   TO_VAL = TO_VAL > average_range(0, i) < average_range(1, i)

   widgets = papco_panel_AddVar_Ave_Widgets(avsl_base, average_names(i), $
               FROM_VAL, TO_VAL, $
               average_range(0, i), average_range(1, i), average_range(2, i), $
               average_frmts(i))
   additionalWData=create_struct(widgets, additionalWData)

END 

;button choices for channel selections (used for line plot) 
;use ch as 20 bit integer logical flags
mpb_ch_base = WIDGET_BASE(avsl_base, ROW = 1, /base_align_center, /FRAME)
set = bytarr(16) & names = 'CH'+varprt(indgen(16)+1)
FOR i = 0, 15 DO $
  IF (control.ch AND 2ll^i) EQ 2ll^i THEN set(i)=1 else set(i)=0
mpb_ch = CW_BGROUP(mpb_ch_base, names, COLUMN=8, /NONEXCLUSIVE, $
                   UVALUE='mpb_ch', SET_VALUE=set) 
spb_all = WIDGET_BUTTON(mpb_ch_base, UVALUE='spb_all', VALUE='Select All')


;--- button to choose "select" variable ---------------------------------------
filter_outer_base=WIDGET_BASE(panel_left, /FRAME, COLUMN=1) 
filter_control_base = WIDGET_BASE(filter_outer_base, ROW = 1)

names = 'filter by '+select_names
filter_base=CW_BGROUP(filter_control_base, names, ROW=1, /EXCLUSIVE, $
                      UVALUE='spb_filter', SET_VALUE=control.filter)

names = papco_make_choice_names(control.seg_av_info)
seg_av_base = CW_BGROUP(filter_control_base, names, ROW=1, /EXCLUSIVE, $
                        UVALUE='spb_seg_av', SET_VALUE=control.seg_av)

;--- Add sliders to filter data - BX, BMAG etc -----------------------
;    Multiple map bases. 
select_outer_base=WIDGET_BASE(filter_outer_base, COLUMN=1, /align_center)
select_base=WIDGET_BASE(select_outer_base, /align_center)

;--- make select widgets for all variables in "filter_names" ------------------
;    use papco_panel compound widget for this!
FOR i = 0, n_elements(select_names)-1 DO BEGIN 

   r = execute('RANGE = control.'+select_names(i)+'_SELECT_RANGE')
   r = execute('FROM_VAL = control.'+select_names(i)+'_SELECT_FROM_VAL')
   r = execute('TO_VAL = control.'+select_names(i)+'_SELECT_TO_VAL')

   FROM_VAL = FROM_VAL > select_range(0, i) < select_range(1, i)
   TO_VAL = TO_VAL > select_range(0, i) < select_range(1, i)

   widgets = papco_panel_AddVar_Sel_Widgets(select_base, select_names(i), $
                 RANGE, FROM_VAL, TO_VAL, $
                 select_range(0, i), select_range(1, i), select_range(2, i), $
                 select_frmts(i))             
   additionalWData=create_struct(widgets, additionalWData)

END 

;--- data unit choices --------------------------------------------------------
unit_base = WIDGET_BASE(panel_right, COLUMN=1, /FRAME)
names = papco_make_choice_names(control.cal_info, TITLE = title)
data_base=papco_CW_BGROUP(unit_base, names, COLUMN = 3, /EXCLUSIVE, $
                          LABEL_TOP = title, UVALUE='spb_cal', $
                          BUTTON_UVALUE = cal_uvalues, $
                          SET_VALUE=control.cal) 

;
;---  <<<<<<<<<<  insert your widgets above here!  >>>>>>>>>
;

;--- some panels might need to read/input to text files (event, text ----------
;    module). So a generic papco widget is supplied for this. 
;  widgets=papco_panel_AddFile_Widgets(panel_left, plotinfo)
;  additionalWData=create_struct(widgets, additionalWData)   

;--- add the papco_panel widget for making a batch of all data dates ----------
;  IF friedel THEN BEGIN
;    widgets=papco_panel_AddBatch_Widgets(panel_left, plotinfo,  $
;      REDO = ['Orig', 'MSF', 'DAT', 'RAL_IF', 'CCR C1', 'CCR C2', 'CCR C3', $
;              'CCR C4', 'CCR All e', 'CCR All p', 'CCR BM e'])
;    additionalWData=create_struct(widgets, additionalWData)
;--- add the papco panel widget for specifying own plot description label -----
widgets=papco_panel_AddDescr_Widgets(panel_right, plotinfo, /VERTICAL)
additionalWData=create_struct(widgets, additionalWData)

;--- make base for color bar OR plotstyle widget. One only is mapped ----------
dummy_out_base = WIDGET_BASE(panel_right, COLUMN=1)
dummy_base = WIDGET_BASE(dummy_out_base,/ALIGN_CENTER)

color_base=WIDGET_BASE(dummy_base, COLUMN = 1)
;--- add the papco panel widget for choosing the color bar for color plots ----
widgets=papco_panel_Addcolorbar_Widgets(color_base, plotinfo, /VERTICAL)
additionalWData=create_struct(widgets, additionalWData)

line_base=WIDGET_BASE(dummy_base, COLUMN = 1)
;--- add the  papco panel widget for choosing the plot-style for line plots ---
widgets=papco_panel_add_PlotStyleButtons(line_base, plotinfo, 0,/VERTICAL)
additionalWData=create_struct(widgets, additionalWData)

scaling_base = widget_base(panel_right,  ROW = 1, /base_align_center)
;--- add the papco panel widget for choosing the y-axis-ranges ----------------
widgets=papco_panel_AddYScale_Widgets(scaling_base, plotinfo,/VERTICAL)
additionalWData=create_struct(widgets, additionalWData)
;--- add the papco panel widgets for choosing the z-axis-ranges ---------------
widgets=papco_panel_AddZScale_Widgets(scaling_base, plotinfo,/VERTICAL)
additionalWData=create_struct(widgets, additionalWData)
;--- finally, the Actionbuttons (help, cancel, done) are added to the ---------
;    editor window. This is needed for all panel editors. If you want to 
;    make use of the configure feature, then add the keyword CONFIG
widgets=papco_panel_AddActionButtons(panel_bottom, Action, plotinfo, /CONFIG, $
                                     ROW = 1) 
additionalWData=create_struct(widgets, additionalWData)

;--- now, create the structure that keeps the panel-window data. You always
;    need to specify the help-file, and your structure must contain the
;    'additionalWData'. You should create the helpfile in the papco_interface
;    directory of your module (template is in the cluster_rapid example module)
;    Also add the ID's of your widgets so that you can perform widget_control
;    actions in the events routine or in a refresh routine if needed.

thisWidgetsData={helpfile:'cluster_rapid_panel.help', $
                 color_base:color_base, $
                 line_base:line_base, $
                 filename:PlotInfo.filename, $
                 if_base:if_base, if_ids:if_ids, $
                 lanl_base:lanl_base, sci_base:sci_base, $
                 mpb_ch:mpb_ch, $
                 mpb_ch_base:mpb_ch_base, $
                 plot_base:plot_base, $
                 pl_type_base:pl_type_base, pl_type_ids:pl_type_ids, $
                 outer_grad_base:outer_grad_base, $
                 coord_base:coord_base, $
                 grad_base:grad_base, $
                 av_lbl_base:av_lbl_base, $
                 fld_mu:fld_mu, $
                 ef_smooth:ef_smooth, $
                 ef_pa_bin:ef_pa_bin, $
                 sc4_base:sc4_base,$
                 toggle_base:toggle_base}

; add control struct, papco_panel structs
widgetData=papco_panel_CreateWidgetData(additionalWData, thisWidgetsData)
widgetData=papco_panel_CreateWidgetData(widgetData, control)

;--- return size of the window only here --------------------------------------
IF keyword_set(SIZE_ONLY) THEN BEGIN
    size=WIDGET_INFO(panel_base, /GEOMETRY)
    size_only = size & return, 1
ENDIF 

;--- realize the window -------------------------------------------------------
papco_Panel_Realize, panel_base, widgetData
WIDGET_CONTROL, panel_base, HOURGLASS=1
cluster_rapid_refresh,  /ALL

papco_xmanager, 'cluster_rapid_panel', panel_base

;now return data. Since all the widget's info in in a control structure,
;only load it's pointer into USR_PTR1 and return that.

;control is part of the widgetData structure. Extract it
struct_assign, widgetData, *plotinfo.USR_PTR1

;work out if there has been a change in the control structure
plotinfo.modified = plotinfo.modified + $
                    papco_compare_struct(control, *plotinfo.USR_PTR1)

ReturnData={USR_PTR1:plotinfo.USR_PTR1, $
            MODIFIED:plotinfo.modified}

additionalRData=papco_panel_ReturnData(widgetData)

return, create_struct(ReturnData, additionalRData)
   
END

;******************************************************************************
;* PROCEDURE:     
;*      PRO cluster_rapid_refresh 
;* 
;* DESCRIPTION:  
;*      Takes refreesh actions on panel widgets controlled by keywords.
;*
;* INPUTS:       
;*      none
;*                  
;* KEYWORDS:
;*      
;* 
;* CALLING SEQUENCE:
;*      cluster_rapid_refresh, /keywords
;******************************************************************************
PRO cluster_rapid_refresh,  ALL = ALL, PROD = PROD, PLOT = PLOT, $
  SC4 = SC4, SELECTORS = SELECTORS, CHANNELS = CHANNELS, $
  STYLE_PLOT=STYLE_PLOT, AVERAGE = AVERAGE, COLOR_LINE = COLOR_LINE, $
  AV_LBL = AV_LBL
  
COMMON cluster_rapid_panel_data, widgetData
COMMON cluster_rapid            ;see cluster_rapid_commons.pro for contents

IF keyword_set(ALL) THEN BEGIN
    PROD = 1 & PLOT = 1 & SC4 = 1 & SELECTORS = 1 & STYLE_PLOT=1
    AVERAGE = 1 & COLOR_LINE = 1 & AV_LBL = 1
ENDIF

IF keyword_set(PROD) THEN BEGIN
    wid_ids = [widgetData.if_base, widgetData.sci_base, widgetData.lanl_base]
    CASE widgetData.data_src OF
        0: BEGIN 
            map = [1, 0, 0] & pl_type_ids = [1, 1, 1, 0, 0, 1]
            IF widgetData.pa_src EQ 0 THEN pl_type_ids =[1, 1, 1, 1, 1, 1]
        end
        1: BEGIN 
            map = [0, 1, 0] & pl_type_ids = [1, 1, 0, 0, 0, 0]
        END
        2:  BEGIN 
            map = [0, 0, 1] & pl_type_ids = [0, 0, 0, 1, 1, 0]
        END
    ENDCASE 
    IF pl_type_ids(widgetData.pl_type) EQ 0 THEN BEGIN
        idx = where(pl_type_ids EQ 1) &  widgetData.pl_type = idx(0)
    ENDIF

    FOR i = 0, n_elements(pl_type_ids)-1 DO $
      IF pl_type_ids(i) THEN widget_control, widgetData.pl_type_ids(i), sensitive=1 $
        ELSE widget_control, widgetData.pl_type_ids(i), sensitive=0
    FOR i = 0, n_elements(map)-1 DO widget_control, wid_ids(i), MAP = map(i)
    widget_control, widgetData.pl_type_base, SET_VALUE = widgetData.pl_type
    
    CASE 1 OF 
        (widgetData.pl_type EQ 3) OR (widgetData.pl_type EQ 4): BEGIN
            
        END
        ELSE: BEGIN
            
        END  
    ENDCASE   
    
ENDIF

IF keyword_set(PLOT) THEN BEGIN
    IF widgetData.spec EQ 0 THEN GOTO, done_plot
    CASE widgetData.pl_type OF 
        0: widgetData.ylinlog = 1
        1: widgetData.ylinlog = 0
        2: widgetData.ylinlog = 0
        3: widgetData.ylinlog = 1
        4: widgetData.ylinlog = 0
        5: widgetData.ylinlog = 1
        ELSE: BEGIN & End 
    ENDCASE 
    papco_panel_Refresh, widgetData, /YSCALE
    done_plot:
ENDIF     

IF keyword_set(SC4) THEN BEGIN
    wid_ids = [widgetData.outer_grad_base, widgetData.plot_base, $
               widgetData.grad_base,widgetData.coord_base]
    n_wid = n_elements(wid_ids) & map = bytarr(n_wid) & sen = bytarr(n_wid)
    CASE widgetData.prod_4sc OF
        0: BEGIN
            sen = [1, 1, 0,0]
        END
        1: BEGIN
            sen = [0, 1, 0,0]
        END
        2: BEGIN
            sen = [0, 1, 0,0]
        END
        3: BEGIN
            sen = [0, 1, 0,0]
        END
        4: BEGIN
            sen = [1,1,1,1]
            widgetData.YLINLOG = 0
        END
        5: BEGIN
            sen = [0, 1, 0,0]
        END
        6: BEGIN
            sen = [0, 1, 0,0]
        END
    ENDCASE
  
    FOR i = 0, n_wid-1 DO WIDGET_CONTROL, wid_ids(i), SENSITIVE = sen(i) 
ENDIF

IF keyword_set(SELECTORS) THEN BEGIN

ENDIF 

IF keyword_set(CHANNELS) THEN BEGIN
    set = bytarr(16) 
    FOR i=0,15 DO IF (widgetData.ch AND 2ll^i) EQ 2ll^i THEN $
      set(i)=1 else set(i)=0
    WIDGET_CONTROL, widgetData.mpb_ch, SET_VALUE=set
ENDIF

; ---- COLORBAR or PLT SYYLE widget -------------------------------------------
IF keyword_set(COLOR_LINE) THEN BEGIN    
    IF widgetData.spec EQ 0 THEN BEGIN 
        widget_control, widgetData.line_base, MAP=1
        widget_control, widgetData.color_base, MAP=0
    ENDIF ELSE BEGIN 
        widget_control, widgetData.line_base, MAP=0
        widget_control, widgetData.color_base, MAP=1
    ENDELSE
ENDIF 

; --- Filter Variable multiple bases ------------------------------------------
IF keyword_set(STYLE_PLOT) THEN BEGIN
    ;make a list of widget ID's for the filter variable bases
    FOR i = 0, n_elements(select_names)-1 DO BEGIN
        idx = where(tag_names(widgetData) EQ select_names(i)+'_SELECT_BASE') 
        IF i EQ 0 THEN wid_ids = [widgetData.(idx(i))] ELSE $
           wid_ids = [wid_ids, widgetData.(idx(0))]
    ENDFOR 
    n_wid = n_elements(wid_ids) & map = bytarr(n_wid)
    map(widgetData.filter) = 1
    FOR i = 0, n_wid-1 DO WIDGET_CONTROL, wid_ids(i), MAP = map(i) 
ENDIF

; --- Average Variable multiple bases -----------------------------------------
IF keyword_set(AVERAGE) THEN BEGIN
    ;make a list of widget ID's for the average variable bases
    FOR i = 0, n_elements(average_names)-1 DO BEGIN
        idx = where(tag_names(widgetData) EQ average_names(i)+'_AVERAGE_BASE') 
        IF i EQ 0 THEN wid_ids = [widgetData.(idx(i))] ELSE $
           wid_ids = [wid_ids, widgetData.(idx(0))]
    ENDFOR 
    wid_ids = [wid_ids, widgetData.mpb_ch_base]
    n_wid = n_elements(wid_ids) & map = bytarr(n_wid)
    map(widgetData.average) = 1
    FOR i = 0, n_wid-1 DO WIDGET_CONTROL, wid_ids(i), MAP = map(i) 
ENDIF

; --- Average Label update ----------------------------------------------------
IF keyword_set(AV_LBL) THEN BEGIN
    s1 = varprt(fix(widgetData.SECTOR_AVERAGE_FROM_VAL))
    s2 = varprt(fix(widgetData.SECTOR_AVERAGE_TO_VAL))
    d1 = varprt(fix(widgetData.DETECTOR_AVERAGE_FROM_VAL))
    d2 = varprt(fix(widgetData.DETECTOR_AVERAGE_TO_VAL))
    e1 = varprt(fix(widgetData.ENERGY_AVERAGE_FROM_VAL))
    e2 = varprt(fix(widgetData.ENERGY_AVERAGE_TO_VAL))
    p1 = varprt(fix(widgetData.PITCH_AVERAGE_FROM_VAL))
    p2 = varprt(fix(widgetData.PITCH_AVERAGE_TO_VAL))
    out_str = 'Det: '+d1+' - '+d2+'      Sec: '+s1+' - '+s2+$
              '      Ech: '+e1+' - '+e2+'      PA: '+p1+' - '+p2 
    out_str = strformat(out_str, 65, /center)
    widget_control, widgetData.av_lbl_base, set_value = out_str
ENDIF

END


;******************************************************************************
;* PROCEDURE:     
;*      PRO cluster_rapid_panel_event, event
;* 
;* DESCRIPTION:  
;*      This processes events from the XMANAGER.
;*      As this example does not extend the basic functionality of plot-
;*      editors, it directly calls the default-handler, papco_PANEL_Event
;*
;* INPUTS:       
;*      event            an XManager event
;*                  
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      called by XMANAGER
;******************************************************************************
PRO cluster_rapid_panel_event, event
  
COMMON cluster_rapid_panel_data, widgetData

WIDGET_CONTROL,GET_UVALUE=uval, event.id

;test uvalue for structure - select variable sliders return structure uvals.
r = size(uval, /st)
IF r.type_name EQ 'STRUCT' THEN BEGIN
    uval_str = uval
    uval = uval.uvalue
ENDIF

; in this editor, we only need to process the channel slider events. events
; other than those from your own widgets need to be passed on to 
; 'papco_panel_Event', which handles the events generated
; by the papco-provided panel widgets. Your own added panel widgets events
; need to be processed here:  you should check 'event' for your events and call
; 'papco_panel_Event', if event was not generated by your own panel widgets.
; The best way to learn is to look at some of the other module's panel editors.

CASE uval OF 
    'spb_all': BEGIN
        widgetData.ch = 0
        FOR i=0,15 DO widgetData.ch = widgetData.ch+ 2ll^i
        cluster_rapid_refresh, /CHANNELS
    END     

    'ef_av_3dd' :    widgetData.mus = string(event.value)
    'spb_pitch_3dd': widgetData.pa_ctr_3dd = string(event.select)
    'spb_label_3dd': widgetData.pa_lbl_3dd = string(event.select)

    'spb_pa_norm': widgetData.pa_norm=event.select
    'spb_pa_smpl': widgetData.pa_smpl=event.select
    'spb_epad_interp': widgetData.epad_ek=event.select
    'spb_sc_id' : widgetData.sc_id=event.value
    'spb_reload' : widgetData.reload=event.value
    'spb_pa_src' : BEGIN 
        widgetData.pa_src = event.value
        cluster_rapid_refresh, /PROD
    END  
    'spb_make' : widgetData.make = event.value
    'spb_mask' : widgetData.mask = event.value
    'spb_bin'  : widgetData.product = event.value
    'spb_if'   : widgetData.product = event.value
    'spb_sci'  : widgetData.product = event.value
    'spb_lanl' : widgetData.product = event.value

    'spb_data_src' : BEGIN 
        widgetData.data_src=event.value
        cluster_rapid_refresh, /PROD
    END  
     
    'spb_plot':BEGIN 
        widgetData.pl_type=event.value
        cluster_rapid_refresh, /PLOT
    END 

    'spb_spec':BEGIN 
        widgetData.spec = event.value
        cluster_rapid_refresh, /COLOR_LINE
    END

    'mpb_ch': BEGIN
        WIDGET_CONTROL, widgetData.mpb_ch, GET_VALUE=channel_event
        widgetData.ch=0
        FOR i=0,n_elements(channel_event)-1 do if channel_event(i) then $
            widgetData.ch=widgetData.ch+2^i
    END 

    'spb_sc4': BEGIN
        widgetData.prod_4sc=event.value
        cluster_rapid_refresh,  /SC4
    END 

    'ef_smooth': widgetData.smooth=event.value  

    'spb_coord':BEGIN
        widget_control, widgetData.grad_base, sensitive = 0
        widget_control, widgetData.coord_base,GET_VALUE=channel_event
        widgetData.grad_xyz=0 
        widgetData.grad_prj=0
        for i=0,n_elements(channel_event)-1 do if channel_event(i) then $
        widgetData.grad_xyz=widgetData.grad_xyz+2^i  
    END 

    'spb_vec':BEGIN
        widgetData.grad_prj = event.value
        cluster_rapid_refresh,  /SC4

        widget_control, widgetData.coord_base, sensitive = 0
        WIDGET_CONTROL,widgetData.grad_base,GET_VALUE=channel_event
        coord_set = [0,0, 0]
    END

    'spb_cal'      : BEGIN
        widgetData.cal = event.value
        ;polar_ceppad_refresh, /AVERAGE
    END 

    'spb_filter' : begin
        widgetData.filter = event.value
        cluster_rapid_refresh, /STYLE_PLOT
    END 

    'spb_average' : begin
        widgetData.average = event.value
        cluster_rapid_refresh, /AVERAGE
    END 

    'spb_seg_av':widgetData.seg_av = event.value

    ELSE: BEGIN
       WIDGET_CONTROL, widgetData.fld_mu, GET_VALUE=x
       widgetData.mus = x  
       WIDGET_CONTROL, widgetData.ef_smooth, GET_VALUE=x
       widgetData.smooth = x  
       WIDGET_CONTROL, widgetData.ef_pa_bin, GET_VALUE=x
       widgetData.pa_bin = x  
       widgetData=papco_panel_Event(event, widgetData)
       IF xregistered('cluster_rapid_panel') THEN $
         cluster_rapid_refresh, /AV_LBL
    END 

ENDCASE 
    
END


;******************************************************************************
FUNCTION cluster_rapid_slice_thumbs_panel, plotinfo, GROUP=GROUP, $
                                           ACTION=ACTION, MODULE = MODULE

module = 'cluster_rapid'

plotinfo = papco_slice_panel( plotinfo, $
           GROUP=GROUP, ACTION=ACTION, MODULE = module)

return, plotinfo

END
