;******************************************************************************
;* file cluster_rapid_slice.pro
;*      USER_3DD_Slice_cluster_rapid
;* end
;*
;* Listed above are those slices available for the thumb plots. 
;******************************************************************************

;******************************************************************************
;* PROCEDURE:     
;*      PRO vert_slice_cluster_rapid, panelNr, seconds, yValue
;* 
;* DESCRIPTION:  
;*      Draw a slice for cluster_rapid
;*      calling a separate window for the plot.
;*      slice_type in draw_cluster_rapid must be 
;*                                      set to +2 to enable this feature.
;*
;* INPUTS:       
;*      panelNr     an integer, the number of the panel for which the slice 
;*                  is to be drawn
;*      seconds     time of mouse click, a float
;*      yvalue      y-value of mouse click, a float
;*      
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      called by papco_DrawSLice (in $papco_PATH/papco_cursor.pro)
;*
;* MODIFICATION HISTORY:       
;*      December 1995, written by R.Friedel at Goddard
;******************************************************************************
PRO vert_slice_cluster_rapid, panelNr, seconds, yValue,  READDATA=READDATA, $
                              PANEL = PANEL

COMMON papco_vert_slice, slice_widget
COMMON plot_composer, widgetdata

; add in here any other common blocks you would need to make your plot data
; available. The common block included here is a dummy for illustration.
; papco uses the the getdatacall you set in papco_add_cluster_rapid to read the
; data you need for the slice. As long as you stay in one panel with the 
; mouse, the data is not read again. When you swittch panels, the new data is 
; read. 

common cluster_rapid_data, input_header, input_data
common cluster_rapid_slice_1, time, zmat, data, yarr, utitle, uytitle, uztit
common cluster_rapid
common mjdt, mjdt_start, mjdt_end    ;common time limit in mjdt

WIDGET_CONTROL,/HOURGLASS
thisPlot=widgetData.plotsDrawn(panelNr)

; plotinfo contains the full papco structure of plot information for
; the panel in which the mouse was clicked in. 
plotinfo =widgetData.plotinfos(panelNr)
plotdrawn=widgetData.plotsDrawn(panelNr)

; set the panel window for the slice to full window

!p.position=[0.1,0.1,0.9,0.9]

; select the index for the time chosen out of the time array

DiffTimes=ABS(time(*, 0)- seconds)     ; this sets timeIndex to the index which
result=MIN(DiffTimes, timeIndex)       ; is closest to 'seconds'


;choose data to plot
  type=plotinfo.typeVector
  data_source = control.data_src
  data_type = control.product
  plot_type =  control.pl_type
  detector = control.det
  sector = control.sect
  energy = control.ek
  
  CASE data_source OF 
  2: BEGIN ;RAL IF files
    IF_type = IF_names(data_type)
    CASE 1 OF
      (IF_type EQ 'IES_CLEANBM'): BEGIN 
        xarr = fltarr(n_det, 2)  & xtitle = 'detector #'
        xarr(*, 0) = findgen(n_det)+0.5 & xarr(*, 1) = xarr(*, 0)+1
        CASE plot_type OF
          0: BEGIN            ;spectrum - make energy v. detector array
              zmat =  transpose(one_data.data(sector,*,*))
              zmat = transpose(zmat)
              utitle = utitle+'!CSec '+varprt(sector)
              ztitle = uztit & uytitle='Energy (keV)'
              xlog = 0 & ylog = 1
          END
          1: BEGIN            ;roll - make sector v. detector array
              zmat =  transpose(one_data.data(*,*,energy-1))
              ;utitle = utitle+'!CEK '+ $
              ;strcompress( $
              ;string(input_header.CH_POSITIONS(0,energy-1,detector-1,0), $
              ;       input_header.CH_POSITIONS(1,energy-1,detector-1,0), $
              ;       format = "(f6.1,'-',f6.1)"), /remove_all) + $
              ;       '!CDet(x) v. Sec(y)'
              ztitle = uztit & uytitle='sector #'
              xlog = 0 & ylog = 0
          END
        ENDCASE
      END
      ELSE: BEGIN
        message,  'IF file type "' + IF_type+'" not handled here', /cont
        return
      END
    ENDCASE
  END
  3: BEGIN ;MPAE SCI files
    sci_type = sci_names(data_type)
    CASE input_header.DATA_CLASS OF
      'I3DD': BEGIN 
        ;select which species of I3DD
        result = strsplit(sci_type, ' ', /extract)
        species_str = result(1)
        CASE species_str OF
          'H': ion = 0
          'He': ion = 1
          'CNO': ion = 2
        ENDCASE

        CASE plot_type OF
          0: BEGIN            ;spectrum - make flux v. energy array
              xray = (yarr(*, 0)+yarr(*, 1))/2.0
              xtitle='Energy (keV)'
              xlog = 1 & ylog = 1
              xst = yarr(0, 0) & xen =yarr(n_elements(xray)-1, 1)
          END
          1: BEGIN            ;roll - make flux v. sector array
              xray = findgen(16)+1 
              xtitle='sector #'
              xlog = 0 & ylog = 1
              xst = 0 & xen = 18
          END
        ENDCASE
        yray = zmat(timeIndex, *)
        ytitle = uztit
        yst=plotdrawn.ZSCL_MIN  &  yen=plotdrawn.ZSCL_MAX 
      END
      ELSE: BEGIN
        message,  'SCI file type "' + sci_type+'" not handled here', /cont
        return
      END
    ENDCASE  
  END
  ELSE: BEGIN
    message,'Data source not handled here', /cont
    return
  END
  ENDCASE

; now plot data

  time_str=papco_cursor_time(seconds,panelNr)
  time_mjdt=papco_cursor_time(seconds,panelNr,/mjdt)

; set the panel window for the slice to full window
  IF keyword_set(PANEL) THEN panel = panel ELSE panel = [0, 1, 1]
  panelset, panel

  extra_plotPar={xrange:[xst,xen], yrange:[yst,yen], $
                 xlog:xlog, ylog:ylog, xticks:0, yticks:0, yminor:0, $
                 color:1, xtitle:xtitle, ytitle:ytitle}  
  erase
  plot, xray, yray, _extra=extra_plotPar, xticks = 0, yticks = 0
  right_side_label,panel,utitle+'!C'+time_str
   
  info1='You got to the default slice window'
  info2=string(time_str, yValue, format="('time: ',A20,'   Y-value: ',f9.2)")

; these calls put your info strings in the provided labels of the slice widget

  WIDGET_CONTROL, slice_widget.lbl_slice1, SET_VALUE=info1
  WIDGET_CONTROL, slice_widget.lbl_slice2, SET_VALUE=info2

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO levelzero_cluster_rapid, seconds, yvalue
;*                                     
;* DESCRIPTION:  
;*      This procedure calls the widget for level zero data access 
;*      for this plottype. 
;*      slice_type in draw_cluster_rapid must be set to +4 to enable this feature.
;*
;* INPUTS:       
;*      seconds           T90 time at which mouse is clicked
;*      yvalue            y-value at which mouse is clicked
;*
;* OUTPUTS: 
;*      none
;*
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      levelzero_cluster_rapid, seconds, yvalue
;*
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel at Goddard
;******************************************************************************
pro levelzero_cluster_rapid, seconds, yvalue

; The actual procedure is very dependent on what the level zero widget 
; application needs. An example is in $papco_MODULES/addpapco_swe_plots.pro.
; Provided as parameters to be passed to the new application, are the values
; of the mouse pointer when clicked. 
;
; The main widget remains active and can be swittched to (e.g. to get a new
; time from the graph) by using a "Parent" button in the new application
; which then executes the line
;            current_appl=xregistered('papco_main') 
; to swittch back to the papco main calling widget. 

; call the levelzero widget.  
;   cluster_rapid_levelzero, seconds, yvalue

; the following line is printed on your xterm as a check to see if papco
; correctly called this routine. If there is a problem make sure your naming 
; convention is O.K. : This routine MUST be called levelzero_cluster_rapid, 
; where cluster_rapid is the name of the panel you have set in papcoadd_cluster_rapid.

    print, 'got to the levelzero interface  ',seconds, yvalue

END


;******************************************************************************
;* PROCEDURE:     
;*      PRO WriteSlice_cluster_rapid, panelNr, seconds, yValue
;* 
;* DESCRIPTION:  
;*      Draw a Write Paneldata slice for cluster_rapid
;*      calling a seperate window for the widget
;*      slice_type in draw_ceppad must be set to +32 to enable this feature.
;*
;* INPUTS:       
;*      panelNr     an integer, the number of the panel for which the slice 
;*                  is to be drawn
;*      seconds     time of mouse click, a float
;*      yvalue      y-value of mouse click, a float
;*      
;* KEYWORDS:
;*      READDATA    if set the data needs to be re-read from file
;* 
;* CALLING SEQUENCE:
;*      called by papco_DrawSLice (in $papco_PATH/papco_cursor.pro)
;*
;* MODIFICATION HISTORY:       
;*      may 1997, written by R.Friedel
;******************************************************************************
PRO WriteSlice_cluster_rapid, panelNr, seconds, yValue,READDATA=READDATA

   COMMON PAPCO_WRITE_PANELDATA, sliceWidgetData
   COMMON PLOT_COMPOSER, widgetData

; THIS TEMPLATE ACTUALLY DOES NOTHING, ONLY THE DEFAULT WRITE SLICE 
; FUNCTIONALITY WILL WORK (WRITE TIME AND Y-VALUE OF MOUSECLICK TO FILE). THE 
; EXAMPLE HERE IS TO SHOW HOW ONE COULD ADD ADDITIONAL INFORMATION.

; add in here any other common blocks you would need to make your additional
; data available (ie ephemeris).

; papco uses the the getdatacall you set in papco_add_cluster_rapid to read the
; plotdata For your panel. As long as you stay in one panel with the 
; mouse, the data does not need to be read again (READDATA=0). When you swittch
; panels, new data needs to be read (READDATA=0). It's up to the user to read 
; in the data he needs here. 

;   common cluster_rapid_data, input_header, input_data
;   common cluster_rapid_ephem, cluster_rapid_ephem
;   common mjdt, mjdt_start, mjdt_end    ;common time limits in mjdt

   WIDGET_CONTROL,/HOURGLASS
   thisPlot=widgetData.plotsDrawn(panelNr)

; use the same typevector, swittch etc as is used in the call to your plot
; routine. thisPlot contains the full papco structure of plot information for
; the panel in which the mouse was clicked in. 

   thisPlot=widgetData.plotsDrawn(panelNr)
   type=thisPlot.typeVector
   getdatacall=thisPlot.get_data_call

;read the data if needed

;   if READDATA then begin 
;      callgood=execute(getdatacall)
;      r_cluster_rapid_ephem
;   endif

; select the index for the time chosen out of the time array

;   DiffTimes=ABScluster_rapid(.time - seconds)
                                        ; this sets timeIndex to the index 
;   result=MIN(DiffTimes, tidx)         ; closest to 'seconds'

; these strings are user-definable output strings to label the slice window.
; The "write Panelinfo" window papco supplies has 6 lines of user-definable
; text. 

   info1='User label 1'
   info2='User data  1'
   info3='User label 2'
   info4='User data 2'
   info5='User label 3'
   info6='User data 3'

; Time and y-value of mouse click are automatically put into labels.
; The calls below put your info strings in the provided user labels (max 6).
;
; NB NB NB:
; However, only info2, info4, info6 are written to file in the form
; info2+info4+info6.
;
; The idea is that info1 will have the labels for the data in info2, info3 the
; labels for the data in info4, and info5 the labels for the data in info6.
; Time and y-value of mouse click are written first, then info2+info4+info6
; followed by a user defined comment from the Write Paneldata window.

   WIDGET_CONTROL, sliceWidgetData.lbl_slice1, SET_VALUE=info1
   WIDGET_CONTROL, sliceWidgetData.lbl_slice2, SET_VALUE=info2
   WIDGET_CONTROL, sliceWidgetData.lbl_slice3, SET_VALUE=info3
   WIDGET_CONTROL, sliceWidgetData.lbl_slice4, SET_VALUE=info4
   WIDGET_CONTROL, sliceWidgetData.lbl_slice5, SET_VALUE=info5
   WIDGET_CONTROL, sliceWidgetData.lbl_slice6, SET_VALUE=info6

END

;******************************************************************************
;* PROCEDURE:
;*      PRO USER_3DD_Slice_cluster_rapid_Event, Event
;*
;* DESCRIPTION:
;*	This slice displays a 3DD slice through the data, when the user clicks
;*      in the drawing-widget. This procedure handles events for that window.
;*
;* INPUTS:
;*	Event	 	an XMANAGER-event
;*
;* KEYWORDS:
;* 	none
;*
;* CALLING SEQUENCE:
;*	called by XMANAGER
;*
;* MODIFICATION HISTORY:
;*     written November 2000, Reiner Friedel
;******************************************************************************
PRO USER_3DD_Slice_cluster_rapid_Event, Event

COMMON USER_3DD_Slice_cluster_rapid, slice_widget, det_pitch
COMMON plot_composer, widgetdata
COMMON slice_calls, slice_calls

widget_control,event.id,get_uvalue=ev
  
CASE ev OF 
    'ef_av' : BEGIN
        slice_widget.average = Event.value
    END 
    'spb_pitch': slice_widget.pitch = event.select
    'spb_label': slice_widget.label = event.select
    'pb_done' :  widget_control, slice_widget.base, /destroy
    'pb_help':   xdisplayfile, papco_helpfile('papco_slice.help'), $
                  title='papco online-help', group=event.top
    'pb_print':  begin ;loop through slices for all panels if needed
      slice_widget.rsl_lines=0
      for i=0,n_elements(slice_calls)-1 do begin
        close_dev=[i,n_elements(slice_calls)-1]
        papco_printmenu_event,slice_calls(i), CLOSE_DEV=close_dev
      endfor  
    end
    'pb_output': begin
      ;establish error handler
      if widgetData.default_trap eq 1 then $
        catch, error_status else error_status=0
      if error_status ne 0 then begin    ;call error handler
        papco_error_handler, $
          error_status, proc_call, '', [0,0,0], /NO_ERR_DRAW
        error_status=0 & catch, /CANCEL ;disable error catching
        return
      endif
      tmp_str=slice_widget.slice_call  &  tmp_str=str_sep(tmp_str,',')
      proc_call=tmp_str(0)  &  panelNr=fix(tmp_str(1))
      seconds=float(tmp_str(2))  &  yValue=float(tmp_str(3))
      CALL_PROCEDURE,proc_call,panelNr,seconds,yValue,/OUTPUT
      catch, /CANCEL        ;disable error catching 
    end  
    else : begin & end
ENDCASE 

END


;******************************************************************************
;* PROCEDURE:     
;*      PRO USER_3DD_Slice_cluster_rapid, panelNr, seconds, yValue
;* 
;* DESCRIPTION:  
;*      Draw a slice for cluster_rapid
;*      calling a separate window for the plot.
;*
;* INPUTS:       
;*      panelNr     an integer, the number of the panel for which the slice 
;*                  is to be drawn
;*      seconds     time of mouse click, a float
;*      yvalue      y-value of mouse click, a float
;*      
;* KEYWORDS:
;*      OVERPLOT  endables oberplot into same slice window. Lineplots only.
;*      READDATA  set to indicate data needs reading
;*      NOWIDGET  produce the plot only, no widget! Used for slice_thumb panel.
;* 
;* CALLING SEQUENCE:
;*      called by papco_DrawSLice (in $papco_PATH/papco_cursor.pro)
;*
;* MODIFICATION HISTORY:       
;*      November 2000, written by R.Friedel 
;******************************************************************************
PRO USER_3DD_Slice_cluster_rapid, panelNr, seconds, yValue, CLOSE=CLOSE, $
                                  OVERPLOT=OVERPLOT, READDATA=READDATA, $
                                  NOWIDGET=NOWIDGET, PANEL=PANEL, $
                                  NUTITLE=NUTITLE, _EXTRA=E

COMMON get_error, get_err_no, get_err_msg  
COMMON USER_3DD_Slice_cluster_rapid, slice_widget, det_pitch
COMMON PLOT_COMPOSER, widgetData
COMMON papco_color_names

; add in here any other common blocks you would need to make your plot data
; available. The common block included here is a dummy for illustration.
; papco uses the the getdatacall you set in papco_add_cluster_rapid to read the
; data you need for the slice. As long as you stay in one panel with the 
; mouse, the data is not read again. When you swittch panels, the new data is 
; read. 
  
COMMON cluster_rapid
COMMON cluster_rapid_data, input_header, input_data
COMMON cluster_rapid_slice              ;see p_cluster_rapid_spec for contents
COMMON mjdt, mjdt_start, mjdt_end       ;common time limit in mjdt
COMMON yscale, yscl			;man/auto yscaling
COMMON zscale, zscl			;man/auto zscaling
  
; plotinfo contains the full papco structure of plot information for
; the panel in which the mouse was clicked in. 
plotinfo =widgetData.plotinfos(panelNr)
plotdrawn=widgetData.plotsDrawn(panelNr)

;--- get the module's control structure
control = *plotinfo.USR_PTR1

if keyword_set(OVERPLOT) then overplot=OVERPLOT else overplot=0
if keyword_set(NOWIDGET) THEN BEGIN
    ;initialize slice widget structure
    IF n_elements(slice_widget) GT 0 THEN BEGIN
        tags = tag_names(slice_widget)
        idx = where(tags EQ 'NOWIDGET_CALL', c)
        IF c EQ 0 THEN wid_init = 0 ELSE wid_init = slice_widget.nowidget_call
    ENDIF ELSE wid_init = 0

    IF NOT wid_init THEN $
      slice_widget={average:fix(control.av_3dd), $
                    pitch:fix(control.pa_ctr_3dd), $
                    label:fix(control.pa_lbl_3dd), $
                    have_pitch:0, $
                    overplot:overplot, $
                    slice_call:'', $
	            panelNr:panelNr, $
                    lastPanelDrawn:-1, $
                    nowidget_call:1}
    goto, plotonly
ENDIF

; see if slice window needs to be closed --------------------------------------
if keyword_set(CLOSE) then begin
    if xregistered('USER_3DD_Slice_cluster_rapid') then $
      if n_elements(slice_widget) gt 0 then begin
	if slice_widget.base ne 0 then $
          widget_control, slice_widget.base, /destroy
        slice_widget.base=0
      endif
    return
ENDIF 

; if slice window exists, bring it in front. ELSE, create it ------------------
if not xregistered('USER_3DD_Slice_cluster_rapid') then begin
    
    slice_base=widget_base(column=1, group=widgetdata.main_base, $
                           title='3DD_Slice_cluster_rapid')
    
;-- setup draw area -----------------------------------------------------------
    draw_base=WIDGET_BASE(Slice_base, COLUMN=1, /FRAME)
    dw_Slice  =WIDGET_DRAW(draw_base, XSIZE=600, YSIZE=500)

;-- set up label fields for user data -----------------------------------------
    user_label_base=WIDGET_BASE(slice_base, COLUMN=1, /FRAME)
    lbl_slice1=WIDGET_LABEL(user_label_base,VALUE=strcenter('uservalues',80))
    lbl_slice2=WIDGET_LABEL(user_label_base,VALUE=strcenter('uservalues',80))
   
;-- set up some additional options for this slice -----------------------------
    Opt_base = WIDGET_BASE(Slice_Base, ROW=1, /FRAME)
    ef_av = CW_FIELD(Opt_base, /INTEGER, /RETURN_EVENTS, XSIZE = 4, $
                     TITLE = 'Frames to average (+/-)', $
                     VALUE = fix(control.av_3dd), $
                     UVALUE='ef_av')

;-- set up some toggle switch options for this slice --------------------------
    toggle_base=WIDGET_BASE(Slice_Base, /FRAME, ROW=1, /NONEXCLUSIVE)
    spb_pitch = WIDGET_BUTTON(toggle_base, UVALUE='spb_pitch', $
                             VALUE='Overlay Pitch Angle Contour')
    WIDGET_CONTROL, spb_pitch, SET_BUTTON = fix(control.pa_ctr_3dd)

;-- set up some toggle switch options for this slice --------------------------
    spb_label = WIDGET_BUTTON(toggle_base, UVALUE='spb_label', $
                             VALUE='Label cells with Pitch Angle')
    WIDGET_CONTROL, spb_label, SET_BUTTON = fix(control.pa_lbl_3dd)

    
;-- set up buttons for the Actions --------------------------------------------
    Butt_Base=WIDGET_BASE(Slice_Base, ROW=1, /FRAME)
    pb_help=WIDGET_BUTTON(Butt_Base, Value=strcenter('Help', 12), $
                          UVALUE='pb_help')
    pb_print=WIDGET_BUTTON(Butt_Base, Value=strcenter('Print', 12), $
                           UVALUE='pb_print')
    pb_output=WIDGET_BUTTON(Butt_Base, Value=strcenter('Output', 12), $
                           UVALUE='pb_output')
    pb_done=WIDGET_BUTTON(Butt_Base, Value=strcenter('Close', 12), $
                          UVALUE='pb_done')

    slice_widget={rsl_lines:0, $ 
                  average:fix(control.av_3dd), $
                  pitch:fix(control.pa_ctr_3dd), $
                  label:fix(control.pa_lbl_3dd), $
                  have_pitch:0, $
                  overplot:overplot, $
                  base:slice_base, $
                  dw_slice:dw_slice, $
		  lbl_slice1:lbl_slice1, $
                  lbl_slice2:lbl_slice2, $
                  pb_output:pb_output, $
                  slice_call:'', $
	          panelNr:panelNr, $
                  lastPanelDrawn:-1}
    
    widget_control, slice_base, /realize
    xmanager, 'USER_3DD_Slice_cluster_rapid', slice_base, /JUST_REG
    
ENDIF                           ; end construct slice window
    
IF keyword_set(NOPLOT) THEN return

WIDGET_CONTROL,/HOURGLASS

;--- now, we need to get the data --------------------------------------------
;    but we will only read the data, if the panel the user clicked is not the
;    panel that he last clicked into (in this case, the data is already read).
IF slice_widget.lastPanelDrawn NE panelNr THEN begin
    papco_get_slice_data, panelNr
    READDATA=1
endif else READDATA=0  
  
;--- prepare the drawing widget for plotting ----------------------------------
if !D.name eq 'X' then begin
    widget_control, slice_widget.dw_slice, get_value=drawindex
    wset, drawindex
endif  
  
; set the panel window for the slice to full window
!p.position=[0.1,0.1,0.75,0.95]  
IF keyword_set(PANEL) THEN panel = panel ELSE panel = [0, 1, 1]
panelset, panel

plotonly:

;--- store the current coordinate system --------------------------------------
old_x = !x  &  old_y = !y  &  old_p = !p
  
; NOW do plot into draw window!!! ---------------------------------------------

; select the index for the time chosen out of the time array

;make time of middle of spins:
t = (time(*, 0)+time(*, 1))/2.0

DiffTimes=ABS(t - seconds)           ; this sets timeIndex to the index with
result=MIN(DiffTimes, timeIndex)     ; time(timeIndex,0) closest to 'seconds'

;check for nearest time found. Needs to be within diff*2
diff = 60
if result gt diff*2 then begin
    message, 'No data found at: '+time_str,/cont
    xl=!p.position(0)  &  xr=!p.position(2)
    yb=!p.position(1)  &  yt=!p.position(3)
    plots, [xl, xr, xr, xl, xl], [yb, yb, yt, yt, yb], COLOR=1, /NORMAL
    goto, no_plot
endif    

one_data = data(timeIndex).data

IF slice_widget.average GT 0 THEN BEGIN 
    av_count = 0
    one_data(*) = 0
    FOR i = (timeIndex-slice_widget.average) > 0, $
      (timeIndex+slice_widget.average) < (n_elements(data)-1) DO BEGIN
        one_data = one_data + data(i).data
        av_count = av_count+1
    ENDFOR 
    IF av_count NE 0 THEN one_data = one_data/av_count
ENDIF

data_source = control.data_src
data_type = control.product
plot_type =  control.pl_type
detector = control.DETECTOR_AVERAGE_FROM_VAL > 1
sector = control.SECTOR_AVERAGE_FROM_VAL > 0
ek1 = control.ENERGY_AVERAGE_FROM_VAL > 1
nodata = 1e-20 & xticks = 0 & xminor = 0 & tickv = 0

n_ch =  input_header.NBANDS
n_sec = input_header.NSECTORS
n_det = input_header.NDETECTORS

utitle = 'Cluster '+varprt(control.sc_id+1)

CASE data_source OF 
    0: BEGIN                    ;RAL IF files
        IF_type = IF_names(data_type)
        CASE 1 OF
            (IF_type EQ 'IES_CLEANBM'): BEGIN 
                species = 0 ;for pitch overlay
                xarr = fltarr(n_det, 2)  & xtitle = 'detector #'
                xarr(*, 0) = findgen(n_det)+0.5 & xarr(*, 1) = xarr(*, 0)+1
                xtickv = findgen(n_det)+1
                xticks = n_det-1
                CASE plot_type OF
                    0: BEGIN          ;spectrum - make energy v. detector array
                        zmat = transpose(one_data(sector,*,*))
                        zmat = transpose(zmat)
                        utitle = utitle+'!CSec '+varprt(sector)
                        ztitle = uztit & uytitle='Energy (keV)'
                        xlog = 0 & ylog = 1 
                    END
                    1: BEGIN          ;roll - make sector v. detector array
                        zmat =  transpose(one_data(*,*,ek1-1))
                        utitle = utitle+'!CEK '+ $
                          strcompress(string(yarr(ek1-1, 0), yarr(ek1-1, 1), $
                          format = "(f6.1,'-',f6.1)"), /remove_all) + $
                               '!CDet(x) v. Sec(y)'
                        ztitle = uztit & uytitle='sector #'
                        xlog = 0 & ylog = 0 
                    END
                ENDCASE
                idx = where(zmat NE nodata, c )
                IF c NE 0 THEN zmat(idx) = zmat(idx) * 16
            END
            
            ELSE: BEGIN
                message, 'IF file type "' + IF_type+'" not handled here', /cont
                return
            END
        ENDCASE
    END

    1: BEGIN                    ;MPAE SCI files
        ch = input_header.CH_POSITIONS
        sci_type = sci_names(data_type)
    CASE 1 OF
      (strmid(sci_type, 0, 4) EQ 'I3DD'): BEGIN 
        ;select which species of I3DD
        result = strsplit(sci_type, ' ', /extract)
        species_str = result(1)
        CASE species_str OF
            'H': BEGIN ion = 0 & species = 2 & END  
            'He': BEGIN ion = 1 & species = 3 & END  
            'CNO': BEGIN ion = 2 & species = 4 & END
        ENDCASE
        xarr = fltarr(n_det, 2)  & xtitle = 'detector #'
        xarr(*, 0) = findgen(n_det)+0.5 & xarr(*, 1) = xarr(*, 0)+1
        CASE plot_type OF
          0: BEGIN            ;spectrum - make energy v. detector array
              zmat =  transpose(one_data(sector,*,*))
              zmat = transpose(zmat)
              utitle = utitle+'!CSec '+varprt(sector)
              ztitle = uztit & uytitle='Energy (keV)'
              xlog = 0 & ylog = 1
          END
          1: BEGIN              ;roll - make sector v. detector array
              zmat = fltarr(12, 16)
              FOR sec = 0, 15 DO FOR det = 0, 11 DO $
                zmat(det, sec) = one_data(det,energy,ion,sec)
              utitle = utitle+'!CEK '+ $
              strcompress( $
              string(ch(0,energy,detector-1,ion), $
                     ch(1,energy,detector-1,ion), $
                     format = "(f6.1,'-',f6.1)"), /remove_all) + $
                     '!CDet(x) v. Sec(y)'
              ztitle = uztit & uytitle='sector #'
              xlog = 0 & ylog = 0
          END
        ENDCASE
      END
      ELSE: BEGIN
        message,  'SCI file type "' + sci_type+'" not handled here', /cont
        return
      END
    ENDCASE  
  END
  ELSE: BEGIN
    message,'Data source not handled here', /cont
    return
  END
ENDCASE

zmin=plotdrawn.ZSCL_MIN  &  zmax=plotdrawn.ZSCL_MAX
zscl(panelNr,3) = plotdrawn.ZLINLOG

;do automatic scaing for full dynamic range
;papco_autorange,zmat,zmin,zmax, log=zscl(panelNr,3), exclude=0, nodata=nodata

yst=plotdrawn.YSCL_MIN * 0.99  &  yen=plotdrawn.YSCL_MAX * 1.01
xst = xarr(0, 0) &  xen = xarr(n_elements(xarr(*, 0))-1, 1)
papco_y_label, yst, yen, log=ylog
 
extra_plotPar={xrange:[xst,xen], yrange:[yst,yen], zrange:[zmin,zmax], $
               xlog:xlog, ylog:ylog, zlog:zscl(panelNr,3), $
               ztitle:ztitle, color:1,  $
               xtickformat:'',ztickformat:'papco_color_bar_log_ticks', $
               xtitle:xtitle, xticks:xticks, xminor:xminor, xtickv:xtickv}

if keyword_set(NOWIDGET) then begin
    extra_plotPar.xtickformat='noticks'
    extra_plotPar.xtitle=''
    extra_plotPar=create_struct(extra_plotPar, {ytickformat:'noticks'})
endif ELSE erase 
  
; should return the correct time.
time_str=papco_cursor_time(seconds,panelNr)
time_mjdt=papco_cursor_time(seconds,panelNr,/mjdt)

; do color plot and color bar plot. nodata specifies the "nodata" flag
; in your data. Here we do not plot axis and data seperate, as
; papco_plot_colorspec cannot be used as an overplot!
  
;check the routine papco_plot_colorspec in papco_XX\papco_lib\color_plot
;for all the possible options of papco_plot_colorspec.
;Hint: you might want to use the resx, resy keywords, which set the
;resolution to plot add. Without these, the routine will try to use
;device resolution, which for postscript may result in HUGE files!
;resx, resy are in data coordinates. If the axis is logarithmic, 
;resx, resy need to be too.
;resx=4000  &  resy=(alog10(yen)-alog10(yst))/n_channels
papco_plot_colorspec, zmat, xarr, yarr, nodata=nodata, _extra=extra_plotPar  

;now, overlay pitch angle info if required
IF (slice_widget.pitch OR slice_widget.label) AND plot_type THEN BEGIN
    header_buf = input_header
    
    ;make the detector/sector pitch angle maps

    IF NOT slice_widget.have_pitch THEN BEGIN 
        message, 'Getting Pitch Angle Info', /cont
        get_cluster_rapid_ies_pitch, control, p
        IF get_err_no NE 0 THEN GOTO, no_plot
        slice_widget.have_pitch = 1
        ;default pitch angle is pitch angle detector is looking at.
        ;if control.pa_bin is negative, sort into pitch angle particle
        ;is coming from!
        IF control.pa_bin LT 0 THEN BEGIN
            idx = where(p NE nodata, c)
            p(idx) = (180 - p(idx)) MOD 180.0
            det_pitch = p
        ENDIF
        det_pitch = p
    ENDIF 
 
    IF control.pa_bin LT 0 THEN utitle2 = '!CPA flow'
    this_det_pitch = transpose(det_pitch(*, *, timeindex))
    
    IF slice_widget.label THEN BEGIN ;label cells with pitch angle
        FOR k = 0, 8 DO BEGIN
            FOR j = 0, 15 DO BEGIN
                x = float(k+1)-0.2 & y = float(j+1)-0.1
                out_str = varprt(fix(this_det_pitch(k, j)))
                xyouts, x, y, out_str, color = violet, /data 
            ENDFOR
        ENDFOR
    ENDIF
    
    IF slice_widget.pitch THEN BEGIN ;draw contours of pitch angle
        levels = [30, 60, 90, 120, 150]
        c_labels = [varprt(30), varprt(60), varprt(90), $
                    varprt(120), varprt(150)]
        c_thick = 1.5 & c_charsize = 1.0
        x = findgen(9)+1 & y = findgen(16) +1

        contour, this_det_pitch, x, y, $
          xrange = [0.5, 9.5], yrange = [0.5, 16.65], $
          xstyle = 5, ystyle = 5, $
          LEVELS = levels, C_LABELS = c_labels, $
          color = violet, c_thick = c_thick, c_charsize = c_charsize, $
          /closed
    ENDIF     
ENDIF ELSE utitle2 = ''

no_plot:
utitle2 = ''

IF keyword_set(NOWIDGET) THEN BEGIN
    if NOWIDGET eq 2 then BEGIN ;last thumb also plots right side label
      panelset, PANEL  ;set to normal panel size!
      ;papco_color_bar, _extra=extra_plotPar
      utitle = utitle+utitle2
      right_side_label, panel, utitle, /rot90
      slice_widget.nowidget_call = 0
    ENDIF
    ;restore the coordinate-system
    !x = old_x  &  !y = old_y  &  !p = old_p
    return
endif  
 
papco_color_bar, _extra=extra_plotPar

; plot y-axis label at left of plot. Use scalable routine!  
left_side_label,panel,uytitle,/rot90
  
; Since here papco_colorbars are plotted, labels are rotated by 90
; degrees and then plotted. No allowance is made for overplots, as
; color plots cannot be an overplot.
right_side_label,panel,utitle,/rot90
  
; these two info strings are user-definable output strings to label the slice 
; window. To calculate the time use the start time in mjdt and the seconds 
; passed in. If your plot uses the papco philosophy then the time axis is in 
; running seconds since the start of the start day.
   
info1='This string not used yet'
info2=string(time_str, yValue, format="('time: ',A20,'   Y-value: ',f9.2)")
   
; these calls put your info strings in the provided labels of the slice widget
WIDGET_CONTROL, slice_widget.lbl_slice1, SET_VALUE=info1
WIDGET_CONTROL, slice_widget.lbl_slice2, SET_VALUE=info2 
  
;--- restore the coordinate-system --------------------------------------------
!x = old_x  &  !y = old_y  &  !p = old_p 
  
;--- keep old panel number- used for detecting panel changed of mouse ---------
slice_widget.lastPanelDrawn = panelNr   

END 



;******************************************************************************
;* PROCEDURE:
;*      PRO USER_Cluster_Boundary_cluster_rapid_Event, Event
;*
;* DESCRIPTION:
;*	This slice displays a 3DD slice through the data, when the user clicks
;*      in the drawing-widget. This procedure handles events for that window.
;*
;* INPUTS:
;*	Event	 	an XMANAGER-event
;*
;* KEYWORDS:
;* 	none
;*
;* CALLING SEQUENCE:
;*	called by XMANAGER
;*
;* MODIFICATION HISTORY:
;*     written November 2000, Reiner Friedel
;******************************************************************************
PRO USER_Cluster_Boundary_cluster_rapid_Event, Event 

COMMON USER_Cluster_Boundary_cluster_rapid, slice_widget 
COMMON plot_composer, widgetdata 
COMMON slice_calls, slice_calls 

widget_control, event.id, get_uvalue=ev 
  
CASE ev OF 
    'pb_done' :  widget_control, slice_widget.base, /destroy
    'pb_help':   xdisplayfile, papco_helpfile('papco_slice.help'), $
                  title='papco online-help', group=event.top

    'pb_reset': BEGIN
        FOR i = 0, 3 DO BEGIN 
            value = strcenter('Cluster '+varprt(i+1)+' time ',80)
            slice_widget.user_times(i).time_str = value
            WIDGET_CONTROL, slice_widget.user_times(i).lbl_slice, $
              SET_VALUE=slice_widget.user_times(i).time_str
        ENDFOR     
    END 
    
    'pb_call_timing': BEGIN
        print,  'inhere'
        ;get boundary times in required format for timing program, ie
        ;hr:mn:sc, in array  [(cl1),(cl2),(cl3),(cl4)]
        boundary_times = slice_widget.USER_TIMES.time_str
        ;assume each boundary_times array is in format year/mn/dy hr/mn/sc
        ; so time str is 11 thru 15 index of str
        boundary_times=strmid(boundary_times,11,15) 
        ; widget_control, slice_widget.base, /destroy
        papco_cluster_timing,  boundary_times
    END 

    'pb_print':  begin ;loop through slices for all panels if needed
      slice_widget.rsl_lines=0
      for i=0,n_elements(slice_calls)-1 do begin
        close_dev=[i,n_elements(slice_calls)-1]
        papco_printmenu_event,slice_calls(i), CLOSE_DEV=close_dev
      endfor  
    end

    'pb_output': begin
      ;establish error handler
      if widgetData.default_trap eq 1 then $
        catch, error_status else error_status=0
      if error_status ne 0 then begin    ;call error handler
        papco_error_handler, $
          error_status, proc_call, '', [0,0,0], /NO_ERR_DRAW
        error_status=0 & catch, /CANCEL ;disable error catching
        return
      endif
      tmp_str=slice_widget.slice_call  &  tmp_str=str_sep(tmp_str,',')
      proc_call=tmp_str(0)  &  panelNr=fix(tmp_str(1))
      seconds=float(tmp_str(2))  &  yValue=float(tmp_str(3))
      CALL_PROCEDURE,proc_call,panelNr,seconds,yValue,/OUTPUT
      catch, /CANCEL        ;disable error catching 
    end  
    else : begin & end
ENDCASE 

END 


;******************************************************************************
;* PROCEDURE:     
;*      PRO USER_Cluster_Boundary_cluster_rapid, panelNr, seconds, yValue
;* 
;* DESCRIPTION:  
;*      Draw a slice for cluster_rapid
;*      calling a separate window for the plot.
;*
;* INPUTS:       
;*      panelNr     an integer, the number of the panel for which the slice 
;*                  is to be drawn
;*      seconds     time of mouse click, a float
;*      yvalue      y-value of mouse click, a float
;*      
;* KEYWORDS:
;*      OVERPLOT  endables oberplot into same slice window. Lineplots only.
;*      READDATA  set to indicate data needs reading
;*      NOWIDGET  produce the plot only, no widget! Used for slice_thumb panel.
;* 
;* CALLING SEQUENCE:
;*      called by papco_DrawSLice (in $papco_PATH/papco_cursor.pro)
;*
;* MODIFICATION HISTORY:       
;*      November 2000, written by R.Friedel 
;******************************************************************************
PRO USER_Cluster_Boundary_cluster_rapid, panelNr, seconds, yValue, $
                                  CLOSE=CLOSE, $
                                  OVERPLOT=OVERPLOT, READDATA=READDATA, $
                                  NOWIDGET=NOWIDGET, PANEL=PANEL, $
                                  NUTITLE=NUTITLE, _EXTRA=E

COMMON USER_Cluster_Boundary_cluster_rapid, slice_widget
COMMON PLOT_COMPOSER, widgetData

; add in here any other common blocks you would need to make your plot data
; available. The common block included here is a dummy for illustration.
; papco uses the the getdatacall you set in papco_add_cluster_rapid to read the
; data you need for the slice. As long as you stay in one panel with the 
; mouse, the data is not read again. When you swittch panels, the new data is 
; read. 
  
COMMON cluster_rapid
COMMON cluster_rapid_data, input_header, input_data
COMMON cluster_rapid_slice_1, time, zmat, data, yarr, utitle, uytitle, uztit
COMMON mjdt, mjdt_start, mjdt_end     ;common time limit in mjdt


IF keyword_set(OVERPLOT) then overplot=OVERPLOT else overplot=0
  
; see if slice window needs to be closed --------------------------------------
IF keyword_set(CLOSE) THEN BEGIN
    IF xregistered('USER_Cluster_Boundary_cluster_rapid') THEN $
      IF n_elements(slice_widget) GT 0 THEN BEGIN
	IF slice_widget.base NE 0 THEN $
          widget_control, slice_widget.base, /destroy
        slice_widget.base=0
      endif
    return
ENDIF

; plotinfo contains the full papco structure of plot information for
; the panel in which the mouse was clicked in. 
plotinfo =widgetData.plotinfos(panelNr)
plotdrawn=widgetData.plotsDrawn(panelNr)
control=*plotinfo.USR_PTR1 
; if slice window exists, bring it in front. ELSE, create it ------------------
IF NOT xregistered('USER_Cluster_Boundary_cluster_rapid') THEN BEGIN 
    
    slice_base=widget_base(column=1, group=widgetdata.main_base, $
                           title='Cluster_Boundary_cluster_rapid')

;---set up some variables to "remember" which panel was pressed ---------------
    dummy = {BOUNDARY_TIMES_2, $
             SC_id:99, lbl_slice:0l, time:0.0d, time_str : ''}
    user_times = replicate(dummy, 4)

;-- set up label fields for user times ----------------------------------------
    user_label_base=WIDGET_BASE(slice_base, COLUMN=1, /FRAME)
    FOR i = 0, 3 DO BEGIN 
        value = strcenter('Cluster '+varprt(i+1)+' time ',80)
        user_times(i).time_str = value
        user_times(i).lbl_slice = WIDGET_LABEL(user_label_base, VALUE = value)
    END

;-- set up buttons for the Actions --------------------------------------------
    Butt_Base=WIDGET_BASE(Slice_Base, ROW=1, /FRAME)
    pb_help=WIDGET_BUTTON(Butt_Base, Value=strcenter('Help', 12), $
                          UVALUE='pb_help')
    pb_print=WIDGET_BUTTON(Butt_Base, Value=strcenter('Reset', 12), $
                           UVALUE='pb_reset')
    pb_output=WIDGET_BUTTON(Butt_Base, Value=strcenter('Do Timing', 12), $
                           UVALUE='pb_call_timing')    
    pb_done=WIDGET_BUTTON(Butt_Base, Value=strcenter('Close', 12), $
                          UVALUE='pb_done')
    slice_widget={CLUSTER_BOUNDARY_5, $
                  user_times:user_times, $
                  base:slice_base, $
	          panelNr:panelNr, $
                  lastPanelDrawn:-1}
    
    widget_control, slice_base, /realize
    xmanager, 'USER_Cluster_Boundary_cluster_rapid', slice_base, /JUST_REG
    
ENDIF                           ; end construct slice window

SC_id = control.sc_id

; should return the correct time.
time_str=papco_cursor_time(seconds,panelNr)
time_mjdt=papco_cursor_time(seconds,panelNr,/mjdt) 

idx = where(slice_widget.user_times.SC_id EQ SC_id, c)
IF c NE 0 THEN BEGIN
    slice_widget.user_times(SC_id).time_str = time_str
    slice_widget.user_times(SC_id).time = seconds
ENDIF ELSE BEGIN
    slice_widget.user_times(SC_id).SC_id = SC_id
    slice_widget.user_times(SC_id).time_str = time_str
    slice_widget.user_times(SC_id).time = seconds
ENDELSE

; these calls put your info strings in the provided labels of the slice widget
FOR i = 0, 3 DO WIDGET_CONTROL, slice_widget.user_times(i).lbl_slice, $
  SET_VALUE=slice_widget.user_times(i).time_str
END 


;******************************************************************************
;* PROCEDURE:     
;*      PRO USER_CLUSTER_4SC_3D_VIEW_CLUSTER_RAPID, panelNr, seconds, yValue
;* 
;* DESCRIPTION:  
;*      Call Steve Monk's 3D viewc routine
;*
;* INPUTS:       
;*      panelNr     an integer, the number of the panel for which the slice 
;*                  is to be drawn
;*      seconds     time of mouse click, a float
;*      yvalue      y-value of mouse click, a float
;*      
;* KEYWORDS:
;*      OVERPLOT  endables oberplot into same slice window. Lineplots only.
;*      READDATA  set to indicate data needs reading
;*      NOWIDGET  produce the plot only, no widget! Used for slice_thumb panel.
;* 
;* CALLING SEQUENCE:
;*      called by papco_DrawSLice (in $papco_PATH/papco_cursor.pro)
;*
;* MODIFICATION HISTORY:       
;*      November 2003, written by R.Friedel 
;******************************************************************************
PRO USER_CLUSTER_4SC_3D_VIEW_CLUSTER_RAPID, panelNr, seconds, yValue, $
                   CLOSE=CLOSE, OVERPLOT=OVERPLOT, READDATA=READDATA, $
                   NOWIDGET=NOWIDGET, PANEL=PANEL, NUTITLE=NUTITLE, _EXTRA=E

COMMON PLOT_COMPOSER, widgetData
COMMON USER_CLUSTER_4SC_3D_VIEW_CLUSTER_RAPID, slice_widget
COMMON mjdt, mjdt_start, mjdt_end       ;common time limit in mjdt

; see if slice window needs to be closed --------------------------------------
if keyword_set(CLOSE) then begin
    if xregistered('rave_main') then $
      if n_elements(slice_widget) gt 0 then begin
	if slice_widget.base ne 0 then $
          widget_control, slice_widget.base, /destroy
        slice_widget.base=0
      endif
    return
ENDIF ; if slice window exists, bring it in front. ELSE, create it -------------


tai1=utc2tai({mjd:mjdt_start.mjd, time:mjdt_start.t*1000})
tai2=utc2tai({mjd:mjdt_end.mjd, time:mjdt_end.t*1000})  

str = tai2utc( tai1,  /ecs)
space_index =  strpos( str, ' ')
str = strmid( str, 0, space_index) + '-' + strmid( str, space_index+1)
tcad_time_string__parse_date_time, str, dt, flag
start_time_str = dt_to_string( dt)

str = tai2utc( tai2,  /ecs)
space_index =  strpos( str, ' ')
str = strmid( str, 0, space_index) + '-' + strmid( str, space_index+1)
tcad_time_string__parse_date_time, str, dt, flag
stop_time_str = dt_to_string( dt)

; plotinfo contains the full papco structure of plot information for
; the panel in which the mouse was clicked in. 
plotinfo =widgetData.plotinfos(panelNr)

; should return the correct time.
str=papco_cursor_time(seconds,panelNr)
space_index =  strpos( str, ' ')
str = strmid( str, 0, space_index) + '-' + strmid( str, space_index+1)
tcad_time_string__parse_date_time, str, dt, flag
event_time_str = dt_to_string( dt)

IF NOT xregistered('rave_main') THEN BEGIN
    rave_main, start_time_str, stop_time_str, event_time_str, base, $
               RUN_ENV='PAPCO' 
    slice_widget = {base: base}
ENDIF ELSE BEGIN
    WIDGET_CONTROL, slice_widget.base, get_uvalue=state
    state.event_time = str_to_t( event_time_str)
    rave_update, state, /SET_NEW_EVENT_TIME
ENDELSE
END
