;******************************************************************************
;* FUNCTION:     
;*      FUNCTION  cluster_rapid_panel, aPlotInfo, GROUP=group, ACTION=ACTION
;* 
;* DESCRIPTION:  
;*	Open an editor widget for a panel, using the methods defined in the 
;*	file $papco_XX/papco_panel.pro .
;*	When the user presses the 'Done'- or the 'Cancel'-Button, the values 
;*	changed by the event-methods of papco_panel.pro are returned.
;*	The returned value is a structure containing those values and a flag,
;*	'canceled', which the caller may read out to check, whether the 
;*	window was closed using the 'Cancel' or the 'Done'-Button.
;*
;* INPUTS:       
;*      aPlotInfo       a structure of the papco_plotinfo-type
;*                      this structure is edited in the panel editor window.
;* KEYWORDS:
;*      GROUP           a WIDGET_BASE
;*	                when specified, this is used as GROUP_LEADER
;*      ACTION          a string
;*	                when specified, the 'Done'-Button is labeled with it
;*		        if not specified, the 'Done'-Button is labeled 'Done'
;*
;* OUTPUT:	 
;*      a default-structure of type papco_panel_editor_return. Though this
;*      type is nowhere defined, it is a virtually existing data-type. 
;*      All the functions in $papco_XX/papco.pro that edit plots or add new
;*      ones, expect the opened editor-windows to return a structure containing
;*      at least some fields. Typically this structure contains fields
;*      used by the papco_panel widgets, and some fields "inherited"
;*      from aplotinfo but modified by the user.
;*
;* CALLING SEQUENCE:
;*      newPlotData=papco_PANEL_TEMPLATE(aplotInfo, ACTION=, GROUP=)
;******************************************************************************
FUNCTION  cluster_rapid_panel, aPlotInfo, GROUP=group, ACTION=ACTION

COMMON cluster_rapid_panel_data, widgetData
COMMON cluster_rapid

;--- check if the current user is "friedel" by checking PAPCO_PATH
current = getenv('PAPCO_PATH') & pos=strpos(current, 'friedel') 
IF pos NE -1 THEN friedel = 1 ELSE friedel = 0

spawn,'domainname',result &  pos=strpos(result,'lanl')
if pos(0) ne -1 THEN friedel = 1
;friedel = 0

;--- if GROUP is not specified, the group-leader will be the desktop
IF NOT KEYWORD_SET(GROUP) THEN GROUP=0

;--- set the label for the action-button
IF NOT KEYWORD_SET(ACTION) THEN ACTION='Done'

;--- the routines used by papco_panel.pro define tags for a structure. Those
;    tags will be stored in the structure 'additionalWData'. When you realize 
;    the window, you have to add your own window-specific data to that  
;    structure and afterwards assign it to widgetData.
  additionalWData={canceled:1}

;--- create the base-window ---------------------------------------------------
;    since this can be a large widget, make a left and right side.
title = aPlotInfo.panelKind + ', ' + papco_module_version('cluster_rapid')
panel_base=papco_WIDGET_BASE(title=title, column=1, GROUP_LEADER=group, /MODAL)
panel_top =    widget_base(panel_base,column = 1, /base_align_center)
panel_middle = widget_base(panel_base,row = 1, /base_align_center)
panel_left   = widget_base(panel_middle,column=1, /base_align_center)
panel_right  = widget_base(panel_middle,column=1, /base_align_center)
panel_bottom = widget_base(panel_base,column = 1, /base_align_center)

;--- All papco_panel_add routines are papco panel widgets that can be included
;    in the panel editor or not. They provide functionality that papco 
;    supports. They dynamically extend the panel editors widget structure, and
;    all the events they generate are supported by papco_panel_event, 
;    transparent to the user. 

;--- the first part of the window will consist of the widget that -------------
;    controls the height of that panel. This is needed for all panel editors.
widgets=papco_panel_AddHeightWidgets(panel_top, aPlotInfo)
additionalWData=create_struct(widgets, additionalWData)
   
;
;---  <<<<<<<<<<  insert your widgets below here!  >>>>>>>>>
;
sc_id_base=CW_BGROUP(panel_left, sc_names, COLUMN=4, /EXCLUSIVE, /FRAME, $
                     UVALUE='spb_sc_id', LABEL_LEFT = 'Spacecraft #:', $
                     SET_VALUE=control.sc_id) 

;--- some choices for data source ---------------------------------------------
data_source=widget_base(panel_left, ROW=1)

data_names=['MPAe Bin','CDF', 'RAL IF', 'MPAe Sci', 'LANL Pitch']
data_base=papco_CW_BGROUP(data_source, data_names, ROW = 2, /EXCLUSIVE, /FRAME, $
                      LABEL_LEFT='Data source:', $
                      UVALUE='spb_data_source', BUTTON_IDS = data_ids, $
                      BUTTON_UVALUE = data_uvalues, $
                      SET_VALUE=control.data_src) 
have = ['RAL IF', 'MPAe Sci', 'LANL Pitch']
IF NOT friedel THEN BEGIN 
FOR i = 0,  n_elements(data_names)-1 DO BEGIN
    pos = strpos(have, data_names(i)) & index = where(pos NE -1,  c) 
    IF c LT 1 THEN widget_control, data_ids(i),  sensitive = 0
ENDFOR
ENDIF
 
cdf_names=['MPAe','Kiruna']
cdf_base=CW_BGROUP(data_source, cdf_names, COLUMN=2, /EXCLUSIVE, /FRAME, $
                     LABEL_LEFT='CDF:', $
                     UVALUE='spb_cdf_source', $
                     SET_VALUE=acontrol.ch)

data_behavior=widget_base(panel_left, ROW=1)  
reload_names=['Reload', 'Adopt']
reload_base=CW_BGROUP(data_behavior, reload_names, /FRAME, ROW=1, /EXCLUSIVE, $
                        LABEL_LEFT='Input:', UVALUE='spb_reload', $
                        SET_VALUE=acontrol.reload)

IF friedel THEN begin
force_names = ['Yes', 'No', 'Force']
force_base = CW_BGROUP(data_behavior, force_names, /FRAME, ROW=1, /EXCLUSIVE, $
                        LABEL_LEFT='Make:', UVALUE='spb_force', $
                        SET_VALUE=acontrol.make)

;--- smoothing of data --------------------------------------------------------
smooth_base=CW_FIELD(data_behavior, $
                       ROW=1, /INTEGER, /RETURN_EVENTS, $
                       TITLE = 'Smooth :', XSIZE = 3, $
                       UVALUE='ef_smooth', VALUE=acontrol.smooth)
ENDIF


;--- make a toggle base for data type selection, RAL, bin etc ----------------
prod_out_base=WIDGET_BASE(panel_left, column=1, /base_align_center) 
prod_top_base=WIDGET_BASE(prod_out_base, /base_align_center) 

if_base=papco_CW_BGROUP(prod_top_base, IF_names, COLUMN=5, $
                          /EXCLUSIVE, /FRAME, $ 
                          LABEL_TOP='RAL IF Data Choice:', $
                          UVALUE='spb_if', BUTTON_IDS = IF_ids, $
                          BUTTON_UVALUE = if_uvalues, $
                          SET_VALUE=acontrol.product) 
  
have = ['IIMS_HSPCT', 'IIMS_ISPCT_He', 'IIMS_ISPCT_CNO', 'IES_ESPCT', $
          'IES_CLEANBM', 'IES_CLEAN', 'IES_EPAD_16', 'IES_PEDESTAL']
FOR i = 0,  n_elements(IF_names)-1 DO BEGIN
    pos = strpos(have, IF_names(i)) & index = where(pos NE -1,  c) 
    IF c LT 1 THEN widget_control, IF_ids(i),  sensitive = 0
ENDFOR

bin_base=CW_BGROUP(prod_top_base, bin_names, COLUMN=5, /EXCLUSIVE, /FRAME, $
                      LABEL_TOP='Particle Species', $
                      UVALUE='spb_bin', $
                      SET_VALUE=acontrol.product <  4) 

sci_base =  papco_CW_BGROUP(prod_top_base, sci_names, COLUMN=7, $
                /EXCLUSIVE, /FRAME, $ 
                LABEL_TOP='MPAe Sci Data Choice:', $
                UVALUE='spb_sci', BUTTON_IDS = sci_ids, $
                BUTTON_UVALUE = sci_uvalues, $
                SET_VALUE=acontrol.product < (n_elements(sci_names)-1)) 

lanl_base =  papco_CW_BGROUP(prod_top_base, lanl_names, COLUMN=7, $
                /EXCLUSIVE, /FRAME, $ 
                LABEL_TOP='LANL Pitch Data Choice:', $
                UVALUE='spb_lanl', BUTTON_IDS = lanl_ids, $
                BUTTON_UVALUE = lanl_uvalues, $
                SET_VALUE=acontrol.product < (n_elements(lanl_names)-1))


;--- add choices for plot types -----------------------------------------------
plot_types_base = WIDGET_BASE(panel_left, ROW=1, /base_align_center) 

;types of plots
plot_names = ['Spectrum',  'Roll',  'LUT', $
              'Energy @ Pitch Angle',  'Pitch Angle @ Energy']
plot_base = papco_CW_BGROUP(plot_types_base, plot_names, $
                        COLUMN=1, /EXCLUSIVE, /FRAME, $
                        LABEL_TOP='Plot type:', BUTTON_IDS = plot_ids, $
                        UVALUE='spb_plot', $
                        SET_VALUE=acontrol.pl_type) 

IF friedel THEN begin
;other single toggle button choices
toggle_base=WIDGET_BASE(plot_types_base, /FRAME, COLUMN=1, /NONEXCLUSIVE)
spb_pitch_stats=WIDGET_BUTTON(toggle_base, UVALUE='spb_pitch_stats', $
                                VALUE='pitch samples / bin')
WIDGET_CONTROL, spb_pitch_stats, SET_BUTTON=acontrol.pa_bin

spb_epad_interp=WIDGET_BUTTON(toggle_base, UVALUE='spb_epad_interp', $
                                VALUE='EPAD energy interpolate')
WIDGET_CONTROL, spb_epad_interp, SET_BUTTON=acontrol.epad_ek

;type of 4 spacecraft plots 
sc4_names=['None', 'Standard Deviation', $
           'Total variance', 'Spacecraft Intercal', 'Gradiometer']
sc4_base = papco_CW_BGROUP(plot_types_base, sc4_names, $
                        COLUMN=1, /EXCLUSIVE, /FRAME, $
                        LABEL_TOP='Four SC products:', $
                        UVALUE='spb_sc4', BUTTON_IDS = sc4_ids, $
                        SET_VALUE=acontrol.prod_4sc)
widget_control,sc4_ids(2), sensitive = 0
ENDIF 

;--- make toggle base for energy, channel, etc selection ----------------------
sel_out_base=WIDGET_BASE(panel_left, ROW=1, /base_align_center) 
sel_top_base=WIDGET_BASE(sel_out_base, /base_align_center) 

  ;select for spec - detector/sector
  spec_base = WIDGET_BASE(sel_top_base, ROW = 1, /base_align_center, /FRAME)
  sl_det1=WIDGET_SLIDER(spec_base, XSIZE=250, $
		       /DRAG, TITLE='detector #', $
		       VALUE=acontrol.det < 12 > 1, $
                       UVALUE='sl_det', MINIMUM=1, MAXIMUM=12)
  sl_sec=WIDGET_SLIDER(spec_base, XSIZE=250, $
		       /DRAG, TITLE='sector #', $
		       VALUE=acontrol.sect < 16 > 1, $
                       UVALUE='sl_sec', MINIMUM=1, MAXIMUM=16)

  ;select for roll -  detector/energy
  ek_base = WIDGET_BASE(sel_top_base, ROW = 1, /base_align_center, /FRAME)
  sl_det2=WIDGET_SLIDER(ek_base, XSIZE=250, $
		       /DRAG, TITLE='detector #', $
		       VALUE=acontrol.det < 12 > 1, $
                       UVALUE='sl_det', MINIMUM=1, MAXIMUM=12)
  sl_ek =WIDGET_SLIDER(ek_base, XSIZE=250, $
		       /DRAG, TITLE='energy channel #', $
		       VALUE=acontrol.ek < 8 > 1, $
                       UVALUE='sl_ek', MINIMUM=1, MAXIMUM=8)

  ;select for pitch - energy, pitch angle
  pitch_base = WIDGET_BASE(sel_top_base, ROW = 1, /base_align_center, /FRAME)
  sl_pa =WIDGET_SLIDER(pitch_base, XSIZE=250, $
		       /DRAG, TITLE='pitch angle bin #', $
		       VALUE=acontrol.pa < 19 > 1, $
                       UVALUE='sl_pa', MINIMUM=1, MAXIMUM=19)

  sl_ek =WIDGET_SLIDER(pitch_base, XSIZE=250, $
		       /DRAG, TITLE='energy channel #', $
		       VALUE=acontrol.ek < 8 > 1, $
                       UVALUE='sl_ek', MINIMUM=1, MAXIMUM=8)

;--- now, add the widgets for choosing the data units -------------------------
widgets=papco_PANEL_AddUnits_Widgets(panel_left, aPlotInfo)
additionalWData=create_struct(widgets, additionalWData)  

;
;---  <<<<<<<<<<  insert your widgets above here!  >>>>>>>>>
;

;--- some panels might need to read/input to text files (event, text ----------
;    module). So a generic papco widget is supplied for this. 
;  widgets=papco_panel_AddFile_Widgets(panel_left, aPlotInfo)
;  additionalWData=create_struct(widgets, additionalWData)   

;--- add the papco_panel widget for making a batch of all data dates ----------
;  IF friedel THEN BEGIN
;    widgets=papco_panel_AddBatch_Widgets(panel_left, aPlotInfo,  $
;      REDO = ['Orig', 'MSF', 'DAT', 'RAL_IF', 'CCR C1', 'CCR C2', 'CCR C3', $
;              'CCR C4', 'CCR All e', 'CCR All p', 'CCR BM e'])
;    additionalWData=create_struct(widgets, additionalWData)
IF friedel THEN BEGIN
;--- add the papco panel widget for specifying own plot description label -----
  widgets=papco_panel_AddDescr_Widgets(panel_right, aPlotInfo, /VERTICAL)
  additionalWData=create_struct(widgets, additionalWData)
ENDIF
;--- add the papco panel widget for choosing the color bar for color plots ----
  widgets=papco_panel_Addcolorbar_Widgets(panel_right, aPlotInfo, /VERTICAL)
  additionalWData=create_struct(widgets, additionalWData)
;--- add the  papco panel widget for choosing the plot-style for line plots ---
  widgets=papco_panel_add_PlotStyleButtons(panel_right, aPlotInfo, 0,/VERTICAL)
  additionalWData=create_struct(widgets, additionalWData)

  scaling_base = widget_base(panel_right,  ROW = 1, /base_align_center)
;--- add the papco panel widget for choosing the y-axis-ranges ----------------
  widgets=papco_panel_AddYScale_Widgets(scaling_base, aPlotInfo,/VERTICAL)
  additionalWData=create_struct(widgets, additionalWData)
;--- add the papco panel widgets for choosing the z-axis-ranges ---------------
  widgets=papco_panel_AddZScale_Widgets(scaling_base, aPlotInfo,/VERTICAL)
  additionalWData=create_struct(widgets, additionalWData)
;--- finally, the Actionbuttons (help, cancel, done) are added to the ---------
;    editor window. This is needed for all panel editors. If you want to 
;    make use of the configure feature, then add the keyword CONFIG
  widgets=papco_panel_AddActionButtons(panel_bottom, Action, aPlotInfo, $
                                       /CONFIG) 
  additionalWData=create_struct(widgets, additionalWData)

;--- now, create the structure that keeps the panel-window data. You always
;    need to specify the help-file, and your structure must contain the
;    'additionalWData'. You should create the helpfile in the papco_interface
;    directory of your module (template is in the cluster_rapid example module)
;    Also add the ID's of your widgets so that you can perform widget_control
;    actions in the events routine or in a refresh routine if needed.
  thisWidgetsData={helpfile:'cluster_rapid_panel.help', $
                   ioptions: aplotinfo.ioptions, $
                   typevector:aPlotInfo.typevector, $
                   filename:aPlotInfo.filename, $
                   cdf_base:cdf_base, $
                   if_base:if_base, bin_base:bin_base, $
                   lanl_base:lanl_base, sci_base:sci_base, $
                   spec_base:spec_base, ek_base:ek_base, $
                   pitch_base:pitch_base, plot_base:plot_base, $
                   plot_ids:plot_ids}
  widgetData=papco_panel_CreateWidgetData(thisWidgetsData, additionalWData)

;--- realize the window -------------------------------------------------------
  cluster_rapid_panel_refresh,  /ALL
  papco_Panel_Realize, panel_base, widgetData
  papco_xmanager, 'cluster_rapid_panel', panel_base
  
;--- the returned data consists of the standard-data inherited from
;    papco_panel.pro and your new data. Stick to the tags provided in
;    aPlotInfo, there are ample variables for the user
  returnData={typevector:widgetData.typevector,$
              ioptions:widgetData.ioptions}
  additionalRData=papco_panel_ReturnData(widgetData)

  return, create_struct(ReturnData, additionalRData)
   
END

;******************************************************************************
;* PROCEDURE:     
;*      PRO cluster_rapid_panel_event_refresh 
;* 
;* DESCRIPTION:  
;*      Takes refreesh actions on panel widgets controlled by keywords.
;*
;* INPUTS:       
;*      none
;*                  
;* KEYWORDS:
;*      
;* 
;* CALLING SEQUENCE:
;*      cluster_rapid_panel_refresh, /keywords
;******************************************************************************
PRO cluster_rapid_panel_refresh,  ALL = ALL,  PROD = PROD,  PLOT = PLOT, $
                                  SC4 = SC4

COMMON cluster_rapid_panel_data, widgetData

IF keyword_set(ALL) THEN BEGIN
    PROD = 1 & PLOT = 1 & SC4 = 1
ENDIF

IF keyword_set(PROD) THEN BEGIN
    prod_bases = [widgetData.bin_base, widgetData.cdf_base, widgetData.if_base, $
                  widgetData.sci_base, widgetData.lanl_base]
    CASE widgetData.data_src OF
        0: BEGIN 
            map = [1, 0, 0, 0, 0]
            plot_ids = [1, 1, 0, 0, 0]
        end
        1: BEGIN 
            map = [0, 1, 0, 0, 0]
            plot_ids = [1, 1, 0, 0, 0]
        END
        2:  BEGIN 
            map = [0, 0, 1, 0, 0]
            plot_ids = [1, 1, 1, 0, 0]
        END
        3:  BEGIN 
            map = [0, 0, 0, 1, 0]
            plot_ids = [1, 1, 0, 0, 0]
        END
        4:  BEGIN 
            map = [0, 0, 0, 0, 1]
            plot_ids = [0, 0, 0, 1, 1]
        END
    ENDCASE 
    FOR i = 0, n_elements(plot_ids)-1 DO $
      IF plot_ids(i) THEN widget_control, widgetData.plot_ids(i), sensitive=1 $
        ELSE widget_control, widgetData.plot_ids(i), sensitive=0
    FOR i = 0, n_elements(map)-1 DO widget_control, prod_bases(i), MAP = map(i)
    cluster_rapid_panel_refresh, /PLOT
ENDIF

IF keyword_set(PLOT) THEN BEGIN
    CASE widgetData.pl_type OF
        0: BEGIN 
            widget_control, widgetData.spec_base, MAP=1
            widget_control, widgetData.ek_base, MAP=0
            widget_control, widgetData.pitch_base, MAP=0
            widgetData.yLinLog = 1
       end
        1: BEGIN 
            widget_control, widgetData.spec_base, MAP=0
            widget_control, widgetData.ek_base, MAP=1
            widget_control, widgetData.pitch_base, MAP=0
            widgetData.yLinLog = 0
        END
        2:  BEGIN 
            widget_control, widgetData.spec_base, MAP=0
            widget_control, widgetData.ek_base, MAP=0
            widget_control, widgetData.pitch_base, MAP=0
        END
        3:  BEGIN 
            widget_control, widgetData.spec_base, MAP=0
            widget_control, widgetData.ek_base, MAP=0
            widget_control, widgetData.pitch_base, MAP=1
            widgetData.YLINLOG = 1
        END
        4:  BEGIN 
            widget_control, widgetData.spec_base, MAP=0
            widget_control, widgetData.ek_base, MAP=0
            widget_control, widgetData.pitch_base, MAP=1
            widgetData.YLINLOG = 0
        END
    ENDCASE 
    papco_panel_Refresh, widgetData, /YSCALE
ENDIF  

IF keyword_set(SC4) THEN BEGIN
    IF widgetData.prod_4sc EQ 0 THEN BEGIN 
        cluster_rapid_panel_refresh,  /PLOT
        widget_control, widgetData.plot_base, sensitive=1
    ENDIF ELSE BEGIN 
        old = widgetData.pl_type
        widgetData.pl_type = 1
        cluster_rapid_panel_refresh,  /PLOT
        widgetData.pl_type = old
        widget_control, widgetData.plot_base, sensitive=0
    ENDELSE 
ENDIF

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO cluster_rapid_panel_event, event
;* 
;* DESCRIPTION:  
;*      This processes events from the XMANAGER.
;*      As this example does not extend the basic functionality of plot-
;*      editors, it directly calls the default-handler, papco_PANEL_Event
;*
;* INPUTS:       
;*      event            an XManager event
;*                  
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      called by XMANAGER
;******************************************************************************
PRO cluster_rapid_panel_event, event
  
COMMON cluster_rapid_panel_data, widgetData

WIDGET_CONTROL,GET_UVALUE=uval, event.id

; in this editor, we only need to process the channel slider events. events
; other than those from your own widgets need to be passed on to 
; 'papco_panel_Event', which handles the events generated
; by the papco-provided panel widgets. Your own added panel widgets events
; need to be processed here:  you should check 'event' for your events and call
; 'papco_panel_Event', if event was not generated by your own panel widgets.
; The best way to learn is to look at some of the other module's panel editors.

CASE uval OF 
 
    'spb_pitch_stats': widgetData.pa_bin=event.select
    'spb_epad_interp': widgetData.epad_ek=event.select
    'spb_sc_id' : widgetData.sc_id=event.value
    'spb_reload' : widgetData.reload=event.value
    'spb_force'  : widgetData.make=event.value
    'spb_bin' : widgetData.product=event.value
    'spb_if'  : widgetData.product=event.value
    'spb_sci' : widgetData.product=event.value
    'spb_lanl': widgetData.product=event.value
    'spb_data_source' : BEGIN 
        widgetData.data_src=event.value
        cluster_rapid_panel_refresh,  /PROD
    END 
    'spb_cdf_source': widgetData.ch=event.value
    'spb_plot':BEGIN 
        widgetData.pl_type=event.value
        cluster_rapid_panel_refresh,  /PLOT
    END 
    'spb_sc4': BEGIN
        widgetData.prod_4sc=event.value
        cluster_rapid_panel_refresh,  /SC4
    END 
    'sl_det': widgetData.det=event.value
    'sl_ek': widgetData.ek=event.value
    'sl_pa': widgetData.pa=event.value
    'sl_sec': widgetData.sect=event.value
    'ef_smooth': widgetData.smooth=event.value
    ELSE: widgetData=papco_panel_Event(event, widgetData)

ENDCASE 
    
END
