;******************************************************************************
;*  NAME:        a_cluster_rapid
;*
;*  DESCRIPTION: example routine for access to cluster_rapid module's data
;*               For use inside the PAPCO environment 
;*               (papco is compiled, required modules are loaded)
;*               
;*                                  Uses MJDT time
;*                                  papco read philosophy
;*                
;*               NB!! Don't modify this routine, copy to own space,
;*                    then alter at will!
;*
;*  INPUT:       none
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  KEYWORDS:    VERBOSE  -  enables print comments, save/restore comments
;*               T1       -  start date/time in any PAPCO format, string
;*               T2       -  end date/time in any PAPCO format, string
;*         
;*  Enter a time in one of these formats: date time, date
;*  where time is in one of those formats: h:m:s, h:m
;*  and date is in one of those formats: m/d/y, doy/y, d.m.y, m/d/y, yyyymmdd
;*  where d is day and doy is day of year, h:m:s is the time
;*  (year > 1990 if you use T90 format)
;*  e.g: 1/90 14:11, 1/1990 1:1:1, 350/1991 01:00:03
;*       15.1.90 14:11, 1/15/1990 15:3:1                  
;*
;*  DATE:        March 2003
;*
;*  AUTHOR:      R. Friedel
;*
;*  CHANGES:     
;******************************************************************************
PRO a_cluster_rapid, T1 = T1, T2 = T2

;first, we need to call papco once so that it initializes itself. This
;sets up a bunch of common blocks, structures and environmental
;variables that are needed by various modules.

papco, /set_only ;only initializes papco then exists.

IF NOT keyword_set(T1) THEN BEGIN
    message, 'No start time (T1) defined', /cont & return
ENDIF 

IF NOT keyword_set(T2) THEN BEGIN
    message, 'No end time (T2) defined', /cont & return
ENDIF 

;PAPCO sets the start and end time in a common block, which controls
;the time range for all read routines and all plot routines in all
;papco modules that are loaded.
COMMON mjdt, mjdt_start, mjdt_end

;The time format in mjdt_start, mjdt_end is modified Julian date and
;seconds of day, in a structure:
; mjdt_start = {MJD:0l, T:0.0}

;PAPCO provides a set of time conversion routines that can convert
;between string times to papco time and a variety of other formats:
;look in ../papco/papco_lib/convert_time.pro

;for the example here, if the input string is in any of the PAPCO time
;formats, use the function convert_timeToSeconds to convert to
;mjdt_start, mjdt_end. This function returns a structure containing
;the tags err, value and errtext. If err is zero, the time string was
;recognized, and the converted time is in value. If err eq -1 there
;was an error, the error message is in errtext. This errtext the also
;contains a list of the "allowed" formats...

result = convert_timeToSeconds(T1, /T90, /MJDT)
IF result.err EQ 0 THEN mjdt_start =result.value ELSE BEGIN
    message, 'Error in start time (T1)', /cont
    print,  result.errtext
    return
ENDELSE

result = convert_timeToSeconds(T2, /T90, /MJDT)
IF result.err EQ 0 THEN mjdt_end =result.value ELSE BEGIN
    message, 'Error in end time (T2)', /cont
    print,  result.errtext
    return
ENDELSE 

;if you don't like the papco time formats you can also use the time
;routines provided under
;../papco/papco_lib/cdssw/time/
;these will come in handy in any case, since a lot of the papco
;modules use the TAI format internally - international atomic time.
;There's an informative readme file in that directory too.
;Here's a quick way of converting mjdt_start, mjdt_end to TAI:
tai1=utc2tai({mjd:mjdt_start.mjd, time:mjdt_start.t*1000})
tai2=utc2tai({mjd:mjdt_end.mjd, time:mjdt_end.t*1000})    

;Once the time common block is set, the module routines can be called.

;In general, the module's read routine will return data for the time
;set in common mjdt. You'll need to look at that read routine to see
;what parameters it needs to be called with. This can be as little as
;nothing, to something like a Satellite ID, or a more complete
;plotinfo structure( see explanation below...). In the read routine
;you'll also find the common block that the data is returned in, in
;general it looks something like:
COMMON cluster_rapid_data, input_header, input_data

;where  input_header is a structure containing info about the data
;       input_data is an array of structures containing the data
;investigate by using  help,input_header,/st

;This data might not be in the format you want, as it generally will
;need some calibration/conversion etc applied to it. Often this is
;done by a module's plot routine. 
;A lot (but possibly not all) modules now support "getting" data from
;the plot routine, if the keyword OUTPUT is set to 2. basically this
;calls the plot routine, and returns from it just before the actual
;plot is made, at which point all the plot's data is left in a common
;block. 
;Again, you need to look at the plot routine to see what the call is,
;and what the common block is. may plot routines now use the same
;plotinfo structure as the read routine as input. The data is returned
;in a common block that looks something like this:
COMMON cluster_rapid_slice, xut1, xut2, time, zmat, data, yarr, $
                            utitle, title, uytitle, uztit, mag_fgm, aux, fgm, $
                            ndat, nodata, s1, s2, d1, d2, e1, e2, p1, p2

;The same common block is also used by the slice routines...


;In case this isn't confusing enough, one can combine both of the
;above steps. PAPCO internally keeps a list of all the plotinfo
;structures for each plot it needs to make, and calls one interface
;routine when it needs to make a plot. That routine first calls the
;read routine, and then the plot routine. You can call it yourself,
;with the keyword OUTPUT set to 2. This routine is always called
;draw_module_name, and always has the same set of input parameters and
;keywords. It resides in the modules
;../papco_interface/papcoadd_module_name.pro file:
;
;pro draw_cluster_rapid, panel, PlotInfo, orbitNo, $
;                   get_data_call, slice_type, IsRead, $
;                   OUTPUT=OUTPUT

;when using this routine "manually" many of the above parameters are
;not important. panel is a vector specifying the panel position of the
;plot, set it to a default (panel=[0,1,1]). PlotInfo is the
;panel's structure containing all it's plot information, that needs to
;be set (see below). orbitNo is used if the module you're using allows
;time specification by orbit number, generally that is not the case,
;just set it to zero. get_data_call, slice_type, are return
;values which are not needed further but need to be in the call. 
;IsRead needs to be set to zero, meaning the data is not read. if
;successful, it is returned set to 1, and if you call the routine
;again with isread set to 1 it won't read the data again as it is
;already in the read common block. Useful if you're re-reading the
;same data often but getting different plot data made... use with
;care, or always set to zero to force a read.
;OUTPUT is the keyword than needs to be set to 2!


;now, the explanation of the PlotInfo structure. This is a general
;structure that represents all the data needed to specify the plot
;the module needs to make. This structure contains tags used by PAPCO,
;and tags used by the user who wrote the module. It is this structure
;that is modified by the module's panel editor (the widget you get with
;all the choices for a given module). So by setting this structure
;correctly, you can control exactly what the data is that is going to
;be plotted, in exactly the same manner as you would by clicking the
;options in the widget.

;So in general, you need to "make" a plotinfo structure, set it
;according to what data you want, and then call the draw_module_name
;routine.

;The information used by polar ceppad is set in cluster_rapid_commons -
;here we need to set the information in the cluster_rapid_control structure.
COMMON cluster_rapid

;To make a plotinfo structure there's a papco routine you can use:
new_plotinfo = papco_getplotinfostruct()
new_control = cluster_rapid_control

;So for the cluster_rapid module you might do something like this:
new_control.sc_id = 0        ;Cluster 1
new_control.data_src = 0     ;ral if data
new_control.product = 0      ;IES_ESPCT
new_control.cal = 3          ;Differential flux
new_control.reload = 0       ;always reload data
new_control.prod_4sc = 0     ;normal singel sc data
new_control.pl_type = 0      ;no pitch
new_control.spec = 1         ;full spectra
new_plotinfo.USR_PTR1 = ptr_new(new_control)


;Finally you're set to read / make the data!

panel = [0, 1, 1] & orbitNo = 0 & IsRead = 0 & slice_type = 0
;plotinfo we set above
output = 2 ;"tells" the module to make the plot data only, then to return

draw_cluster_rapid, $
             panel, new_plotinfo, orbitNo, get_data_call, slice_type, IsRead, $
             OUTPUT = output

;you can check if the above call was successful. All module read
;routines set an error common block that can be interrogated:
COMMON get_error, get_err_no, get_err_msg

;get_err_no is zero if successful, non-zero if not, then get_err_msg
;contains the error message.
IF get_err_no NE 0 THEN BEGIN
    message, get_err_msg, /cont
    return
ENDIF

;if you get this far then the data was read and resides in the common
;block from the plot routine. You can interrogate this by:
help, time, yarr, zmat, data, extra_plotPar, utitle, uytitle, uztitle

;time contains the time array, "standard" is to have time in seconds, starting
;from zero ALWAYS. You need to add in the start time to get the actual
;time. Some modules return a one-dimension array, others a 2D array
;containing start and end times.
time = time+tai1
print, ''
print, 'Startime: '
print,  tai2utc(time(0), /ecs)  ;prints the start time.

;ymat can be a linear array or a 2D array depending on what data you
;selected. generally the first dimension is time, the second data
;channel number.

;utitle, uytitle are strings that papco would have used to label the
;panel, utitle describes the data in short, uytitle is the y-axis
;label, Some other modules might also include the z\uztitle, label for
;the color bar. yarr here is info for the energies of the channels.


;Now you're done, have the data you need in common blocks, and can do
;what you want with it...


END 
