;******************************************************************************
;* pro based on Jim Roederer's code to make pitch angles for IES.
;* Inputs needed are FGM mag field and spin axis GSE lat and lon angles.
;*
;* Here we read a full IES CLEAN file for the day to find out what
;* data is available, then calculate ALL pitch angles for all detector /
;* sector combinations once and save to file.
;******************************************************************************
PRO cluster_rapid_ies_pitch, plotinfo

COMMON mjdt, mjdt_start, mjdt_end
COMMON cluster_rapid_data, input_header, input_data
COMMON get_error, get_err_no, get_err_msg  
COMMON cluster_rapid

IF NOT (papco_check_data_env('CLUSTER_RAPID_PITCH', PATH=path)) THEN return

;get the module's control structure
control = *plotinfo.USR_PTR1
datestr = plotinfo.filename
sc_id = control.SC_ID

;first, see if pa file for this day already exists.
fln = path+'C'+varprt(sc_id+1)+'_'+datestr+'_IES_PITCH.idl'

r = findfile(fln, count = c)

IF c EQ 1 THEN BEGIN            ;file found, read
    message, 'Reading', /info & print, '  ', fln
    restore, file = fln
    return
ENDIF     

message, 'constructing pitch angle for each det/sec (slow)', /info

;have to make pitch file. Do for whole day of IES data!
orig_mjdt_start = mjdt_start & orig_mjdt_end = mjdt_end

r = convert_timeToSeconds(datestr, /T90, /MJDT)
mjdt_start = r.value & mjdt_end = mjdt_start & mjdt_end.mjd = mjdt_start.mjd+1

;read the IES_EPAD_16 file for this spacecraft to get times of data
idx = where(if_names EQ 'IES_EPAD_16')
control.product =  idx(0)
control.pl_type = 0
control.data_src = 0
new_plotinfo = papco_getplotinfostruct()
new_plotinfo.USR_PTR1 = ptr_new(control)
r_cluster_rapid, new_plotinfo
IF get_err_no NE 0 THEN return

;make time array of midpoints of spins, get mag and aux data for these times
tai1=utc2tai({mjd:mjdt_start.mjd, time:mjdt_start.t*1000})
time = (input_data.time+input_data.endtime)/2.0 - tai1

ndat = n_elements(time)

;NOTE: Both get_cluster_aux and get_cluster_mag return data on the
;time basis of the input time, so all is on data time res from the beginning!

; --- get spin axis GSE lat and lon angles from cluster cdf aux files
;     use papco _cdf read  for this: aux(ndat,2) where aux(*,0) is long
;     and aux(*,1) is lat
IF input_header.version NE 'Adopt' THEN BEGIN
    get_cluster_aux, control, aux, time
    IF get_err_no NE 0 THEN return
ENDIF

;	Spherical coordinates of unit vectors is (lon, lat, rad)
spin_sph = fltarr(ndat, 3)
spin_sph(0, 0) = aux(*, 0) ;lon
spin_sph(0, 1) = aux(*, 1) ;lat 
spin_sph(0, 2) = replicate(1., ndat) ;rad
spin_sph = transpose( spin_sph)

;	Convert spin axis angles to rectangular unit vectors
spin_xyz = cv_coord( from_sphere = spin_sph, /TO_RECT, /DEGREES)

; --- get magnetic field gse from cluster cdf mfi files
;     use papco _cdf read  for this. fgm(ndat,4) where mfe(*,0:2) is BxByBz
;     and fgm(*,4) is magnitude.
IF input_header.version NE 'Adopt' THEN BEGIN
    get_cluster_mag, control, fgm, time
    IF get_err_no NE 0 THEN return
ENDIF
;make magnetic field unit vectors in bmag (ndat,3)
bmag = fgm(*, 0:2) / [fgm(*,3),fgm(*,3),fgm(*,3)]

;	Spin axis polar angles from latitudes in radians
theta_rot = (90. - spin_sph( *, 1)) * !dtor

;	Rotation angles in radians
phi_rot = spin_sph(*, 0) * !dtor

;	GSE y projection of x sc axis must be zero gives third angle in radians
psi_rot = atan( tan( phi_rot * !dtor) / cos( theta_rot * !dtor))

;	Rotate magnetic field to non-spinning spacecraft coordinates
rot_vect, phi_rot, theta_rot, psi_rot, bmag, bmag_sc

;now have mag field vector in spacecraft frame, and data in array per
;sector, per detector. 

;get an array of fov of IES in s/c coordinates
m_ies_look, ies_look

;need to make an array for each time in data (each spin) that has
;pitch angles for each detector/sector
p = fltarr(16, 9, ndat)

;now construct pa array for each detector/sector combination.
FOR i = 0, ndat-1 DO BEGIN
    FOR d = 0, 8 DO BEGIN
        FOR s = 0, 15 DO BEGIN
            pang = acos( bmag_sc(i, 0) * ies_look(s, d, 0) + $
                         bmag_sc(i, 1) * ies_look(s, d, 1) + $
	                 bmag_sc(i, 2) * ies_look(s, d, 2)) / !dtor
            IF finite(pang) THEN p(s, d, i) = pang ELSE p(s, d, i) = -99
        ENDFOR  
    ENDFOR 
ENDFOR  

;now save pitch array in file for future use.
dummy = {time:0.0d, data: fltarr(16, 9)}
input_data = replicate(dummy, ndat)
input_data.time = time+tai1
input_data.data = temporary(p)

message, 'Saving', /info & print, '  ', fln
save, file = fln, input_header, input_data, /verb

;restore original time range
mjdt_start = orig_mjdt_start & mjdt_end = orig_mjdt_end

END 


;******************************************************************************
PRO m_ies_look,  det_coords,  OFFSET = OFFSET

print, '' & message, 'Making array of detector/sector look directions', /cont
;make an array of detector look direction coordinates in sector/detector
;coordinate system for each sector. Here we can also add a sector
;offset to compensate for mag data timing offsets...

;from Steve Monk
;Rotate the GlobeTransform about the z axis so that sector 13 is
;approximately centered on the +x (sun) direction.  Positive rotations
;go counter-clockwise when looking down from the N pole. Without
;rotation, the start of sector 0 is at -x, the start of sector 1 is
;22.5 degrees clockwise of -x (when looking down from the N pole),
;etc.  A 11.25 degree rotation centers sector 0 on -x, 180 more
;centers it on +x, and -3*22.5 centers sector 13 instead of sector 0
;on +x.  An additional angle of -3.917 accounts for the sun
;being slightly off-center in sector 13. 
;    angle = 11.25 + 180 - 71.417

dno = 9
sec_ang =  findgen(16)*22.5 + 11.25 + 180 - 71.417

det_ang = findgen(dno)*(180./dno) + (180./dno)/2
det_coords = fltarr(16, dno, 3) ; 16 sectors, dno look directions, x,y,z

IF keyword_set(OFFSET) THEN BEGIN
    message, 'Sector offset used: '+varprt(offset), /cont
    sec_ang =  sec_ang+offset
ENDIF

FOR s = 0, 15 DO BEGIN
    FOR d = 0, dno-1 DO BEGIN
        det_coords(s, d, 0) = sin(det_ang(d)*!DTOR)*cos(sec_ang(s)*!DTOR)
        det_coords(s, d, 1) = sin(det_ang(d)*!DTOR)*sin(sec_ang(s)*!DTOR)
        det_coords(s, d, 2) = cos(det_ang(d)*!DTOR)
    ENDFOR
ENDFOR

END


;******************************************************************************
; returns the calculated pitch angle in degrees as the angle between b vector
; and detecor vector, both in spacecraft coordinates. Since both vectors are
; unit vectors, we use the dotproduct a . b = cos (theta)
;******************************************************************************
function m_pitch, bvec, dvec

res = total(bvec*dvec)
pa  = acos(res) * !RADEG
return,pa

END 
