;------------------------------------------------------------------------------
; File:    IIMS_FOV.PRO   RAPID IIMS Fields-of-View Unit Vectors
; Revision 02/20/03       J L Roeder
;------------------------------------------------------------------------------
pro iims_fov, polindex, spinindex, vectors, err

;       RAPID IIMS field-of-view direction unit vectors in s/c coordinates
;		from info in instrument description [Wilken et al., 1997]

; Inputs:
;   polindex	Polar angle index, 0-11
;   spinindex	Spin angle index (sectors), 0-15

; Outputs:
;	vectors		Unit vectors of fov directions in rectangular s/c coordinates
;	err			Error string array is null for no error

;	Number of data points
nvects = n_elements( polindex)

;	Assume no error
if nvects gt 0 then err = '' else err = ['No angle indices', 'available']

;	Validate range of indices
if (min( polindex) lt 0L) or (max( polindex) gt 11) or $
	(min( spinindex) lt 0L) or (max( spinindex) gt 15) then $
	err = ['Angular index out of range:', $
		'Polar Min = ' + string( min( polindex)), $
		'Polar Max = ' + string( max( polindex)), $
		'Spin Min = ' + string( min( spinindex)), $
		'Spin Max = ' + string( max( spinindex))]

if err(0) eq '' then begin

;	Polar angle from spin axis in radians
  polar_angle = (polindex * 15. + 7.5) * !dtor

;	Sector spin angle in radians
  spin_angle = (spinindex * (360. / 16.) + 89.) * !dtor

;	Trig factors
  cp = cos( polar_angle)
  sp = sin( polar_angle)
  cs = cos( spin_angle)
  ss = sin( spin_angle)

;	Unit vectors
  vectors = fltarr( nvects, 3)
  vectors(0, 0) = reform( cs * sp, nvects)
  vectors(0, 1) = reform( ss * sp, nvects)
  vectors(0, 2) = reform( cp, nvects)

endif

end
