;run setup_test_graphics to set up the graphics test window(s)
;
;run m_cluster_rapid_pitch to make for one energy a full det/sec array
;of corresponding pitch angles.
;
;run intercal_det to get the same pitch angles for each detector
;
;run det_intercal to get the intercalibration factors using gen_fit
;
;
;


;******************************************************************************
;program to compute pitch angles and to do intercallibration work for
;cluster rapid IIMS - extensible for IES too.
;Need to read I3DD or E3DD data from Pat's science files
;Need to reed mag dir info from MDATA_K science file
;being part of the APPCO suite the time is expected to be set in the
;papco time common block. papco read routines are utilized.
;;
;Keywords:  SPECIES  (same ordering as lanl_names for pitch)
;                    0 for electrons (if data, E3DD, default)
;                    1 for electrons (if data, IES_EPAD_16)
;                    2 for ions (protons, science, I3DD)
;                    3 for ions (helium, science, I3DD)
;                    4 for ions (CNO, science, I3DD)
;
;           LLIMIT   lower L-value for data to use
;           NO_TEST  don't make single energy test array 
;    DET_PITCH_ONLY  makes array of pitch for all det/sec combinations only
;******************************************************************************
PRO m_cluster_rapid_pitch, SPECIES = SPECIES,  ENERGY = ENERGY,  $
                           HAVE_3DD = HAVE_3DD, HAVE_MDATA = HAVE_MDATA, $
                           HAVE_DET_PITCH = HAVE_DET_PITCH, SC_ID = SC_ID, $
                           DIAGNOSTIC = DIAGNOSTIC,  VERBOSE = VERBOSE, $
                           FORCE = FORCE, HR_FGM = HR_FGM, $
                           HAVE_HR_FGM =HAVE_HR_FGM, $
                           OFFSET = OFFSET, LLMIT = LLMIT, NO_TEST = NO_TEST, $
                           UNITS = UNITS, DET_MASK = DET_MASK, $
                           DET_PITCH_ONLY = DET_PITCH_ONLY

COMMON mjdt, mjdt_start, mjdt_end     ;common time limit in mjdt
COMMON papco_color_names
COMMON cluster_rapid_data, input_header, input_data
COMMON cluster_eph_data, eph_header, eph_data

COMMON cluster_fgm_data, fgm_header, fgm_data
COMMON get_error, get_err_no, get_err_msg  
COMMON cluster_rapid
COMMON cluster_fgm, fgm_time, fgm_dir
COMMON m_cluster_rapid_pitch_info,  OFFSET1,  SC_ID1, ENERGY1

COMMON cluster_rapid_pitch, M_header, M_data, DD3_header, DD3_data
COMMON cluster_rapid_pitch_output,  data, det_pitch

FORWARD_FUNCTION m_pitch, find_bsec, z_rot, vec_ang,  mfunct
papco,/set_only

IF keyword_set(UNITS) THEN units = UNITS ELSE units = 1 ;cts / sec
IF keyword_set(OFFSET) THEN rapid_offset = OFFSET ELSE rapid_offset = 0
OFFSET1 = rapid_offset
IF keyword_set(VERBOSE) THEN verbose = 1 ELSE verbose = 0
IF keyword_set(SC_ID) THEN sc_id = SC_ID-1 ELSE sc_id = 0
SC_ID1 = SC_ID
IF keyword_set(FORCE) THEN force = 1 ELSE force = 0
IF keyword_set(SPECIES) THEN species = SPECIES ELSE species = 0 ;(IES_CLEANBM)
IF keyword_set(ENERGY) THEN  energy = ENERGY ELSE energy = 0
ENERGY1 = ENERGY
IF keyword_set(DIAGNOSTIC) THEN BEGIN
    window, 3, ypos = 500 & wset, 3 & !p.position = [0.1, 0.1, 0.9, 0.9]
ENDIF

;prepare time array 
tai_1=utc2tai({mjd:mjdt_start.mjd, time:mjdt_start.t*1000})
tai_2=utc2tai({mjd:mjdt_end.mjd, time:mjdt_end.t*1000})    

;make dummy plotinfo struct needed for read routines
plotinfo = papco_getplotinfostruct()
control.sc_id = sc_id ;sets scpacecraft number
control.data_src = 3     ;sets MPAe science files for input
control.reload = 0      ;force reread
control.make = 0      ;force make if data does not exist
control.units =  units    ;adopt data units set

;read ephemeris data if required, get L-array
IF keyword_set(LLMIT) THEN BEGIN
    r_cluster_eph, VERBOSE=VERBOSE
    IF get_err_no NE 0 THEN return
    ltag = 'eph_data.C'+varprt(sc_id+1)+'_L_VALUE'
    result = execute('larr='+ltag)
    ;eph times are in cdf epoch,  data is in TAI. work out offset
    tai=utc2tai('2002-04-01')
    cdf_epoch,epoch,2002,04,01,/compute_epoch & epoch = epoch/1000.0
    time_offset = epoch-tai
    eph_tai = (eph_data.epoch/1000.0)-time_offset
    index = where( (eph_tai GE tai_1) AND (eph_tai LE tai_2), c)
    IF c NE 0 THEN BEGIN
        eph_tai = eph_tai(index) 
        larr = larr(index)    
    ENDIF ELSE message, 'No eph data'
ENDIF    

IF keyword_set(HAVE_3DD) THEN GOTO, have_data
;read the input data
pitch_type = lanl_names(species)
print, '' &  message, 'Reading data for '+pitch_type, /cont
CASE 1 OF 
    (strmid(pitch_type, 0, 3) EQ 'IES'): BEGIN 
        control.data_src = 2   ;sets RAL IF files for input
        CASE control.units OF
            0: output_info = {div:0, rat:0, cal:0, eff:1} 
            1: output_info = {div:0, rat:1, cal:0, eff:1} 
            2: output_info = {div:0, rat:1, cal:1, eff:1} 
            3: output_info = {div:1, rat:1, cal:1, eff:1}     
        ENDCASE
        CASE 1 OF
            (pitch_type EQ 'IES_CLEANBM'): BEGIN 
                control.product = 20  ;IES CLEAN Burst Mode data
                r_cluster_rapid, plotinfo
            END 
            (pitch_type EQ 'IES_EPAD_16'): BEGIN 
                control.product = 21  ;IES EPAD 16 data (NM + BM)
                control.epad_ek = 1     ;to energy intercal
                r_cluster_rapid, plotinfo
            END 
        ENDCASE 
        IF get_err_no NE 0 THEN return  
        message, 'Doing IES flux conversion', /cont
        result = rcs_docal( input_header, input_data, output_info)
        input_data.data = result
        DD3_header = input_header & DD3_data= input_data 
        DD3_header.DIV = output_info.div
        DD3_header.RAT = output_info.rat
        DD3_header.CAL = output_info.cal
    END 
    (strmid(pitch_type, 0, 4) EQ 'I3DD'): BEGIN 
        control.units = 1
        CASE species OF ;set science product depending on species chosen
            0: BEGIN    ;H
                pos=strpos(sci_names,'I3DD H')  
                index=where(pos ne -1,c)
                control.product = index(0)
            END 
            1: control.product = 3 ;He
            2: control.product = 4 ;CNO
        ENDCASE
        r_cluster_rapid, plotinfo, FORCE = FORCE
        IF get_err_no NE 0 THEN return
        DD3_header = input_header & DD3_data= input_data       
    END 
ENDCASE    
have_data:

IF keyword_set(DET_PITCH_ONLY) AND keyword_set(HAVE_3DD) THEN BEGIN
    DD3_header = input_header & DD3_data= input_data 
ENDIF 

dno = dd3_header.NDETECTORS
IF keyword_set(DET_MASK) THEN det_mask = DET_MASK ELSE $
  det_mask = intarr(dno) & det_mask(*) = 1

;mask out all detectors not used for finding pitch angles.
IF keyword_set(DET_MASK) THEN BEGIN
    
ENDIF

;read MDATA
IF keyword_set(HR_FGM) THEN BEGIN ; read high res baunschweig data
    IF NOT keyword_set(HAVE_HR_FGM) THEN BEGIN
        print, '' & message, 'Reading High Res mag Data', /cont
        info = papco_getplotinfostruct()
        info.sc_id = sc_id
        r_cluster_fgm, info
        IF get_err_no NE 0 THEN return
        ;now, the resolution of this data is way more than we need.
        ;reduce this to time res of sectors, use dd3_data time for this.
        ;first make a new time array in sector resolution. 
;        dd3_header = m_header ;for ions, to get spin resolution...
;        dd3_data = m_data
        message, 'making sector time resolution time array', /cont
        fgm_time = dblarr(dd3_header.npoints*16l, 2)
        FOR i = 0l, dd3_header.npoints-1 DO BEGIN
            diff = dd3_data(i).endtime-dd3_data(i).time
            sec_time = diff/16.0
            FOR j = 0l, 15l DO BEGIN
                fgm_time(i*16+j, 0) =  dd3_data(i).time+sec_time*j
                fgm_time(i*16+j, 1) =  fgm_time(i*16+j, 0)+sec_time
            ENDFOR
        ENDFOR    

        ;now fold fgm data into this time resolution. Average!
        message, 'folding into mfe data into sector time resolution',/cont
        print, '  ', 'and rotating to RAPID coord system, normalizing'
        rapid_sun = 60.167+rapid_offset    ;rapid offset from sun direction
        rapid_sun_rad = rapid_sun /!RADEG
        fgm_dir = fltarr(3, dd3_header.npoints*16)
        k = 0l ; search index in fgm_data array
        FOR i = 0l, n_elements(fgm_time)/2 -1 DO BEGIN
            ;don't use "where" - too slow
            WHILE fgm_data(k).time LT fgm_time(i, 0) DO BEGIN
                k = k+1 
                IF k EQ n_elements(fgm_data) THEN BEGIN
                    c = 0 & k = st_idx
                    GOTO, onon
                ENDIF
            ENDWHILE
            st_idx = k
            WHILE fgm_data(k).time LE fgm_time(i, 1) DO BEGIN
                k = k+1 
                IF k EQ n_elements(fgm_data) THEN BEGIN
                    c = 0 & k = st_idx
                    GOTO, onon
                ENDIF
            ENDWHILE
            en_idx = k
            c = en_idx-st_idx
            onon:
            ;print, i, c, st_idx, en_idx
            IF c NE 0 THEN BEGIN
                x = total(fgm_data(st_idx:en_idx-1).data(0))/c
                y = total(fgm_data(st_idx:en_idx-1).data(1))/c
                z = total(fgm_data(st_idx:en_idx-1).data(2))/c
                ;r = moment(fgm_data(idx).data(0), MDEV = x_dev) & x = r(0) 
                ;r = moment(fgm_data(idx).data(1), MDEV = y_dev) & y = r(0) 
                ;r = moment(fgm_data(idx).data(2), MDEV = z_dev) & z = r(0) 
                fgm = [x, y, z] & fgm_mag = sqrt(total(fgm*fgm))   
                fgm_dir(*, i) = fgm / fgm_mag
                fgm_dir(*, i) = z_rot(fgm_dir(*, i), rapid_sun_rad)
            ENDIF ELSE BEGIN 
                fgm_dir(*, i) = [-2, 0, 0]
            ENDELSE
        ENDFOR
    ENDIF 
ENDIF 

IF NOT keyword_set(HAVE_MDATA) THEN BEGIN
    print, '' & message, 'Reading Mag Data (MDATA) Science file', /cont
    input_header = 0 & input_data = 0
    control.data_src = 1  ;sets MPAe science files for input
    pos=strpos(sci_names,'MDATA') & idx=where(pos ne -1)
    control.product = idx(0) ;science data type MDATA
    control.units = 0
    r_cluster_rapid, plotinfo
    IF get_err_no NE 0 THEN return
    M_header = input_header & M_data = input_data
    IF m_header.npoints NE 0 THEN $
      print, tai2utc(m_data(0).time, /ECS),  ' to ', $
      tai2utc(m_data(m_header.npoints-1).time, /ECS) $
      ELSE message, 'No M-data in timerange', /cont
END

IF keyword_set(HAVE_DET_PITCH) THEN GOTO,  done_det_pitch

;now compute direction of magnetic field from MDATA. Here we work in a
;fixed frame of sectors. Middle of sector 1 is +ve X,
;middle of sector 4 is -ve Y, middle of sector 8 is -ve X, middle of
;sector 12 is +ve Y. Z is along spin axis.
;We find the mag field direction during one spin in that fixed frame.
;Then we compute the pitch angles with all detectors as the detectors
;spin around from sector 0 to 15. 
;This actually means that we're working in a fixed fram of reference
;in which the direction of the magnetic field is constant during one spin.

;M_data.b_sec gives the sector number of the sector containing the
;magnetic field vector.
;M_data.data(M_data.b_sec,0) gives the angle of a vector perpendicular
;to the magnetic field, that lies in the plane defined by the spin
;axis and the senter of sector M_data.b_sec. This angle is quantized
;in 16 steps: 0 is along spin axis (+ve Z, 15 is anti-spin axis (-ve Z)

;make arrays of sector and mag directions
sec_ang =  findgen(16)*22.5 + 11.25
sec_ang_hr =  findgen(360)
mag_ang = findgen(16)*12
mag_ang = findgen(16)*11.25 + 5.625

;restrict M_data to valid b_sec (invalid data is -ve)
index = where(M_data.b_sec GT 0, c)
IF c NE 0 THEN M_data = M_data(index) ELSE BEGIN
    get_err_msg = 'Found no valid M_data' & get_err_no = 1
    message, get_err_msg, /cont    
    return
ENDELSE
M_header.NPOINTS = n_elements(M_data)

;set up variables needed for fitting M
weights = findgen(16) & weights(*) = 1
x = sec_ang/!RADEG  ;x values for function, in radians

;make array of normalized b direction components x, y, z
print, '' & message, 'Making high res sector B vectors from MDATA', /cont
print, '  ', '(fitting m_data to 1 degree resolution)'
m_dir = fltarr(M_header.NPOINTS, 3)
FOR i = 0l, M_header.NPOINTS-1 DO BEGIN
    ; fit curve to m - data, get high res bsec
    y = M_data(i).data(*, 0) & s = M_data(i).data(*, 1)
    A = [15, 0]   ;intial guess. Amplitude 15, offset zero
    y_fit = curvefit(x, y, weights, A, SIGMA, FUNCTION_NAME = 'mfunct')
    ;mfit_test, A, x, y, s
    b_sec_hr = find_bsec(A, s)
    b_sec = M_data(i).b_sec-1
    m = M_data(i).data(b_sec, 0) & ms = -90
    m_dir(i, 0)= sin((mag_ang(m)+ms)/!RADEG)*cos(sec_ang_hr(b_sec_hr)/!RADEG)
    m_dir(i, 1)= sin((mag_ang(m)+ms)/!RADEG)*sin(sec_ang_hr(b_sec_hr)/!RADEG)
    m_dir(i, 2)= cos((mag_ang(m)+ms)/!RADEG)
    ;print, b_sec,  m,  mag_ang(m), mag_ang(m)+ms
    ;print,i, b_sec, m, m_dir(i, 0), m_dir(i, 1), m_dir(i, 2), $
    ;  sqrt(total(m_dir(i, *)*m_dir(i, *)))
ENDFOR

IF keyword_set(HR_FGM) AND keyword_set(DIAGNOSTIC) THEN BEGIN   
    ;now check against fgm data if available...
    print, '' & message, 'Comparing B data...', /cont
    ;fgm data is now in sector resolution, for all dd3-data times.    
    ;b data from rapid data stream is at spin resolution only!
    err_arr = fltarr(m_header.NPOINTS)

    FOR i = 0l, M_header.NPOINTS-1 DO BEGIN
        ;select out range of fgm_time for each m_data spin
        idx = where((fgm_time(*, 0) GE m_data(i).time) AND $
                    (fgm_time(*, 0) LE m_data(i).endtime), c)
        ;print, tai2utc(m_data(i).time, /ECS), c
        ;print, i, c
        IF c NE 0 THEN BEGIN ;c should always be 16...
            ;get mean direction for this spin!
            r = moment(fgm_dir(0, idx), MDEV = x_dev) & x = r(0) 
            r = moment(fgm_dir(1, idx), MDEV = y_dev) & y = r(0) 
            r = moment(fgm_dir(2, idx), MDEV = z_dev) & z = r(0) 
            fgm = [x, y, z] & fgm_dev = [x_dev, y_dev, z_dev]
            ;compute error, degrees between these two vectors. 
            error_deg = vec_ang(transpose(m_dir(i, *)), fgm)*!RADEG
            ;print, strFormat('mag dir error', 20, /left),error_deg
            err_arr(i) = error_deg
            IF err_arr(i) GT 90 THEN err_arr(i) = abs(err_arr(i)-180)
        ENDIF ELSE err_arr(i) = -1
    ENDFOR
    b_dir_test, m_data.time, err_arr, rapid_offset
ENDIF 

print, '' & message, 'Making array of detector/sector look directions', /cont
;make an array of detector look direction coordinates in sector/detector
;coordinate system for each sector. Here we can also add a sector
;offset to compensate for mag data timing offsets...
det_ang = findgen(dno)*(180./dno) + (180./dno)/2
det_coords = fltarr(16, dno, 3) ; 16 sectors, dno look directions, x,y,z
IF keyword_set(OFFSET) THEN BEGIN
    message, 'Sector offset used: '+varprt(offset), /cont
    sec_ang =  sec_ang+offset
ENDIF

FOR s = 0, 15 DO BEGIN
    FOR d = 0, dno-1 DO BEGIN
        det_coords(s, d, 0) = sin(det_ang(d)/!RADEG)*cos(sec_ang(s)/!RADEG)
        det_coords(s, d, 1) = sin(det_ang(d)/!RADEG)*sin(sec_ang(s)/!RADEG)
        det_coords(s, d, 2) = cos(det_ang(d)/!RADEG)
        det_vec = transpose(det_coords(s,d,*))  
        ;print, s, d, sqrt(total(det_vec*det_vec))
    ENDFOR
ENDFOR

IF keyword_set(HR_FGM) THEN BEGIN  ;use high res mag data to find pitch angles.
    ;fgm data is now in sector resolution, for all dd3-data times. 
    print, '' & message, 'doing fgm pitch angle for all detector,sector', /cont
    det_pitch = fltarr(DD3_header.NPOINTS, 16, dno)
    FOR i = 0l, DD3_header.NPOINTS-1 DO BEGIN
        FOR s = 0l, 15 DO BEGIN
            mag_dir = fgm_dir(*, i*16+s)
            IF mag_dir(0) NE -2 THEN BEGIN ;valid mfe data
                FOR d = 0,  dno-1 DO BEGIN  
                    det_vec = transpose(det_coords(s,d,*))
                    det_pitch(i, s, d) = m_pitch(mag_dir, det_vec)
                ENDFOR
            ENDIF ELSE det_pitch(i, s, *) = -1
        ENDFOR
    ENDFOR
ENDIF ELSE BEGIN 

;now, for all detector directions at each sector, compute the angle
;between the mag field direction determined above and the detector,
;i.e., the pitch angle.
;before we can loop through the data, we need to fold the mag dir
;information onto the time resolution of the 3DD data. For this to
;work, we need the mag field to be more or less steady not only for a
;spin period, but for the whole data accumulation period, i.e 
; 32 spins for I3DD normal mode
;  8 spins for I3DD burst mode
;  no product for E3DD normal mode that can be used 
;  1 spin  for E3DD bust mode

acc_no = 1 ;for E3DD data

;first restrict DD3 data to times where there is mdata
index = where((DD3_data.time GE m_data(0).time) AND $
              (DD3_data.time LE m_data(n_elements(m_data)-1).endtime), c)
IF c NE 0 THEN BEGIN
    DD3_header.NPOINTS = c & DD3_data = DD3_data(index)
ENDIF

det_pitch = fltarr(DD3_header.NPOINTS, 16, dno)
mag_dir_arr = fltarr(3, DD3_header.NPOINTS)

print,'' & message,'Computing pitch angle for all det/sec combinations',/cont
FOR i = 0, DD3_header.NPOINTS-1 DO BEGIN
    ;find mag data for time period of data 
    index = where( (m_data.time GE DD3_data(i).time) AND $
                   (m_data.time LE DD3_data(i).endtime), c)
    ;print, tai2utc(dd3_data(i).time, /ECS), i, c
    ;continue

    IF c GE acc_no THEN BEGIN     ;find AVERAGE direction in accumulation time!
        ;print,m_dir(index, 0) & print,m_dir(index, 1) & print,m_dir(index, 2)
        mag_dir = [total(m_dir(index, 0))/c, $
                   total(m_dir(index, 1))/c, $
                   total(m_dir(index, 2))/c]
        mag_dir_arr(*, i) = mag_dir
        ;print,i, c, DD3_data(i).endtime-DD3_data(i).time,  mag_dir

      ;now compute a pitch angle for each detector direction for this spin.
      FOR s = 0, 15 DO BEGIN
          FOR d = 0,  dno-1 DO BEGIN
              det_vec = transpose(det_coords(s,d,*))
              det_pitch(i, s, d) = m_pitch(mag_dir, det_vec)
          ENDFOR
      ENDFOR
    ENDIF ELSE BEGIN
      IF verbose THEN message, varprt(i)+': no valid MDATA '+varprt(c), /cont
      det_pitch(i, *, *) = -1.0
    ENDELSE

    IF keyword_set(DIAGNOSTIC) THEN BEGIN
        det_test, i
        wait, 0.05
    ENDIF
    
ENDFOR

ENDELSE 

done_det_pitch:

IF keyword_set(DET_PITCH_ONLY) THEN return

;now make some pitch angle distributions with data to check if they
;make sense! 

;get data for one energy into same array format as det_pitch
IF NOT keyword_set(NO_TEST) THEN BEGIN 
    print, '' & message, 'Making test data for energy # '+varprt(energy), /cont
    data = fltarr(DD3_header.NPOINTS, 16, dno)
    CASE 1 OF 
        (strmid(pitch_type, 0, 3) EQ 'IES'): $
            FOR i = 0, DD3_header.NPOINTS-1 $
              DO FOR s = 0, 15 DO FOR d = 0, dno-1 DO $
                data(i, s, d) = DD3_data(i).DATA(s,d,energy)
        (strmid(pitch_type, 0, 4) EQ 'I3DD'): $
            FOR i = 0, DD3_header.NPOINTS-1 DO $
              FOR s = 0, 15 DO FOR d = 0, dno-1 DO $
                data(i, s, d) = DD3_data(i).DATA(s,d,energy)
    ENDCASE
ENDIF 

;limit data to selected L-range if needed
IF keyword_set(LLMIT) THEN BEGIN
    ;set -1 L-values to 1e6
    idx = where(larr EQ -1, cl)
    IF cl NE 0 THEN larr(idx) = 1e6
    ;interpolate L times 
    l_arr = interpol(larr, eph_tai, DD3_data.time)
    idx = where(l_arr GE LLMIT, cl)
    data = data(idx, *, *)
    DD3_data = DD3_data(idx)
    DD3_header.NPOINTS = cl-1
ENDIF

END



;******************************************************************************
;rotates a 3d vector around the z axis 
FUNCTION z_rot, invec, rad
 
   new_x =    invec(0)*cos(rad) + invec(1)*sin(rad)
   mew_y = -1*invec(0)*sin(rad) + invec(1)*cos(rad)
   
   return, [new_x, mew_y, invec(2)]

END 

;******************************************************************************
;computes angle between two vectors.
FUNCTION vec_ang, a, b

  ma = sqrt(total(a*a)) & mb = sqrt(total(b*b))
  a_dot_b = TRANSPOSE(A) # B
  return, acos(a_dot_b/(ma*mb))

END
;******************************************************************************
;offset puts an offset between the sector assignments of pitch and
;data, this is equivalent to a timing offset between mag and data.
PRO pitch_test, tidx,  YRANGE = YRANGE,  PS = PS, CONNECT = CONNECT

COMMON m_cluster_rapid_pitch_info,  OFFSET,  SC_ID, ENERGY
COMMON cluster_rapid_pitch_output,  data, det_pitch
COMMON cluster_rapid_pitch, M_header, M_data, DD3_header, DD3_data
COMMON papco_color_names

IF keyword_set(ps) THEN BEGIN
    set_plot, 'PS'
    device, filename = '/u/friedel/rapid_pitch_test.ps', /color,  ysize = 18, $
      yoffset = 4
    !p.charsize = 1
ENDIF ELSE BEGIN
  set_plot, 'X'
  !p.charsize = 1.6
  window, 3, ypos = 330, xpos = 30, xsize = 800, ysize = 800 & wset, 3 
  erase
ENDELSE

!p.position = [0.1, 0.1, 0.9, 0.85]
!p.position = 0
!P.REGION = [0.1, 0.1, 0.9, 0.85]
!P.REGION = 0
!p.multi = [0, 3, 3, 0, 0]
!X.margin = [6, 6]
!Y.margin = [5, 5]
!Y.ticks=0 & !X.ticks=0 

time_str =  tai2utc(DD3_data(tidx).time, /ECS)
offset_str = 'sector offset: '+varprt(offset)+' degrees'
sc_id_str  = 'CLUSTER '+varprt(SC_ID)
e_st =  'Energy # '+varprt(energy)
title = sc_id_str+', '+e_st+'!C'+time_str+'!C'+offset_str

IF keyword_set(YRANGE) THEN yrange = YRANGE ELSE yrange = [1000,100000]

FOR det = 0, 8 DO BEGIN
    !p.multi(0) = det+1
    pitch = det_pitch(tidx,*,det)
    value = data(tidx,*,det)
    plot, pitch, value, xr=[0,180], yr=yrange, $
     /ylog, /nodata, xtitle = 'pitch angle', ytitle = 'flux', $
     title = title+' Det: '+varprt(det+1), color = black
    FOR i=0,15 DO xyouts, pitch(i), value(i), varprt(i), color= black,  $
      charsize = 0.5
    IF keyword_set(CONNECT) THEN $
      oplot, pitch, value, color = black, linestyle = 1

ENDFOR 

IF keyword_set(ps) THEN BEGIN
    device, /close
    IF ps EQ 2 THEN spawn, 'lpr -P w108argb /u/friedel/rapid_pitch_test.ps'
ENDIF

END 

;******************************************************************************
PRO b_dir_test, time, error_deg, offset

COMMON papco_color_names
setup_test_graphics

title = 'Error between FGM and B DATA B-field direction'
title = title+'!CRAPID Offset: '+varprt(offset)
plot, time, error_deg,  yrange = [0, 15], xtitle = 'points', $
  ytitle = 'degrees', title = title, color = black

END

;******************************************************************************
PRO sector_test, tidx, det, yrange=yrange

COMMON cluster_rapid_pitch_output,  data, det_pitch
COMMON cluster_rapid_pitch, M_header, M_data, DD3_header, DD3_data
COMMON papco_color_names

if keyword_set(yrange) then yr=yrange else yr=[1000,100000]
erase
x = findgen(16)+1
plot, x, data(tidx,*,det), $
  xr=[0,17], yrange=yr, /ylog, /nodata, xtitle = 'sector', $
  ytitle = 'flux', title = DD3_data(tidx).TIMESTR,  color = black
FOR i=0,15 DO xyouts,x(i),data(tidx,i,det),varprt(i), color = black

end

;******************************************************************************
PRO det_test, tidx

COMMON cluster_rapid_pitch_output,  data, det_pitch
COMMON cluster_rapid_pitch, M_header, M_data, DD3_header, DD3_data
COMMON papco_color_names

erase
dno = dd3_header.NDETECTORS
timestr =  tai2utc(DD3_data(tidx).time, /ECS)

plot, findgen(16), transpose(det_pitch(tidx, *, 0)), $
  yr=[0,180], xr=[-1,16], psym=0, col=black, xtitle = 'sector', $
  ytitle = 'pitch angle', title = timestr

FOR det=1,dno-1 DO oplot, findgen(16), transpose(det_pitch(tidx, *, det)), $
           psym=0, col=black+det

FOR det = 0, dno-1  DO xyouts, 17, 30+det*10, varprt(det), color = black+det

end

;******************************************************************************
;Function for curvefit to fit M data. Uses a shifted sin curve.
;  F(x)=  a * sin(b+x)
;
;  df/da = sin(b+x)
;  df/db = a * cos(b+x)
;  
; In procedure  call, A is array containg [a,b]
PRO mfunct,  X,  A,  F,  pder

;define function
F = A[0] * sin(A[1]+x)

;If the procedure is called with four parameters, calculate the
;partial derivatives.  
IF N_PARAMS() GE 4 THEN pder = [ [sin(A[1]+x)], [A[0] * cos(A[1]+x)] ]

END

;******************************************************************************
PRO mfit_test, A, x, y, s

COMMON papco_color_names

x_fit = findgen(360)/!radeg 

max = max([y, A[0] * sin(A[1]+x_fit)])+1
min = min([y, A[0] * sin(A[1]+x_fit)])-1

erase
plot,x_fit, A[0] * sin(A[1]+x_fit), color = black, yr = [min, max]
oplot, x, y, color = red
oplot, x, s, color = green

x_bsec = find_bsec(A, s)/!radeg 
oplot, [x_bsec, x_bsec], [min, max], color = blue, linestyle = 3

END

;******************************************************************************
;find extremes of functions fitted to m, (one degree resolution). 
;test agains signs to see which extreme has s of zero -> that's the 
;one degree resolution sector of the where the field is. 
function find_bsec, A, s

x_fit = findgen(360)/!radeg 
y_fit = A[0] * sin(A[1]+x_fit)

max = max(A[0] * sin(A[1]+x_fit), max_idx)
min = min(A[0] * sin(A[1]+x_fit), min_idx)

IF s(fix(max_idx/22.5)) EQ 0 THEN return, max_idx
IF s(fix(min_idx/22.5)) EQ 0 THEN return, min_idx

END


;******************************************************************************
PRO intercal_det,  VERBOSE = VERBOSE, PANGLE = PANGLE,  REF_DET = REF_DET, $
                   STEP = STEP,  PA_BIN = PA_BIN, COUNT_RATE = COUNT_RATE, $
                   AVERAGE = AVERAGE,  XRANGE = XRANGE,  YRANGE = YRANGE,  $
                   PS = PS

COMMON cluster_rapid_pitch_output,  data, pitch
COMMON cluster_rapid_pitch, M_header, M_data, DD3_header, DD3_data
COMMON papco_color_names
COMMON intercals, intcal_arr, one_pitch
COMMON all_pitchs,  all_pitch, pa_rg_str

IF keyword_set(VERBOSE) THEN verbose = VERBOSE ELSE verbose = 0

dno = dd3_header.NDETECTORS

;find average values for each detector in a given range of pitch angles.

;sets the pitch angle range; default 80-100 degrees -, normally has best coverage
IF keyword_set(PANGLE) THEN pangle = PANGLE ELSE pangle = 4 
;sets reference detector, that has cal factor 1. 
IF keyword_set(REF_DET) THEN ref_det = REF_DET ELSE ref_det = 3
;sets running average for data
IF keyword_set(STEP) THEN step = STEP ELSE step = 1
;sets pitch angle bin to look at
IF keyword_set(PA_BIN) THEN pa_bin = PA_BIN ELSE pa_bin = 20 

n_pa_bins = 180/pa_bin
pa_rg = fltarr(n_pa_bins, 2)
pa_rg_str = strarr(n_pa_bins)
FOR i = 0, n_pa_bins-1 DO BEGIN
    pa_rg(i, 0) = pa_bin*i
    pa_rg(i, 1) = pa_bin*(i+1)
    pa_rg_str(i) = string(pa_rg(i, 0), pa_rg(i, 1), format = "(i3.3,'-',i3.3)")
ENDFOR

;find averages in PA range per detector for a period
;data AND pitch are in same format,  data(i, s, d)

;NB. It seems that inter factors vary with count rate. So, to test
;this divide out data into several count rate "regions". First try is
;three ranges: 0 to 20 c/s, 20-80 c/s, and 80 c/s upward.

IF NOT keyword_set(COUNT_RATE) THEN cr_rng = [1, 10000] ELSE cr_rng = COUNT_RATE
cr_rng(0) = cr_rng(0) > 1
cr_rng_str = varprt(cr_rng(0))+'-'+varprt(cr_rng(1))

step = 1
intcal_arr = fltarr(dno, n_elements(DD3_data)-(step-1))
one_pitch = intcal_arr
all_pitch = fltarr(dno, n_pa_bins, n_elements(DD3_data)-(step-1))
all_count = fltarr(dno, n_pa_bins, n_elements(DD3_data)-(step-1))
one_pitch = intcal_arr
dno = dd3_header.NDETECTORS

;now collext data from each detector into each pa bin
FOR count = 0, n_elements(DD3_data)-step  DO BEGIN 
    
    intercal = fltarr(dno, n_pa_bins)
    intercal_c = intarr(dno, n_pa_bins)
    intercal_factors= fltarr(dno, n_pa_bins)

    FOR det = 0, dno-1 DO BEGIN
        det_data  = data(count:count+step-1, *, det)
        det_pitch = pitch(count:count+step-1, *, det)
        FOR i = 0, n_pa_bins-1 DO BEGIN
            idx = where((det_pitch GE pa_rg(i, 0)) AND $
                        (det_pitch LT pa_rg(i, 1)), c)
            IF c NE 0 THEN BEGIN ;detector covers pitch angle range 
                ;see if there are any counts in this PA range 
                d = det_data(idx)
                ;select range of counts
                idx = where((d GE cr_rng(0)) AND (d LE cr_rng(1)), c)         
                IF c NE 0 THEN BEGIN 
                    d = det_data(idx)
                    intercal(det, i) = total(d(idx))/c
                    intercal_c(det, i) = c 
                ENDIF						
            ENDIF    
        ENDFOR
    ENDFOR

   IF keyword_set(VERBOSE) THEN BEGIN 
     ;print matrix of counts / det / pa range
     out_str = '    '
     FOR i = 0, n_pa_bins-1 DO $
       out_str = [out_str, string(pa_rg(i, 0), pa_rg(i, 1), $
                                  format = "(' ',i3.3,'-',i3.3)")]
     print, out_str
     FOR det = 0, dno-1 DO print, det+1, intercal_c(det, *), $
       format = "(i2,' ',9(i9))"
     print, ''
     FOR det = 0, dno-1 DO print, det+1, intercal(det, *), $
       format = "(i2,' ',9(f9.1))"
   ENDIF 

   one_pitch(*, count) = intercal(*, pangle)
   all_pitch(*, *, count) = intercal
   all_count(*, *, count) = intercal_c
   
   max = max(intercal, max_idx)
   index = where(intercal NE 0, c)
   IF c NE 0 THEN intercal_factors(index) = max/intercal(index)
   FOR det = 0, dno-1 DO BEGIN
       alldets = intercal_factors(det,*)
       index = where(alldets NE 0, c)
       IF c NE 0 THEN intcal_arr(det, count) = total(alldets(index)) / c
   ENDFOR

   IF keyword_set(VERBOSE) THEN BEGIN 
     print, ''
     FOR det = 0, dno-1 DO $
       print, det+1, intercal_factors(det, *), intcal_arr(det, count), $
       format = "(i2,' ',9(f9.3),'  ',f8.2)"
   ENDIF 

   d = max_idx MOD  dno+ 1
   r = max_idx/dno

ENDFOR

;one_pitch contains average data from each detector at one pitch angle range 
;for each time step.
;find the detector with the largest average counts, this one will be
;used to normilze others and get cal factor "1"
max_data = fltarr(dno)
FOR det = 0, dno-1 DO BEGIN
    one_det_one_pitch = one_pitch(det, *)
    idx = where(one_det_one_pitch NE 0, c)
    IF c NE 0 THEN max_data(det) = total(one_det_one_pitch(idx)) / c
ENDFOR

;find detector with max counts
max = max(max_data,  max_idx)
message, 'Maximum counts are in detector '+varprt(max_idx), /cont

max_idx = ref_det

;find average flux/counts per detector, then do intercal!
IF keyword_set(AVERAGE) THEN BEGIN
    av_det = fltarr(dno)
    ;first find average clounts/flux for each detector
    FOR det = 0, dno-1 DO BEGIN
        ndat = xrange(1)-xrange(0)+1
        sum = total(one_pitch(det, xrange(0):xrange(1)))
        av_det(det) = sum/ndat
        print,  'av det '+varprt(det), +': '+varprt(sum/ndat)
    ENDFOR
    print, 'intercal, ref_det ='+varprt(ref_det)
    print,  av_det/ av_det(ref_det), format = "(9(f6.2))"
ENDIF

IF verbose THEN $ 
    test_intercal, title = 'Pitch angle range:!C'+pa_rg_str(pangle)+$
      '!CRate range:!C'+cr_rng_str,  XRANGE = XRANGE,  YRANGE = YRANGE, PS = PS

END
 
;******************************************************************************
PRO test_pitch_sort, pa_range, $
                     TITLE = TITLE, YRANGE = YRANGE, PS = PS

COMMON mjdt, mjdt_start, mjdt_end     ;common time limit in mjdt
COMMON papco_color_names
COMMON all_pitchs,  all_pitch, pa_rg_str;   all_pitch(det,pa,*)
COMMON cluster_rapid_pitch, M_header, M_data, DD3_header, DD3_data
COMMON m_cluster_rapid_pitch_info,  OFFSET,  SC_ID, ENERGY

!xticks = 6

IF keyword_set(ps) THEN BEGIN
    set_plot, 'PS'
    fln = '/u/friedel/rapid_pitch_plots/rapid_pitch_sort_' $ 
      +pa_rg_str(pa_range)+'_'+'EK'+varprt(energy)+'.ps'
    device, filename = fln, /color,  $
      ysize = 22, yoffset = 4
    !p.charsize = 1
ENDIF ELSE BEGIN
    set_plot, 'X'
    !p.charsize = 2
    window, 4, ypos = 330, xpos = 30, xsize = 800, ysize = 800 & wset, 4
ENDELSE

panel = [0, 1, 1] & ;panelset, panel

convert_t90_to_date, mjdt_start, yr,doy,hr,min,sec,mo,cmo,dom, /mjdt
tai_1=UTC2TAI({YEAR:yr, MONTH:mo, DAY:dom, HOUR:hr, MINUTE:min, $
               SECOND:sec, MILLISECOND:0})
convert_t90_to_date, mjdt_end, yr,doy,hr,min,sec,mo,cmo,dom, /mjdt
tai_2=UTC2TAI({YEAR:yr, MONTH:mo, DAY:dom, HOUR:hr, MINUTE:min, $
               SECOND:sec, MILLISECOND:0})

xut1=0  &  xut2=tai_2-tai_1  
time = DD3_data.time-tai_1

IF keyword_set(TITLE) THEN title = TITLE ELSE title = ''
IF keyword_set(YRANGE) THEN yrange = YRANGE ELSE BEGIN
    max = max(all_pitch)
    yrange = [0, max]
ENDELSE

extra_plotPar={xrange:[xut1,xut2], yrange:yrange, ylog:0, color:black, $
               psym:1, xtickformat:'noticks', symsize:0.2}
top_title = 'CLUSTER '+varprt(SC_ID+1)+'   PA: '+pa_rg_str(pa_range)+$
  '   EK # '+varprt(energy)

xyouts, 0.5, 0.98, top_title, align = 0.5, /normal, color = black, $
  charsize = !p.charsize*.75

FOR det = 0, 8 DO BEGIN
    !p.position = 0
    !P.REGION = 0
    !p.multi = [0, 3, 3, 0, 0]
    !X.margin = [5, 5]
    !Y.margin = [5, 5]
    !p.multi(0) = det+1
    title ='Det '+varprt(det+1)
    papco_y_label, yrange(0), yrange(1), log=0
    papco_draw_time_axis, panel, OVERPLOT=OVERPLOT, _extra=extra_plotPar
    plot, time, all_pitch(det, pa_range, *), xstyle=5, ystyle=5, $
      title = title, _extra=extra_plotPar
ENDFOR 

IF keyword_set(ps) THEN BEGIN
    device, /close
    message, 'Created '+fln, /cont
    IF ps EQ 2 THEN spawn, 'lpr -P w108argb '+fln
    IF ps EQ 3 THEN spawn, 'lpr -P w108argb_t '+fln
ENDIF

end

;******************************************************************************
PRO test_intercal, TITLE = TITLE,  XRANGE = XRANGE,  YRANGE = YRANGE,  PS = PS

COMMON papco_color_names
COMMON intercals, intcal_arr, one_pitch
COMMON cluster_rapid_pitch, M_header, M_data, DD3_header, DD3_data
COMMON mjdt, mjdt_start, mjdt_end     ;common time limit in mjdt
COMMON m_cluster_rapid_pitch_info,  OFFSET,  SC_ID, ENERGY

cols = [black, red, green, blue, magenta, cyan, $
        burgundy, olive, dark_green, teal, royal_blue, violet]

old_charsize=!p.charsize
!xticks = 6

IF keyword_set(ps) THEN BEGIN
    set_plot, 'PS'
    fln = '/u/friedel/rapid_intercal.ps'
    print, fln
    device, filename = 'fln', /color,  $
      ysize = 16, yoffset = 4
    !p.charsize = 2
ENDIF ELSE BEGIN
    set_plot, 'X'
    !p.charsize = 1
    window, 3, ypos = 500, xsize = 650 
    wset, 3 & !p.position = [0.1, 0.2, 0.85, 0.9]    
ENDELSE

panel = [0, 1, 1] & panelset, panel

convert_t90_to_date, mjdt_start, yr,doy,hr,min,sec,mo,cmo,dom, /mjdt
tai_1=UTC2TAI({YEAR:yr, MONTH:mo, DAY:dom, HOUR:hr, MINUTE:min, $
               SECOND:sec, MILLISECOND:0})
convert_t90_to_date, mjdt_end, yr,doy,hr,min,sec,mo,cmo,dom, /mjdt
tai_2=UTC2TAI({YEAR:yr, MONTH:mo, DAY:dom, HOUR:hr, MINUTE:min, $
               SECOND:sec, MILLISECOND:0})

xut1=0  &  xut2=tai_2-tai_1  
time = DD3_data.time-tai_1

IF keyword_set(TITLE) THEN title = TITLE ELSE title = ''
IF keyword_set(XRANGE) THEN xrange = XRANGE $
ELSE xrange = [0, n_elements(intcal_arr(0, *))-1]
IF keyword_set(YRANGE) THEN yrange = YRANGE ELSE BEGIN
    max = max(intcal_arr)+0.5
    yrange = [0, max]
ENDELSE
dno = dd3_header.NDETECTORS

offset_str = 'Offset: '+varprt(offset)+' deg.'
sc_id_str  = 'CLUSTER '+varprt(SC_ID+1)
e_st =  'Energy # '+varprt(energy)
title = sc_id_str+'!C'+e_st+'!C'+offset_str+'!C'+title


extra_plotPar={xrange:[xut1,xut2], yrange:yrange, ylog:0, color:black, $
               psym:1, xtickformat:'noticks', symsize:0.5}

papco_y_label, yrange(0), yrange(1), log=0
papco_draw_time_axis, panel, OVERPLOT=OVERPLOT, _extra=extra_plotPar

plot, time, intcal_arr(0, *), xstyle=5, ystyle=5, _extra=extra_plotPar

FOR det = 1, dno-1 DO oplot, time, intcal_arr(det, *), $
   color = cols(det), psym = 1,  symsize = 0.5
FOR det = 0, dno-1 DO xyouts, 0.85, 0.2+det*0.05, varprt(det), $
  color = cols(det), /norm
xyouts, 0.85, 0.2+det*0.05, 'Det #', color = black, /norm

right_side_label,panel, title, _extra=extra_Par

IF keyword_set(ps) THEN BEGIN
    device, /close
    IF ps EQ 2 THEN spawn, 'lpr -P w108argb /u/friedel/rapid_intercal.ps'
    IF ps EQ 3 THEN spawn, 'lpr -P w108argb_t /u/friedel/rapid_intercal.ps'
ENDIF

!p.charsize=old_charsize

END

;******************************************************************************
PRO det_intercal, TITLE = TITLE,  XRANGE = XRANGE,  YRANGE = YRANGE,  PS = PS,$
                  HEAD = HEAD, VERBOSE = VERBOSE, PAIRS = PAIRS, CALS = CALS

COMMON papco_color_names
COMMON intercals, intcal_arr, one_pitch
COMMON cluster_rapid_pitch, M_header, M_data, DD3_header, DD3_data
COMMON mjdt, mjdt_start, mjdt_end     ;common time limit in mjdt
COMMON m_cluster_rapid_pitch_info,  OFFSET,  SC_ID, ENERGY

n_det = DD3_header.NDETECTORS

IF keyword_set(verbose) THEN BEGIN 
    cols = [black, red, green, blue, magenta, cyan, $
            burgundy, olive, dark_green, teal, royal_blue, violet]
    old_charsize=!p.charsize

    IF keyword_set(ps) THEN BEGIN
        set_plot, 'PS'
        fln='/u/friedel/tmp/rapid_det_intercal.ps'
        device, filename = fln, /color, ysize=22, yoffset = 4
        !p.charsize = 1.0
    ENDIF ELSE BEGIN
        set_plot, 'X'
        !p.charsize = 1.5
        window, 3, ypos = 330, xpos = 30, xsize = 800, ysize = 800 & wset, 3 
        erase
    ENDELSE

    !p.position = 0
    !P.REGION = 0
    IF n_det EQ 12 THEN BEGIN
        !p.multi = [0, 4, 4, 0, 0]
        title = 'I3DD Detector intercal (rates)'
        det_per_head = 4
    ENDIF   ELSE BEGIN 
        !p.multi = [0, 3, 3, 0, 0]
        title = 'E3DD Detector intercal (rates)'
        det_per_head = 3
    ENDELSE
    !X.margin = [7, 6]
    !Y.margin = [4, 6]
    !Y.ticks=0 & !X.ticks=0 

    t_str = tai2utc(DD3_data(0).time, /ecs) + ' to '+ $
      tai2utc(DD3_data(n_elements(DD3_data)-1).time, /ecs)

    xyouts, 0.02, 0.98, 'Time Range: '+t_str, /normal,  color = black, $
      charsize = 1.5, charthick = 1.5
ENDIF 

if keyword_set(HEAD) then head=HEAD else head=0

IF keyword_set(PAIRS) THEN intercal_pairs = PAIRS ELSE BEGIN 
    IF n_det EQ  12 THEN $
       intercal_pairs=[[0,0],[0,1],[0,2],[0,3], $
                      [1,0],[1,1],[1,2],[1,3], $
                      [2,0],[2,1],[2,2],[2,3], $
                      [3,0],[3,1],[3,2],[3,3]] $
    ELSE $
       intercal_pairs=[[0,0],[0,1],[0,2], $
                      [1,0],[1,1],[1,2], $
                      [2,0],[2,1],[2,2]] 
ENDELSE 

res=size(intercal_pairs)
slopes=fltarr(res(2))

FOR i = 0, res(2)-1 DO BEGIN
    !p.multi(0) = i+1
    xdet=intercal_pairs(0,i)+head*det_per_head
    ydet=intercal_pairs(1,i)+head*det_per_head
    xarr=transpose(one_pitch(xdet,*))
    yarr=transpose(one_pitch(ydet,*))

    index=where(xarr gt 0,c)
    if c ne 0 then begin
      xarr=xarr(index)  &  yarr=yarr(index)
    endif
    index=where(yarr gt 0,c)
    if c ne 0 then begin
      xarr=xarr(index)  &  yarr=yarr(index)
    endif

    ;call gen_fit_release to calculate actual slope!
    sigma_x=sqrt(xarr) & sigma_y=sqrt(yarr)
    gen_fit_release, x=xarr, y=yarr,$
      sigma_x=sigma_x, sigma_y=sigma_y, nparms=1, n_iterations=10000,$
      slope = out_slope, intercept=out_intercept, status=status
     
    ;y=mx + c  where m is slope, c in intercept
    print,'gen_fit ',xdet,ydet,out_slope, out_intercept, status

    IF keyword_set(verbose) THEN BEGIN 
        IF keyword_set(YRANGE) THEN yrange = YRANGE ELSE BEGIN
          yr = [0,max([xarr,yarr])] & xr=yr
        ENDELSE

        plot, xarr, yarr, yrange=yr, xrange=xr, psym=1, color=black, $
          xtitle='Det '+varprt(xdet), ytitle='Det '+varprt(ydet), $
          title=title 
        oplot,[0,xr(1)],[0,yr(1)], linestyle=2, color=red 
        
        x=(findgen(100) / 100.0) * yr(1)    
        y=out_slope*x + out_intercept
        oplot, x, y, linestyle=3, color=blue
        
        x=(xr(1)-xr(0))*0.05 & y=(yr(1)-yr(0))*0.95
        xyouts,x,y,'y=a+bx',charsize=!P.charsize*0.75, color=dark_green
        x=(xr(1)-xr(0))*0.05 & y=(yr(1)-yr(0))*0.85
        xyouts,x,y,'a='+varprt(out_intercept),charsize=!P.charsize*0.75, $
           color=dark_green
        x=(xr(1)-xr(0))*0.05 & y=(yr(1)-yr(0))*0.75
        xyouts,x,y,'b='+varprt(out_slope), charsize=!P.charsize*0.75, $
           color=dark_green
    ENDIF 
    
    slopes(i)=out_slope

ENDFOR

cals=slopes

IF keyword_set(verbose) THEN BEGIN 
    IF keyword_set(ps) THEN BEGIN
        device, /close
        IF ps EQ 2 THEN spawn, 'lpr -P w108argb '+fln
        IF ps EQ 3 THEN spawn, 'lpr -P n114rgb '+fln
    ENDIF
    !p.charsize=old_charsize
ENDIF

END


;******************************************************************************
PRO test_binned_pitch,  t_idx,  ENERGY = ENERGY

common cluster_rapid_data, input_header, input_data
COMMON cluster_rapid_pitch, M_header, M_data, DD3_header, DD3_data
COMMON papco_color_names

IF keyword_set(ENERGY) THEN ek = ENERGY ELSE ek = 0

setup_test_graphics
erase
timestr =  tai2utc(input_data(t_idx).time, /ECS)

xray = (input_header.PA_POSITIONS(*, 0)+input_header.PA_POSITIONS(*, 1))/2.0
data = transpose(input_data(t_idx).data(ek, *))
count = transpose(input_data(t_idx).count(ek, *))

index = where((count NE 0) AND (count NE -1), c)

xray = xray(index)
yray = data(index)/count(index)

index = where(yray NE 0)
xray = xray(index)
yray = yray(index)

plot_io, xray, yray, $
  xr=[0,180], psym=-1, col=black, xtitle = 'pitch angle', $
  ytitle = 'rate', title = timestr

stop

END


;******************************************************************************
PRO setup_test_graphics

window, 3, ypos = 630, xpos = 30, xsize = 700, ysize = 500
wset, 3 
!p.position = [0.1, 0.1, 0.8, 0.85]
!Y.ticks=0 & !X.ticks=0

END


;******************************************************************************
; Here we apply any intercallibration between detectors, and assemble
; data into pitch angle/energy arrays.
; if keyword no_bin is set, save ALL data plus PA for each det, sect.
;           DET_MASK a mask for which detectors to use, default all
;
PRO w_cluster_rapid_pitch, SPECIES = SPECIES, PA_BIN = PA_BIN, $
                           NO_SAVE = NO_SAVE, NO_BIN = NOBIN, $
                           DET_MASK =  DET_MASK

COMMON cluster_rapid_pitch_output,  data, det_pitch
COMMON cluster_rapid_pitch, M_header, M_data, DD3_header, DD3_data
COMMON m_cluster_rapid_pitch_info,  OFFSET,  SC_ID, ENERGY
COMMON cluster_rapid_data, input_header, input_data
COMMON cluster_rapid

IF NOT (papco_check_data_env('CLUSTER_RAPID_PITCH',path=PATH)) THEN return

IF keyword_set(PA_BIN) THEN pa_bin = float(PA_BIN) ELSE pa_bin = 10.0

name = lanl_names(species) ;use for file nameing

;set up pitch angle ranges for binning. Always make sure that center
;of "middle" bin has 90 degree value in it's center!
pa_range = fltarr(1, 2) & top = 90.0+pa_bin/2.0 & bot = 90.0-pa_bin/2.0
pa_range(0, *) = [bot, top] & next_top = 90.0
WHILE next_top LT 180 DO BEGIN
    next_top = (top+pa_bin) < 180.0 & next_bot = (bot-pa_bin) > 0.0
    pa_range = [transpose([next_bot, bot]), pa_range, transpose([top, next_top])]
    top = next_top & bot = next_bot
ENDWHILE

n_bin = n_elements(pa_range)/2 & pitch = fltarr(DD3_header.NBANDS, n_bin)
n_dat = DD3_header.NPOINTS
dummy = create_struct(name = 'CLUSTER_RAPID_PITCH_'+name+'_'+varprt(n_bin), $
                      'time', 0.0d, 'endtime', 0.0d, $
                      'data', pitch, 'count', pitch, 'lut', 0l, 'flag', 0l)
input_data = replicate(dummy, n_dat)

input_data.time = DD3_data.time &  input_data.endtime = DD3_data.endtime
input_data.lut = DD3_data.lut & input_data.flag = DD3_data.flag

;bin into pitch angle ranges AND apply intercallibration factors to
;data. Make sure only to count bins with data in them!!!!!!

intercal = [1.08, 1.00, 0.92, 1.08, 1.00, 1.10, 1.03, 1.00, 1.09]

message,'Binning data into pitch angle ranges (slow)',/cont

FOR i = 0, n_dat-1 DO BEGIN 
   
   IF det_pitch(i, 0, 0) EQ -1 THEN BEGIN ; no mag field dir -> no pitch angles
       input_data(i).data = -1
       input_data(i).count = -1
       continue 
   ENDIF
   FOR det = 0, DD3_header.NDETECTORS-1 DO BEGIN 
      FOR sec = 0, DD3_header.NSECTORS-1 DO BEGIN 
         pa = round(det_pitch(i, sec, det))
         pa_idx = where(pa LT pa_range(*,1)) & pa_idx = pa_idx(0)
         FOR ek = 0, DD3_header.NBANDS-1 DO BEGIN
             IF DD3_data(i).DATA(sec, det, ek) GT 0 THEN BEGIN
                 input_data(i).data(ek, pa_idx) = $
                   input_data(i).data(ek, pa_idx) + $
                   DD3_data(i).DATA(sec, det, ek) * intercal(det)
               input_data(i).count(ek, pa_idx) = $
                 input_data(i).count(ek, pa_idx)+1
             ENDIF
         ENDFOR
      ENDFOR
   ENDFOR
ENDFOR

input_header={ $
                DATATYPE     : 'CLUSTER/RAPID/'+name+'PITCH ANGLES', $
                NPOINTS      : n_dat, $
                NDETECTORS   : DD3_header.NDETECTORS, $
                NBANDS       : DD3_header.NBANDS, $
                NSECTORS     : long(n_bin), $
                NLUTS        : 'not used' , $
                DETECTORS    : 0l, $
                TSTART       : input_data(0).time, $
                TSTOP        : input_data(n_dat-1).endtime, $
                CH_POSITIONS : DD3_header.CH_POSITIONS, $
                PA_POSITIONS : pa_range, $
                COUNT_CAL    : DD3_header.COUNT_CAL , $  
                ENERGY_CAL   : DD3_header.ENERGY_CAL, $  
                DETECTOR_CAL : intercal, $
                PEDESTALS    : 'not used' , $  
                CAL          : DD3_header.CAL, $  
                RAT          : DD3_header.RAT, $  
                DIV          : DD3_header.DIV, $  
                CLEAN        : DD3_header.CLEAN, $  
                SUBTRACT     : 'not used' , $  
                TITLE        : name+'_PITCH', $
                XTITLE       : 'UT', $
                YTITLE       : 'Pitch Angle', $
                ZTITLE       : 'diff flux', $
                PTITLE       : 'not used'          } 
    
input_header = create_struct(name='CLUSTER_RAPID_PITCH_'+name+'_'+ $
                             varprt(n_bin)+'_head', input_header)

; now write out pitch angle array to file. Keep file structure similar to 
; RAL if format for compatibility: input_header and input_data structures
IF NOT keyword_set(NO_SAVE) THEN BEGIN
t = tai2utc(input_data(0).time,/ext)
date = string(t.year, t.month, t.day, format = "(i4.4,i2.2,i2.2)")
print,''
print,'% -------------------------------------------------------------------'
fln=path + 'C'+varprt(SC_ID+1)+'_'+date + '_'+name+'_pitch.idl'
message,'Saving pitch angle data to:', /cont
print,'  ',fln

save, FILENAME=fln, input_header, input_data, /COMPRESS,  VERBOSE=VERBOSE
print,'% -------------------------------------------------------------------'
ENDIF   


END


;******************************************************************************
PRO rapid_pitch_batch, sc_id = sc_id,  HAVE = HAVE,  PS = PS,  EK = EK, $
                       OFFSET = OFFSET, TIMEmcp_times = TIME, HR_FGM = HR_FGM

IF keyword_set(OFFSET) THEN offset = OFFSET ELSE offset = 0
IF keyword_set(SC_ID) THEN sc_id = SC_ID ELSE sc_id = 0
IF keyword_set(PS) THEN ps = PS ELSE ps = 0
IF keyword_set(EK) THEN energy = EK ELSE energy = 0
IF keyword_set(HR_FGM) THEN hr_fgm = 1 ELSE hr_fgm = 0

IF keyword_set(TIME) THEN BEGIN ;make papco time common block!
   common mjdt, mjdt_start, mjdt_end
   result = convert_timeToSeconds(TIME(0), /t90, /mjdt)
   IF result.err EQ 0 THEN BEGIN
       message, 'Start time '+ $
         convert_secondsToTime(result.value, /mjdt, /t90),  /cont
       mjdt_start = result.value
   ENDIF ELSE message, 'wrong start time format'
   result = convert_timeToSeconds(TIME(1), /t90, /mjdt)
   IF result.err EQ 0 THEN BEGIN
       message, 'End time   '+ $
         convert_secondsToTime(result.value, /mjdt, /t90),  /cont
       mjdt_end = result.value
   ENDIF ELSE message, 'wrong end time format'
ENDIF

IF keyword_set(HAVE) THEN BEGIN
    HAVE_I3DD = 1
    HAVE_MDATA = 1
    HAVE_E3DD = 1
    HAVE_HR_FGM = 1
    HAVE_DET_PITCH = 1
ENDIF ELSE BEGIN
    HAVE_I3DD = 0
    HAVE_MDATA = 0
    HAVE_E3DD = 0
    HAVE_HR_FGM = 0
    HAVE_DET_PITCH = 0
ENDELSE

m_cluster_rapid_pitch, HAVE_I3DD = HAVE_I3DD, HAVE_MDATA = HAVE_MDATA, $
  HAVE_E3DD = HAVE_E3DD, HAVE_HR_FGM = HAVE_HR_FGM,  OFFSET = OFFSET, $
  SPECIES = 3,  ENERGY = ENERGY,  HR_FGM = HR_FGM, SC_ID = SC_ID,  $
  HAVE_DET_PITCH = HAVE_DET_PITCH

intercal_det, ref_det=4, pa_bin=20, pangle=4, COUNT_RATE=[0,10000], $
  yrange=[0,3], ps = ps

END

;******************************************************************************
;procedure to evaluate intercalibration factors for IIMS for several
;timeperiods. 
;Starts with the times when magnetometer patches were uploaded.
;
PRO rapid_iims_intercal,  WRITE_MCP_TIMES = WRITE_MCP_TIMES

COMMON mjdt, mjdt_start, mjdt_end     ;common time limit in mjdt
COMMON get_error, get_err_no, get_err_msg  

papco, /set_only
 
path = '/u/friedel/tmp/cluster_rapid/'

;get today's UTC date
caldat,systime(/utc,/julian),month,day,year 
now = string(year, month, day, format = "(i4.4,i2.2,i2.2)")
message, 'Today is '+now, /cont

;list of times for changes to MCP voltage levels. from beginning of
;sicence mission, 01 Feb 2001
dummy = {sc_id:0, from:'', to:'', label:''}

mcp_times = replicate(dummy, 20)

;start times - times of magnetometer patch uploads
i = 0
mcp_times(i).sc_id = 1
mcp_times(i).from  = '20010518 09:07'
mcp_times(i).to    = '20010818' 
mcp_times(i).label = 'Start/Stop Voltage 4/5'
i = i+1
mcp_times(i).sc_id = 2
mcp_times(i).from  = '20010518 12:40'
mcp_times(i).to    = '20011016'
mcp_times(i).label = 'Start/Stop Voltage 7/7'
i = i+1
mcp_times(i).sc_id = 3
mcp_times(i).from  = '20010426 15:07' 
mcp_times(i).to    = '20010818' 
mcp_times(i).label = 'Start/Stop Voltage 4/4'
i = i+1
mcp_times(i).sc_id = 4
mcp_times(i).from  = '200100518 07:21' 
mcp_times(i).to    = '20010818' 
mcp_times(i).label = 'Start/Stop Voltage 5/5'


i = i+1
mcp_times(i).sc_id = 1
mcp_times(i).from  = '20010818'
mcp_times(i).to    = '20011101'
mcp_times(i).label = 'Start/Stop Voltage 6/7'
i = i+1
mcp_times(i).sc_id = 2
mcp_times(i).from  = '20011016'
mcp_times(i).to    = '20011202'
mcp_times(i).label = 'Start/Stop Voltage 8/8'
i = i+1
mcp_times(i).sc_id = 3
mcp_times(i).from  = '20010818'
mcp_times(i).to    = '20011101'
mcp_times(i).label = 'Start/Stop Voltage 6/6'
i = i+1
mcp_times(i).sc_id = 4
mcp_times(i).from  = '20010818'
mcp_times(i).to    = '20011101'
mcp_times(i).label = 'Start/Stop Voltage 7/7'


i = i+1
mcp_times(i).sc_id = 1
mcp_times(i).from  = '20011101'
mcp_times(i).to    = now
mcp_times(i).label = 'Start/Stop Voltage 7/8'
i = i+1
mcp_times(i).sc_id = 2
mcp_times(i).from  = '20011202'
mcp_times(i).to    = now
mcp_times(i).label = 'Start/Stop Voltage 9/9'
i = i+1
mcp_times(i).sc_id = 3
mcp_times(i).from  = '20011101'
mcp_times(i).to    = '20020408'
mcp_times(i).label = 'Start/Stop Voltage 7/7'
i = i+1
mcp_times(i).sc_id = 4
mcp_times(i).from  = '20011101'
mcp_times(i).to    = now
mcp_times(i).label = 'Start/Stop Voltage 8/8'


i = i+1
mcp_times(i).sc_id = 3
mcp_times(i).from  = '20020408'
mcp_times(i).to    = now
mcp_times(i).label = 'Start/Stop Voltage 8/8'


mcp_times = mcp_times(0:i)

IF keyword_set(WRITE_MCP_TIMES) THEN BEGIN
   save, file = path+'mcp_times.idl',mcp_times, /verb
   return 
ENDIF 

;run through calibration procedure for each time period of a given MCP
;voltage combination, in 5 day increments to gather "lots" of data.
step = 5

;setup which reference detector, calibration pairs
ref_det = 2 & pairs = intarr(2, 12)
pairs(0, *) = ref_det & pairs(1, *) = findgen(12)
cals = fltarr(12)

result = convert_timeToSeconds(mcp_times(0).from, /t90, /mjdt)

;setup arrays to keep calibration factors as a function of time
dummy = {no:0, sc_id:0, from:result.value, to:result.value, cals:cals}
count = 0

FOR sc_id = 1, 4 DO BEGIN

    message, 'Doing for Cluster '+varprt(sc_id), /cont

    index = where(mcp_times.sc_id EQ sc_id)
    times = mcp_times(index)
    fln = 'C'+varprt(sc_id)+'_detector_intercal.idl'
 
    file = findfile(path+fln, count = n)
    IF n NE 0 THEN BEGIN
        restore, file = path+fln, /verb
        time_idx = max(cals_trend.no) +1
    ENDIF ELSE BEGIN
        time_idx = 0
        cals_trend = replicate(dummy, 300)
    ENDELSE

    FOR j = 1, n_elements(times)-1 DO BEGIN
        
        ;get range of times
        result = convert_timeToSeconds(times(j).from, /t90, /mjdt)
        print, convert_secondsToTime(result.value, /t90, /mjdt)
        mjdt_start = result.value
        result = convert_timeToSeconds(times(j).to, /t90, /mjdt)
        print, convert_secondsToTime(result.value, /t90, /mjdt)
        mjdt_end = result.value
 
        mjdt_time =  mjdt_start  
        mjdt_final = mjdt_end
        WHILE mjdt_time.mjd LT mjdt_final.mjd DO BEGIN
            mjdt_start.mjd = mjdt_time.mjd
            mjdt_end.mjd = mjdt_time.mjd+step
            print, '  ', convert_secondsToTime(mjdt_start, /t90, /mjdt), $
              ' - ', convert_secondsToTime(mjdt_end, /t90, /mjdt)

            IF count LT time_idx THEN BEGIN
                message, 'already done...', /cont
                print, ''
                count = count+1
                mjdt_time.mjd = mjdt_time.mjd+step
                continue
            ENDIF
                   
            sc = sc_id
            ;now call intercal / pitch angle routines
            m_cluster_rapid_pitch,llmit=7.0, SC_ID = sc
            IF get_err_no NE 0 THEN BEGIN
                mjdt_time.mjd = mjdt_time.mjd+step
                continue
            ENDIF 
            intercal_det
            det_intercal, pairs = pairs, cals = cals
            message, 'Calibration factors:', /cont
            print, cals/cals(ref_det)

            cals_trend(count).no =  count
            cals_trend(count).sc_id = sc_id
            cals_trend(count).from = mjdt_start
            cals_trend(count).to = mjdt_end
            cals_trend(count).cals = cals

            mjdt_time.mjd = mjdt_time.mjd+step
            count = count+1
            save, file = path+fln, cals_trend, /verb
            IF count GT n_elements(cals_trend) THEN $
              cals_trend = [cals_trend, dummy]

        ENDWHILE    

    ENDFOR

    ;save cals file for this spacecraft
    cals_trend = cals_trend(0:count-1)
    save, file = path+fln, cals_trend, /verb

ENDFOR 

END


;******************************************************************************
PRO plot_rapid_iims_intercal,  SC_ID = SC_ID,  PS = PS

COMMON mjdt, mjdt_start, mjdt_end     ;common time limit in mjdt
COMMON papco_color_names
COMMON shift_label, down

sc_id = papco_keyword(SC_ID, 1)
path = '/u/friedel/tmp/cluster_rapid/'

!p.position = [0.1, 0.1, 0.8, 0.85]
!Y.ticks=0 & !X.ticks=6
!p.position = 0
!P.REGION = 0
!p.multi = [0, 2, 2, 0, 0]
!X.margin = [7, 6]
!Y.margin = [4, 6]

IF keyword_set(ps) THEN BEGIN
    set_plot, 'PS'
    file='/u/friedel/tmp/cluster_rapid/rapid_iims_det_intercal_trend.ps'
    device, filename = file, /color, ysize=22, yoffset = 4
    !P.charsize = 0.8
    !x.charsize = 0.9
ENDIF ELSE BEGIN
    set_plot, 'X'
    !P.charsize = 1.5
    !x.charsize = 0.9
    window, 3, ypos = 300, xpos = 30, xsize = 900, ysize = 900
    wset, 3      
    erase
ENDELSE

restore, path+'mcp_times.idl', /verb


FOR k = 0, 3 DO BEGIN
    !p.multi(0) = k+1
    sc_id = 4-k

    fln = 'C'+varprt(sc_id)+'_detector_intercal.idl'
    restore, file = path+fln

    n = n_elements(cals_trend)
    mjdt_start = cals_trend(0).from
    mjdt_end = cals_trend(n-1).from

    time = cals_trend.from.mjd+cals_trend.from.t/86400.0d
    xut1 = time(0) & xut2 = time(n-1)
    yst = 0 & yen = 2.0
    panel = [0, 1, 1]
    down = 0

    title = 'RAPID IIMS Detector Intercallibration: C'+varprt(sc_id)+$
      '!CRef Det: 2'

    papco_y_label, yst, yen, log=0
    extra_plotPar={xrange:[xut1,xut2], yrange:[yst,yen], color: black, $
                   ylog:0, xtickformat:'noticks', title:title}

    papco_draw_time_axis, panel, OVERPLOT=OVERPLOT, _extra=extra_plotPar    

    plot, time, cals_trend.cals(0), _extra=extra_plotPar, psym = -5, $
      xstyle = 5, ystyle = 5

    FOR i = 1, 11 DO oplot, time, cals_trend.cals(i), color = black+i

    ;indicate changes in MCP Voltages
    index = where(mcp_times.sc_id EQ sc_id)
    times = mcp_times(index) &  nt = n_elements(times)
    mcpt = dblarr(nt)
    FOR i = 0, nt-1 DO BEGIN
        result = convert_timeToSeconds(times(i).from, /t90, /mjdt)
        mcpt(i) = result.value.mjd+result.value.t/86400.0d
        oplot, [mcpt(i), mcpt(i)], [yst,yen], color = red, thick = 2, $
          linestyle = 2
    ENDFOR

ENDFOR

FOR det = 0, 11  DO xyouts, 0.475, 0.8-det*0.05, 'Det '+varprt(det), $
  color = black+det, /normal

IF keyword_set(ps) THEN BEGIN
    device, /close
    IF ps EQ 2 THEN spawn, 'lpr -P w108argb '+file
    IF ps EQ 3 THEN spawn, 'lpr -P n114rgb '+file
ENDIF

END
