;******************************************************************************
;* PROCEDURE:     
;*      p_cluster_rapid_4sc, panel, plotinfo, OUTPUT=OUTPUT, $
;*           OVERPLOT=OVERPLOT,$
;*           PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           SUBTABLE=SUBTABLE, _EXTRA=extra_par
;* 
;* DESCRIPTION:  
;*	This procedure plots data of type data_type for inclusion in papco.  
;*      Data must have been read and placed into common blocks before this
;*      routine is called. In general, the following plot conventions are used:
;*      	- if the panel is at the bottom plot time axis plus any
;*                additional axis (ie ephemeris) needed.
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a papco_colorbar
;*              - panels all have common time
;*              - y-scaling (and/or z-scaling for color plots) is either
;*              - automatic or manually set using common yscale or zscale    
;*
;* INPUTS:       
;*	panel	three element integer vector. Controls positioning of the plot
;*              (see papco_conventions.text in $papco)
;*      type    optional parameter for options. Normally is a four element
;*              integer vector. This is normally equivalent to the field
;*              typeVector of the papco structure PAPCD_PLOTINFO which is set
;*              by the panel editor. Additional fields that can be used if
;*              needed are papco_plotinfo.ioptions (20 element vector) and 
;*              PAPCO_PLOTINFO.channel (integer). If your plot routine needs
;*              more input fields than these you have to make use of
;*              an extra common block.
;*              
;* OUTPUTS:   	none
;*
;* KEYWORDS:
;*      OUTPUT  if set call the papco routine that outputs plot data
;*              to a file. If equal to 2, make plot data and return,
;*              do not plot anything! Used by slices.
;*    OVERPLOT  if you support this you can alter the appearance of your
;*              plot if it is an overplot to an exisiting panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mag. field
;*              (see p_crres_mag for an example).
;* PLOTS_ATTHISPOSITION  used to control the left side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and line style.   
;* 
;* CALLING SEQUENCE:
;*       p_cluster_rapid, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY:       
;*     written april 1996, Reiner Friedel
;*     added OUTPUT support, July 1998, Reiner Friedel
;******************************************************************************
pro p_cluster_rapid_4sc, panel, PlotInfo, OUTPUT=OUTPUT, $
                      OVERPLOT=OVERPLOT, $
                      PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
                      SUBTABLE=SUBTABLE, _EXTRA=extra_par

; The following common blocks are needed for PAPCO:
  common mjdt, mjdt_start, mjdt_end     ;common time limit in mjdt
  common yscale, yscl			;man/auto yscaling
  common zscale, zscl			;man/auto zscaling
  common shift_label, down              ;common for x-axis label shifting
  common coordinateSystems, plotted_x, plotted_y  
                                        ;info on coords used by mouse
  COMMON papco_color_names
  cl_col = [black, red, green, blue]

; One of the common blocks should contain the plot data as returned by the
; data read procedure r_cluster_rapid. 
  common cluster_rapid_data, input_header, input_data
  common cluster_rapid_4sc, data4sc_timed
  common cluster_rapid_slice_1, time, zmat, data, yarr, utitle, uytitle, uztit
  common cluster_rapid

  sc_id = control.sc_id
  data_source = control.data_src
  data_type = control.product
  plot_type =  control.pl_type
  detector = control.det > 0 < 8
  sector = control.sect
  energy = control.ek-1
  units = control.units
  sc4_flag =  control.prod_4sc
  
  convert_t90_to_date, mjdt_start, yr,doy,hr,min,sec,mo,cmo,dom, /mjdt
  tai_1=UTC2TAI({YEAR:yr, MONTH:mo, DAY:dom, HOUR:hr, MINUTE:min, $
                 SECOND:sec, MILLISECOND:0})
  convert_t90_to_date, mjdt_end, yr,doy,hr,min,sec,mo,cmo,dom, /mjdt
  tai_2=UTC2TAI({YEAR:yr, MONTH:mo, DAY:dom, HOUR:hr, MINUTE:min, $
                 SECOND:sec, MILLISECOND:0})

  ;4sc data already with time ready for plotting - comes from 
  ;p_cluster_rapid_spec.
  xut1=0  &  xut2=tai_2-tai_1  
  time = data4sc_timed.time
  ndat=n_elements(time(*, 0))

  ;select data to plot
  IF_type = IF_names(data_type)
  n_ch = input_header.NBANDS
  n_sec = input_header.NSECTORS
  n_det = input_header.NDETECTORS
  utitle = 'RAPID '+IF_type


  if control.smooth ne 0 then begin ;set smoothing
    sm = control.smooth
    utitle=utitle+'!CSmooth '+varprt(sm)
  endif else sm=0

  CASE sc4_flag OF             ;compute the 4sc product needed
    1:BEGIN  ;deviation from average
      ;here find average of data at each time then find standard
      ;deviation of data from that average. 
      ;Gives measure of "differntness" of data to flag interesting times.
      ymat = fltarr(ndat, 4)
      IF sm LT 2 THEN FOR i= 1,4 DO ymat(*, i-1)=data4sc_timed.(i)(*, energy) $
      ELSE FOR i =1,4 DO ymat(*, i-1)=smooth(data4sc_timed.(i)(*, energy), sm)
      ek_str = strcompress( $
                string(input_header.CH_POSITIONS(0,energy,0,0), $
                       input_header.CH_POSITIONS(1,energy,0,0), $
                       format = "(f6.1,'-',f6.1,' keV')"), /remove_all)
      utitle=sc_names(sc_id)+'!C'+utitle+'!CSt. Dev./ mean!C'+ek_str
      yarr = fltarr(ndat)
      FOR i = 0l, ndat-1 DO BEGIN
        vec_4sc = ymat(i, *)
        result = moment(vec_4sc, SDEV = sdev) 
        mean = result(0)
        vec_4sc_diff = abs(vec_4sc-mean)
        yarr(i) = (max(vec_4sc_diff)-min(vec_4sc_diff))/mean
        yarr(i) = sdev/mean
      ENDFOR
      uytitle = 'St. Dev./ mean'
    END
    2:BEGIN  ;total variance

    END
    3:BEGIN  ;spacecraft Intercal
      ;for a given channel, plot line plots of data for all 4 spacecraft
      ymat = fltarr(ndat, 4)
      IF sm LT 2 THEN FOR i= 1,4 DO ymat(*, i-1)=data4sc_timed.(i)(*, energy) $
      ELSE FOR i =1,4 DO ymat(*, i-1)=smooth(data4sc_timed.(i)(*, energy), sm)
      ek_str = strcompress( $
                string(input_header.CH_POSITIONS(0,energy,0,0), $
                       input_header.CH_POSITIONS(1,energy,0,0), $
                       format = "(f6.1,'-',f6.1,' keV')"), /remove_all)
      utitle=sc_names(sc_id)+'!C'+utitle+'!CIntercal!C'+ek_str
    END
    ELSE: BEGIN
    END
  ENDCASE

  if keyword_set(OUTPUT) then if output eq 2 then return  
  
  papco_Set_SubTable, SUBTABLE          ;sets the papco color table index
  panelset,panel		        ;sets the panel position viewport

  CASE sc4_flag OF             ;plot the 4sc product needed
    1:BEGIN  ;deviation from average
      if (yscl(panel(0),0) eq 1) then begin 
        yst=yscl(panel(0),1)  &  yen=yscl(panel(0),2)
      endif else BEGIN
        papco_autorange,yarr,yst,yen, $
          log=yscl(panel(0),3), exclude=0, nodata=1e-20
        yscl(panel(0),1)=yst  &  yscl(panel(0),2)=yen
      endelse      

      papco_y_label, yst, yen, log=yscl(panel(0),3) 
    
      extra_plotPar={xrange:[xut1,xut2], yrange:[yst,yen], $
                     ylog:yscl(panel(0),3), xtickformat:'noticks'}  
      ;add keyword structure set locally and the one passed in
      extra_plotPar=create_struct(extra_plotPar, extra_par)

      down=0
      papco_draw_time_axis, panel, OVERPLOT=OVERPLOT, _extra=extra_plotPar

      idx = where(yarr NE 0)
      plot, time(idx), yarr(idx), xstyle=5, ystyle=5, _extra=extra_plotPar

      right_side_label,panel,utitle
    END
    2:BEGIN  ;total variance

    END
    3:BEGIN  ;spacecraft Intercal
    
      if (yscl(panel(0),0) eq 1) then begin 
        yst=yscl(panel(0),1)  &  yen=yscl(panel(0),2)
      endif else BEGIN
        papco_autorange,ymat,yst,yen, $
          log=yscl(panel(0),3), exclude=0, nodata=1e-20
        yscl(panel(0),1)=yst  &  yscl(panel(0),2)=yen
      endelse      

      papco_y_label, yst, yen, log=yscl(panel(0),3) 
    
      extra_plotPar={xrange:[xut1,xut2], yrange:[yst,yen], $
                     ylog:zscl(panel(0),3), xtickformat:'noticks'}  
      ;add keyword structure set locally and the one passed in
      extra_plotPar=create_struct(extra_plotPar, extra_par)

      down=0
      papco_draw_time_axis, panel, OVERPLOT=OVERPLOT, _extra=extra_plotPar
    
      FOR i = 0, 3 DO BEGIN
        idx = where(ymat(*, i) NE 0, c)
        extra_plotPar.color = cl_col(i)
        IF c LT 2 THEN GOTO,  out_plot
        plot, time(idx), ymat(idx, i), xstyle=5, ystyle=5, _extra=extra_plotPar
        xyouts,!P.position(0)+0.01+0.135*i,!P.position(3)-0.02,$
          sc_names(i), $
          /normal,charsize=!p.charsize,color=extra_plotPar.color
      ENDFOR
      right_side_label,panel,utitle
      uytitle = uztit
    END
  ENDCASE

  out_plot:
; plot y-axis label at left of plot. Use scalable routine!  
  left_side_label,panel,uytitle,/rot90

; store the coordinate information into common block
  plotted_x = !x  &  plotted_y = !y


; check if data needs to be output to file. Use the optional keywords
; to add more information to the data file
;      DESCRIPTION   a string or string array describing the data
;      TIME_LABEL    a string decribing the x-axis format
;      CHANNELS      an (ch,2) array giving the channel start/end
;      Y_LABEL       a string with the y-axis label
;      Z_LABEL       a string with the z-axis label
;      RS_LABEL      a string with the righ-side label of the plot
;      NODATA        contains the no data flag value.
  if KEYWORD_SET(OUTPUT) then begin
    message,'Writing plot data out to file'
    description='cluster_rapid template sample data'
    time_label='Seconds since start of day'
    channels=yarr
    y_label=uytitle
    z_label=uztit
    rs_label=utitle
    nodata=0
    papco_write_data, time, zmat, $
                      DESCRIPTION=description, TIME_LABEL=time_label, $
                      CHANNELS=channels, Y_LABEL=y_label, Z_LABEL=z_label, $
                      RS_LABEL=rs_label, NODATA=nodata
  endif
   
end 
