;******************************************************************************
;* PROCEDURE:     
;*      p_cluster_rapid_4sc, panel, plotinfo, OUTPUT=OUTPUT, $
;*           OVERPLOT=OVERPLOT,$
;*           PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           SUBTABLE=SUBTABLE, _EXTRA=extra_par
;* 
;* DESCRIPTION:  
;*	This procedure plots data of type data_type for inclusion in papco.  
;*      Data must have been read and placed into common blocks before this
;*      routine is called. In general, the following plot conventions are used:
;*      	- if the panel is at the bottom plot time axis plus any
;*                additional axis (ie ephemeris) needed.
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a papco_colorbar
;*              - panels all have common time
;*              - y-scaling (and/or z-scaling for color plots) is either
;*              - automatic or manually set using common yscale or zscale    
;*
;* INPUTS:       
;*	panel	three element integer vector. Controls positioning of the plot
;*              (see papco_conventions.text in $papco)
;*      type    optional parameter for options. Normally is a four element
;*              integer vector. This is normally equivalent to the field
;*              typeVector of the papco structure PAPCD_PLOTINFO which is set
;*              by the panel editor. Additional fields that can be used if
;*              needed are papco_plotinfo.ioptions (20 element vector) and 
;*              PAPCO_PLOTINFO.channel (integer). If your plot routine needs
;*              more input fields than these you have to make use of
;*              an extra common block.
;*              
;* OUTPUTS:   	none
;*
;* KEYWORDS:
;*      OUTPUT  if set call the papco routine that outputs plot data
;*              to a file. If equal to 2, make plot data and return,
;*              do not plot anything! Used by slices.
;*    OVERPLOT  if you support this you can alter the appearance of your
;*              plot if it is an overplot to an exisiting panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mag. field
;*              (see p_crres_mag for an example).
;* PLOTS_ATTHISPOSITION  used to control the left side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and line style.   
;* 
;* CALLING SEQUENCE:
;*       p_cluster_rapid, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY:       
;*     written april 1996, Reiner Friedel
;*     added OUTPUT support, July 1998, Reiner Friedel
;******************************************************************************
pro p_cluster_rapid_4sc, panel, PlotInfo, OUTPUT=OUTPUT, $
                      OVERPLOT=OVERPLOT, $
                      PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
                      SUBTABLE=SUBTABLE, _EXTRA=extra_par

; The following common blocks are needed for PAPCO:
COMMON mjdt, mjdt_start, mjdt_end     ;common time limit in mjdt
COMMON yscale, yscl			;man/auto yscaling
COMMON zscale, zscl			;man/auto zscaling
COMMON shift_label, down              ;common for x-axis label shifting
COMMON coordinateSystems, plotted_x, plotted_y  
                                        ;info on coords used by mouse
COMMON papco_color_names
cl_col = [black, red, green, blue]

; One of the common blocks should contain the plot data as returned by the
; data read procedure r_cluster_rapid. 
COMMON cluster_rapid_data, input_header, input_data
COMMON cluster_rapid_4sc, data4sc_timed
COMMON cluster_mag_curv, curv_header, curv_data
COMMON cluster_gyro_rad, gyro_header, gyro_data

COMMON cluster_rapid_slice_1, time, zmat, data, yarr, utitle, uytitle, uztit
COMMON cluster_rapid 
control = *plotinfo.USR_PTR1
sc_id = control.sc_id
data_source = control.data_src
data_type = control.product
plot_type =  control.pl_type

;detector = control.detector_average_from_val > 0 < 8
;sector = control.sect
;energy = control.ek-1>0
;units = control.units
sc4_flag =  control.prod_4sc
grad_coords =  control.grad_xyz
grad_vec =  control.grad_prj
energy = control.energy_average_from_val
units = control.cal_info

tai_1=utc2tai({mjd:mjdt_start.mjd, time:mjdt_start.t*1000})
tai_2=utc2tai({mjd:mjdt_end.mjd, time:mjdt_end.t*1000})    

;4sc data already with time ready for plotting - comes from 
;p_cluster_rapid_spec.
xut1=0  &  xut2=tai_2-tai_1  

CASE sc4_flag OF
    5: time = curv_data.tai-tai_1
    6: time = gyro_data.tai-tai_1
    ELSE: time = data4sc_timed.time
ENDCASE

ndat=n_elements(time(*, 0))

;select data to plot
CASE data_source OF 
    0: BEGIN            ;-- prepare plot arrays for RAL IF files ---
        type = if_names(data_type)
    END 
    1: BEGIN            ;-- prepare plot arrays for MPAE Science files --
        type = sci_names(data_type)
    END 
    2: BEGIN            ;--LANL pitch files - calib info is set by m_pitch 
        type = lanl_names(data_type)
    END 
ENDCASE

n_ch = input_header.NBANDS
n_sec = input_header.NSECTORS
n_det = input_header.NDETECTORS
utitle = 'RAPID '+type

if control.smooth ne 0 then begin ;set smoothing
    sm = control.smooth
    utitle=utitle+'!CSmooth '+varprt(sm)
endif else sm=0

CASE sc4_flag OF             ;compute the 4sc product needed
    1:BEGIN  ;deviation from average
      ;here find average of data at each time then find standard
      ;deviation of data from that average. 
      ;Gives measure of "differntness" of data to flag interesting times.
      ymat = fltarr(ndat, 4)
      IF sm LT 2 THEN FOR i= 1,4 DO ymat(*, i-1)=data4sc_timed.(i)(*, energy) $
      ELSE FOR i =1,4 DO ymat(*, i-1)=smooth(data4sc_timed.(i)(*, energy), sm)
      ek_str = strcompress(string(input_header.CH_POSITIONS(0,energy,0,0), $
                                  input_header.CH_POSITIONS(1,energy,0,0), $
                           format = "(f6.1,'-',f6.1,' keV')"), /remove_all)
      utitle=sc_names(sc_id)+'!C'+utitle+'!CSt. Dev./ mean!C'+ek_str
      yarr = fltarr(ndat)
      FOR i = 0l, ndat-1 DO BEGIN
          vec_4sc = ymat(i, *)
          result = moment(vec_4sc, SDEV = sdev) 
          mean = result(0)
          vec_4sc_diff = abs(vec_4sc-mean)
          yarr(i) = (max(vec_4sc_diff)-min(vec_4sc_diff))/mean
          yarr(i) = sdev/mean
      ENDFOR
      uytitle = 'St. Dev./ mean'
    END
    2:BEGIN  ;total variance

    END
    3:BEGIN  ;spacecraft Intercal
      ;for a given channel, plot line plots of data for all 4 spacecraft
      ymat = fltarr(ndat, 4)
      IF sm LT 2 THEN FOR i= 1,4 DO ymat(*, i-1)=data4sc_timed.(i)(*, energy) $
      ELSE FOR i =1,4 DO ymat(*, i-1)=smooth(data4sc_timed.(i)(*, energy), sm)
      ek_str = strcompress(string(input_header.CH_POSITIONS(0,energy,0,0), $
                                  input_header.CH_POSITIONS(1,energy,0,0), $
                           format = "(f6.1,'-',f6.1,' keV')"), /remove_all)
      utitle=sc_names(sc_id)+'!C'+utitle+'!CIntercal!C'+ek_str
    END
    4:BEGIN 
        GRADIOMETER_CALC, ymat,  idx4scdata, mesocentre, interp_all_sc_pos
        ek_str = strcompress( $
                  string(input_header.CH_POSITIONS(0,energy,0,0), $
                         input_header.CH_POSITIONS(1,energy,0,0), $
                         format = "(f6.1,'-',f6.1,' keV')"), /remove_all)
        utitle='Gradient'+'!C'+'calculation'+'!C'+utitle+'!C'+ek_str
    END 
    5: BEGIN ;mag field curvature
        uytitle = 'Radius (Re)'
        yarr = curv_data.radius
        utitle='Field Line Curvature!Cfrom MFE!CAv. over ' + $
               varprt(control.smooth*4)+' sec'
    END     
    6: BEGIN ;hyro radius
        uytitle = 'Gyro Radius (Re)'
        yarr = transpose(gyro_data.data) / 6731.0
        utitle='Electron!CGyroradius!CAv. over ' + $
               varprt(control.smooth*4)+' sec'
    END     
    ELSE: BEGIN
    END
ENDCASE

if keyword_set(OUTPUT) then if output eq 2 then return  
  
papco_Set_SubTable, SUBTABLE          ;sets the papco color table index
panelset,panel		        ;sets the panel position viewport

CASE sc4_flag OF             ;plot the 4sc product needed
    1:BEGIN  ;deviation from average
      if (yscl(panel(0),0) eq 1) then begin 
        yst=yscl(panel(0),1)  &  yen=yscl(panel(0),2)
      endif else BEGIN
        papco_autorange,yarr,yst,yen, $
          log=yscl(panel(0),3), exclude=0, nodata=1e-20
        yscl(panel(0),1)=yst  &  yscl(panel(0),2)=yen
      endelse      

      papco_y_label, yst, yen, log=yscl(panel(0),3) 
    
      extra_plotPar={xrange:[xut1,xut2], yrange:[yst,yen], $
                     ylog:yscl(panel(0),3), xtickformat:'noticks'}  
      ;add keyword structure set locally and the one passed in
      extra_plotPar=create_struct(extra_plotPar, extra_par)

      down=0
      papco_draw_time_axis, panel, OVERPLOT=OVERPLOT, _extra=extra_plotPar

      idx = where(yarr NE 0)
      plot, time(idx), yarr(idx), xstyle=5, ystyle=5, _extra=extra_plotPar

      right_side_label,panel,utitle
    END
    2:BEGIN  ;total variance

    END
    3:BEGIN  ;spacecraft Intercal
        if (yscl(panel(0),0) eq 1) then begin 
            yst=yscl(panel(0),1)  &  yen=yscl(panel(0),2)
        endif else BEGIN
            papco_autorange,ymat,yst,yen, $
              log=yscl(panel(0),3), exclude=0, nodata=1e-20
            yscl(panel(0),1)=yst  &  yscl(panel(0),2)=yen
        endelse      

        papco_y_label, yst, yen, log=yscl(panel(0),3) 
    
        extra_plotPar={xrange:[xut1,xut2], yrange:[yst,yen], $
                       ylog:zscl(panel(0),3), xtickformat:'noticks'}  
        ;add keyword structure set locally and the one passed in
        extra_plotPar=create_struct(extra_plotPar, extra_par)
        down=0
        papco_draw_time_axis, panel, OVERPLOT=OVERPLOT, _extra=extra_plotPar
    
        FOR i = 0, 3 DO BEGIN
          idx = where(ymat(*, i) NE 0, c)
          extra_plotPar.color = cl_col(i)
          IF c LT 2 THEN GOTO,  out_plot
          plot, time(idx), ymat(idx, i), xstyle=5, ystyle=5, $
            _extra=extra_plotPar
          xyouts,!P.position(0)+0.01+0.135*i,!P.position(3)-0.02,$
            sc_names(i), $
            /normal,charsize=!p.charsize,color=extra_plotPar.color
        ENDFOR
        right_side_label,panel,utitle
        uytitle = uztit
        print, uztit
    END

    4:BEGIN  ;GRADIOMETER

        reset_yscl = yscl

        ;*select components to plot
        gradcoord_names = ['X COMP', 'Y COMP',  'Z COMP']
        planeplot_names = ['X-Z', 'Y-Z', 'X-Y']
        ;*this bit selects which set of indices to use for a given energy
        ;channel, the structure contains all 6 energy channel indexes
        idx = ({idx0:idx4scdata(0, *), idx1:idx4scdata(1, *), $
                idx2:idx4scdata(2, *), idx3:idx4scdata(3, *), $
                idx4:idx4scdata(4, *), idx5:idx4scdata(5,*)})
        idx0_n = where(idx.idx0 NE 0)
        idx0 = idx.idx0(idx0_n)
        idx1_n = where(idx.idx1 NE 0)
        idx1 = idx.idx1(idx1_n)
        idx2_n = where(idx.idx2 NE 0)
        idx2 = idx.idx2(idx2_n)
        idx3_n = where(idx.idx3 NE 0)
        idx3 = idx.idx3(idx3_n)
        idx4_n = where(idx.idx4 NE 0)
        idx4 = idx.idx4(idx4_n)
        idx5_n = where(idx.idx5 NE 0)
        idx5 = idx.idx5(idx5_n)

        CASE energy OF
            0:idx = idx0
            1:idx = idx1
            2:idx = idx2
            3:idx = idx3
            4:idx = idx4
            5:idx =  idx5
        ENDCASE 

        ;DO AVERAGING HERE - let averaging input be :
        ;equal to number of data points to average over
        smoothing_value = control.smooth

        IF smoothing_value GT 2 THEN begin
        smooth_data_x = smooth(ymat(0,*, energy), smoothing_value,/edge_truncate, /NAN)
        smooth_data_y = smooth(ymat(1,*, energy), smoothing_value, /edge_truncate, /NAN)
        smooth_data_z = smooth(ymat(2,*, energy), smoothing_value, /edge_truncate,/NAN)
        ymat(0, *, energy) = smooth_data_x
        ymat(1, *, energy) = smooth_data_y
        ymat(2, *, energy) = smooth_data_z
        endif
        if (control.grad_xyz eq 0) THEN BEGIN

        original_P = !P.POSITION;

        height = original_P(3)-original_P(1);find numbers defining dim of orig Panel
        width =  original_P(2)-original_P(0)
        original_centre_y = original_P(1)+(height/2)
        vec_width = .99*(original_P(0));scale box width to 90% of original x(0)of original panel
        vec_height = vec_width
        start_y = original_centre_y -(vec_height/2)
        scale_y_label = (height)/(vec_height)
        new_ylabel_max =(scale_y_label)
        yst = -new_ylabel_max
        yen = new_ylabel_max

        extra_plotPar={xrange:[xut1,xut2], yrange:[yst,yen], $
                       ylog:yscl(panel(0),3), xtickformat:'noticks'}  

        index = size(time);find number of step required
        index = index(1)
        time_stamps = (xut2-xut1)/4
        delta_x = width/(time_stamps-1);set up step length
        step_x = dblarr(time_stamps);define new 'time' series,in panel
        step_x(0) = original_P(0); set first point as first x in original

        FOR j = 1, index-1 DO BEGIN
            step_x(j) = step_x(j-1)+delta_x
        END

        new_start_x = step_x-(vec_width/2); set new start coordinat such that window is centred on first time step

FOR k = 0, index-2 DO begin

!P.POSITION = [new_start_x(k), start_y, new_start_x(k)+vec_width, start_y+vec_height]
xrange_scale = 1
!x.range=[-xrange_scale,xrange_scale]
!y.range =[-xrange_scale,xrange_scale] 
;plot,([0, 1], [0, 1])


if (control.grad_prj) EQ 0 THEN begin
plot,[0, (ymat(0,k, energy))], [0, (ymat(2, k, energy))], xstyle=5, ystyle=5, color = 1;_extra=extra_plotPar
planeplot_names1 = planeplot_names(0)
ENDIF
IF (control.grad_prj) EQ 1 THEN begin
plot,[0, (ymat(1,k, energy))], [0, (ymat(2, k, energy))], xstyle=5, ystyle=5, color = 1;_extra=extra_plotPar
planeplot_names1 = planeplot_names(1)
endif
IF (control.grad_prj) EQ 2 THEN begin
plot,[0, (ymat(0,k, energy))], [0, (ymat(1, k, energy))], xstyle=5, ystyle=5, color = 1; _extra=extra_plotPar
planeplot_names1 = planeplot_names(2)
ENDIF

ENDFOR
!P.POSITION = original_P
xyouts,!P.position(0)+0.01+0.13,!P.position(3)-0.02,$
planeplot_names1, $$
          /normal,charsize=!p.charsize, color = 1

;------------------------------------------
;----------------------------------------------------------

     right_side_label,panel,utitle
; uytitle = 'delta '+ uytitle+'/delta km'   
uytitle =  'normalised' + '!C'+'units'


ENDIF ELSE begin
     if (yscl(panel(0),0) eq 1) then begin 
        yst=yscl(panel(0),1)  &  yen=yscl(panel(0),2)

      endif else BEGIN
        papco_autorange,ymat(*, *, energy),yst,yen, $
          log=yscl(panel(0),3), exclude=0, nodata=1e-20
        yscl(panel(0),1)=yst  &  yscl(panel(0),2)=yen
;******for normalised gradients units , use 1.1 and -1.1 as limits 
    yst = -1.1 & yen =  1.1

 endelse    

      papco_y_label, yst, yen, log=yscl(panel(0),3)
endelse  

;      papco_y_label, yst, yen, log=yscl(panel(0),3) 
    
      extra_plotPar={xrange:[xut1,xut2], yrange:[yst,yen], $
                     ylog:yscl(panel(0),3), xtickformat:'noticks'}  
      ;add keyword structure set locally and the one passed in
      extra_plotPar=create_struct(extra_plotPar, extra_par)

      down=0
      papco_draw_time_axis, panel, OVERPLOT=OVERPLOT, _extra=extra_plotPar




  for i=0,n_elements(gradcoord_names)-1 do BEGIN
    if (control.grad_xyz and 2l^i) eq 2l^i THEN BEGIN 
;--------------------------------------------------------------------
;--------------------------------------------

       extra_plotPar.color = cl_col(i);idx4scdata calculated in GRADIOMETER_CALC.pro, disgards any 


stop

plot, time(*), ymat(i, *, energy), xstyle=5, ystyle=5, _extra=extra_plotPar

   xyouts,!P.position(0)+0.01+0.135*i,!P.position(3)-0.02,$
gradcoord_names(i), $$
          /normal,charsize=!p.charsize,color=extra_plotPar.color
     right_side_label,panel,utitle
 uytitle = 'delta '+ uytitle+'/delta km'   
uytitle =  'normalised' + '!C'+'units'
ENDIF

ENDFOR

    END ;gradiometer 

    5: BEGIN                    ;RADIUS OF CURVATURE
        nodata = 1e-20

        if (yscl(panel(0),0) eq 1) then begin 
            yst=yscl(panel(0),1)  &  yen=yscl(panel(0),2)
        endif else BEGIN
            papco_autorange, yarr, yst, yen, $
              log=yscl(panel(0),3), exclude=0, nodata=nodata
            yscl(panel(0),1)=yst  &  yscl(panel(0),2)=yen
        endelse      
        papco_y_label, yst, yen, log=yscl(panel(0),3)

        extra_plotPar={xrange:[xut1,xut2], yrange:[yst,yen], $
                       ylog:yscl(panel(0),3), xtickformat:'noticks'}  
        ;add keyword structure set locally and the one passed in
        extra_plotPar=create_struct(extra_plotPar, extra_par)
        
        down=0
        papco_draw_time_axis, panel, OVERPLOT=OVERPLOT, _extra=extra_plotPar

        idx = where(yarr NE nodata)
        plot, time(idx), yarr(idx), xstyle=5, ystyle=5, _extra=extra_plotPar

        right_side_label,panel,utitle

    END 

    6: BEGIN                    ;gyro radius
        nodata = 1e-20

        if (yscl(panel(0),0) eq 1) then begin 
            yst=yscl(panel(0),1)  &  yen=yscl(panel(0),2)
        endif else BEGIN
            papco_autorange, yarr, yst, yen, $
              log=yscl(panel(0),3), exclude=0, nodata=nodata
            yscl(panel(0),1)=yst  &  yscl(panel(0),2)=yen
        endelse      
        papco_y_label, yst, yen, log=yscl(panel(0),3)

        extra_plotPar={xrange:[xut1,xut2], yrange:[yst,yen], $
                       ylog:yscl(panel(0),3), xtickformat:'noticks'}  
        ;add keyword structure set locally and the one passed in
        extra_plotPar=create_struct(extra_plotPar, extra_par)
        
        down=0
        papco_draw_time_axis, panel, OVERPLOT=OVERPLOT, _extra=extra_plotPar

        idx = where(yarr NE nodata)        

        plot, time(idx), smooth(yarr(idx, 0), sm), xstyle=5, ystyle=5, $
                         _extra=extra_plotPar
        oplot, time(idx), smooth(yarr(idx, 6), sm), _extra=extra_plotPar


        right_side_label,panel,utitle

    END 

ENDCASE

  out_plot:
; plot y-axis label at left of plot. Use scalable routine!  
  left_side_label,panel,uytitle,/rot90

; store the coordinate information into common block
  plotted_x = !x  &  plotted_y = !y


; check if data needs to be output to file. Use the optional keywords
; to add more information to the data file
;      DESCRIPTION   a string or string array describing the data
;      TIME_LABEL    a string decribing the x-axis format
;      CHANNELS      an (ch,2) array giving the channel start/end
;      Y_LABEL       a string with the y-axis label
;      Z_LABEL       a string with the z-axis label
;      RS_LABEL      a string with the righ-side label of the plot
;      NODATA        contains the no data flag value.
  if KEYWORD_SET(OUTPUT) then begin
    message,'Writing plot data out to file'
    description='cluster_rapid template sample data'
    time_label='Seconds since start of day'
    channels=yarr
    y_label=uytitle
    z_label=uztit
    rs_label=utitle
    nodata=0
    papco_write_data, time, zmat, $
                      DESCRIPTION=description, TIME_LABEL=time_label, $
                      CHANNELS=channels, Y_LABEL=y_label, Z_LABEL=z_label, $
                      RS_LABEL=rs_label, NODATA=nodata
  endif
   
end 




;******************************************************************************
;******************************************************************************
;***              GRADIOMETER CALULATION                                      *
;***            MATT TAYLOR October 2002 LANL                                 *
;***    Using Harvey method, least squared calculation  of                    *
;***             interspacecraft gradients                                    *
;***                                                                          *
;******************************************************************************
PRO GRADIOMETER_CALC, gradient_data,  idx4scdata, mesocentre, interp_all_sc_pos

common cluster_rapid_data, input_header, input_data
common cluster_rapid_4sc, data4sc_timed
;common cluster_rapid_slice_1, time, zmat, data, yarr, utitle, uytitle, uztit
COMMON cluster_rapid_slice
common cluster_rapid
common cluster_eph_data, eph_header, eph_data
common mjdt, mjdt_start, mjdt_end     ;common time limit in mjdt

r_cluster_eph

;get TAI of papco start time of plot
convert_t90_to_date, mjdt_start, yr,doy,hr,min,sec,mo,cmo,dom, /mjdt
tai_1=UTC2TAI({YEAR:yr, MONTH:mo, DAY:dom, HOUR:hr, MINUTE:min, $
                 SECOND:sec, MILLISECOND:0})

;get CDF epoch of papco start time of plot
CDF_EPOCH,epoch_1, yr, mo, dom, hr, min, sec, 0, /COMPUTE_EPOCH

convert_t90_to_date, mjdt_end, yr,doy,hr,min,sec,mo,cmo,dom, /mjdt
tai_2=UTC2TAI({YEAR:yr, MONTH:mo, DAY:dom, HOUR:hr, MINUTE:min, $
                 SECOND:sec, MILLISECOND:0})

;cdf epoch is in milliseconds since 01 Jan 00:00:00.000
offset = tai_1-(epoch_1/1000.0);offset between start time of plot and epoch time

;4sc time is in TAI

data4sc_timed.time = data4sc_timed.time+tai_1
tai_4sc = (data4sc_timed.time(*, 0)+data4sc_timed.time(*, 1)) /2.0

;EPH is in CDF EPOCH time 
tai_eph=(eph_data.EPOCH)/1000.0+offset

;print, tai2utc(tai_4sc(0), /ecs)
;print, tai2utc(tai_eph(0), /ecs)
;reduce eph time to papco start end
index = where((tai_eph GE tai_1) AND (tai_eph LE tai_2), count)
IF count EQ 0 THEN message, 'no eph data found' ELSE message,  'eph data loaded', /continue


eph_data_4sc = eph_data(index)

ref_sc_pos = eph_data_4sc.sc_r_xyz_gse
delta_sc1 = eph_data_4sc.sc_dr1_xyz_gse
delta_sc2 = eph_data_4sc.sc_dr2_xyz_gse
delta_sc3 = eph_data_4sc.sc_dr3_xyz_gse
delta_sc4 = eph_data_4sc.sc_dr4_xyz_gse

sc1_pos = ref_sc_pos + delta_sc1
sc2_pos = ref_sc_pos + delta_sc2
sc3_pos = ref_sc_pos + delta_sc3
sc4_pos = ref_sc_pos + delta_sc4



size_arr = size(tai_4sc)
sc1_ipos =dblarr(3, size_arr(1))
sc2_ipos =dblarr(3, size_arr(1))
sc3_ipos =dblarr(3, size_arr(1))
sc4_ipos =dblarr(3, size_arr(1))

;interpolate spacecraft position on to same res as time (ie 4 seconds)

sc1_ipos(0, *)= INTERPOL(sc1_pos(0, *), tai_eph(index),tai_4sc, /spline)
sc1_ipos(1, *)= INTERPOL(sc1_pos(1, *), tai_eph(index),tai_4sc, /spline)
sc1_ipos(2, *)= INTERPOL(sc1_pos(2, *), tai_eph(index),tai_4sc, /spline)

sc2_ipos(0, *)= INTERPOL(sc2_pos(0, *), tai_eph(index),tai_4sc, /spline)
sc2_ipos(1, *)= INTERPOL(sc2_pos(1, *), tai_eph(index),tai_4sc, /spline)
sc2_ipos(2, *)= INTERPOL(sc2_pos(2, *), tai_eph(index),tai_4sc, /spline)

sc3_ipos(0, *)= INTERPOL(sc3_pos(0, *), tai_eph(index),tai_4sc, /spline)
sc3_ipos(1, *)= INTERPOL(sc3_pos(1, *), tai_eph(index),tai_4sc, /spline)
sc3_ipos(2, *)= INTERPOL(sc3_pos(2, *), tai_eph(index),tai_4sc, /spline)

sc4_ipos(0, *)= INTERPOL(sc4_pos(0, *), tai_eph(index),tai_4sc, /spline)
sc4_ipos(1, *)= INTERPOL(sc4_pos(1, *), tai_eph(index),tai_4sc, /spline)
sc4_ipos(2, *)= INTERPOL(sc4_pos(2, *), tai_eph(index),tai_4sc, /spline)

interp_all_sc_pos =dblarr(4, 3, size_arr(1))
interp_all_sc_pos(0, *, *) = sc1_ipos
interp_all_sc_pos(1, *, *) = sc2_ipos
interp_all_sc_pos(2, *, *) = sc3_ipos
interp_all_sc_pos(3, *, *) = sc4_ipos


;have fourspacecraft data and eph time and position at same resolution
;define position of tetrahedron mesocentre
mesocentre = (sc1_ipos +sc2_ipos+sc3_ipos+sc4_ipos)/4.0

;relative spacecraft separations

r1 = sc1_ipos-mesocentre
r2 = sc2_ipos-mesocentre
r3 = sc3_ipos-mesocentre
r4 = sc4_ipos-mesocentre


; for four spacecraft analysis need to caluclate Volumetric tensor, a
; mathematical description of the dimensions of the tetrahedron

rr = dblarr(4, 3, size_arr(1))
R = dblarr(3,3, size_arr(1))
RA = dblarr(4, size_arr(1))
invR = R
r_meso_check =dblarr (4, 3, size_arr(1))
rr(0, *, *) = r1
rr(1, *, *) = r2
rr(2, *, *) = r3
rr(3, *, *) = r4


; set up rr here then volumetric tensor
N = 4.0;number of spacecraft
j = 0
k = 0
alpha = 0
message,  'CALCULATING VOLUMETRIC TENSOR FOR SELECTED TIME PERIOD', /continue
WHILE (j LE 2) DO begin
 WHILE (k LE 2) DO begin
  WHILE (alpha LE N-1) DO begin
  RA(alpha, *) = rr(alpha, j, *)*rr(alpha, k, *)
  alpha = alpha+1
  endwhile
 alpha = 0;
message, '.', /continue
;sumRrows = total(RA, 2);add  components up for each time step
R(j, k, *) = (1/N)*total(RA, 1)
k = k+1
endwhile
k = 0
j = j+1
ENDWHILE
size_R = size(R)
nn = size_R(3)-1
FOR i = 0,nn, 1 DO begin
;print,   i
invR(*, *, i) = invert(R(*, *, i))
ENDFOR

message,  'done', /continue
message, 'now do gradient calculation', /continue

;start gradiometer calculation
;***********need four values to interpolate across spacecraft tetrahedron.
;CALCULATE GRADIENT FOR ALL CHANNELS
; data is in data4sc_timed.zmat1,2,3,4 in time versus 6 channel array
;zmat=[6,datapoints]

data_sc1 = data4sc_timed.zmat1
data_sc2 = data4sc_timed.zmat2
data_sc3 = data4sc_timed.zmat3
data_sc4 = data4sc_timed.zmat4


data_size = size(data4sc_timed.zmat1)
data_all = dblarr (4,data_size(2),data_size(1))
data_all(0, *, *) =transpose(data4sc_timed.zmat1)
data_all(1, *, *) =transpose(data4sc_timed.zmat2)
data_all(2, *, *) =transpose(data4sc_timed.zmat3)
data_all(3, *, *) =transpose(data4sc_timed.zmat4)

sum1 = dblarr(4, 4, data_size(1))
sum2 = dblarr(3,1, data_size(1))
max_channel = data_size(2)
summer = dblarr(3,1, data_size(1))
gradient_data = dblarr(3, data_size(1), max_channel)
KK = dblarr(3, data_size(1))
channel = 0                                                     
l = 0
k = 0
alpha = 0
beta = 0
WHILE channel LE max_channel-1 DO begin
   WHILE (l LE 2) DO BEGIN
      WHILE (k LE 2) DO BEGIN
         WHILE (alpha LE N-1) DO BEGIN
            WHILE (beta LE N-1) DO begin
            sum1(alpha, beta, *) = (data_all(alpha, channel, *) - data_all(beta, channel, *))*(rr(alpha, k, *)-rr(beta,k, *))
            beta = beta+1
            ENDWHILE
         alpha = alpha+1
         beta = 0
     ENDWHILE
      summer(k,0, *) = total(total(sum1, 1), 1)
FOR i = 0,nn, 1 DO begin
      sum2(k,0, i) = (summer(k,0, i))#(invR(l, k, i))
ENDFOR

      k = k+1
      alpha = 0
      ENDWHILE
   KK(l, *) = (1/(2*N^2))*total(total(sum2, 1), 1)

   l = l+1
   k = 0
  ENDWHILE
l = 0
gradient_data(*, *, channel) = KK;
print,  'gradients for channel',channel+1,' calculated'
channel = channel+1

ymat = gradient_data




ENDWHILE

;put spacecraft positions + mesocentre in RE
RE = 6400
interp_all_sc_pos = interp_all_sc_pos/RE
mesocentre = mesocentre/RE
;********need to get rid of bad data caused by zeros in count rates.
idx3= INTARR( data_size(1))
help,  idx3
idx4scdata = INTARR(6, data_size(1))
;rint,  idx4scdata(0, 0)
help, data_all
FOR ii = 0, data_size(1)-1 DO BEGIN
;idx1(ii) =  ii
;idx2(ii) =  ii
;idxx(ii) = ii
idx3(ii) =  ii
ENDFOR
resetidx = idx3
FOR iii =  0, max_channel-1 DO BEGIN
idx3 = resetidx

  idx0 = where (data_all(0, iii, idx3) gt 0)
  idx1 = where (data_all(1, iii, idx0) gt 0)
idx1 = idx0(idx1)
  idx2 = where (data_all(2, iii, idx1) gt 0) 
idx2 = idx1(idx2)
  idx3 = where (data_all(3, iii, idx2) gt 0)
idx3 = idx2(idx3)

idxsize = size(idx3)
idx4scdata[iii, 0:idxsize(1)-1] = idx3

;************************************
;****************
ENDFOR


;***************************************************************8
;normalise gradients coordinates
size_graddata = size(ymat)
ymat1 = ymat
FOR energy = 0, max_channel-1 DO begin
FOR i = 0, size_graddata(2)-1 DO BEGIN
magnitude = sqrt( (ymat(0, i, energy))^2 + (ymat(1, i, energy))^2 + (ymat(2, i, energy))^2)
 
ymat1(*, i, energy) = ymat(*, i, energy)/magnitude
ENDFOR
ENDFOR

gradient_data = ymat1
;*************************************************************

;IF channel EQ 
;idx_allchannel = ({idx0:idx0, idx1:idx1, idx2:idx2, idx3:idx3, idx4:idx4, idx5:idx5})
;*check array OF strutures
END 





