;******************************************************************************
;* PROCEDURE:     
;*      p_cluster_rapid_spec, panel, plotinfo, OUTPUT=OUTPUT, $
;*           OVERPLOT=OVERPLOT,$
;*           PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           SUBTABLE=SUBTABLE, _EXTRA=extra_par
;* 
;* DESCRIPTION:  
;*	Plots data of type cluster_rapid  for inclusion in papco.  
;*      Data must have been read and placed into common blocks before this
;*      routine is called. 
;*
;* INPUTS:       
;*	panel	three element integer vector. Controls positioning of the plot
;*              (see papco_conventions.text in $papco)
;*      plotinfo structure controlling the plot, same as set by panel editor
;*              
;* OUTPUTS:   	none
;*
;* KEYWORDS:
;*      OUTPUT  if set call the papco routine that outputs plot data
;*              to a file. If equal to 2, make plot data and return,
;*              do not plot anything! Used by slices.
;*    OVERPLOT  if you support this you can alter thecasecase of
;*              plot if it is an overplot to an exisiting panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mag. field
;*              (see p_crres_mag for an example).
;* PLOTS_ATTHISPOSITION  used to control the left side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and line style.   
;* 
;* CALLING SEQUENCE:
;*       p_cluster_rapid, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY:       
;*     written april 1996, Reiner Friedel
;*     added OUTPUT support, July 1998, Reiner Friedel
;******************************************************************************
PRO p_cluster_rapid_spec, panel, PlotInfo, OUTPUT=OUTPUT, $
                          OVERPLOT=OVERPLOT, $
                          PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
                          SUBTABLE=SUBTABLE, _EXTRA=extra_par

; The following common blocks are needed for PAPCO:
COMMON mjdt, mjdt_start, mjdt_end     ;common time limit in mjdt
COMMON yscale, yscl		      ;man/auto yscaling
COMMON zscale, zscl		      ;man/auto zscaling
COMMON shift_label, down              ;common for x-axis label shifting
COMMON coordinateSystems, plotted_x, plotted_y  
                                        ;info on coords used by mouse
COMMON papco_color_names
color_indx = [black, red, green, blue, magenta, cyan, burgundy, $
              olive, dark_green, teal, royal_blue, violet]

; One of the common blocks should contain the plot data as returned by the
; data read procedure r_cluster_rapid. 
COMMON cluster_rapid_data, header, input_data
COMMON cluster_rapid_slice, xut1, xut2, time, zmat, data, yarr, $
                            utitle, title, uytitle, uztit, mag_fgm, aux, fgm, $
                            ndat, nodata, s1, s2, d1, d2, e1, e2, p1, p2
COMMON cluster_rapid
COMMON get_error, get_err_no, get_err_msg  

if keyword_set(OUTPUT) then output=OUTPUT else output=0

;--- get the module's control structure, and set some variables --------------
control = *plotinfo.USR_PTR1

s1 = fix(control.SECTOR_AVERAGE_FROM_VAL)
s2 = fix(control.SECTOR_AVERAGE_TO_VAL)
d1 = fix(control.DETECTOR_AVERAGE_FROM_VAL)
d2 = fix(control.DETECTOR_AVERAGE_TO_VAL)
e1 = fix(control.ENERGY_AVERAGE_FROM_VAL)
e2 = fix(control.ENERGY_AVERAGE_TO_VAL)
p1 = fix(control.PITCH_AVERAGE_FROM_VAL)
p2 = fix(control.PITCH_AVERAGE_TO_VAL)
d1 = d1 > 0 < (header.NDETECTORS-1) & d2 = d2 > 0 < (header.NDETECTORS-1)
s1 = s1 > 0 < (header.NSECTORS -1) & s2 = s2 > 0 < (header.NSECTORS -1)
p1 = p1 > 0 < 180 & p2 = p2 > 0 < 180

;make title strings for det and sec ranges
IF d1 EQ d2 THEN tit_det = 'Det '+varprt(d1) ELSE $
                 tit_det = 'Det '+varprt(d1)+'-'+varprt(d2)
IF s1 EQ s2 THEN tit_sec = 'Sec '+varprt(s1) ELSE $
                 tit_sec = 'Sec '+varprt(s1)+'-'+varprt(s2)
tit_pa = 'PA '+ varprt(p1)+'-'+varprt(p2)

valid = 0 & count=0 & nodata=1e-20 & ch_str = ''

;--- make index of channels to plot - from binary number in control.ch --------
to_plot = bytarr(16) 
FOR i = 0, 16-1 DO IF (control.ch and 2^i) EQ 2^i THEN to_plot(i) = 1
pl_idx = where(to_plot eq 1, nplot)


;--- set timerange requested in TAI -------------------------------------------
tai1=utc2tai({mjd:mjdt_start.mjd, time:mjdt_start.t*1000})
tai2=utc2tai({mjd:mjdt_end.mjd, time:mjdt_end.t*1000})    
;restrict data in time 
index=where((input_data.time GE tai1) AND (input_data.time LE tai2),c)
IF c GT 2 THEN data=input_data(index) ELSE BEGIN
    get_err_msg = 'No data to plot' & GOTO, noplot
ENDELSE
xut1=0  &  xut2=tai2-tai1  
ndat = n_elements(data) & time = dblarr(ndat,2)
time(*,0) = data.TIME-tai1 & time(*,1) = data.ENDTIME-tai1

;--- check if any filtering or PSD is needed - get ephemeris/mag data first ---
IF control.B_X_SELECT_RANGE OR control.B_TOT_SELECT_RANGE OR $
   (control.cal EQ 4) OR (control.cal EQ 5) THEN BEGIN $
   get_cluster_mag, control, mag_fgm, time
ENDIF


;--- prepare data arrays for plotting ----------------------------------------
CASE control.data_src OF

    0: BEGIN            ;-- prepare plot arrays for RAL IF files ---
        product = if_names(control.product)
        utitle = 'RAPID '+product
        idx = where(data.data LE 0, cc)
        ;check for EPAD_16 and Ek interpolate - do CLEAN on it first. 
        ;This makes it "look" like CLEAN BM data,
        ;so we can use the same code as CLEAN BM from here onward.
        pos = strpos(product, 'IES_EPAD_16')
        IF (pos NE -1) AND (control.epad_ek EQ 1) THEN BEGIN
            message, 'Interpolating EPAD...', /info
            cluster_rapid_ies_epad_ecal, control
            ;now epad data resampled to 8 energy channels 
            utitle = 'RAPID '+product+'_CLEAN'
            idx = where(data.data EQ nodata, cc)
        ENDIF

        CASE control.cal OF
              0: output_info = {div:0, rat:0, cal:0, eff:1} 
              1: output_info = {div:0, rat:1, cal:0, eff:1} 
              2: output_info = {div:0, rat:1, cal:1, eff:1} 
              3: output_info = {div:1, rat:1, cal:1, eff:1} 
              4: output_info = {div:1, rat:1, cal:1, eff:1} 
              5: output_info = {div:1, rat:1, cal:1, eff:1}
              6: output_info = {div:1, rat:1, cal:1, eff:1}              
              ELSE: output_info = {div:0, rat:0, cal:0, eff:0}  
        ENDCASE

        ;select data to plot
        n_ch = header.NBANDS
        n_sec = header.NSECTORS 
        n_det = header.NDETECTORS
        yarr = fltarr(n_ch, 2)
        uztit='[cts]' & uytitle = 'E (keV)'

        ;do IES channel conversion from ADC units to keV
        pos = strpos(product, 'IES')
        IF pos NE -1 THEN BEGIN
            ch = header.CH_POSITIONS * 2.2 + 6.5
            yarr = transpose(ch(*,*,d1,0))
            IF header.version EQ 'Adopt' THEN GOTO, done_cal
            message, 'Doing IES flux conversion', /info 
            FORWARD_FUNCTION rcs_docal
            result = rcs_docal(header, data, output_info)
            IF n_det EQ 1 THEN data.data = result $
              ELSE IF n_sec ne 1 THEN data.data = reform(result)
            IF cc NE 0 THEN BEGIN 
                temp = data.data
                temp(idx) = nodata
                data.data = temporary(temp)
            ENDIF 
            done_cal:
        ENDIF 

        e1 = e1 >  0 <  (n_ch-1) & e2 = e2 >  0 <  (n_ch-1) 
        tit_ek = 'EK '+ strcompress(string(ch(0,e1,d1,0),ch(1,e2,d1,0),$
                        format = "(f6.1,'-',f6.1)"), /remove_all)

        CASE 1 OF
            (product EQ 'IES_CLEAN'): BEGIN ;spin averaged detectors
                ;average detector range chosen
                zmat = fltarr(ndat, n_ch) & numb = zmat
                FOR i = 0, ndat-1 DO BEGIN 
                    FOR d = d1, d2 DO BEGIN
                        idx = where(data(i).data(*,d,*) NE nodata, c)
                        IF c NE 0 THEN BEGIN
                            zmat(i,idx) = zmat(i,idx)+data(i).data(*,d,idx)
                            numb(i,idx) = numb(i,idx)+1
                        ENDIF 
                    ENDFOR 
                ENDFOR 
                idx1 = where(numb NE 0, c1, comp = idx2, nc = c2)
                IF c1 NE 0 THEN zmat(idx1) =  zmat(idx1) / numb(idx1)
                IF c2 NE 0 THEN zmat(idx2) = nodata
                IF d1 EQ d2 THEN title = 'Det '+varprt(d1) ELSE $
                  title = 'Det '+varprt(d1)+'-'+varprt(d2)
                yarr = transpose(ch(*,*,d1,0))
            END 

            (product EQ 'IES_EPAD_16'): BEGIN
                m_IES_CEANBM_zmat, control

                IF NOT papco_Draw_CheckResult(PlotInfo.panelKind, panel, $
                              0, PlotInfo.typeVector, PlotInfo.channel) $
                THEN return

                CASE control.pl_type OF    
                    0: BEGIN    ;spectrum: average det, sec range chosen.
                        title = tit_det+', '+tit_sec
                        yarr = transpose(ch(*,*,d1,0))
                    END
                    1: BEGIN     ;roll
                        yarr = fltarr(n_sec, 2)
                        yarr(*, 0) = findgen(n_sec)+0.5
                        yarr(*, 1) = yarr(*, 0)+1
                        title = tit_ek + ', '+ tit_det
                        uytitle='sector #'
                        valid=[1, 2, 4, 8]
                    END 
                    3: BEGIN    ;pitch E @ Pa: Sum energy, Pa range chosen.
                        yarr = transpose(ch(*,*,d1,0))
                        IF get_err_no NE 0 THEN GOTO, noplot
                        title = tit_pa + '!C'+ tit_det + ', ' + tit_sec
                    END
                    4: BEGIN    ;pitch Pa @ E: Average Pa, E range chosen  
                        IF get_err_no NE 0 THEN GOTO, noplot
                        title = tit_ek + '!C'+ tit_det + ', ' + tit_sec
                        uytitle = 'Pitch Angle (deg)'
                        valid=[1, 2, 3, 4.5, 6, 9]
                    END 
                ENDCASE                
            END

            (product EQ 'IES_EPAD_16_PA'): BEGIN
                m_IES_CEANBM_zmat, control, /HAVE_PA

                IF NOT papco_Draw_CheckResult(PlotInfo.panelKind, panel, $
                              0, PlotInfo.typeVector, PlotInfo.channel) $
                THEN return

                CASE control.pl_type OF
                    0: BEGIN    ;spectrum: average det, sec range chosen.
                        title = tit_det+', '+tit_sec
                        yarr = transpose(ch(*,*,d1,0))
                    END
                    1: BEGIN     ;roll
                        yarr = fltarr(n_sec, 2)
                        yarr(*, 0) = findgen(n_sec)+0.5
                        yarr(*, 1) = yarr(*, 0)+1
                        title = tit_ek + ', '+ tit_det
                        uytitle='sector #'
                        valid=[1, 2, 4, 8]
                    END
                    3: BEGIN    ;pitch E @ Pa: Sum energy, Pa range chosen.
                        yarr = transpose(ch(*,*,d1,0))
                        IF get_err_no NE 0 THEN GOTO, noplot
                        title = tit_pa + '!C'+ tit_det + ', ' + tit_sec
                    END
                    4: BEGIN    ;pitch Pa @ E: Average Pa, E range chosen  
                        IF get_err_no NE 0 THEN GOTO, noplot
                        title = tit_ek + '!C'+ tit_det + ', ' + tit_sec
                        uytitle = 'Pitch Angle (deg)'
                        valid=[1, 2, 3, 4.5, 6, 9]
                    END 
                ENDCASE
            END

            (product EQ 'IES_CLEANBM'): BEGIN
                m_IES_CEANBM_zmat, control

                IF NOT papco_Draw_CheckResult(PlotInfo.panelKind, panel, $
                              0, PlotInfo.typeVector, PlotInfo.channel) $
                THEN return

                CASE control.pl_type OF
                    0: BEGIN    ;spectrum: average det, sec range chosen.
                        title = tit_det+', '+tit_sec
                        yarr = transpose(ch(*,*,d1,0))
                    END
                    1: BEGIN     ;roll
                        yarr = fltarr(n_sec, 2)
                        yarr(*, 0) = findgen(n_sec)+0.5
                        yarr(*, 1) = yarr(*, 0)+1
                        title = tit_ek + ', '+ tit_det
                        uytitle='sector #'
                        valid=[1, 2, 4, 8]
                    END
                    3: BEGIN    ;pitch E @ Pa: Sum energy, Pa range chosen.
                        yarr = transpose(ch(*,*,d1,0))
                        IF get_err_no NE 0 THEN GOTO, noplot
                        title = tit_pa + '!C'+ tit_det + ', ' + tit_sec
                    END
                    4: BEGIN    ;pitch Pa @ E: Average Pa, E range chosen  
                        IF get_err_no NE 0 THEN GOTO, noplot
                        title = tit_ek + '!C'+ tit_det + ', ' + tit_sec
                        uytitle = 'Pitch Angle (deg)'
                        valid=[1, 2, 3, 4.5, 6, 9]
                    END 
                ENDCASE
            END
            
            (product EQ 'IES_CLEANBM2'): BEGIN
                m_IES_CEANBM_zmat, control
                IF NOT papco_Draw_CheckResult(PlotInfo.panelKind, panel, $
                              0, PlotInfo.typeVector, PlotInfo.channel) $
                THEN return

                CASE control.pl_type OF
                    0: BEGIN    ;spectrum: average det, sec range chosen.
                        title = tit_det+', '+tit_sec
                        yarr = transpose(ch(*,*,d1,0))
                    END
                    1: BEGIN     ;roll
                        yarr = fltarr(n_sec, 2)
                        yarr(*, 0) = findgen(n_sec)+0.5
                        yarr(*, 1) = yarr(*, 0)+1
                        title = tit_ek + ', '+ tit_det
                        uytitle='sector #'
                        valid=[1, 2, 4, 8]
                    END
                    3: BEGIN    ;pitch E @ Pa: Sum energy, Pa range chosen.
                        yarr = transpose(ch(*,*,d1,0))
                        IF get_err_no NE 0 THEN GOTO, noplot
                        title = tit_pa + '!C'+ tit_det + ', ' + tit_sec
                    END
                    4: BEGIN    ;pitch Pa @ E: Average Pa, E range chosen  
                        IF get_err_no NE 0 THEN GOTO, noplot
                        title = tit_ek + '!C'+ tit_det + ', ' + tit_sec
                        uytitle = 'Pitch Angle (deg)'
                        valid=[1, 2, 3, 4.5, 6, 9]
                    END 
                ENDCASE
            END            

            (product EQ 'IES_CLEANBM2_PA'): BEGIN
                m_IES_CEANBM_zmat, control, /HAVE_PA
                IF NOT papco_Draw_CheckResult(PlotInfo.panelKind, panel, $
                              0, PlotInfo.typeVector, PlotInfo.channel) $
                THEN return

                CASE control.pl_type OF
                    0: BEGIN    ;spectrum: average det, sec range chosen.
                        title = tit_det+', '+tit_sec
                        yarr = transpose(ch(*,*,d1,0))
                    END
                    1: BEGIN     ;roll
                        yarr = fltarr(n_sec, 2)
                        yarr(*, 0) = findgen(n_sec)+0.5
                        yarr(*, 1) = yarr(*, 0)+1
                        title = tit_ek + ', '+ tit_det
                        uytitle='sector #'
                        valid=[1, 2, 4, 8]
                    END
                    3: BEGIN    ;pitch E @ Pa: Sum energy, Pa range chosen.
                        yarr = transpose(ch(*,*,d1,0))
                        IF get_err_no NE 0 THEN GOTO, noplot
                        title = tit_pa + '!C'+ tit_det + ', ' + tit_sec
                    END
                    4: BEGIN    ;pitch Pa @ E: Average Pa, E range chosen  
                        IF get_err_no NE 0 THEN GOTO, noplot
                        title = tit_ek + '!C'+ tit_det + ', ' + tit_sec
                        uytitle = 'Pitch Angle (deg)'
                        valid=[1, 2, 3, 4.5, 6, 9]
                    END 
                ENDCASE
            END            

            (product EQ 'IES_ESPCT') : BEGIN
                IF control.pl_type EQ 2 THEN BEGIN ;LUT
                    yarr = data.lut
                    title = 'LUT #'
                ENDIF ELSE BEGIN 
                    data.data = data.data * header.NSECTORS
                    yarr(*, 0) = ch(0,*,0,0)
                    yarr(*, 1) = ch(1,*,0,0)
                    title = 'Sum all Det'
                    zmat = transpose(data.data(0,0,*))
                    ch_str = strarr(n_ch)
                    FOR i = 0, n_ch-1 DO ch_str(i) = strcompress( $
                       string(yarr(i, *),format="(f6.1,'-',f6.1)"),/remove_all)
                       spike_value=1e6
                       idx_spike=where(zmat gt spike_value,count)
                    if count gt 1 then zmat(idx_spike)=0

                    ; do psd calc to put data in PSD
                    if control.cal eq 4 then begin
                        low_energy_channels=yarr(*,0);
                        high_energy_channels=yarr(*,1);
                        papco_hilmer_phase_space_density,low_energy_channels, $
                                          high_energy_channels,zmat,PSD,/GEM
                        zmat=PSD
                    ENDIF  

                    ;do PSD @const mu
                    IF control.cal EQ 5 and control.pl_type NE 5 THEN BEGIN 
                        ;make array of mu's  
                        papco_m_mu_arr, control.mus, 100, 10000, 15, $
                                        mus, mus_str, yminarr, ymaxarr
                        n_ch = n_elements(yminarr) 
                        psd_mat = fltarr(ndat,n_ch)
                        ;now call the mu / psd calc routine...
                        FOR i = 0, n_ch-1 DO BEGIN
                            papco_mu_calculation, yarr(*, 0), yarr(*, 1), $
                              mus(i), zmat, mag_fgm(*, 3), PSD, /GEM
                            psd_mat(*, i) = PSD
                        ENDFOR 

                        zmat = psd_mat 
                        idx = where(finite(psd_mat) EQ 0, c)
                        IF c NE 0 THEN zmat(idx) = nodata 
                        idx = where(zmat LT nodata, c)
                        IF c NE 0 THEN zmat(idx) = nodata 

                        yarr = fltarr(n_ch, 2)
                        yarr(*, 0) = yminarr & yarr(*, 1) = ymaxarr
                        
                        uztit = 'Hilmer PSD s!U3!N km!U-6!N'
                        uztit = 'Hilmer PSD c!u3!N / MeV!u3!N cm!u3!N'
                        uytitle =  '!4l!3 (MeV/G)'
                        ch_str = mus_str
                    ENDIF    
                ENDELSE  
            END

            (product EQ 'IES_PEDESTAL') : BEGIN
                yarr(*, 0) = findgen(n_ch)
                yarr(*, 1) = yarr(*, 0)+1.0
                zmat =  fltarr(ndat, n_ch)
                title = 'Det '+varprt(detector)
                uytitle = 'channel'
                FOR i=0, ndat-1 DO zmat(i,*)=data(i).data(0,detector-1,*)
            END 

            ELSE: BEGIN
                message,  'IF file type "' + product+'" not handled yet', /cont
                plotted_x = !x  &  plotted_y = !y & return
            END 

        ENDCASE  
 
        IF output_info.rat THEN uztit = uztit + '!Ns!E-1!N'
        IF output_info.cal THEN uztit = uztit + '!Ncm!E-2!Nsr!E-1!N'
        IF output_info.div THEN uztit = uztit + '!NkeV!E-1!N'  
        IF control.cal EQ 4 THEN uztit = 'Hilmer PSD c!u3!N/MeV!u3!Ncm!u3!N'
        IF control.cal EQ 5 THEN uztit = 'Hilmer PSD c!u3!N/MeV!u3!Ncm!u3!N'
                         
        IF control.pa_norm THEN uztit= 'Normalized'

        ;check for Mu range plot
        IF control.pl_type EQ 5 THEN BEGIN
            yarr(*, 0) = ch(0,*,0,0)
            yarr(*, 1) = ch(1,*,0,0)
            idx = [ek1-1, ek2-1] & yarr =yarr[idx, *]
            ch_str = strarr(2)
            FOR i = 0, 1 DO ch_str(i) = strcompress( $
                string(yarr(i, *), format="(f6.1,'-',f6.1)"),/remove_all)
            ch_str =ch_str+' keV' 
            zmat = fltarr(ndat, 2)
            ;need to get mag field data
;             get_cluster_mag, mag_bt, tai1, tai2, time, plotinfo
            get_cluster_mag, control, mag_fgm(*,3), time
            mag_bt=mag_fgm(*,3)
            ;MATT CHANGE.......
            ;treat relativistically 
            ;mu= <p2c2>/(B*2*Er)  where Er is rest energy
            ;<p2c2>= energy*(energy + (2*rest_energy))
            energy=[yarr(0,0),yarr(1,1)]
            rest_energy=0.511*1000;keV
             for i=0,1 do begin
               p2c2= energy(i)*(energy(i) + (2*rest_energy))
                zmat(*,i)=p2c2/(mag_bt*rest_energy*2) 
            ENDFOR 
 
          ;*****CONVERT TO MeV/G!!!!!!!!
           zmat=zmat*100.0
           ;*******
            control.pl_style = 1       
           uztit =  '!4l!3 (MeV/G)'
           title = '!4l!3 Range'
        ENDIF  
        utitle = utitle+'!C'+title

    END                          ; RAL IF files


    1: BEGIN ;-- prepare plot arrays for MPAE Science files --

        ;select data to plot
        sci_type = sci_names(control.product)
        n_ch = header.NBANDS & n_sec = header.NSECTORS
        n_det = header.NDETECTORS
        yarr = fltarr(n_ch, 2)
        utitle = 'RAPID '+sci_type
        uztit=header.CONTENTS
        uytitle='E (keV)' 

        CASE header.DATA_CLASS OF
            
            'I3DD': BEGIN       ;3d distribution of ion species
                ;select which species of I3DD
                result = strsplit(sci_type, ' ', /extract)
                species_str = result(1)
                CASE species_str OF
                    'H': ion = 0
                    'He': ion = 1
                    'CNO': ion = 2
                ENDCASE
 
                CASE control.pl_type OF
                    0: BEGIN            ;spectrum
                        yarr = fltarr(n_ch, 2)
                        yarr(*, 0) = header.CH_POSITIONS(0,*,detector-1,ion)
                        yarr(*, 1) = header.CH_POSITIONS(1,*,detector-1,ion) 
                        utitle = utitle+'!CDet '+varprt(detector)+ $
                                 ', Sec '+varprt(sector)
                        zmat = transpose(data.data(detector-1,*,ion,sector))
                    END 
                    1: BEGIN            ;roll
                        n_ch = n_sec
                        yarr = fltarr(n_sec, 2)
                        yarr(*, 0) = findgen(n_sec)+0.5
                        yarr(*, 1) = yarr(*, 0)+1
                        zmat =  fltarr(ndat, header.NSECTORS)
                        utitle = utitle+'!CDet '+varprt(detector)+', EK '+ $
                          strcompress( $
                    string(header.CH_POSITIONS(0,energy-1,detector-1,ion), $
                           header.CH_POSITIONS(1,energy-1,detector-1,ion), $
                           format = "(f6.1,'-',f6.1)"), /remove_all)
                        uytitle='sector #'
                        zmat = transpose(data.data(detector-1,energy-1,ion,*)) 
                    END 
                ENDCASE 
            END 
            'HSPCT':BEGIN    ;spin averaged,detector averaged proton spectrum
                yarr = fltarr(n_ch, 2)
                yarr(*, 0) = header.CH_POSITIONS(0,*)
                yarr(*, 1) = header.CH_POSITIONS(1,*)
                utitle = utitle
                zmat = transpose(data.data)
            END
            'ISPCT':BEGIN    ;spin averaged,detector averaged He/CNO spectrum
                yarr = fltarr(n_ch, 2)
                CASE sci_type OF
                    'ISPCT He': species = 0 
                    'ISPCT CNO':species = 1
                ENDCASE
                yarr(*, 0) = header.CH_POSITIONS(0, *, species)
                yarr(*, 1) = header.CH_POSITIONS(1, *, species)
                utitle = utitle
                zmat = transpose(data.data(*, species))
            END 
      
        ENDCASE                 ;science data classes

    END                         ;science files 

    2: BEGIN ;LANL pitch files - calib info is set by m_pitch routine

    ;select data to plot
    lanl_type = lanl_names(control.product)
    n_ch = header.NBANDS
    n_pa = header.NSECTORS
    n_det = header.NDETECTORS
    utitle = 'RAPID '+header.TITLE
    uztit='[cts]'
    IF header.rat THEN uztit = uztit + '!Ns!E-1!N'
    IF header.cal THEN uztit = uztit + '!Ncm!E-2!Nsr!E-1!N'
    IF header.div THEN uztit = uztit + '!NkeV!E-1!N'  
    nodata = -99.0

    ;do IES channel converion to keV
    ;channel info is in ADC units - convert to KeV
    pos = strpos(lanl_type, 'IES')
    IF pos NE -1 THEN header.CH_POSITIONS = header.CH_POSITIONS* 2.2 + 6.5

    ;select data to plot
    CASE control.pl_type OF
        3: BEGIN                ;spectrum @ pitch
            yarr = fltarr(n_ch, 2)
            yarr(*, 0) = header.CH_POSITIONS(0,*,detector-1,0)
            yarr(*, 1) = header.CH_POSITIONS(1,*,detector-1,0) 
            utitle = utitle
            count = transpose(data.count(*,control.pa)) 
            IF control.pa_bin THEN BEGIN 
                zmat = count
                idx = where(zmat LE 0, c)
                IF c NE 0 THEN zmat(idx) = -99   
                uztit = '# samples/bin'
            ENDIF ELSE BEGIN 
                pitch = transpose(data.data(*,control.pa)) 
                idx1 = where(count GT 0, c, complement= idx2, ncomplement= c2)
                IF c NE 0 THEN BEGIN
                    zmat = fltarr(ndat, n_ch)
                    zmat(idx1) = pitch(idx1)/count(idx1)
                    IF c2 NE 0 THEN  zmat(idx2) = -99.0
                ENDIF 
            ENDELSE 
            utitle = utitle+'!CPA '+ strcompress( $
              string(header.PA_POSITIONS(control.pa,0), $
                     header.PA_POSITIONS(control.pa,1), $
                     format = "(f6.1,'-',f6.1)"), /remove_all)
            uytitle='E (keV)' 
        END
        4: BEGIN                ;pitch @ energy
            yarr = fltarr(n_pa, 2)
            yarr(*, 0) = header.PA_POSITIONS(*,0)
            yarr(*, 1) = header.PA_POSITIONS(*,1)
            utitle = utitle
            count = transpose(data.count(e1,*)) 
            IF control.pa_bin THEN BEGIN 
                zmat = count
                idx = where(zmat LE 0, c)
                IF c NE 0 THEN zmat(idx) = -99           
                uztit = '# samples/bin'
            ENDIF ELSE BEGIN 
                pitch = transpose(data.data(e1,*)) 
                idx1 = where(count GT 0, c, complement= idx2, ncomplement= c2)
                IF c NE 0 THEN BEGIN
                    zmat = fltarr(ndat, n_pa)
                    zmat(idx1) = pitch(idx1)/count(idx1)
                    IF c2 NE 0 THEN zmat(idx2) = -99.0
                ENDIF ELSE BEGIN
                    msg = 'No valid data to plot'
                    message, msg, /cont
                    papco_Draw_ErrorPanel, panel, msg
                    return
                ENDELSE     
            ENDELSE 
            utitle = utitle+'!CEK '+ strcompress( $
              string(header.CH_POSITIONS(0,e1,d1,0), $
                     header.CH_POSITIONS(1,e1,d1,0), $
                     format = "(f6.1,'-',f6.1)"), /remove_all)
            uytitle='Pitch Angle (deg)' 
            valid=[1, 2, 3, 4.5, 6, 9]
        END
    ENDCASE
    idx_1 = where(zmat EQ 1e-20, n1)
    IF n1 NE 0 THEN zmat(idx_1) = nodata 
    idx_0 = where(zmat EQ 0, c, complement = idx_n0)
    ;data is based on sectored input, need to multiply by sectors.
    zmat(idx_n0) = zmat(idx_n0) * 16  ;multiply by sectors
    ;set zero's to no data flag
    idx = where(zmat LE 0, c) & IF c NE 0 THEN zmat(idx) = nodata
    
    END ;LANL pitch files

ENDCASE ;end prepare data arrays for data types

utitle='CLUSTER '+varprt(control.sc_id+1)+'!C'+utitle

n_ch=n_elements(yarr)/2
yst=yarr(0,0)  &  yen=yarr(n_ch-1,1)

;filter the data using the ephemeris/mag info ---------------------------------
;Handle different time bases - set non-filtered data to nodata flag.

;filter for B_X
IF control.B_X_SELECT_RANGE THEN BEGIN
    filt_var = mag_fgm(*, 0)
    v1 = control.B_X_SELECT_FROM_VAL
    v2 = control.B_X_SELECT_TO_VAL
    frmt = control.B_X_SELECT_RANGE_FRMT
    idx = where((filt_var LE v1) OR (filt_var GE v2), c)
    IF c NE 0 THEN zmat(idx, *) = nodata
    utitle = utitle+'!C'+string(v1, v2, format=frmt)
ENDIF

;filter for B_TOT
IF control.B_TOT_SELECT_RANGE THEN BEGIN
    filt_var = mag_fgm(*, 0)
    v1 = control.B_TOT_SELECT_FROM_VAL
    v2 = control.B_TOT_SELECT_TO_VAL
    frmt = control.B_TOT_SELECT_RANGE_FRMT
    idx = where((filt_var LE v1) OR (filt_var GE v2), c)
    IF c NE 0 THEN zmat(idx, *) = nodata
    utitle = utitle+'!C'+string(v1, v2, format=frmt)
ENDIF

if keyword_set(OUTPUT) then if output eq 2 then return  

;actual plotting of data is done below from here -----------------------------

panelset,panel		              ;sets the panel position viewport

IF control.pl_type EQ 2 THEN BEGIN    ;LUT line plot
    panelset,panel		  ;sets the panel position viewport
    yst = -1 & yen = 16
    if (yscl(panel(0),0) eq 1) then begin 
      yst=yscl(panel(0),1)  &  yen=yscl(panel(0),2)
    endif else begin
      yscl(panel(0),1)=yst  &  yscl(panel(0),2)=yen
    endelse    
    papco_y_label, yst, yen, log=yscl(panel(0),3)
    extra_plotPar={xrange:[xut1,xut2], yrange:[yst, yen], $
                   ylog:yscl(panel(0),3), xtickformat:'noticks'}
    extra_plotPar=create_struct(extra_plotPar, extra_par)
    down=0
    papco_draw_time_axis, panel, OVERPLOT=OVERPLOT, _extra=extra_plotPar

    plot, time, yarr, xstyle=5, ystyle=5, _extra=extra_plotPar
    plotted_x = !x  &  plotted_y = !y
    right_side_label, panel, utitle, _extra=extra_Par
    left_side_label,panel,''    
    return
ENDIF

;--- do the plot - either color or line ---------------------------------------
IF control.spec EQ 1 THEN BEGIN   ;plot color specs
    rot = 1 &  line = 0
    papco_Set_SubTable, SUBTABLE          ;sets the papco color table index

    exclude = plotinfo.MANUALZSCALING

    IF (zscl(panel(0),0) EQ 1) THEN BEGIN 
        zmin=zscl(panel(0),1)  &  zmax=zscl(panel(0),2)
    ENDIF ELSE BEGIN
        papco_autorange, zmat, zmin, zmax, $
          log=zscl(panel(0),3), exclude=exclude, nodata=nodata, /verb
        zscl(panel(0),1)=zmin  &  zscl(panel(0),2)=zmax
    ENDELSE 
    if (yscl(panel(0),0) eq 1) then begin 
        yst=yscl(panel(0),1)  &  yen=yscl(panel(0),2)
    endif else begin
        yscl(panel(0),1)=yst  &  yscl(panel(0),2)=yen
    endelse  

    papco_y_label, yst, yen, log=yscl(panel(0),3),  VALID = valid, /verb
    yst = yst*0.99 & yen = yen*1.01

    extra_plotPar={xrange:[xut1,xut2], yrange:[yst,yen], zrange:[zmin,zmax], $
                ylog:yscl(panel(0),3), zlog:zscl(panel(0),3), $
                ztitle:uztit, $
                xtickformat:'noticks',ztickformat:'papco_color_bar_log_ticks'}

    ; add keyword structure set locally and the one passed in
    extra_plotPar=create_struct(extra_plotPar, extra_par)
     
    ; do color plot and color bar plot. nodata specifies the "nodata" flag
    ; in your data. Here we do not plot axis and data seperate, as
    ; papco_plot_colorspec cannot be used as an overplot!
    resx=2 & dx = 4
    IF yscl(panel(0),3) THEN resy=(alog10(yen)-alog10(yst))/n_ch / 10 ELSE $
        resy=((yen)-(yst))/n_ch / 10

    papco_plot_colorspec,zmat,time,yarr, nodata=nodata, resx=resx, resy=resy, $
        dx = dx, xstyle=5, ystyle=5, _extra=extra_plotPar
 
    ; store the coordinate information into common block
    plotted_x = !x  &  plotted_y = !y

    down=0
    papco_draw_time_axis, panel, OVERPLOT=OVERPLOT, _extra=extra_plotPar      
  
    papco_color_bar, _extra=extra_plotPar
   
ENDIF ELSE BEGIN                ;plot multiple lines for chosen channels.
    rot = 0 & line = 1

    if (yscl(panel(0),0) eq 1) then begin 
        yst=yscl(panel(0),1)  &  yen=yscl(panel(0),2)
    endif else begin
        papco_autorange,zmat, yst, yen, $
                        log=zscl(panel(0),3), exclude=exclude, nodata=nodata
        yscl(panel(0),1)=yst  &  yscl(panel(0),2)=yen
    endelse  

    r = size(zmat, /st)
    IF r.N_DIMENSIONS EQ 1 THEN n_ch = 1 ELSE n_ch = r.DIMENSIONS(1)
    to_plot = to_plot(0:n_ch-1)
    pl_idx=where(to_plot eq 1, nplot)

    uytitle = uztit
    papco_y_label, yst, yen, log=yscl(panel(0),3),  VALID = valid
    
    extra_plotPar={xrange:[xut1,xut2], yrange:[yst,yen], $
                   ylog:yscl(panel(0),3), xtickformat:'noticks'}
    ; add keyword structure set locally and the one passed in
    extra_plotPar=create_struct(extra_plotPar, extra_par)     
    old_color = extra_plotPar.color

    ;draw time axis only, no data
    down=0
    papco_draw_time_axis, panel, OVERPLOT=OVERPLOT, _extra=extra_plotPar

    t = (time(*, 0)+time(*, 1)) / 2.0
    plot, t, t, _extra=extra_plotPar,  xstyle=5, ystyle=5, /nodata
    
    ;loop line plot. Assume data is in zmat(time, ch)
    FOR i = 0, nplot-1 DO BEGIN
        idx = pl_idx(i)
        IF nplot GT 1 THEN $
          extra_plotPar.color=color_indx(i mod n_elements(color_indx))

        y = zmat(*, idx) 
        idx1 = where(y NE nodata, c)
        IF c EQ 0 THEN continue
        IF n_elements(idx1) gt control.smooth THEN  BEGIN 
            IF control.smooth GT 2 THEN $
              y(idx1) = papco_smooth(y(idx1), control.smooth, nodata = nodata)
        ENDIF 
        rst=''

        papco_gap_plot, t, y, nodata, _extra=extra_plotpar, $
            /OPLOT, AVERAGE = control.seg_av

        c=count/5 & mult=count mod 5 
        IF ch_str(0) EQ '' THEN name = 'Ch '+varprt(idx+1) ELSE $
          name = ch_str(idx)
        IF (nplot NE 1) THEN  BEGIN 
            FOR j=1,c DO  name='!C'+name
            xyouts,!P.position(0)+0.01+0.135*mult, $
                   !P.position(3)-0.015, name, /normal, $
                   charsize=!p.charsize, color=extra_plotPar.color
        ENDIF ELSE utitle = utitle+'!C'+name
        count=count+1

    ENDFOR   
    IF control.smooth GT 2 THEN BEGIN 
        IF control.smooth GE 25 THEN  $
          utitle = utitle+'!CSmooth: '+varprt(control.smooth/25)+' mins' ELSE $
          utitle = utitle+'!CSmooth: '+varprt(control.smooth*4)+' secs' 
    ENDIF 

    extra_plotPar.color = old_color

    ; store the coordinate information into common block
    plotted_x = !x  &  plotted_y = !y

ENDELSE 

papco_rs_lbl, panel, utitle, ROT90 = rot, LINE = line, _extra=extra_plotpar

; plot y-axis label at left of plot. Use scalable routine!  
IF NOT keyword_Set(overplot) THEN left_side_label, panel, uytitle, /ROT90

; check if data needs to be output to file. Use the optional keywords
; to add more information to the data file
;      DESCRIPTION   a string or string array describing the data
;      TIME_LABEL    a string decribing the x-axis format
;      CHANNELS      an (ch,2) array giving the channel start/end
;      Y_LABEL       a string with the y-axis label
;      Z_LABEL       a string with the z-axis label
;      RS_LABEL      a string with the righ-side label of the plot
;      NODATA        contains the no data flag value.
IF KEYWORD_SET(OUTPUT) THEN BEGIN 
    message,'Writing plot data out to file',  /cont
    description='cluster_rapid data'
    time_label='TAI'
    channels=yarr
    y_label=uytitle
    z_label=uztit
    rs_label=utitle
    nodata=1e-20
    papco_write_data, time+tai1, zmat, $
                      DESCRIPTION=description, TIME_LABEL=time_label, $
                      CHANNELS=channels, Y_LABEL=y_label, Z_LABEL=z_label, $
                      RS_LABEL=rs_label, NODATA=nodata
ENDIF
 

return

;jump to here if no data to plot 
noplot: 
  get_err_no = 10
  message, get_err_msg, /cont
  papco_Draw_ErrorPanel, panel, plotinfo.panelkind+': ' + get_err_msg
  plotted_x = !x  &  plotted_y = !y
  
END


;******************************************************************************
;* PROCEDURE: m_IES_CEANBM_zmat
;*
;* DESCRIPTION:  Make plot arrays for IES_CLEANBM data. Also does Pitch!
;*               Also works for IES_EPAD_16 data (needs to be
;*               converted to same energy ranges as IES_CLEANBM prior
;*               to calling this rotuine)
;******************************************************************************
PRO m_IES_CEANBM_zmat,  control,  HAVE_PA = HAVE_PA

COMMON cluster_rapid_data, header, input_data
COMMON cluster_rapid_slice
COMMON get_error, get_err_no, get_err_msg  

;use mask of bad det/sec/ek to remove contamination
IF control.mask THEN BEGIN 
    f = data.data
    message, 'Masking contaminated data', /info
    cluster_rapid_ies_mask, control, mask, nodata_mask
    IF header.NBANDS EQ 1 THEN $
        FOR i = 0l, ndat-1 DO f(*,*,i) = f(*,*,i)*mask + nodata_mask $
    ELSE $
        FOR i = 0l, ndat-1 DO f(*,*,*,i) = f(*,*,*,i)*mask + nodata_mask
    utitle = utitle+ '+ mask'
    data.data = temporary(f)
ENDIF         

CASE control.pl_type OF

    0: BEGIN    ;spectrum: average det, sec range chosen.
        zmat = fltarr(ndat, header.NBANDS) & numb = zmat & numb(*) = 0
        FOR i = 0l, ndat-1 DO BEGIN
            FOR d = d1, d2 DO BEGIN
                FOR s = s1, s2 DO BEGIN
                    idx=where(data(i).data(s,d,*) NE nodata, c)
                    IF c NE 0 THEN BEGIN
                        zmat(i,idx) = $
                          zmat(i,idx)+data(i).data(s,d,idx)
                        numb(i,idx) = numb(i,idx)+1
                    ENDIF 
                ENDFOR  
            ENDFOR  
        ENDFOR       
        idx1 = where(numb NE 0, c1, comp = idx2, nc = c2)
        IF c1 NE 0 THEN zmat(idx1) =  zmat(idx1) / numb(idx1)
        IF c2 NE 0 THEN zmat(idx2) = nodata
        idx = where(zmat NE nodata, c )
        IF c NE 0 THEN zmat(idx) = zmat(idx) * 16 

    END

    1: BEGIN                    ;roll
        zmat =  fltarr(ndat, header.NSECTORS)
        FOR i = 0l, ndat-1 DO zmat(i, *) = data(i).data(*,d1,e1)
        idx = where(zmat NE nodata, c )
        IF c NE 0 THEN zmat(idx) = zmat(idx) * 16 
    END

    3: BEGIN    ;pitch E @ Pa: Sum energy, Pa range chosen.

        ;read the ies det/sec pitch data.
        IF keyword_set(HAVE_PA) THEN p = data.pa *!RADEG $
        ELSE get_cluster_rapid_ies_pitch, control, p
        IF get_err_no NE 0 THEN return

        ;make data array - here average pitch angle range chosen
        ;need to preserve nodata flags
        f = data.data
        n_e = n_elements(yarr)/2
        zmat = fltarr(ndat, n_e) & numb = zmat
        
        message, 'Folding data into spectrum at pitch angle ranges '+ $
                  varprt(p1)+' to '+varprt(p2), /info
        FOR i = 0l, ndat-1 DO BEGIN
            FOR s = s1, s2 DO BEGIN
                FOR d = d1, d2 DO BEGIN 
                    pa = p(s, d, i)
                    IF (pa GE p1) AND (pa LE p2) THEN BEGIN
                        FOR e = 0, n_e-1 DO BEGIN
                            IF f(s, d, e, i) NE nodata THEN BEGIN
                                zmat(i, e) = zmat(i, e) + f(s, d, e, i)
                                numb(i, e) = numb(i, e) + 1
                            ENDIF
                        ENDFOR 
                    ENDIF 
                ENDFOR
            ENDFOR
        ENDFOR        

        idx1 = where(numb EQ 0, c1, COMPLEMENT = idx2, NCOMPLEMENT=c2)
        IF c2 NE 0 THEN zmat(idx2) = zmat(idx2) / numb(idx2)
        IF c1 NE 0 THEN zmat(idx1) = nodata

        idx = where(zmat NE nodata, c )
        IF c NE 0 THEN zmat(idx) = zmat(idx) * 16  ;mult by sectors

    END
    4: BEGIN    ;pitch Pa @ E: Average Pa, E range chosen.

        ;read the ies det/sec pitch data.
        IF keyword_set(HAVE_PA) THEN p = data.pa *!RADEG $
        ELSE get_cluster_rapid_ies_pitch, control, p
        IF get_err_no NE 0 THEN return

        ;make data array - here sum energy range chosen
        ;since nodata is 1e-20, summing through it makes no differecne
        f = data.data(*, *, 0) & f(*) = 0            
        FOR e = e1, e2 DO f = f + data.data(*, *, e)
        idx = where(f LE (e2-e1+1)*nodata, c)
        IF c NE 0 THEN f(idx) = nodata
       
        idx = where((p LT 0) OR (f LE 0), c)
        IF c NE 0 THEN f(idx) = nodata

        ;default pitch angle is pitch angle detector is looking at.
        ;if control.pa_bin is negative, sort into pitch angle particle
        ;is coming from!
        IF control.pa_bin LT 0 THEN BEGIN
            idx = where(p NE nodata, c)
            p(idx) = (180 - p(idx)) MOD 180.0
            utitle = utitle+', from'
        ENDIF 

        yarr = papco_pitch_bin(abs(control.pa_bin),  PA_IDX = pa_idx)
        n_ch = n_elements(yarr(*, 0))
        zmat = fltarr(ndat, n_ch) & numb = zmat

        IF control.cal EQ 6 THEN intercal = 1 ELSE intercal = 0

        IF NOT intercal THEN BEGIN

            ;fold data into pitch angle bins
            message, 'Folding data into Pitch angle ranges...', /info
            FOR i = 0l, ndat-1 DO BEGIN
                FOR s = s1, s2 DO BEGIN
                    FOR d = d1, d2 DO BEGIN 
                        IF f(s, d, i) NE nodata THEN BEGIN
                            pa = pa_idx(p(s, d, i)) <  179
                            zmat(i, pa) = zmat(i, pa) + f(s, d, i)
                            numb(i, pa) = numb(i, pa) + 1
                        ENDIF     
                    ENDFOR
                ENDFOR
            ENDFOR

            idx1 = where(numb EQ 0, c1, COMPLEMENT = idx2, NCOMPLEMENT=c2)
            IF c2 NE 0 THEN zmat(idx2) = zmat(idx2) / numb(idx2)
            IF c1 NE 0 THEN zmat(idx1) = nodata

            IF control.pa_norm THEN BEGIN ;normilze pitch angle distrib 0->1
                message, 'Normalizing PA distributions', /info
                FOR i = 0l, ndat-1 DO BEGIN
                    pa_distrib = zmat(i, *)
                    idx = where(pa_distrib NE nodata, c)
                    IF c NE 0 THEN BEGIN
                        max_pa = max(pa_distrib(idx))
                        zmat(i, idx) = zmat(i, idx) / max_pa
                        print, i, max(zmat(i, idx))
                    ENDIF
                ENDFOR  
            ENDIF ELSE BEGIN    ;do nor normalize 
                idx = where(zmat NE nodata, c )
                IF c NE 0 THEN zmat(idx) = zmat(idx) * 16  ;mult by sectors
            ENDELSE 
        ENDIF ELSE BEGIN

            ;fold data into pitch angle bins for d1
            message, 'Folding Det '+varprt(d1)+$
                     ' into Pitch angle ranges...', /info
            FOR i = 0l, ndat-1 DO BEGIN
                FOR s = 0, header.NSECTORS-1 DO BEGIN
                    FOR d = d1, d1 DO BEGIN 
                        IF (f(s, d, i) NE nodata) THEN BEGIN
                            pa = pa_idx(p(s, d, i))
                            zmat(i, pa) = zmat(i, pa) + f(s, d, i)
                            numb(i, pa) = numb(i, pa) + 1
                        ENDIF     
                    ENDFOR
                ENDFOR
            ENDFOR
            
            idx1 = where(numb EQ 0, c1, COMPLEMENT = idx2, NCOMPLEMENT=c2)
            IF c2 NE 0 THEN zmat(idx2) = zmat(idx2) / numb(idx2)
            IF c1 NE 0 THEN zmat(idx1) = nodata

            zmat1 = zmat
            zmat = fltarr(ndat, n_ch) & numb = zmat
            
            ;fold data into pitch angle bins for d2
            message, 'Folding Det '+varprt(d2)+$
                     ' into Pitch angle ranges...', /info
            FOR i = 0l, ndat-1 DO BEGIN
                FOR s = 0, header.NSECTORS-1 DO BEGIN
                    FOR d = d2, d2 DO BEGIN 
                        IF (f(s, d, i) NE nodata) THEN BEGIN
                            pa = pa_idx(p(s, d, i))
                            zmat(i, pa) = zmat(i, pa) + f(s, d, i)
                            numb(i, pa) = numb(i, pa) + 1
                        ENDIF     
                    ENDFOR
                ENDFOR
            ENDFOR

            idx1 = where(numb EQ 0, c1, COMPLEMENT = idx2, NCOMPLEMENT=c2)
            IF c2 NE 0 THEN zmat(idx2) = zmat(idx2) / numb(idx2)
            IF c1 NE 0 THEN zmat(idx1) = nodata
            
            zmat2 = zmat

            idx = where((zmat1 NE nodata) AND (zmat2 NE nodata),  $
                          c, COMPLEMENT = idx2)

            message, 'Finding ratio between det '+$
                     varprt(d1)+', '+ varprt(d2), /info
            IF c EQ 0 THEN BEGIN
                msg = 'No common pitch angles'
                GOTO, noplot
            ENDIF  
            zmat(idx) = zmat1(idx)/zmat2(idx)
            zmat(idx2) = nodata
            utitle=utitle+'!CRatio: '+varprt(d1)+'/'+varprt(d2)

            ;if average is set, and line plot, here average
            ;togther ratios from all pitch angles!
            IF (control.seg_av EQ 1) AND (control.spec EQ 0) THEN BEGIN
                yray = fltarr(ndat) & yray(*) = nodata
                FOR i = 0l, ndat-1 DO BEGIN
                    y = zmat(i, *)
                    idx1 = where(y NE nodata, c1)
                    y = alog10(y)
                    IF c1 NE 0 THEN BEGIN
                        min = min(y(idx1), max = max)
                        IF (max-min) GT 0.5 THEN continue
                        yray(i) = 10^(total(y(idx1))/c1)
                    ENDIF 
                ENDFOR     
                ;control.seg_av = 0
                n_ch = 1
                zmat = yray
                uytitle = 'Av. all pitch'
            ENDIF

        ENDELSE ;do intercal

    END ;do PA @ Ek        

ENDCASE

return

noplot:
stop

END 


;******************************************************************************
;* EPAD IES has two energy channels. Attempt to interpolate to common energy
;* range. Use an exponential fit : F(x) = a * exp(b*x)
;*
;* Need to determine actual anergy range of the two channels. use the
;* pedestal information in IES PEDESTAL to do this!
;*
;* Option to use spectrum from ESPECT if only one energy is available
;*
;* Interpolate onto 8 energy channels to make this data product "look"
;* like IES CLEAN BM data!
;******************************************************************************
PRO cluster_rapid_ies_epad_ecal, in_control, VERBOSE = VERBOSE

COMMON mjdt, mjdt_start, mjdt_end     ;common time limit in mjdt
COMMON cluster_rapid
COMMON cluster_rapid_slice
COMMON cluster_rapid_data, input_header, input_data
COMMON channels, f1, f2, ek1, ek2 ;fluxes and energy ranges for channels
COMMON new_time, tt1, tt2
COMMON get_error, get_err_no, get_err_msg  

verbose = 0
EPAD_header = input_header & EPAD_data = data & control = in_control

;check if file exists
SC_id=varprt(control.sc_id+1)
product = 'IES_CLEAN_EPAD_16_PA'
ndat =  n_elements(input_data)
tt = tai2utc(input_data(ndat/2).time,/ext)
yymmdd = string(tt.year-2000, tt.month, tt.day, format = "(i2.2,i2.2,i2.2)")
IF NOT (papco_check_data_env('CLUSTER_RAPID_IF', PATH=path)) THEN return
sub_path = 'RAPID_' + SC_id + papco_get_pathsep() + product
fln = 'RAPID_' + SC_id + papco_get_pathsep() + $
                product + papco_get_pathsep() + yymmdd
ext = 'vv1.0.dat'                           
file = path + fln + ext
IF NOT file_test(path+sub_path, /directory) THEN file_mkdir, path+sub_path

result = file_search(file, count = c)
IF c EQ 1 THEN BEGIN
    restore, file = file, /verb
    ;--- set timerange requested in TAI ---------------------------------------
    tai1=utc2tai({mjd:mjdt_start.mjd, time:mjdt_start.t*1000})
    tai2=utc2tai({mjd:mjdt_end.mjd, time:mjdt_end.t*1000})    
    ;restrict data in time 
    index=where((input_data.time GE tai1) AND (input_data.time LE tai2),c)
    IF c GT 2 THEN data=input_data(index) ELSE BEGIN
        get_err_no = 3 & get_err_msg = 'No data to plot' & return
    ENDELSE
    return
ENDIF

message, 'Doing IES EPAD energy interpolation (slow)', /cont 

;set desired fixed energy ranges to compute. Same as IES CLEAN BM
yarr = fltarr(8, 2)
yarr[*, 0] = [41.70, 52.70, 70.30, 96.70, 127.5, 171.5, 237.5, 325.5]
yarr[*, 1] = [52.70, 70.30, 96.70, 127.5, 171.5, 237.5, 325.5, 453.1]
spec = fltarr(8)

;make new data array for expanded EPAD data (8 interpolated energies)
new_data = fltarr(16, 9, 8, ndat) & new_data(*) = nodata

;read in IES PEDESTAL info
new_plotinfo = papco_getplotinfostruct()
new_control = in_control 
idx = where(if_names EQ 'IES_PEDESTAL')
new_control.product = idx[0]     ;IES PEDESTAL data, spin averaged per detector
new_plotinfo.USR_PTR1 = ptr_new(new_control)
r_cluster_rapid, new_plotinfo

PEDESTAL_header = input_header & PEDESTAL_data = input_data

;We know the counts for two ADC ranges. Use PEDESTAL info to find
;actual energy ranges, then use these energy ranges this to find b of
;the exponent of fit. Do for each detector seperately, then evaluate for a
;fixed energy range.
   
FOR det = 0, 8 DO BEGIN
    message, 'Doing for det '+varprt(det), /info
    
    ;set nominal ADC bin range for this detector
    b1 =   EPAD_header.CH_POSITIONS(*, 0, det, 0)
    b2 =   EPAD_header.CH_POSITIONS(*, 1, det, 0)
 
    ;find actual energy channels using IES PEDESTAL info
    ;data is in ADC bin number, each ADC step = 2.2 keV, offset = 6.5 keV
    ped_nominal = PEDESTAL_data.data(0, det, 0)
    ped_offset  = PEDESTAL_data.data(0, det, 1)
    ped_det = ped_nominal+ped_offset

    ;ped_det is on a slightly different time base. Iterpolate onto EPAD16 time
    tai1=utc2tai({mjd:mjdt_start.mjd, time:mjdt_start.t*1000})
    t_ped_det = PEDESTAL_data.time-tai1
    t_epad16  = EPAD_data.time-tai1
    ped_det = interpol(ped_det,t_ped_det, t_epad16)

    ek_arr_1 = fltarr(ndat, 2)
    ek_arr_2 = fltarr(ndat, 2)

    ;subtract off pedestal ADC and convert to keV
    ek_arr_1(*, 0) = ( b1(0) - ped_det) * 2.2 + 6.5
    ek_arr_1(*, 1) = ( b1(1) - ped_det) * 2.2 + 6.5
    ek_arr_2(*, 0) = ( b2(0) - ped_det) * 2.2 + 6.5
    ek_arr_2(*, 1) = ( b2(1) - ped_det) * 2.2 + 6.5

    ;now have actual energy assignments at each timestep. Loop time
    FOR i = 0l, n_elements(EPAD_data)-1 DO BEGIN 

        ;set energy channels for this spin
        ek1 = ek_arr_1(i, *) & ek2 = ek_arr_2(i, *) 

        ;test if lowest energy bound of first channel turns
        ;out negative -> pedestal in first channel, ignore
        IF ek1(0) LT 0 THEN BEGIN
            IF keyword_set(VERBOSE) THEN $
               message, 'Det '+varprt(det)+' contaminated', /info
            continue
        ENDIF 

        ;loop through all sectors.
        FOR j = 0, 15 DO BEGIN 
            
            ;get data for this sector
            f1 = EPAD_data(i).data(j, det, 0)
            f2 = EPAD_data(i).data(j, det, 1)

            ;skip of no data in either or high channel has more counts
            IF (f1 EQ 0) OR (f2 EQ 0) OR (f2 GE f1) THEN continue 

            ;find b - root of function find_exp 
            x = [-1, -2, -3]*0.01
            b = papco_FX_ROOT(x, 'papco_spec_area')

            IF get_err_no NE 0 THEN  continue
            ;once we have b, can find a
            a = f1*b / ( exp(b*ek1(1)) - exp(b*ek1(0)) )
              
            ;now find interpolated f's at chosen energy ranges
            FOR e = 0, 7 DO spec(e) = papco_spec_rng(a, b, yarr(e, *))

            IF keyword_set(VERBOSE) THEN print, i, j, f1, f2, spec, $
                   format = "(2(i4),2(f8.3,'  '),8(f8.3,'  '))"

            ;load new_data with new values
            new_data(j, det, *, i) = spec

        ENDFOR 

    ENDFOR

ENDFOR

idx = where(new_data LE 0, c)
IF c NE 0 THEN new_data(idx) = nodata

;need to make a data structure and header that looks like CLEAN BM
;data. See if PA tag is present, and copy if there!

struct_name = 'DATA_RAPID_'+varprt(control.sc_id+1)+'_IES_CLEAN_EPAD_V2.0'
dummy = {TIME:0.0d, ENDTIME:0.0d, DATA:new_data[*, *, *, 0], $
         LUT:0l, FLAG:0l}

epad_tags = tag_names(epad_data)
idx = where(epad_tags EQ 'PA', have_pitch)
IF have_pitch EQ 1 THEN BEGIN
    struct_name = struct_name+'_PA'
    dummy = create_struct(dummy, 'mvals', bytarr(11), 'pa', fltarr(16, 9))
ENDIF

dummy = create_struct(name = struct_name, dummy)
data = replicate(dummy, n_elements(EPAD_data))
data.TIME = EPAD_data.TIME
data.ENDTIME = EPAD_data.ENDTIME
data.data = new_data
data.LUT = EPAD_data.LUT
data.FLAG = EPAD_data.FLAG

IF have_pitch EQ 1 THEN BEGIN
    data.mvals = EPAD_data.mvals
    data.pa =  EPAD_data.pa
ENDIF 

struct_name = 'HEADER_RAPID_'+varprt(control.sc_id+1)+'_IES_CLEAN_EPAD_V1.0'
dummy = { $
   VERSION           :  'xa1.0', $
   DATATYPE          :  'RAPID_'+varprt(control.sc_id+1)+'_IES_CLEAN_EPAD', $
   NPOINTS           :  epad_header.NPOINTS, $
   NDETECTORS        :  epad_header.NDETECTORS, $
   NBANDS            :  8l, $
   NSECTORS          :  epad_header.NSECTORS, $
   NLUTS             :  epad_header.NLUTS, $
   DETECTORS         :  epad_header.DETECTORS, $
   TSTART            :  epad_data(0).TIME, $
   TSTOP             :  epad_data(epad_header.NPOINTS-1).ENDTIME , $
   CH_POSITIONS      :  fltarr(2, 8, 9, 8), $
   COUNT_CAL         :  epad_header.COUNT_CAL, $
   ENERGY_CAL        :  epad_header.ENERGY_CAL, $
   ENERGY_EFFICIENCY :  fltarr(8, 8), $
   LUT_MAP           :  epad_header.LUT_MAP, $
   LUT_TIMES         :  epad_header.LUT_TIMES, $
   CAL               :  epad_header.CAL, $
   RAT               :  epad_header.RAT, $
   DIV               :  epad_header.DIV, $
   CLEAN             :  1, $
   SUBTRACT          :  epad_header.SUBTRACT, $
   TITLE             :  'E-PAD CLEAN', $
   XTITLE            :  'Time', $
   YTITLE            :  'Energy', $
   ZTITLE            :  'Counts', $
   PTITLE            :  epad_header.PTITLE  }

input_header = create_struct(name = struct_name, dummy)

;fill CH_POSITION with interpolated energies
FOR det = 0, 8 DO FOR lut = 0, 7 DO $
  input_header.CH_POSITIONS[*, *, det, lut] = transpose((yarr-6.5)/2.2)

input_header.ENERGY_EFFICIENCY(*) = 1.0

message, 'Done.',/cont
print, ''

;save data to saveset

input_data = data
result = file_search(file, count = c)
IF c EQ 1 THEN return
message, 'Saving ifset', /info
print, '  ', file
save, file = file, input_header, input_data, /compress, /verbose

END

;******************************************************************************
;* Returns a detector, sector, energy mask for the spacecraft that is set to
;* zero for contaminated detector/sectors. 
;*
;* Sun contamination, Pedestool contaminaion can be masked out here
;*
;* works for IES CLEAN BM data and interpolated IES EPAD data
;******************************************************************************
PRO cluster_rapid_ies_mask, control, mask, nodata_mask

COMMON cluster_rapid_slice

;setup default mask, all set to 1
mask = fltarr(16, 9, 8) & mask(*) = 1
nodata_mask = mask & nodata_mask(*) = 0

CASE control.sc_id OF

    0:BEGIN
    END     

    1:BEGIN
    END     

    2:BEGIN
        mask(12:13, 0, 0:1) = 0 
        mask(12:13, 1, 0:1) = 0 
        mask(12:13, 2, 0:2) = 0
    END     

    3:BEGIN
    END     

ENDCASE 

idx1 = where(mask EQ 0, c1, COMPLEMENT = idx2, NCOMPLEMENT=c2)
IF c1 NE 0 THEN nodata_mask(idx1) = nodata
IF c2 NE 0 THEN nodata_mask(idx2) = 0

END
