;******************************************************************************
;*  NAME:        r_cluster_rapid
;*
;*  DESCRIPTION: Routine to read cluster_rapid files. Adheres to
;*               papco conventions: Error common block
;*                                  Uses MJDT time
;*                                  papco read philosophy
;*
;*  INPUT:       none (uses common block time from papco)
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  KEYWORDS:    VERBOSE  -  enables print comments, save/restore comments
;*
;*  DATE:        January 2000
;*
;*  AUTOR:       R. Friedel
;*
;*  MODIFIED     R. Friedel, Steve Monk, May 2003
;*                 added wget capability
;*                 operating system independence:
;*                   changed science file saveset names to single extension
;*                   use of papco_opsys routines throughout
;******************************************************************************
PRO r_cluster_rapid, plotinfo, VERBOSE=VERBOSE

COMMON cluster_rapid_data, input_header, input_data
COMMON mjdt, mjdt_start, mjdt_end
COMMON get_error, get_err_no, get_err_msg
COMMON cluster_rapid

get_err_no=0  &  get_err_msg='cluster_rapid read succesfully'

;get the module's control structure
control = *plotinfo.USR_PTR1

; see if 4sc data required - then don't read, read loop done by plot routine
sc4_flag =  control.prod_4sc
IF sc4_flag NE 0 THEN return

; check VERBOSE keyword and set
IF keyword_set(VERBOSE) THEN verbose=VERBOSE ELSE verbose=0

; clear data arrays for reuse
input_header=0  &  input_data=0
input_header={clust_rapid_header,sc_id:0}

; loop through from start to end time concatonating data files into one
; array. Here we assume that the data is stored in files per day.
time=mjdt_start  &  no_files_read=0

WHILE mjdt_end.mjd GE time.mjd DO BEGIN

    ;check if there is any time to be read on the end day.
    ;this avoids reading the next day if a whole day is specified
    IF mjdt_end.mjd EQ time.mjd THEN IF mjdt_end.t EQ 0 THEN goto, next

    ;assemble the data file from the time range requested
    convert_t90_to_date, time, year, day_oy, hour, minute, second, $
                         month, cmonth, day_om, /mjdt
    datestr=string(year,month,day_om,format='(i4.4,i2.2,i2.2)')

    plotinfo.filename=datestr

    ;read one day of data
    r_cluster_rapid_onefile, plotinfo, VERBOSE = VERBOSE

    ;if a file was succesfully read, concatonate!
    IF get_err_no EQ 0 THEN BEGIN
        IF no_files_read EQ 0 THEN BEGIN   ;start the array buffer
            input_data_buf=temporary(input_data)
        ENDIF ELSE BEGIN                   ;otherwise concatonate arrays
            input_data_buf=[temporary(input_data_buf),temporary(input_data)]
        ENDELSE
        input_data=0                      ;clear array for reuse
        no_files_read=no_files_read+1
    ENDIF
    next:
    time.mjd=time.mjd+1         ; next day
ENDWHILE

; check if any files were read
IF (no_files_read NE 0) AND (get_err_no ne 1) THEN BEGIN
    get_err_no=0
    get_err_msg=string(no_files_read,format="(i2)")+ ' files read succesfully'
    ;clear arrays and load buffer into output array
    input_data=0  &  input_data=temporary(input_data_buf)
ENDIF ELSE BEGIN
    ;get_err_no=1
    ;get_err_msg='No files found for timerange'
    return
ENDELSE

; restrict data to actual time range requested - works for time in TAI
tai_1=utc2tai({mjd:mjdt_start.mjd, time:mjdt_start.t*1000})
tai_2=utc2tai({mjd:mjdt_end.mjd, time:mjdt_end.t*1000})

index=where((input_data.time GE tai_1) AND (input_data.time LE tai_2),c)
IF c NE 0 THEN input_data=input_data(index) ELSE input_data = input_data
input_header.npoints = n_elements(input_data)

;do some calibration, energy interpolation for IF data only!
IF NOT (control.data_src EQ 2) THEN return

END

;******************************************************************************
;*  NAME:        r_cluster_rapid_onefile,fln
;*
;*  DESCRIPTION: reads one file of type plot_file. Assumes data has been
;*               stored in IDL saveset using structures input_header and
;*               input_data
;*
;*  INPUT:       fln  - string of filename
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        October 1997
;*
;*  AUTOR:       R. Friedel
;******************************************************************************
PRO r_cluster_rapid_onefile, plotinfo, VERBOSE=VERBOSE

COMMON cluster_rapid_data, input_header, input_data
COMMON get_error, get_err_no, get_err_msg
COMMON cluster_rapid

get_err_no=0  &  get_err_msg='cluster_rapid onefile read succesfully'

;get the module's control structure
control = *plotinfo.USR_PTR1

;------------------------------------------------------------------------------
data_source = control.data_src
data_type = control.product
plot_type =  control.pl_type
Sc4_flag =  control.prod_4sc
force = control.make
;------------------------------------------------------------------------------

;clear data arrays for reuse
input_data=0

;make filenames
SC_id=varprt(control.sc_id+1)  &  ext = '.idl'

IF force EQ 2 THEN GOTO, read_original

CASE control.data_src OF
    0: data_src = 'ifdata'
    1: data_src = 'science'
    2: data_src = 'pitch'
    3: data_src = 'det_sec_pitch'
ENDCASE

ext = '' ;sm
CASE data_src OF
    'ifdata': BEGIN ;RAL intermediate files, IDL savesets
          IF NOT (papco_check_data_env('CLUSTER_RAPID_IF', PATH=path)) $
            THEN return
          fln = 'RAPID_' + SC_id + papco_get_pathsep() + $
                if_names(control.product)+ papco_get_pathsep() + $
                strmid(plotinfo.filename,2,8)
          ext = '*.dat'                           ;sm
          file = path + fln + ext
          ;check if '_PA" file was requested. Call ies_reprocess if needed.
          pos = strpos(if_names(control.product), '_PA')
          IF pos NE -1 THEN BEGIN 
              ;see if exists
              result = file_search(file, count = c)
              IF c EQ 0 THEN BEGIN ;call ies_reprocess to make file
                  ;see of "non-PA" file exists
                  product = strmid(if_names(control.product), 0, pos)
                  fln = 'RAPID_' + SC_id + papco_get_pathsep() + $
                        product+ papco_get_pathsep() + $
                        strmid(plotinfo.filename,2,8)
                  result = file_search(path + fln + ext, count = c)
                  IF c NE 0 THEN BEGIN
                      ies_reprocess, /RAPID, /PA, FILES = result
                  ENDIF 
              ENDIF 

          ENDIF

    END
    'science' : begin  ;files are Lindau Daly/Kettman ascii science files.
          if not (papco_check_data_env('CLUSTER_RAPID_SCI_IDL', PATH=path)) $
            then return
          name = strsplit(sci_names(control.product), ' ',/extract)
          CASE control.cal OF
            0: BEGIN & units = '_K' & key = '/COUNTS'  & END
            1: BEGIN & units = '_R' & key = '/RATES'   & END
            2: BEGIN & units = '_I' & key = '/INTFLUX' & END
            3: BEGIN & units = ''   & key = '/FLUX'    & END
            4: BEGIN & units = ''   & key = '/FLUX'    & END
          ENDCASE
          IF name(0) EQ 'MDATA' THEN units = '_K'
          fln = 'C'+SC_id+'_'+plotinfo.filename+'_'+ name(0)
          ext = units + '.idl' ;sm
          file = path + fln + ext  ;sm
    END
    'pitch':BEGIN             ;files are Reiner's LANL pitch angle files
        if not (papco_check_data_env('CLUSTER_RAPID_PITCH', PATH=path)) $
          then return
        name = lanl_names(control.product)
        fln = 'C'+SC_id+'_'+plotinfo.filename+'_'+ name
        ext = '_pitch.idl'       ;sm
        file = path + fln + ext  ;sm
    END
    'det_sec_pitch':BEGIN
        cluster_rapid_ies_pitch, plotinfo
        return
    END
ENDCASE

;if file is gzipped, unzip to temp directroy and read from there
zipflag=0
result = findfile(file, COUNT=count)  ;sm
IF count EQ 0 THEN BEGIN
    zipfile=findfile(path+fln+ext+'*.gz',count=n)
    IF n EQ 1 then BEGIN
        message,'Moving '+ zipfile(0) + ' to ' + getenv('PAPCO_DOWNLOAD') + $
          ' and unzipping, then reading',/cont
        spawn,'cp '+zipfile(0)+ ' ' + getenv('PAPCO_DOWNLOAD')
        papco_sepdir, zipfile(0), dir, name, ext
        spawn, 'gunzip '+getenv('PAPCO_DOWNLOAD')+papco_get_pathsep()+name+ext
        file=getenv('PAPCO_DOWNLOAD')+papco_get_pathsep()+name+'.dat'
        zipflag=1
    ENDIF
ENDIF else file=result(0)


;check if file exists - try to get remotely if not.
;only try this if the "make" option is set to "No".
IF force EQ 1 THEN BEGIN

    result = findfile( path+fln+ext, COUNT=count)  ;sm
    IF count GT 0 THEN file = result[0] $ ;file exisit locally ;sm
    ELSE BEGIN  ;ftp file from CLUSTER_RAPID_DATA_FTP_ARCHIVE. ;sm

        papco_sepdir, file, download_dir, fln, ext
        remote_file_path = strmid(download_dir, strlen(path), $
                           strlen(download_dir)-strlen(path))

        url = papco_getenv('CLUSTER_RAPID_DATA_FTP_ARCHIVE')      ;sm
        url = url + data_src + '/'+ remote_file_path

        papco_get_remote_data, url, fln+ext, download_dir  ;sm

        result = findfile(path+fln+ext, COUNT=count)  ;sm
        IF count GT 0 THEN file = result[0]  ;sm

    ENDELSE                         ;sm

ENDIF

on_ioerror, read_original
message, 'Reading saveset', /info
print, '  ', file
restore, file, VERBOSE=VERBOSE
dealloc_lun,/quiet   ;frees logical units used by save/restore

IF zipflag THEN spawn,'rm -f '+file ;delete unzipped file if needed
return

read_original:

;if the .idl file is not found, try to read the original
r_cluster_rapid_onefile_original, plotinfo, VERBOSE = VERBOSE

;if the mpae dist zip file exists, delete date's files from rdda and
;binary directories.
d_cluster_rapid, plotinfo

END

;******************************************************************************
;*  NAME:        r_cluster_rapid_onefile_original, plotinfo
;*
;*  DESCRIPTION: reads one file of type plot_file_original. Under the
;*               papco read philosophy, files should be read the first
;*               time in whatever format they might have - ascii,
;*               fortran binary, pascal binary, etc. The files might also
;*               be in some compressed format, so they need to be uncompressed
;*               first.
;*
;*               You might also have to change time formats.
;*               A lot of files are stored per day, and have seconds of
;*               that day as the time. This is what is assumed here and
;*               is papco standard. You need to convert the time
;*               to papco standard if you intend to use the plotroutines
;*               supplied with papco (use the routines in
;*               $papco_BASE/papco_Lib/time_convert.pro).
;*               If you use your own plot routines this step is not needed.
;*
;*               The first read might be slow, esp if the format is ascii.
;*               Use the IDL "save" command to make a binary copy of the
;*               file which can then be read quickly using "restore". The
;*               next time only the IDL saveset is read, which is much
;*               faster.
;*
;*               In order to be able to concatonate the arrays easily,
;*               it makes sense to have the data in an array of time-ordered
;*               structures, which makes adding together days of data
;*               trivial using the IDL concatonation operator "[]".
;*
;*  INPUT:       fln  - string of filename
;*
;*  KEYWORDS:    VERBOSE - give verbose progress output to xterm
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        October 1997
;*
;*  AUTOR:       R. Friedel
;******************************************************************************
pro r_cluster_rapid_onefile_original, plotinfo, VERBOSE=VERBOSE

COMMON cluster_rapid_data, input_header, input_data
COMMON get_error, get_err_no, get_err_msg
COMMON cluster_rapid

IF NOT (papco_check_data_env('CLUSTER_RAPID_EXEC', PATH=exe_path)) THEN return

get_err_no=0  &  get_err_msg='cluster_rapid_onefile_orignal read succesfully'

;get the module's control structure
control = *plotinfo.USR_PTR1

if keyword_set(VERBOSE) then verbose=VERBOSE else verbose=0

SC_id=varprt(control.sc_id+1)

;------------------------------------------------------------------------------
data_source = control.data_src
data_type = control.product
plot_type =  control.pl_type
sc4_flag =  control.prod_4sc
force = control.make

;------------------------------------------------------------------------------
; read the original data file - THIS NEEDS TO BE REWRITTEN BY USER!!!!

; define the header and data structures. Edit to suit your data!
  ;input_header.sc_id = SC_ID
dummy={clust_rapid_data, epoch:0.0d, HSPCT:intarr(8), HESPCT:intarr(8), $
       CNOSPCT:intarr(8)}
top:

CASE data_source OF
    0: data_src = 'ral_if'
    1: data_src = 'science'
    2: data_src = 'pitch'
ENDCASE

;------------------------------------------------------------------------------
CASE data_src OF

  'ral_if' : begin  ;files are RAL intermediate , call RAL routine to make

      prob_path = papco_getenv('CLUSTER_RAPID_PROBLEMS')+'if/'
      fln = 'c'+SC_id+'_'+strmid(plotinfo.filename,2,8)

      CASE force OF
          0: BEGIN              ;allow make, but check for problem flag
              message, 'Making original ral_if files', /cont
              result = findfile(prob_path+fln+'.problem', count = c)
              IF c NE 0 THEN BEGIN
                  get_err_no=7
                  get_err_msg=fln+ ' Sat. date flagged, not processed'
                  message,get_err_msg, /cont
                  return
              ENDIF
          END
          1: BEGIN              ;inhibit make, return
              get_err_no = 1
              get_err_msg = 'ral_if make original disabled'
              message, '...not found, '+get_err_msg, /cont
              return
          END
          2: BEGIN              ;allow make, clear problem flags
              message, 'Forcing making of original ral_if files', /cont
              spawn, 'rm -f '+prob_path+fln+'.problem'
          END
      ENDCASE

     ;If not, call RAL routine to make IF data files. Input: lindau bin files.
     ;check if lindau bin files have been made, if not, make them!
     m_cluster_rapid_bin, plotinfo

     tstart = strmid(plotinfo.filename, 2, 6) & tstop = tstart
     CASE control.sc_id OF ;select spacecraft to process
     0: BEGIN & sc1 = 1 & sc2 = 0 & sc3 = 0 & sc4 = 0 & END
     1: BEGIN & sc1 = 0 & sc2 = 1 & sc3 = 0 & sc4 = 0 & END
     2: BEGIN & sc1 = 0 & sc2 = 0 & sc3 = 1 & sc4 = 0 & END
     3: BEGIN & sc1 = 0 & sc2 = 0 & sc3 = 0 & sc4 = 1 & END
     ENDCASE

     IF NOT keyword_set(VERBOSE) THEN silent = 1 ELSE silent = 0
     ;before doing the "make" on if files, check if rcs_batch is loaded
     proc_call = 'rcs_batch'
     error_status=0
     catch, error_status
     if error_status ne 0 then begin ;call error handler
       papco_error_handler, $
         error_status, proc_call, panelkind, panelVector, /NO_ERR_DRAW
       error_status=0 & catch, /CANCEL  ;disable error catching
       get_err_no = 6  &  get_err_msg = 'Cannot access RAL IF routines'
       papco_SetColorTable
       return
     endif

     ;select IF data sets to be made by using corresponding IF
     ;configuration file. This file is automatically created
     ;by m_cluster_rapid_ifcfg.
     m_cluster_rapid_ifcfg, data_type, ifconf_fln

     ; default directory for input data (from LINDAU binary DAT files)
     ; this is set by rcs_init, overwrite if needed here
     ;SETENV, 'RCS_DATA_DIRECTORY=/n/projects/cda/cluster/rapid/binary'

     CALL_PROCEDURE, proc_call, TSTART = tstart,  TSTOP = tstop, $
       SC1 = sc1, SC2 = sc2, SC3 = sc3, SC4 = sc4, SILENT = silent, $
       IFCFG = ifconf_fln, /DAT, /FORCE

     catch, /CANCEL             ;disable error catching
     ;need to reset papco color tables
     papco_SetColorTable

     ;try again to read the idl savesets, but don't try to make originals!
     new_plotinfo = plotInfo
     new_control = *plotinfo.USR_PTR1
     new_control.make = 1
     new_plotinfo.USR_PTR1 = ptr_new(new_control)
     r_cluster_rapid_onefile, new_plotinfo
     IF get_err_no NE 0 THEN BEGIN ;making ral IF files failed
       get_err_no = 1  &  get_err_msg = 'RAL IF file could not be made'
       message, get_err_msg, /cont
       ;flag file as "problem" to avoid trying to make again.
       fln = 'c'+SC_id+'_'+strmid(plotinfo.filename,2,8)
       spawn, 'touch '+prob_path+fln+'.problem'
       message,'Date flagged as problem',/cont
     ENDIF
     return
  END
  ;----------------------------------------------------------------------------
  'science' : BEGIN  ;files are Lindau Daly/Kettman ascii science files.
      if not (papco_check_data_env('CLUSTER_RAPID_SCI', PATH=path)) $
        then return

      prob_path = papco_getenv('CLUSTER_RAPID_PROBLEMS') + $
                  'science' + papco_get_pathsep()

      CASE control.cal of
          0: BEGIN & units = '_K' & key = 'COUNTS'  & END
          1: BEGIN & units = '_R' & key = 'RATES'   & END
          2: BEGIN & units = '_I' & key = 'INTFLUX' & END
          3: BEGIN & units = ''   & key = 'FLUX'    & END
          4: BEGIN & units = ''   & key = 'FLUX'    & END
      ENDCASE

      name1 = strsplit(sci_names(control.product), ' ',/extract)
      name = strsplit(name1(0), '_',/extract)
      date = plotinfo.filename
      fln = 'C'+SC_id+'_'+date+'_'+ name1(0) +  units + '.SCI'

      CASE force OF
          0: BEGIN              ;allow make, but check for problem flag
              message, 'Making original science files', /cont
              prob_fln = prob_path+'C'+SC_id+'_'+date+'_'+name(0)
              result = findfile(prob_fln, count = c)
              IF c NE 0 THEN BEGIN
                  get_err_no=7
                  get_err_msg=fln+ ' Sat. date flagged, not processed'
                  message,get_err_msg, /cont
                  return
              ENDIF
          END
          1: BEGIN              ;inhibit make, return
              get_err_no = 1
              get_err_msg = 'science make original disabled'
              message, '...not found, '+get_err_msg, /cont
              return
          END
          2: BEGIN              ;allow make, clear problem flags
              message, 'Forcing making of original science files', /cont
              spawn, 'rm -f '+prob_path+fln+'.problem'
          END
      ENDCASE

      ;If MPAe ASCII Science file don't exists, spawn a call to make them.
      file = findfile(path+fln, count = c)
      IF (c EQ 0) OR (force EQ 2) THEN BEGIN ;make file
          message, 'Calling MPAe softwae to make files',/cont
          m_cluster_rapid_msf, plotinfo
          exe_cmd = exe_path+'cluster_rapid_msf2sci '
          spawn, exe_cmd+ date+' '+SC_id+' '+name(0)+' /'+key

          ;see if sci files got made
          file = findfile(path+fln, count = c)
          IF c NE 1 THEN BEGIN
              get_err_no=9 & get_err_msg='problem making science files'
              message, get_err_msg, /cont
              spawn, 'echo '+get_err_msg+' >! '+prob_path+'C'+SC_id+'_'+date+$
                '_'+name(0)
              return
          ENDIF
      ENDIF ;make files

      ;now read the science files
      message, 'reading MPAe ASCII Science file (slow)', /cont
      print,'  ',path+fln

      ;open file and read header into input_header structure.
      in_str = ''
      data_name = 'cluster_rapid_science_' + sci_names(control.product)

      hh = create_struct('DATATYPE', 'MPAE Ascii Science files', 'NPOINTS', 0l)
      spin_p = 4.0 ;nominal spin period in seconds

      ;make data array for each data type.
      CASE name1(0) OF
        'MDATA': BEGIN
            data = intarr(16, 2)
            in_data = intarr(16)
            dummy = create_struct(name = data_name, 'time', 0.0d, $
                                  'timestr','',$
                                  'endtime', 0.0d, 'id', '', $
                                  'comment', '',  'data', data, 'b_sec', 0)
            data_size = 2000 & nm_delta = 1 & bm_delta = 1
        END
        'I3DD': BEGIN
            data = fltarr(12,8,3,16)
            ch = fltarr(9)
            hh = create_struct(hh, 'NDETECTORS', 12)
            hh = create_struct(hh, 'NBANDS', 8)
            hh = create_struct(hh, 'NSECTORS', 16)
            hh = create_struct(hh, 'NSPECIES', 3)
            hh = create_struct(hh, 'SPECIES', ['H', 'He', 'CNO'])
            hh = create_struct(hh, 'DETECTORS', indgen(12)+1)
            hh = create_struct(hh, 'CH_POSITIONS', fltarr(2, 8, 12, 3))
            dummy = create_struct(name = data_name, 'time', 0.0d, $
                                  'timestr','',$
                                  'endtime', 0.0d, 'id', '', $
                                  'comment', '',  'data', data)
            data_size = 200 & nm_delta = 32 & bm_delta = 8
        END
        'HSPCT': BEGIN
            data = fltarr(8)
            ch = fltarr(9)
            hh = create_struct(hh, 'NDETECTORS', 1)
            hh = create_struct(hh, 'NBANDS', 8)
            hh = create_struct(hh, 'NSECTORS', 1)
            hh = create_struct(hh, 'NSPECIES', 1)
            hh = create_struct(hh, 'SPECIES', ['H'])
            hh = create_struct(hh, 'CH_POSITIONS', fltarr(2, 8))
            dummy = create_struct(name = data_name, 'time', 0.0d, $
                                  'timestr','',$
                                  'endtime', 0.0d, 'id', '', $
                                  'comment', '',  'data', data)
            data_size = 2000 & nm_delta = 1 & bm_delta = 1
        END
        'ISPCT': BEGIN
            data = fltarr(8, 2)
            ch = fltarr(9)
            hh = create_struct(hh, 'NDETECTORS', 1)
            hh = create_struct(hh, 'NBANDS', 8)
            hh = create_struct(hh, 'NSECTORS', 1)
            hh = create_struct(hh, 'NSPECIES', 2)
            hh = create_struct(hh, 'SPECIES', ['He', 'CNO'])
            hh = create_struct(hh, 'CH_POSITIONS', fltarr(2, 8, 2))
            dummy = create_struct(name = data_name, 'time', 0.0d, $
                                  'timestr','',$
                                  'endtime', 0.0d, 'id', '', $
                                  'comment', '',  'data', data)
            data_size = 1000 & nm_delta = 1 & bm_delta = 1
        END
        ELSE: BEGIN
            get_err_no = 8
            get_err_msg = 'data type ' + name(0) + ' not supported'
            message, get_err_msg, /cont
            return
        end
      ENDCASE

      input_data = replicate(dummy, data_size) & c = 0
      info_str_arr = 'Comment and Information strings'
      delta = nm_delta ; default accumulation time, normal mode.


      ;ON_IOERROR, sci_read_error
      openr, u, path+fln, /get_lun

      readf, u, in_str
      WHILE strmid(in_str, 0, 1) EQ '!' DO BEGIN
          pos = strpos(in_str, ':')
          IF pos NE -1 THEN BEGIN
              tag_name = strsubst(strmid(in_str, 2, pos-2), ' ', '_')
              tag_name = strsubst(tag_name, '/', '_')
              tag_name = strsubst(tag_name, '(', '_')
              tag_name = strsubst(tag_name, ')', '_')
              tag_name = strsubst(tag_name, '-', '_')
              tag_value = strtrim(strmid(in_str,pos+1, strlen(in_str)), 2)
              ;print, tag_name, '  ', tag_value
              hh = create_struct(hh, tag_name, tag_value)
          ENDIF
          readf, u, in_str
      ENDWHILE

      ;now read in the data into corresponding structures...
      WHILE 1 DO begin
        input_data(c).id = strmid(in_str, 0, 1)
        time_str = strmid(in_str, 9, 25)
        input_data(c).timestr = time_str
        input_data(c).comment= strmid(in_str, 38, strlen(in_str))
        input_data(c).time = utc2tai(time_str)
        id = input_data(c).id

        CASE 1 OF
            (id EQ 'C'):BEGIN        ;comments, assemble into info string array
                info_str_arr = [info_str_arr, $
                  input_data(c).id+' '+time_str+' '+input_data(c).comment]
                GOTO, next_record
            END
            (id EQ 'I'):BEGIN        ;info, assemble into info string array
                info_str_arr = [info_str_arr, $
                  input_data(c).id+' '+time_str+' '+input_data(c).comment]
                GOTO, next_record
            END
            (id EQ 'M'):BEGIN        ;mode info - sets delta time of sample!
                CASE input_data(c).comment OF
                    'RAP_ENA: RAPID neutrals mode':delta = nm_delta
                    'RAP_NOM: RAPID nominal mode':delta = nm_delta
                    'RAP_STBY: RAPID standby mode':delta = nm_delta
                    ELSE:stop
                ENDCASE
                info_str_arr = [info_str_arr, $
                  input_data(c).id+' '+time_str+' '+input_data(c).comment]
                GOTO, next_record
            END
            (id EQ 'A'):BEGIN           ;skip coord info
                readf, u, in_str  & GOTO, next_record
            end
            (id EQ 'O'):BEGIN     ;skip orbit info
                readf, u, in_str & GOTO, next_record
            end
            (id EQ 'K') AND (name(0) EQ 'MDATA'):BEGIN   ;read data for MDATA
                pos=strpos(in_str,'Bsec')
                IF pos EQ -1 THEN GOTO, next_record
                b_sec = fix(strmid(in_str, pos+5, 2))
                ;readf, u, data, b_sec
                readf, u, in_data
                data(*, 0) = in_data
                readf, u, in_data
                data(*, 1) = in_data
                input_data(c).data = data
                input_data(c).b_sec = b_sec
                input_data(c).endtime = input_data(c).time + delta*spin_p
            end
            (id EQ 'K') AND (name(0) EQ 'HSPCT'):BEGIN   ;read data for HSPCT
                readf, u, data
                input_data(c).data = data
                input_data(c).endtime = input_data(c).time + delta*spin_p
            end
            (id EQ 'K') AND (name(0) EQ 'ISPCT'):BEGIN   ;read data for ISPCT
                readf, u, data
                input_data(c).data = data
                input_data(c).endtime = input_data(c).time + delta*spin_p
            end
            (id EQ 'D') OR (id EQ 'R') OR (id EQ 'K'):BEGIN ;read data for I3DD
                readf, u, data
                input_data(c).data = data
                input_data(c).endtime = input_data(c).time + delta*spin_p
            end
            (id EQ 'F'):BEGIN      ; ch positions for protons
                IF name1(0) EQ 'HSPCT' THEN BEGIN
                    readf, u, ch
                    hh.ch_positions(0,*) = transpose(ch(0:7))
                    hh.ch_positions(1,*) = transpose(ch(1:8))
                ENDIF ELSE BEGIN
                  FOR head = 0, 2 DO BEGIN ;loop through heads
                    readf, u, ch
                    FOR det = head*4,  head*4+3  DO BEGIN
                      hh.ch_positions(0,*,det,0) = transpose(ch(0:7))
                      hh.ch_positions(1,*,det,0) = transpose(ch(1:8))
                    END
                  END
                ENDELSE
                GOTO, next_record
            end
            (id EQ 'G'):BEGIN   ; ch positions for Helium
                IF name1(0) EQ 'ISPCT' THEN BEGIN
                    readf, u, ch
                    hh.ch_positions(0,*,0) = transpose(ch(0:7))
                    hh.ch_positions(1,*,0) = transpose(ch(1:8))
                ENDIF ELSE BEGIN
                    FOR head = 0, 2 DO BEGIN ;loop through heads
                    readf, u, ch
                    FOR det = head*4,  head*4+3  DO BEGIN
                      hh.ch_positions(0,*,det,1) = transpose(ch(0:7))
                      hh.ch_positions(1,*,det,1) = transpose(ch(1:8))
                    END
                  END
                ENDELSE
                GOTO, next_record
            end
            (id EQ 'H'):BEGIN       ; ch positions for CNO
                IF name1(0) EQ 'ISPCT' THEN BEGIN
                    readf, u, ch
                    hh.ch_positions(0,*,1) = transpose(ch(0:7))
                    hh.ch_positions(1,*,1) = transpose(ch(1:8))
                ENDIF ELSE BEGIN
                    FOR head = 0, 2 DO BEGIN ;loop through heads
                    readf, u, ch
                    FOR det = head*4,  head*4+3  DO BEGIN
                      hh.ch_positions(0,*,det,2) = transpose(ch(0:7))
                      hh.ch_positions(1,*,det,2) = transpose(ch(1:8))
                    END
                  END
                ENDELSE
                GOTO, next_record
            end

            ELSE: BEGIN
            end

        ENDCASE

        c = c+1
        IF c MOD data_size EQ 0 THEN BEGIN
          message, 'reading...', /cont
          input_data = [input_data, replicate(dummy, data_size)]
        ENDIF
        next_record:
        IF eof(u) THEN break
        readf, u, in_str
      ENDWHILE

      close,  u & free_lun, u

      IF c EQ 0 THEN BEGIN
          get_err_no=5 & get_err_msg = 'No data in SCI file'
          message, get_err_msg, /cont & return
      ENDIF

      input_data = input_data(0:c-1)
      ;add some more info to input_header
      head_name = strsubst(data_name,' ','_') + '_' + key + '_C' + $
        hh.SPACECRAFT+'_'+date+'_header'
      hh = create_struct(hh, 'INFO', info_str_arr)
      input_header = create_struct(name = head_name, hh)
      input_header.NPOINTS = c
      GOTO,  no_sci_read_error

      sci_read_error:
      ;report error in reading .sci file
      get_err_no=!ERROR & get_err_msg='!C '+!ERR_STRING
      message,!ERR_STRING , /cont
      spawn, 'echo problem reading science files  >! '+ $
        prob_path+'C'+SC_id+'_'+date+'_'+name(0)
      return

      no_sci_read_error:

      ; save read data as IDL saveset, rename to .idl
      if not (papco_check_data_env('CLUSTER_RAPID_SCI_IDL', PATH=path)) $
          then return

      papco_sepdir, fln, pth, file, ext
      message, 'Saving data in idl saveset', /cont
      print,'  ', path+file+'.idl'

      save, filename = path+file+'.idl', input_header, input_data, $
        /compress, VERBOSE = VERBOSE
      dealloc_lun,/quiet        ;frees logical units used by save/restore
      return

  END
  ;----------------------------------------------------------------------------
  'pitch': BEGIN

      prob_path = papco_getenv('CLUSTER_RAPID_PROBLEMS') + $
                  'pitch'+ papco_get_pathsep()
      fln = 'c'+SC_id+'_'+strmid(plotinfo.filename,2,8)

      CASE force OF
          0: BEGIN              ;allow make, but check for problem flag
              message, 'Making original lanl_pitch files', /cont
              result = findfile(prob_path+fln+'.problem', count = c)
              IF c NE 0 THEN BEGIN
                  get_err_no=7
                  get_err_msg=fln+ ' Sat. date flagged, not processed'
                  message,get_err_msg, /cont & return
              ENDIF
          END
          1: BEGIN              ;inhibit make, return
              get_err_no = 1
              get_err_msg = 'lanl_pitch make original disabled'
              message, '...not found, '+get_err_msg, /cont
              return
          END
          2: BEGIN              ;allow make, clear problem flags
              message, 'Forcing making of original lanl_pitch files', /cont
              spawn, 'rm -f '+prob_path+fln+'.problem'
          END
      ENDCASE

      m_cluster_rapid_pitch, SC_ID = SC_ID, SPECIES = data_type, /NO_TEST, $
        units = control.cal

      IF get_err_no NE 0 THEN BEGIN
          get_err_no = 1 & get_err_msg = 'problem making lanl_pitch'
          message, '...not found, '+get_err_msg, /cont & return
      ENDIF

      w_cluster_rapid_pitch, SPECIES = data_type
      return

  END

ENDCASE

; end of user written section
;------------------------------------------------------------------------------

out:
;if we got here, the files needed might not exist. Try to make!
;first see if this date is "flagged" as not working...
cluster_rapid_make_l0, plotinfo
;if make was succesfull, read the file
if get_err_no eq 0 then goto, top

; if there was an error, report it.
get_err_no=!ERROR & get_err_msg='!C '+!ERR_STRING
message,get_err_msg, /cont

END

;******************************************************************************
;*  NAME:        m_cluster_rapid_msf
;*
;*  DESCRIPTION  check for files needed and call routines to make if needed.
;*               new Jan 04 - get msf' from zip files in dist
;                             archive. 
;*  
;******************************************************************************
PRO m_cluster_rapid_msf, plotinfo, VERBOSE = VERBOSE

COMMON get_error, get_err_no, get_err_msg

;get the module's control structure
control = *plotinfo.USR_PTR1

force = control.make & date = plotinfo.filename
SC_id=+varprt(control.sc_id+1)
IF NOT (papco_check_data_env('CLUSTER_RAPID_PROBLEMS', PATH=prob)) then return
IF NOT (papco_check_data_env('CLUSTER_RAPID_RDDA', PATH=p)) then return
IF NOT (papco_check_data_env('CLUSTER_RAPID_DIST', PATH=dist_path)) then return
IF NOT (papco_check_data_env('CLUSTER_RAPID_EXEC', PATH=exe_path)) THEN  return

mpae_rdda = '/n/projects/cluster/rapid/rdda_lindau_sync/'
;check for msf files first, make if needed
msf_fln = p+'cluster'+SC_id+'/msf_'+SC_id+'/' + strmid(date,2,8)+'*.???'
fls = findfile(msf_fln, count = c)

IF c EQ 0  THEN BEGIN ;get from zip archive
    dist_zip = file_search(dist_path+strmid(date,2,8)+'*', count = c)
    IF c EQ 1 THEN BEGIN
        message, 'Getting msf from MPAE zip archive', /info
        cmd = 'unzip -l '+' '+ dist_zip
        print, cmd & spawn, cmd, result
        pos = strpos(result, 'cluster'+SC_id)
        idx = where(pos NE -1, c)
        files = strmid(result(idx), pos(idx(0)), 27)
        file_list = ''
        FOR i = 0, c-1 DO file_list = file_list + " '"+files(i)+"'"
        cmd = 'unzip -o -d '+p+' '+ dist_zip+file_list
        print, cmd & spawn, cmd
        force = 0
    ENDIF 
ENDIF 

IF c EQ 0 THEN BEGIN            ;get from mpae_rdda temporary directory.
    fln = 'cluster'+SC_id+'/msf_'+SC_id+'/' + strmid(date,2,8)+'*.???'
    fls = findfile(mpae_rdda+fln, count = c)
    IF c NE 0 THEN BEGIN 
        message, 'Getting msf from MPAE rdda download', /info
        file_copy, fls, p+'cluster'+SC_id+'/msf_'+SC_id+'/', /overwrite, /verb
    ENDIF 
    fln = 'cluster'+SC_id+'/aux_'+SC_id+'/' + strmid(date,2,8)+'*.???'
    fls = findfile(mpae_rdda+fln, count = c)
    IF c NE 0 THEN BEGIN 
        message, 'Getting aux from MPAE rdda download', /info
        file_copy, fls, p+'cluster'+SC_id+'/aux_'+SC_id+'/', /overwrite, /verb
    ENDIF 
ENDIF

fls = findfile(msf_fln, count = c)

IF (c EQ 0) OR (force EQ 2) THEN BEGIN ;making or forcing make if needed
    exe_cmd = exe_path+'cluster_rapid_rdda2msf '
    spawn, exe_cmd + date + ' '+SC_id
    ;check if any msf files were made
    fls = findfile(msf_fln, count = c)
    IF c EQ 0 THEN BEGIN
        get_err_no=6 & get_err_msg='msf files ' +date+ ' not made'
        message,get_err_msg, /cont
        spawn, 'touch '+prob+'/msf/C'+SC_id+'_'+date
    ENDIF ELSE message, 'msf made succesfully', /cont
ENDIF ELSE message, 'have msf files', /cont

END

;******************************************************************************
;*  NAME:        m_cluster_rapid_bin
;*
;*  DESCRIPTION  check for files needed and call routines to make if needed.
;******************************************************************************
PRO m_cluster_rapid_bin, plotinfo, VERBOSE = VERBOSE

COMMON get_error, get_err_no, get_err_msg

;get the module's control structure
control = *plotinfo.USR_PTR1

;call routine to make /check for msf files first
message, 'Calling m_cluster_rapid_msf first', /info
m_cluster_rapid_msf, plotinfo, VERBOSE = VERBOSE
IF get_err_no NE 0 THEN return

if not (papco_check_data_env('CLUSTER_RAPID_PROBLEMS', PATH=prob)) then return
IF NOT (papco_check_data_env('CLUSTER_RAPID_EXEC', PATH=exe_path)) THEN  return

force = control.make
date = plotinfo.filename
SC_id=+varprt(control.sc_id+1)

;check for binary files, make if needed
if not (papco_check_data_env('CLUSTER_RAPID_BINARY', PATH=p)) then return
bin_fln = p+'c'+SC_id+'_'+strmid(date,2,8)+'*'
files = findfile(bin_fln, count = c)
IF (c EQ 0) OR force  THEN BEGIN
    exe_cmd = exe_path+'cluster_rapid_msf2bin '
    spawn, exe_cmd + date + ' '+SC_id   
    ;check if any bin files were made
    files = findfile(bin_fln, count = c)
    IF c EQ 0 THEN BEGIN
        get_err_no=7 & get_err_msg='bin files ' +date+ ' not made'
        message,get_err_msg, /cont
        spawn, 'touch '+prob+'/bin/C'+SC_id+'_'+date
    ENDIF ELSE message, 'bin made succesfully', /cont
ENDIF ELSE message, 'have bin files', /cont

END


;******************************************************************************
;*  NAME:        r_cluster_rapid_4SC
;*
;*  DESCRIPTION  Read  data from all four spacecraft and interpolate
;*               onto a common time base.
;*               Returns error if data cannot be found for all SC
;*
;******************************************************************************
PRO r_cluster_rapid_4SC, plotinfo, VERBOSE=VERBOSE, NO_ORIGINAL = NO_ORIGINAL

COMMON cluster_rapid_4sc, data4sc_timed
COMMON cluster_rapid_data, input_header, input_data
COMMON cluster_rapid_slice_1, time, zmat, data, yarr, utitle, uytitle, uztit
COMMON cluster_rapid
COMMON get_error, get_err_no, get_err_msg

;get the module's control structure
control = *plotinfo.USR_PTR1

sc_id = control.sc_id
sc4_flag =  control.prod_4sc
smooth = control.smooth
data4sc = 0

;read data for all four spacecraft then recursively call plot
;routine to make the plot data required. USE the output keyword to
;indicate recursive call - make plot data only.
message, 'Reading data for all spacecraft', /cont
FOR i = 0,  3 DO BEGIN ;loop through spacecraft
    control.sc_id = i & control.prod_4sc = 0
    control.smooth = 0
    r_cluster_rapid, plotinfo
    IF get_err_no ne 0 THEN BEGIN ;check if data read O.K.
        get_err_msg = 'can not read 4 SC data, SC missing'
        GOTO, found_error
    ENDIF
    p_cluster_rapid_spec, [-99, 0, 0], plotInfo, OUTPUT=2
    ;check if any data contained for time period
    IF get_err_no ne 0 THEN BEGIN
        get_err_msg = 'can not read 4 SC data, no common data for time period'
        GOTO, found_error
    ENDIF
    dummy = {time:time, zmat:zmat}
    IF i EQ 0 THEN $ ;make structure for data from 4 spacecraft
      data4sc = create_struct('cl'+varprt(i+1), dummy) $
    ELSE $ ;add spacecraft to structure
      data4sc = create_struct(data4sc,'cl'+varprt(i+1), dummy)
ENDFOR

;now, need to obtain common time axis for data. Use the time of the
;spaceraft chosen as basis for time, interpolate all other spacraft
;data to that time
message, 'Folding data onto time of '+sc_names(sc_id), /cont

data4sc_timed = {time:data4sc.(sc_id).time,  $
                 zmat1:data4sc.(sc_id).zmat,  $
                 zmat2:data4sc.(sc_id).zmat,  $
                 zmat3:data4sc.(sc_id).zmat,  $
                 zmat4:data4sc.(sc_id).zmat }

;get center time from reference spacraft
t_ref = ( data4sc_timed.time(*, 0)+data4sc_timed.time(*, 1) )/2.0
;get number of channels in data array
result = size(data4sc.(sc_id).zmat,  /struct)
ch =  result.dimensions(1)

FOR i = 0,  3 DO BEGIN ;loop through spacecraft
    IF i EQ sc_id THEN continue ;skip for referecne spacecraft
    message, '...doing for Cluster '+varprt(i+1), /cont
    t_center = ( data4sc.(i).time(*, 0)+data4sc.(i).time(*, 1) )/2.0
    ;loop through channels to interpolate
    FOR j = 0, ch-1 DO data4sc_timed.(i+1)(*, j)= $
                         interpol(data4sc.(i).zmat(*, j), t_center, t_ref)
ENDFOR

found_error:
control.sc_id = sc_id
control.prod_4sc = sc4_flag
control.smooth = smooth

END


;******************************************************************************
;*  NAME:        m_cluster_rapid_batch
;*
;*  DESCRIPTION: Makes batch file data containing all the times for
;*               which data can be found, for a given CCR web plot
;*               Allows for checking of file dependencies of the type
;*               "all times for which plots do not exist but for
;*                which x data files exists"
;*
;*  INPUT:       type - CCR plot type, from rapid commons array
;*                      ccr_plot_types
;*
;*  KEYWORDS:    VERBOSE - give verbose progress output to xterm
;*               BEFORE  - string of date to stop at, yyyymmdd
;*               FROM    - string of date to start from, yyyymmdd
;*
;*
;*               SOURCE  - source file type
;*               DEST    - destination file type
;*               SC_ID   - string id of Cluster spacraft to be used for date
;*                         search. If 'A' then find dates for which
;*                         data files exist for all 4 spacraft only
;*               FORCE   - Force all dates in source
;*          PROB_FORCE   - Ignore problem dates
;*               SAME    - instead of missing dates, report same dates
;*
;*  OUTPUT:      batch_times - start / end times for available data
;*               filename    - suggested default filename for batch file
;*
;*  DATE:        January 2002
;*
;*  AUTOR:       R. Friedel
;******************************************************************************
PRO m_cluster_rapid_batch, batch_times, filename, VERBOSE = VERBOSE, $
                           CCR_TYPE = CCR_TYPE, $
                           SC_ID = SC_ID, SOURCE = SOURCE, DEST = DEST, $
                           WRITE = WRITE,  FORCE = FORCE,  $
                           PROB_FORCE = PROB_FORCE,  $
                           FROM = FROM,  BEFORE = BEFORE, $
                           RESTRICT = RESTRICT,  SAME = SAME

COMMON cluster_rapid_data, input_header, input_data
COMMON cluster_rapid
COMMON mjdt, mjdt_start, mjdt_end

if not (papco_check_data_env('CLUSTER_RAPID_RDDA', PATH=path1)) then return
if not (papco_check_data_env('CLUSTER_RAPID_BINARY', PATH=path2)) then return
if not (papco_check_data_env('CLUSTER_RAPID_IF', PATH=path3)) then return
if not (papco_check_data_env('CLUSTER_RAPID_SCI', PATH=path3)) then return
if not (papco_check_data_env('CLUSTER_RAPID_PITCH', PATH=path3)) then return

IF keyword_set(SC_ID) THEN sc_id = SC_ID ELSE sc_id = '1'
IF keyword_set(SOURCE) THEN source = strupcase(SOURCE) ELSE source = 'DIST'
IF keyword_set(DEST) THEN dest = strupcase(DEST) ELSE dest = 'CCR'
IF keyword_set(CCR_TYPE) THEN ccr_type = CCR_TYPE ELSE ccr_type = 'Alle'

rap_base= '/n/projects/cda/cluster/rapid/'
prob_base = rap_base+'problems/' & dir_prob = ''

IF dest EQ 'CCR' THEN BEGIN
    idx = where(ccr_plot_types EQ ccr_type, c)
    IF c NE 1 THEN BEGIN
        message, 'CCR_TYPE '+ccr_type+' not found', /cont
        print, '  Valid CCR_TYPEs: ', ccr_plot_types
        return
    END
    ccr_path  = rap_base+'ccr_plots/'
    ccr_dir = 'ClusterRapid'+ccr_type+'/'
ENDIF

if keyword_set(VERBOSE) then BEGIN
    message, 'Doing for type '+ccr_dir, /cont
    message, 'Using Cluster '+SC_ID+' for dates',/cont
ENDIF

;select the source data files to check against - X is SC #
;X will be replaced by 1,2,3,4 or A - depending on SC_ID
CASE source OF
      'RDDA'   : BEGIN
          dir_source = rap_base+'rdda/clusterX/nsd_X/'
          mask_source = '*rn.1a*'
          dy = '20' & d1 = 0 & d2 = 6 ;for strmid to get date string
      END
      'DIST'   : BEGIN
          dir_source = rap_base+'mpae_dist/'
          mask_source = '*.zip'
          dy = '20' & d1 = 0 & d2 = 6 ;for strmid to get date string
      END
      'RECENT_DIST'  : BEGIN
          dir_source = rap_base+'mpae_recent_dist/'
          mask_source = '*.zip'
          dy = '20' & d1 = 0 & d2 = 6 ;for strmid to get date string
      END 
      'MSF'    : BEGIN
          dir_source = rap_base+'rdda/clusterX/msf_X/'
          mask_source = '*ri.0a*'
          dy = '20' & d1 = 0 & d2 = 6 ;for strmid to get date string
      END
      'BINARY' : BEGIN
          dir_source = rap_base+'binary/'
          mask_source = 'cX_*.desc'
          dy = '20' & d1 = 3 & d2 = 6 ;for strmid to get date string
      END
      'RAL_IF' : BEGIN
          dir_source = rap_base+'ifdata/RAPID_X/'
          IF ccr_type EQ 'AlleBM' THEN dir_source = dir_source+'IES_CLEANBM/' $
                                  ELSE dir_source = dir_source+'IES_CLEAN'
          mask_source = '*xa.dat'
          dy = '20' & d1 = 0 & d2 = 6 ;for strmid to get date string
      END
      'SCIENCE': BEGIN
          dir_source = rap_base+'science/'
          mask_source = 'CX_*_HSPCT.SCI'
          dy = '' & d1 = 3 & d2 = 8 ;for strmid to get date string
      END
      'PITCH'  : BEGIN
          dir_source = rap_base+'pitch/'
          mask_source = 'CX_*_IES_CLEANBM_pitch.idl'
          dy = '' & d1 = 3 & d2 = 8 ;for strmid to get date string
      END
      'MFE'    : BEGIN
          dir_source = getenv('PAPCO_DLNK')+'/cdf/cluster_fgm/*/'
          mask_source = 'cX_pp_fgm_*.cdf'
          dy = '' & d1 = 10 & d2 = 8 ;for strmid to get date string
      END 
  ENDCASE

  ;obtain all dates for files in source
  dir_source = strsubst(dir_source, 'X', SC_ID)
  mask_source = strsubst(mask_source, 'X', SC_ID)

  files = file_search(dir_source+mask_source, count=n_source)
  files = file_basename(files)
  dates_source=dy+strmid(files,d1,d2)

  message, 'Source path: '+dir_source+mask_source, /cont
  message, 'Source '+source+': '+varprt( n_source)+' days', /cont

  ;select the final destination files needed.
  CASE dest OF
      'DIST'   : BEGIN
          dir_dest = rap_base+'mpae_dist/'
          mask_dest = '*.zip'
          dy = '20' & d1 = 0 & d2 = 6 ;for strmid to get date string
      END
      'RECENT_DIST'  : BEGIN
          dir_dest = rap_base+'mpae_recent_dist/'
          mask_dest = '*.zip'
          dy = '20' & d1 = 0 & d2 = 6 ;for strmid to get date string
      END 
      'MSF'    : BEGIN
          dir_dest = rap_base+'rdda/clusterX/msf_X/'
          dir_prob = prob_base+'msf/'
          mask_dest = '*ri.0a*'
      END
      'BINARY' : BEGIN
          dir_dest = rap_base+'binary/'
          dir_prob = prob_base+'bin/'
          mask_dest ='cX_*.desc'
          dy = '20' & d1 = 3 & d2 = 6 ;for strmid to get date string
      END
      'RAL_IF' : BEGIN
          dir_dest = rap_base+'ifdata/RAPID_X/'
          IF ccr_type EQ 'AlleBM' THEN dir_dest = dir_dest+'IES_CLEANBM/' $
                                  ELSE dir_dest = dir_dest+'IES_CLEAN'
          dir_prob = prob_base+'if/'
          mask_dest ='*xa.dat'
          dy = '20' & d1 = 0 & d2 = 6 ;for strmid to get date string
      END
      'SCIENCE': BEGIN
          dir_dest = rap_base+'science/'
          dir_prob = prob_base+'sci/'
          mask_dest ='CX_*_HSPCT.SCI'
          dy = '' & d1 = 3 & d2 = 8 ;for strmid to get date string
      END
      'PITCH'  : BEGIN
          dir_dest = rap_base+'pitch/'
          dir_prob = prob_base+'pitch/'
          mask_dest ='CX_*_IES_CLEANBM_pitch.idl'
          dy = '' & d1 = 3 & d2 = 8 ;for strmid to get date string
      END
      'CCR'    : BEGIN
          dir_dest = rap_base+'SummaryPlots/'+ccr_dir
          mask_dest ='*.ps.gz'
          dy = '' & d1 = 0 & d2 = 8 ;for strmid to get date string
      END
  ENDCASE

  ;obtain all dates for files in dest
  dir_dest = strsubst(dir_dest, 'X', SC_ID)
  mask_dest = strsubst(mask_dest, 'X', SC_ID)
  cd, dir_dest, current = orig_dir
  files=findfile(mask_dest, count=n_dest)
  dates_dest=dy+strmid(files,d1,d2)
  message, 'Dest path: '+dir_dest+mask_dest, /cont
  message, 'Dest '+dest+': '+varprt(n_dest)+' days', /cont
  cd, orig_dir

  ;now find all source files that do NOT have dest files:
  ;Those dates need processing/plotting!
  IF keyword_set(SAME) THEN lbl = '' ELSE lbl = 'no '
  IF keyword_set(FORCE) THEN BEGIN
      dates = dates_source
      message,  'Forcing all source dates', /cont
  ENDIF ELSE BEGIN
      msg = 'All '+source+' dates with '+lbl+dest+' files'
      dates = papco_find_dates_todo(dates_source, dates_dest, $
                                    SAME = SAME, MSG = msg)
  ENDELSE

  IF (n_elements(dates) EQ 1) AND (dates(0) EQ 0) THEN BEGIN
    batch_times = '' & return
  ENDIF

  ;Get a list of "problem" dates for selected destination files.
  ;Those are dates for which processing of the files was attempted
  ;but was not succesfull. Avoids trying to make them again.
  cd, dir_prob, current = orig_dir
  files = findfile('C'+SC_ID+'*', count=n_prob)
  dates_prob = '20'+strmid(files,3,6)
  message, 'Problem '+dest+': '+varprt(n_prob)+' days', /cont
  cd, orig_dir

  IF NOT keyword_set(PROB_FORCE) THEN BEGIN
     ;take out problem dates. These are dates for which the
     ;processing was run, but no output was produced.
     dates = papco_find_dates_todo(dates, dates_prob, $
             MSG = 'All dates with no problems')
     IF (n_elements(dates) EQ 1) AND (dates(0) EQ 0) THEN BEGIN
        batch_times = '' & return
     ENDIF
  ENDIF

;construct default batch filename
IF NOT keyword_set(FILENAME) THEN BEGIN
    filename='cluster_rapid_'+SC_ID+'_'+source+'_'
    IF keyword_set(RESTRICT) THEN $
      filename = filename+'limited_'+dest+'.papco_batch' ELSE $
      filename = filename+dest+'.papco_batch'
ENDIF

  ;restrict to dates FROM/BEFORE if that keyword is set
  IF keyword_set(FROM) THEN BEGIN
    dd = long(dates) & idx = where(dd GE long(from),c)
    message, 'Dates since '+varprt(from)+': '+ varprt(c)+' days', /cont
    IF c EQ 0 THEN BEGIN
      batch_times = '' & RETURN
    ENDIF
    dates = dates(idx)
  ENDIF
  IF keyword_set(BEFORE) THEN BEGIN
    dd = long(dates) & idx = where(dd LE long(before),c)
    message, 'Dates until '+varprt(before)+': '+ varprt(c)+' days', /cont
    IF c EQ 0 THEN BEGIN
      batch_times = '' & RETURN
    ENDIF
    dates = dates(idx)
  ENDIF

;now construct papco-formatted list of times
batch_times=strarr(n_elements(dates))
for i=0, n_elements(dates)-1 do begin
    result=convert_timeToSeconds(dates(i),/mjdt,/t90)
    mjdt_start=result.value
    mjdt_end=mjdt_start  &  mjdt_end.mjd=mjdt_end.mjd+1
    ;here we choose times of actual data only, by reading the files and
    ;determining actual start/end times. Used mainly for burst mode data.

    ;make a sample plotinfo structure for reading data files
    ;setup here for RAL_IF CLEANBM
    plotinfo = papco_getplotinfostruct()
    control = cluster_rapid_control
    control.sc_id = fix(SC_ID)-1
    control.product = 17
    control.data_src = 0
    plotinfo.USR_PTR1 = ptr_new(control)

    IF keyword_set(RESTRICT) THEN BEGIN
       input_data = 0 & input_header = 0
        r_cluster_rapid, plotinfo
        ;get start / end dates OF actual data!
        t1 = tai2utc(input_data(0).time)
        mjdt_start.mjd = t1.mjd & mjdt_start.t = t1.time/1000.0
        t2 = tai2utc(input_data(n_elements(input_data)-1).endtime)
        mjdt_end.mjd = t2.mjd & mjdt_end.t = t2.time/1000.0
        input_data = 0 & input_header = 0
    ENDIF
    t1=convert_secondsToTime(mjdt_start,/mjdt,/t90)
    t2=convert_secondsToTime(mjdt_end,/mjdt,/t90)
    batch_times(i) = t1+' - '+t2
endfor

if keyword_set(WRITE) then begin ;write times to batch file
    papco_write_batch, filename, batch_times
ENDIF

END

;******************************************************************************
;* Batch file to make CLUSTER RAPID CCR web page plots
;*
;* Type is type of ccr plot. Currently supported types:
;*     1, 2, 3, 4 - Summary for each cluster spacecraft
;*     Alle       - All spacecraft, electrons
;*     Allp       - All spacecraft, all ions
;*     AlleBM     - All Spacecraft, electrons Burst Mode
;*     AllePitch  - All Spacecraft, electrons Pitch angles
;*
;* Force overwrites exisiting .png files.
;******************************************************************************
PRO cluster_rapid_make_web_plots, TYPE = TYPE, FORCE = FORCE, CHECK = CHECK, $
                                  ALL = ALL

COMMON cluster_rapid

IF keyword_set(ALL) THEN BEGIN
    FOR i = 0, n_elements(ccr_plot_types)-1 DO $
      cluster_rapid_make_web_plots, TYPE = ccr_plot_types(i)
    spawn, '/u/friedel/bin/sync_ccr_summary_plots'
    return
ENDIF

IF NOT keyword_set(TYPE) THEN BEGIN
    message, 'You need to specify a ccr plot type', /cont
    message, 'Allowed types:', /cont
    print,  ccr_plot_types
    return
ENDIF

;check if type entered is a valid one
pos = strpos(ccr_plot_types, type)
index = where(pos NE -1, c)
IF c EQ 0 THEN BEGIN
    message, 'CCR plot type: '+type+' is not valid', /cont
    message, 'Allowed types:', /cont
    print,  ccr_plot_types
    return
ENDIF

path='/n/projects/cda/cluster/rapid/SummaryPlots/'
dir = 'ClusterRapid'+type & message, 'doing for '+dir, /cont

CD, path+dir+'/', CURRENT = orig_dir
fls=findfile('*.ps',count=c1)
if c1 ne 0 then BEGIN
      for i=0,c1-1 do BEGIN
        papco_sepdir, fls(i), p, f, e
        ;check if plots already exist
        if not keyword_set(FORCE) then begin
          plot_fls=findfile(f + '_*.png',count=c)
          if c eq 4 then begin
            print,f + '   ->  Plotfiles exist'
            goto,next
          endif
      endif
      print, f
        IF keyword_set(CHECK) THEN continue
        spawn,'/u/friedel/bin/m_cluster_rapid_summary_plots '+ f +' png'
        next:
        IF keyword_set(CHECK) THEN continue
        spawn, 'gzip -f '+ f +'*.ps &'
      endfor
endif else begin
      message,'No files found for CCR plot type: '+type, /cont
endelse

CD, orig_dir

end

;******************************************************************************
;make links FOR RAL processing. RAL axpects rdda dat to be in
;directory per CD, with dir name = CD name. At LANL they're all under
;the same directory structure. So, make links for CD names to point to
;the same dir for compatibility.

PRO m_cluster_rapid_ral_links

spawn, 'pwd',  curr_dir
path = '/n/projects/cda/cluster/rapid/rdda'

FOR i = 1, 4 DO BEGIN
    CD,  path+'/cluster'+varprt(i)+'/aux_'+varprt(i)
    result = findfile('*va.1*',count=c)
    IF c NE 0 THEN BEGIN
        dates = strmid(result, 0, 6)
        CD,  path+'/ral_links'
        FOR i = 0,  n_elements(dates)-1 DO BEGIN
            print, 'ln -s '+path+' '+dates(i)+'_1_1a'
            ;spawn, 'ln -s '+path+' '+dates(i)+'_1_1a'
        ENDFOR
    ENDIF
ENDFOR

CD, curr_dir(0)

END

;******************************************************************************
;make IF configuration file needed to produce RAL IF files. Have two
;types of ifcfg files - all of IES and all of IIMS

PRO m_cluster_rapid_ifcfg, data_type, ifconf_fln

COMMON cluster_rapid

use = {papco_ifcfg, E3DD:0, CLEAN:0, E3DDBM:0, CLEANBM:0, CLEANBM2:0, ESPCT:0,$
       PEDESTAL:0, EHIST:0, EHK:0, EPAD:0, I3DD:0, IPAD:0, ISPCT:0, HSPCT:0, $
       MTRX:0, SGLBM:0, SGL01:0, SGL2:0, SGL3:0, HK1:0, HK2:0, HK4:0, HK8:0, $
       HK32:0, SCINFO:0}

path_ifcfg = papco_getenv('RCS_IFCFG_DIRECTORY')
if_name = if_names(data_type)

ifcfg_names = tag_names(use)
FOR i = 0, n_elements(ifcfg_names)-1 DO BEGIN
    pos = strpos(if_name, ifcfg_names(i))
    IF pos NE -1 THEN break
ENDFOR
use.(i) = 1

pos = strpos(IF_name, 'IES')
IF pos NE -1 THEN BEGIN
    use.E3DD = 1 & use.clean = 1 & use.E3DDBM = 1 & use.CLEANBM = 1
    use.CLEANBM2 = 1 & use.ESPCT = 1 & use.PEDESTAL = 1 & use.EHIST = 1
    use.EHK = 1 & use.EPAD = 1
    if_name = 'ALL_IES'
    GOTO, write_ifcg
ENDIF

pos = strpos(IF_name, 'IIMS')
IF pos NE -1 THEN BEGIN
    use.I3DD = 1 & use.IPAD = 1 & use.ISPCT = 1 & use.HSPCT = 1
    if_name = 'ALL_IIMS'
    GOTO, write_ifcg
ENDIF

write_ifcg:
ifconf_fln = path_ifcfg+'papco_'+if_name+'.ifc'
save, file = ifconf_fln, use

END

;******************************************************************************
;* program to sync local rdda with downloaded rdda directory from
;* Lindau - gzip msf files in this process.
;******************************************************************************
PRO c_cluster_rapid_mpae_sync,  FORCE = FORCE

papco, /set_only

IF keyword_set(FORCE) THEN force =  1 ELSE force = 0

IF NOT (papco_check_data_env('CLUSTER_RAPID_RDDA', PATH=rdda_path)) $
       THEN return

mpae_sync_path = '/n/projects/cda/cluster/rapid/rdda_lindau_sync/'

dirs = ['msf_', 'aux_']
exts = ['*.gz', '*.*']

FOR sat = 1, 4 DO BEGIN       ;loop satellites
    FOR i = 0, 1 DO BEGIN     ;loop subdirectories
        subdir = 'cluster'+varprt(sat)+'/'+dirs(i)+varprt(sat)+'/'
        message, 'Doing for '+subdir, /cont
        srce = mpae_sync_path+subdir
        dest = rdda_path+subdir
        srce_files = findfile(srce, count = sc)
        dest_files = findfile(dest, count = dc)

        FOR j = 0, sc-1 DO BEGIN
            papco_sepdir, srce_files(j), dir, fln, ext
            result = strpos(dest_files, fln)
            idx = where(result NE -1, dc)
            IF (dc EQ 0) OR (FORCE EQ 1) THEN BEGIN
                IF i EQ 0 THEN BEGIN
                    f1 = srce+srce_files(j)
                    print, 'gzip -c '+ f1 +' >! '+dest+fln+ext+'.gz'
                    spawn, 'gzip -c '+ f1 +' >! '+dest+fln+ext+'.gz'
                ENDIF ELSE BEGIN
                    print, 'cp '+srce+srce_files(j)+' '+dest
                    spawn, 'cp '+srce+srce_files(j)+' '+dest
                ENDELSE
            ENDIF ELSE print, 'Have '+dest_files(idx(0))
        ENDFOR
    ENDFOR
ENDFOR

END

;******************************************************************************
;* program to copy Lindau latest up-to-date rdda files (downloaded at
;* Lindau from the Cluster Data Center daily) to a zip archive file on
;* the local dist directory/
;* note: these rdda fiels at Lindau may contain data gaps. The full
;* data files are produced from the CDROM's at Lindau once they are
;* received. Thus they may need to be reprocessed at a later stage to
;* get all of the data.
;******************************************************************************
PRO c_cluster_rapid_mpae_rdda_to_local_dist,  FORCE = FORCE

papco, /set_only

IF keyword_set(FORCE) THEN force =  1 ELSE force = 0

IF NOT (papco_check_data_env('CLUSTER_RAPID_DIST', PATH=dist_path)) $
       THEN return

mpae_path = '/n/projects/cda/cluster/rapid/rdda_lindau_sync/'

;get list of exisiting dist files. Only copy from rdda_lindau_sync
;dates not already on dist.

dist_files = file_search(dist_path+'*.zip',count=c1)
dist_dates = strmid(dist_files, strlen(dist_path), 6)

;get a list of dates on rdda_lindau_sync

mpae_files = file_search(mpae_path+'cluster1/msf_1/*rm*',count=c2)
mpae_dates = strmid(mpae_files, strlen(mpae_path+'cluster1/msf_1/'), 6)

FOR i = 0, c2-1 DO BEGIN
    idx = where(mpae_dates(i) EQ dist_dates, c)
    IF c EQ 0  THEN BEGIN
        message, 'Zipping rdda_lindau_sync date '+mpae_dates(i)+$
                 ' to local dist', /info
        ;get a list of files to put into daily zip file
        cd, mpae_path, current = orig_dir
        flns = 'cluster*/*/'+mpae_dates(i)+'*'
        zip_files = file_search(flns, count=c3)
        IF c3 LT 20 THEN BEGIN
            message, 'Not enouigh files', /info
            CONTINUE
        ENDIF 
        dist_zip = mpae_dates(i)+'.zip'
        file_list = ''
        FOR j = 0, c3-1 DO file_list = file_list + " '"+zip_files(j)+"'"
        cmd = 'zip -o '+dist_path+dist_zip+file_list
        print, cmd & spawn, cmd
        cd, orig_dir
    ENDIF
ENDFOR

END


;******************************************************************************
;* program to clean out rdda files (if exisit in zipped archive) and
;* to clean out any produced binary files
;******************************************************************************
PRO d_cluster_rapid, plotinfo

IF NOT (papco_check_data_env('CLUSTER_RAPID_DIST', PATH=dist_path)) then return
IF NOT (papco_check_data_env('CLUSTER_RAPID_RDDA', PATH=rdda_path)) then return
IF NOT (papco_check_data_env('CLUSTER_RAPID_BINARY',PATH=bin_path)) then return

date6 = strmid(plotinfo.filename,2,8)
;get the module's control structure
control = *plotinfo.USR_PTR1
SC_id=varprt(control.sc_id+1)

dist_zip = file_search(dist_path+date6+'*.zip', count = c)

IF c EQ 1 THEN BEGIN
    message, 'Deleting files for '+strmid(plotinfo.filename,2,8)+$
             ' from rdda and binary directories', /info
    flns = rdda_path+'cluster'+SC_id+'/*/'+date6+'*'
    rdda_files = file_search(flns, count = c2)
    FOR i = 0, c2-1 DO file_delete, rdda_files[i], /verb
    
    flns = bin_path+'c'+SC_id+'*'+date6+'*'
    bin_files = file_search(flns, count = c3)
    FOR i = 0, c3-1 DO file_delete, bin_files[i], /verb

ENDIF

END

;******************************************************************************
;* program to remove ccr plots for mpae_recent_dist days when ther
;* full dist files have been downloaded for that day.
;******************************************************************************
PRO d_clustyer_rapid_recent_ccr






END 

;******************************************************************************
;* program to get at mag field curvature from Steve Monk's code.
;******************************************************************************
PRO get_cluster_mag_curv, plotinfo

COMMON cluster_mag_curv, input_header, input_data
COMMON mjdt, mjdt_start, mjdt_end
COMMON get_error, get_err_no, get_err_msg
COMMON cluster_rapid

control = *plotinfo.USR_PTR1

;Steve's programs need time stings like 'yyyy/ddd-hh:mm:ss'
tai_1=utc2tai({mjd:mjdt_start.mjd, time:mjdt_start.t*1000})
tai_2=utc2tai({mjd:mjdt_end.mjd, time:mjdt_end.t*1000})

utc1 = tai2utc(tai_1, /EXTERNAL) & doy1 = utc2doy(utc1)
utc2 = tai2utc(tai_2, /EXTERNAL) & doy2 = utc2doy(utc2)

start_time_str = string(utc1.year, doy1, utc1.hour, utc1.minute, utc1.second, $
                        format = "(i4.4,'/',i3.3,'-',i2.2,':',i2.2,':',i2.2)")
stop_time_str  = string(utc2.year, doy2, utc2.hour, utc2.minute, utc2.second, $
                        format = "(i4.4,'/',i3.3,'-',i2.2,':',i2.2,':',i2.2)")

print, start_time_str
print, stop_time_str

;This is a TCAD time i.e 1 min = 60/86400. - fraction of day.
;smooth is in # of spins - 4 sec intervals
av_sec = control.smooth*4.0
averaging_interval = av_sec/86400.0d

;now get the 4SC mag field data needed
message, 'Getting 4SC mag field averaging every '+varprt(av_sec)+' sec', /info
!QUIET = 1
getInputsForCurvature, start_time_str, stop_time_str, T, $
                       gse_coords, B1, B2, B3, B4, success, $
                       averaging_interval=averaging_interval
!QUIET = 0

IF NOT success THEN BEGIN
    get_err_no = 6
    get_err_msg = 'Problem getting 4SC mag field data'
    return
ENDIF

;now do the curvature calculations
message, 'Calculating curvature', /info
;Outputs: TC - array of TCAD times corresponding to the C array.
;         C  - array of the structures below, same length as TC:
;c = {T        : 0.0d,      $ ;The time in fractional days since 1970
;     B        : fltarr(3), $ ;B at the mesocenter, calculated using k's.
;     B1       : fltarr(3), $ ;The inputs, for debugging.
;     B2       : fltarr(3), $
;     B3       : fltarr(3), $
;     B4       : fltarr(3), $
;     c        : fltarr(3), $ ;curvature vector (x,y,z); a unit vector
;     oplane_n : fltarr(3), $ ;osculating plane normal (x,y,z); a unit vector
;     radius   : 0.0,       $ ;curvature radius, 1/curvatureTot
;     theta_c  : 0.0,       $ ;curvature azimuthal angle, in degrees
;     phi_c    : 0.0,       $ ;curvature polar angle, in degrees
;     theta_n  : 0.0,       $ ;azimuthal angle of osculating plane normal, deg
;     phi_n    : 0.0,       $ ;polar angle of osculating plane normal, in deg
;     theta_b  : 0.0,       $ ;for debugging
;     phi_b    : 0.0}         ;for debugging

calculateCurvature, T, gse_coords, B1, B2, B3, B4, TC, C

;create an output structure for curvature data
dummy= create_struct(name='cluster_mfe_curvature','tai',0.0d,c(0)) 
input_data = replicate(dummy,n_elements(C)) 

input_header = {cluster_mfe_curvature_head, av_sec:av_sec}

;convert TC back to TAI. TC is in TCAD, fractional days since 1970
utc_1970 = anytim2utc('1970/01/01') & utc = utc_1970
FOR i = 0, n_elements(TC)-1 DO BEGIN
    utc.mjd = long(TC(i))+utc_1970.mjd
    utc.time = (TC(i)-long(TC(i))) *86400000.0d
    tai = utc2tai(utc)
    input_data(i).tai = tai
    FOR j = 1, 15 DO input_data(i).(j) = c(i).(j-1)
ENDFOR 

END 


;******************************************************************************
;* program to calculate gyro radius for rapid energies.
;******************************************************************************
PRO get_cluster_gyro_rad, plotinfo

COMMON cluster_gyro_rad, input_header, input_data
COMMON mjdt, mjdt_start, mjdt_end
COMMON get_error, get_err_no, get_err_msg
COMMON cluster_rapid
COMMON cluster_rapid_slice, xut1, xut2, time, zmat, data, yarr, $
                            utitle, title, uytitle, uztit, mag_fgm, aux, fgm, $
                            ndat, nodata, s1, s2, d1, d2, e1, e2, p1, p2

tai1=utc2tai({mjd:mjdt_start.mjd, time:mjdt_start.t*1000})

control = *plotinfo.USR_PTR1

;read rapid data - IES ESPCT to get tiem, energy channels
new_plotInfo = papco_getplotinfostruct()
new_control = cluster_rapid_control    
new_control.sc_id = control.sc_id
new_control.data_src = 0
new_control.product = 0
new_control.cal = 3
new_control.reload = 0
new_plotinfo.USR_PTR1 = ptr_new(new_control)
draw_cluster_rapid,[0, 0, 1], new_plotinfo, 0, get_data_call, $
                              slice_type, 0, OUTPUT = 2
time = ( (time(*, 0)+time(*, 1) ) /2.0 )

;read mag field data
get_cluster_mag, control, mag_fgm, time

Bmag = mag_fgm(*, 3)
energy = [yarr(0, 0), yarr(*, 1)] 
n_ech = n_elements(energy)
data = fltarr(ndat, n_ech)

FOR i = 0, n_ech-1 DO data(*, i) = papco_gyroradius(energy(i), Bmag, 90.0, 0)

;create an output structure for gyroradius data
dummy= create_struct(name='cluster_rapid_gyro','tai',0.0d,'data', fltarr(n_ech))
input_data = replicate(dummy,ndat) 

input_header = {cluster_rapid_gyro_head, label:''}

input_data.tai = time+tai1
input_data.data = transpose(data)

END 
