;+
; Project     :	CLUSTER - RAPID, POLAR - CEPPAD
;
; Name        :	IES_CAL
;
; Purpose     :	Calibrates data and reverses calibration.
;
; Explanation :	Applies count rate calibration depending on LUT mode, converting to flux and reverses calibration.
;
; Use         : < ies_cal, luts, count_cal, output_array >
;
; Inputs      : luts          : BYTARR LUT for each sample point.
;               count_cal     : FLTARR(nluts) calibration factor for each LUT.
;               output_array  : FLTARR(nsectors, ndetectors, nbands, npoints) containing input data
;
; Opt. Inputs : None.
;
; Outputs     : output_array  : FLTARR(nsectors, ndetectors, nbands, npoints) containing processed data.
;
; Opt. Outputs:	None.
;
; Keywords    : REVERSE : flag indicating conversion back to counts required.
;
; Calls       :	None.
;
; Common      : None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 26/9/96
;                 Adapted from ies_calibrate.
;
; Modified    :	Version 0.1, MKC, 4/10/96
;                 Added RAT keyword.
;               Version 0.2, 13/11/96
;                 Separate ies_div and ies_rat.
;             	Version 0.3, 15/4/97
;                 Allowed reverse change.
;             	Version 0.4, 29/4/97
;                 Corrected IPS/HIST wrong calibration factors.
;             	Version 0.5, 18/8/97
;                 Removed fudge for IPS/HIST wrong calibration factors.
;               Version 0.6, 19/8/00
;                 Explicitly allowed for LUT=-1
;               Version 0.7, 20/07/01
;                 Used ies_runs.
;
; Version     :	Version 0.7, 20/07/01
;-
;**********************************************************

PRO ies_cal, luts, count_cal, output_array, REVERSE=reverse

  ; apply count calibration i.e convert from counts to counts/sr/cm2

  ; get list of ends of runs of luts

  lut_list = ies_runs(luts)

  ; loop through runs of each lut

  pos = 0

  previous_lut = 0

  FOR l = 0L, N_ELEMENTS(lut_list)-1 DO BEGIN

    ; get lut and no. of points in run

    lut = luts[lut_list[l]]

    ; check lut

    IF lut GT N_ELEMENTS(count_cal) THEN BEGIN
      PRINT, 'WARNING, invalid lut = ', lut
      lut=0
    ENDIF

    ; if LUT unknown then use previous

    IF lut EQ -1 THEN lut = previous_lut

    previous_lut = lut

    size = lut_list(l) - pos + 1

    ; apply calibration factor

    IF KEYWORD_SET(reverse) THEN BEGIN

      ; reverse calibration for each energy and each detector for this LUT

      output_array(*,*,*,pos:pos+size-1) = output_array(*,*,*,pos:pos+size-1) / count_cal(lut)

    ENDIF ELSE BEGIN

      ; do calibration for each energy and each detector for this LUT

      output_array(*,*,*,pos:pos+size-1) = output_array(*,*,*,pos:pos+size-1) * count_cal(lut)

    ENDELSE

    pos = lut_list(l)+1

  ENDFOR

END
