;+
; Project     :	CLUSTER - RAPID, POLAR - CEPPAD
;
; Name        :	IES_EFFICIENCY
;
; Purpose     :	Applies energy efficiency calibration.
;
; Explanation :	Applies energy efficiency calibration depending on LUT mode and reverses calibration.
;
; Use         : < ies_efficiency, luts, energy_efficiency, output_array >
;
; Inputs      : luts          : BYTARR LUT for each sample point.
;               energy_efficiency : FLTARR(nbands, nluts) efficiency factor for each energy in each LUT.
;               output_array  : FLTARR(nsectors, ndetectors, nbands, npoints) containing input data
;
; Opt. Inputs : None.
;
; Outputs     : output_array  : FLTARR(nsectors, ndetectors, nbands, npoints) containing processed data.
;
; Opt. Outputs:	None.
;
; Keywords    : REVERSE : flag indicating conversion back to counts required.
;
; Calls       :	None.
;
; Common      : None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 18/8/97
;                 Adapted from ies_cal.
;
; Modified    :	Version 0.1, 19/8/00
;                 Explicitly allowed for LUT=-1
;              	Version 0.2, 10/09/01
;                 Extended loop variables to LONG.
;
; Version     :	Version 0.2, 10/09/01
;-
;**********************************************************

PRO ies_efficiency, luts, energy_efficiency, output_array, REVERSE=reverse

  npoints = N_ELEMENTS(luts)

  ; apply count calibration i.e convert from counts to counts/sr/cm2

  ; loop through samples

  previous_lut = 0

  FOR k = 0L, npoints-1 DO FOR s = 0L, N_ELEMENTS(output_array(*,0,0,0))-1 DO FOR d = 0L, N_ELEMENTS(output_array(0,*,0,0))-1 DO BEGIN

    ; get lut and no. of points in run

    lut = luts(k)

    ; check lut

    IF lut GT N_ELEMENTS(energy_efficiency(0,*)) THEN BEGIN
      PRINT, 'WARNING, invalid lut = ', lut
      lut=0
    ENDIF

    ; if LUT unknown then use previous

    IF lut EQ -1 THEN lut = previous_lut

    previous_lut = lut

    ; apply energy efficiency

    IF KEYWORD_SET(reverse) THEN BEGIN

      ; reverse calibration for each energy and each detector for this LUT

      output_array(s,d,*,k) = output_array(s,d,*,k) * energy_efficiency(*,lut)

    ENDIF ELSE BEGIN

      ; do calibration for each energy and each detector for this LUT

      output_array(s,d,*,k) = output_array(s,d,*,k)  / energy_efficiency(*,lut)

    ENDELSE

  ENDFOR

END
