;+
; Project     :	CLUSTER - RAPID, POLAR - CEPPAD
;
; Name        :	IES_RAT
;
; Purpose     :	Converts data from counts to counts per sec and the reverse.
;
; Explanation :	Converts counts to count rate and the reverse.
;               Note that the output_array is not actually count rate but count rate / nsectors.
;
; Use         : < ies_rat, times, nsectors, nbands, ndetectors, output_array, REVERSE=reverse  >
;
; Inputs      : times         : DBLARR TAI times for each sample.
;               nsectors      : INT no. of sectors.
;               nbands        : INT no. of energy channels.
;               ndetectors    : INT no. of detectors.
;               output_array  : FLTARR(nsectors, ndetectors, nbands, npoints) containing input data
;
; Opt. Inputs : None.
;
; Outputs     : output_array  : FLTARR(nsectors, ndetectors, nbands, npoints) containing processed data.
;
; Opt. Outputs:	None.
;
; Keywords    : REVERSE : flag indicating conversion back to counts required.
;               ENDTIMES : DBLARR TAI edntimes of samples.
;
; Calls       :	None.
;
; Common      : None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 13/11/96
;                 Adapted from ies_cal.
;
; Modified    :	Version 0.1, 15/4/97
;                 Allowed reverse change.
;               Version 0.2, 8/5/01
;                 Added ENDTIMES keyword.
;
; Version     :	Version 0.2, 8/5/01
;-
;**********************************************************

PRO ies_rat, times, nsectors, nbands, ndetectors, output_array, REVERSE=reverse, ENDTIMES=endtimes

  npoints = N_ELEMENTS(times)

  ; convert from counts to counts per sec

  ; get time difference array
  ; NB last point is dummy point if full range
  ;    will always be at least two points unless only one point in l1 file

  IF KEYWORD_SET(endtimes) THEN BEGIN

    time_diffs = endtimes-times

  ENDIF ELSE IF npoints GT 1 THEN BEGIN

    time_diffs = - times

    time_diffs[0:npoints-2] = time_diffs[0:npoints-2] + times[1:*]

    time_diffs[npoints-1] = time_diffs[npoints-2]

  ENDIF ELSE BEGIN

    ; only one point guess answer as for survey plots

    PRINT, 'WARNING, guessing sample duration, only one sample'

    time_diffs = [96.0]

  ENDELSE

  ; fix any duff time differences

  list = WHERE ( time_diffs LE 0.001, count )

  IF count GT 0 THEN BEGIN

    PRINT, 'WARNING, bad times encountered'

    mean_diff = TOTAL(time_diffs)/N_ELEMENTS(time_diffs)

    time_diffs(list) = mean_diff

  ENDIF

  IF KEYWORD_SET(reverse) THEN BEGIN

    ; do time difference multiplication for each energy and each output
    ; remember rates are stored as count rate/nsectors, counts are stored as counts

    FOR s = 0L, nsectors-1 DO FOR d = 0L, ndetectors-1 DO FOR e = 0L, nbands-1 DO $
        output_array(s,d,e,*) = output_array(s,d,e,*) * time_diffs

  ENDIF ELSE BEGIN

    ; do time difference division for each energy and each output
    ; remember rates are stored as count rate/nsectors, counts are stored as counts

    FOR s = 0L, nsectors-1 DO FOR d = 0L, ndetectors-1 DO FOR e = 0L, nbands-1 DO $
        output_array(s,d,e,*) = output_array(s,d,e,*) / time_diffs

  ENDELSE

END
