;*********************************************************************************************
;File:    average_data.pro
;
;Purpose: Average time-series data on even time centers.
;Inputs:  T,D       : One-dimensional arrays.
;         binsize   : The width of each averaging bin, same units as T.
;         start_val : The T value of the start of the first bin.
;         /dont_delete_zero_values : Optional keyword prevents deletion of 0 values
;
;Outputs: T,D       : T[0] = start_val + binsize/2.0
;                     T[1] = T[0] + binsize
;                     Etc.
;                     D[0] = Average of D over range start_val..(start_val + binsize)
;*********************************************************************************************

pro average_data, T, D, binsize, start_val, dont_delete_zero_values=dont_delete_zero_values

H = histogram( T, binsize=binsize, reverse_indices=R, min=start_val)
N_BINS = n_elements(H)
TT = lindgen(N_BINS)*binsize + start_val + binsize/2.0

;R is a rather unusual array, in that the first N_BINS*2 elements are start/stop
;indices into the rest of the elements.  The rest of the elements in this case
;are the same as lingen( n_elements(TT)).

DD = fltarr( N_BINS)
for I=0L,N_BINS-1 do begin
  N_PTS = R[I+1] - R[I]
  if N_PTS gt 0 then begin
    indices = R[ R[I]:R[I+1]-1]   ;These are the indices into TT of all the points that
                                  ;went into bin I.
    DD[I] = total( D[ indices]) / N_PTS
  endif
endfor

if not keyword_set( dont_delete_zero_values) then begin
  ;Remove where data is 0.
  W = where( DD ne 0.0)
  T = TT[W]
  D = DD[W]
endif else begin
  T = TT
  D = DD
endelse

return
END


