;*******************************************************************************
; Purpose:   Calculate magnetic field line curvature, display in RAVE.
;
; Routines:  getInputsForCurvature - Get orbit and B field data.
;            calculateCurvature    - Calculate time-series of curvature, etc.
;            getReciprocalVectors  - Helper routine called by calculateCurvature.
;            raveCurvature         - RAVE's interface to all this stuff.
;
; Author:  LASP/Steve Monk
;
; History: 6/1/03 SM - Initial version.
;*******************************************************************************



;*******************************************************************************
;Routine: getInputsForCurvature
;Purpose: Get all 4 Cluster's S/C position and FGM B field data on same time centers.
;Inputs:  start_time_str and stop_time_str are standard TCAD time strings: 'yyyy/ddd-hh:mm:ss'
;         averaging_interval is a TCAD time, e.g. 60/86400. = 1 minute
;Outputs: success is 1 on success, 0 otherwise
;         T, gse_coords, B1, B2, B3, B4 are equal length arrays.
;         T is a dblarr(n) of TCAD times: fractional days since 1970
;         B1, B2, B3, B4 are fltarr(3,n): B averaged/interpolated on times T.
;         gse_coords is an array of: {pos: fltarr(3), dpos: fltarr(3,4)} where
;         pos is the position of the mesocenter in GSE coordinates, and dpos are
;         the x,y,z offsets of each spacecraft from the mesocenter.
;Method:  1) Read S/C position and FGM B field data from CDF's using PAPCO routines.
;         2) Change the S/C reference position and offsets: Cluster 3 => mesocenter.
;         3) Interpolate S/C positions and average B for each S/C on 1 minute
;            time centers (a changable parameter: "averaging_interval").
;*******************************************************************************

pro getInputsForCurvature, start_time_str, stop_time_str, T, gse_coords, B1, B2, B3, B4, success, $
                           averaging_interval=averaging_interval

getOrbitData, start_time_str, stop_time_str, T_orb, gse_coords, success, /TCAD
if not success then return

;Set pos_gse to be the average of the 4 S/C's positions (the mesocenter), instead of the
;position of the reference S/C (Cluster 3).  Adjust dpos accordingly.
mesocenter_gse_coords = gse_coords
for i=0,2 do begin
  pos_c1 = gse_coords[*].pos_gse[i] + gse_coords[*].dpos[i,0]
  pos_c2 = gse_coords[*].pos_gse[i] + gse_coords[*].dpos[i,1]
  pos_c3 = gse_coords[*].pos_gse[i] + gse_coords[*].dpos[i,2]
  pos_c4 = gse_coords[*].pos_gse[i] + gse_coords[*].dpos[i,3]
  mesocenter_gse_coords[*].pos_gse[i] = (pos_c1 + pos_c2 + pos_c3 + pos_c4)/4
  mesocenter_gse_coords[*].dpos[i,0] = pos_c1 - mesocenter_gse_coords[*].pos_gse[i]
  mesocenter_gse_coords[*].dpos[i,1] = pos_c2 - mesocenter_gse_coords[*].pos_gse[i]
  mesocenter_gse_coords[*].dpos[i,2] = pos_c3 - mesocenter_gse_coords[*].pos_gse[i]
  mesocenter_gse_coords[*].dpos[i,3] = pos_c4 - mesocenter_gse_coords[*].pos_gse[i]
endfor

if not keyword_set( averaging_interval) then $
  averaging_interval = 60/86400.  ;1 minute
n = long( (T_orb[ n_elements(T_orb)-1] - T_orb[0]) / averaging_interval)
T = dindgen( n) * averaging_interval + T_orb[0]

;Linearly interpolate gse_coords on T (small source of error since orbit isn't linear)
gse_coords = replicate( gse_coords[0], n)
for i=0,2 do begin
  gse_coords[*].pos_gse[i] = interpol( mesocenter_gse_coords[*].pos_gse[i], T_orb, T)
  for j=0,3 do begin
    gse_coords[*].dpos[i,j] = interpol( mesocenter_gse_coords[*].dpos[i,j], T_orb, T)
  endfor
endfor

;Average each S/C's B on averaging_interval centers; first average the B unit vectors' x,y,z components
;separately, then multiply by the average of the magnitudes.  (It is incorrect to average the x,y,z
;components of non-unit vectors.)  Create T1,B1, T2,B2, T3,B3, T4,B4.
for SC=0,3 do begin
  USE_MODEL_B = 0  ;If 0, use Cluster FGM data.  If 1 use UNILIB field model.
  if USE_MODEL_B then begin
    gse_coords_SC = gse_coords.pos_gse + gse_coords.dpos[*,SC]
    ;Execute a string like:  "T1=T & getModelB, T1, gse_coords_SC, B1, interp=averaging_interval"
    str = 'T' + strtrim(SC+1,2) + '=T & ' + $
          'getModelB, T' + strtrim(SC+1,2) + ', gse_coords_SC, B' + strtrim(SC+1,2) + ', interp=averaging_interval'
    print,'Executing: ' + str
    z = execute( str)
  endif else begin
    read_rapid_B_papco, start_time_str, stop_time_str, strtrim(SC,2), T_SC_original, B_SC, success
    if not success then return
    Btot = sqrt( B_SC[*].(0)^2 + B_SC[*].(1)^2 + B_SC[*].(2)^2) > 1e-6
    ;Average the unit vectors.
    for i=0,2 do begin
      D = B_SC[*].(i) / Btot  ;E.g. D = B_SC[*].Bx / Btot
      T_SC = T_SC_original
      average_data, T_SC, D, averaging_interval, T[0] - averaging_interval/2, /dont_delete_zero_values
      if i eq 0 then B = fltarr(3,n_elements(D))
      B[i,*] = D
    endfor
    ;Average the magnitudes.
    T_SC = T_SC_original
    average_data, T_SC, Btot, averaging_interval, T[0] - averaging_interval/2, /dont_delete_zero_values
    ;Interpolate zero values (gaps).
    W = where( B[0,*] eq 0 and B[1,*] eq 0 and B[2,*] eq 0, count, complement=WW)
    if count gt 0 then begin
      for i=0,2 do begin
        B[i,W] = interpol( B[i,WW], T_SC[WW], T_SC[W])
      endfor
      Btot[W] = interpol( Btot[WW], T_SC[WW], T_SC[W])
    endif
    ;Multiply unit vectors by magnitudes.
    for i=0,2 do B[i,*] = B[i,*] * Btot
    z = execute( 'T' + strtrim(SC+1,2) + '=T_SC  &  B' + strtrim(SC+1,2) + '=B')
  endelse
endfor

;The averaged B arrays from each S/C might have different lengths with different start/stop times,
;and different from gse_coords, but at least all have the same sampling interval with no gaps.
;This meansthat once we find the intersection of their time-tag arrays, they should have the same 
;number of elements.

start_time = max( [T[0], T1[0], T2[0], T3[0], T4[0]]) - 1/86400.
stop_time  = min( [T[ n_elements(T)-1], T1[ n_elements(T1)-1], T2[ n_elements(T2)-1], $
                   T3[ n_elements(T3)-1], T4[ n_elements(T4)-1]]) + 1/86400.
W = where( T gt start_time and T lt stop_time, co)
if co eq 0 then throw,"Orbit data and/or B data don't overlap!"
T = T[W] & gse_coords = gse_coords[W]

W = where( T1 gt start_time and T1 lt stop_time, co)
if co eq 0 then throw,"C1 B data doesn't overlap orbit data and/or other B data!"
B1 = B1[*,W]

W = where( T2 gt start_time and T2 lt stop_time, co)
if co eq 0 then throw,"C2 B data doesn't overlap orbit data and/or other B data!"
B2 = B2[*,W]

W = where( T3 gt start_time and T3 lt stop_time, co)
if co eq 0 then throw,"C3 B data doesn't overlap orbit data and/or other B data!"
B3 = B3[*,W]

W = where( T4 gt start_time and T4 lt stop_time, co)
if co eq 0 then throw,"C4 B data doesn't overlap orbit data and/or other B data!"
B4 = B4[*,W]

if n_elements(T) ne n_elements(B1)/3 or n_elements(B1) ne n_elements(B2) or $
  n_elements(B2) ne n_elements(B3) or n_elements(B3) ne n_elements(B4) then $
  throw,'Error averaging B!'

END


;*******************************************************************************
;Routine: calculateCurvature
;Purpose: Calculate curvature from 4-S/C B field measurements using linear estimator
;         technique for finding the B field gradient from Ref. 1, and formula for
;         curvature from Ref. 2.
;         
;Inputs:  T          - array of n double's (TCAD times, i.e. fractional days since 1970)
;         gse_coords - array of n structures (corresponding to T)
;                      gse_coords[i].pos_gse = fltarr(3)  (x,y,z) mesocenter position
;                      gse_coords[i].dpos = fltarr(3,4)   (x,y,z, c1,c2,c3,c4) offset
;                                                          from mesocenter position
;         B1, B2, B3, B4 - each is fltarr(3,n)  (Bx,By,Bz, n)
;         Notes: - gse_coords and the B's must be on the same time centers, T.
;
;Outputs: TC - array of TCAD times corresponding to the C array.
;         C  - array of the structures below, same length as TC:
;Notes:
;1) References:
;   1. Book: "Analysis Methods for Multi-Spacecraft Data", section 14.2.1
;   2. Paper: "Analyses on the Geometrical Structure of Magnetic Field in the Current
;              Sheet Based on Cluster Measurements", Appendix A.
;*******************************************************************************

pro calculateCurvature, T, gse_coords, B1, B2, B3, B4, TC, C

if n_elements(T) ne n_elements( gse_coords) or $
   n_elements(T) ne n_elements( B1)/3 then begin
  throw,'calculateCurvature: Invalid inputs!'
endif
n = n_elements(T)
c = {T            : 0.0d,      $ ;The time in fractional days since 1970
     B            : fltarr(3), $ ;B at the mesocenter, calculated using k's.
     B1           : fltarr(3), $ ;The inputs, for debugging.
     B2           : fltarr(3), $
     B3           : fltarr(3), $
     B4           : fltarr(3), $
     c            : fltarr(3), $ ;curvature vector (x,y,z); a unit vector
     oplane_n     : fltarr(3), $ ;osculating plane normal (x,y,z); a unit vector
     radius       : 0.0,       $ ;curvature radius, 1/curvatureTot
     theta_c      : 0.0,       $ ;curvature azimuthal angle, in degrees
     phi_c        : 0.0,       $ ;curvature polar angle, in degrees
     theta_n      : 0.0,       $ ;azimuthal angle of osculating plane normal, in degrees
     phi_n        : 0.0,       $ ;polar angle of osculating plane normal, in degrees
     theta_b      : 0.0,       $ ;for debugging
     phi_b        : 0.0}         ;for debugging

c = replicate( c, n)
c.B1 = B1
c.B2 = B2
c.B3 = B3
c.B4 = B4

;Since this is a linear estimator, the output time-tags are the same as the input ones.
TC = T

for i=0,n-1 do begin
  
  getReciprocalVectors, gse_coords[i].dpos, k  
  ;Get B at the mesocenter position.  Plotting this vs. individual S/C B's validates the k's.
  mu_alpha = fltarr(4)  ;Barycentric coordinates of mesocenter.
  for j=0,3 do begin
    mu_alpha[j] = 1 + total( k[*,j] * (-1) * gse_coords[i].dpos[*,j])  ;Ref 1, Equation 14.5
  endfor
  ;Note: in the case that B is fltarr(3) instead of fltarr(3,1) or fltarr(3,n), it's OK to specify
  ;non-existent dimensions, as long as their indices are 0.  E.g. a=1 & print,a[0,0] is OK.
  B = B1[*,i] * mu_alpha[0] + $  ;Ref 1, Equation 14.4
      B2[*,i] * mu_alpha[1] + $
      B3[*,i] * mu_alpha[2] + $
      B4[*,i] * mu_alpha[3]
  Btot = sqrt( total(B^2))
  Bunit = B / Btot  ;Convert B to unit vector.
  
  ;Calculate the elements of the gradient tensor (Ref 1, Equation 14.15), but use unit vectors
  ;as in Ref 2, Equation A1.  Variable naming convention is e.g. dBxdx = dBx/dx
  B1[*,i] = B1[*,i] / sqrt( total( B1[*,i]^2))
  B2[*,i] = B2[*,i] / sqrt( total( B2[*,i]^2))
  B3[*,i] = B3[*,i] / sqrt( total( B3[*,i]^2))
  B4[*,i] = B4[*,i] / sqrt( total( B4[*,i]^2))
 
  dBxdx = total( k[0,*] * [B1[0,i], B2[0,i], B3[0,i], B4[0,i]])
  dBydx = total( k[0,*] * [B1[1,i], B2[1,i], B3[1,i], B4[1,i]])
  dBzdx = total( k[0,*] * [B1[2,i], B2[2,i], B3[2,i], B4[2,i]])
  
  dBxdy = total( k[1,*] * [B1[0,i], B2[0,i], B3[0,i], B4[0,i]])
  dBydy = total( k[1,*] * [B1[1,i], B2[1,i], B3[1,i], B4[1,i]])
  dBzdy = total( k[1,*] * [B1[2,i], B2[2,i], B3[2,i], B4[2,i]])
  
  dBxdz = total( k[2,*] * [B1[0,i], B2[0,i], B3[0,i], B4[0,i]])
  dBydz = total( k[2,*] * [B1[1,i], B2[1,i], B3[1,i], B4[1,i]])
  dBzdz = total( k[2,*] * [B1[2,i], B2[2,i], B3[2,i], B4[2,i]])
  
  ;Calculate the curvature vector.
  ;curvature = (B dot Del)B  where B is the B unit vector at the mesocenter
  ;          = Bx dB/dx + By dB/dy + Bz dB/dz
  ;          = Bx (dBxdx i + dBydx j + dBzdx k) + $
  ;            By (dBxdy i + dBydy j + dBzdy k) + $
  ;            Bz (dBxdz i + dBydz j + dBzdz k)
  curvature = [Bunit[0]*dBxdx + Bunit[1]*dBxdy + Bunit[2]*dBxdz, $
               Bunit[0]*dBydx + Bunit[1]*dBydy + Bunit[2]*dBydz, $
               Bunit[0]*dBzdx + Bunit[1]*dBzdy + Bunit[2]*dBzdz]
  
  ;Resolve curvature into parallel and perpendicular components w.r.t. B, then take the perpendicular one.
  curvatureTot = sqrt( total( curvature^2))
  curvatureUnit = curvature / curvatureTot
  curvature_parallel = curvatureTot * Bunit * total( Bunit * curvatureUnit)
  curvature = curvature - curvature_parallel    ;curvature is now perpendicular component of curvature
  
  curvatureTot = sqrt( total( curvature^2))
  curvatureUnit = curvature / curvatureTot
  osculating_plane_normal = crossp( Bunit, curvatureUnit)
  
  ;Convert GSE to GSM.  This messes things up in ways it shouldn't, e.g. changes angle between B
  ;and curvature, so gave up for now.
  ;split_time, T[i], yr, doy, hh, mm, ss
  ;arr = [B, curvatureUnit, osculating_plane_normal]
  ;gsmgse, yr, doy, hh, mm, ss, arr, -1
  ;B                       = arr[0:2]
  ;curvature               = arr[3:5]
  ;osculating_plane_normal = arr[6:8]
  
  c[i].T        = T[i]
  c[i].B        = B
  c[i].c        = curvatureUnit
  c[i].oplane_n = osculating_plane_normal
  c[i].radius   = 1/curvatureTot/6378.  ;Re
  ;Standard cartesian to spherical coordinate conversions:
  tmp = cv_coord( from_rect=curvature, /to_sphere)
  c[i].theta_c = 90.0 - tmp[1]*!RADEG           ;Convert -90..90 to 0..180 (90 => 0, -90 => 180)
  c[i].phi_c   = (tmp[0]*!RADEG + 360) mod 360  ;Convert -180..180 to 0..360 (-180 => 180, 0 => 360)
  tmp = cv_coord( from_rect=osculating_plane_normal, /to_sphere)
  c[i].theta_n = 90.0 - tmp[1]*!RADEG           ;Ditto above. 
  c[i].phi_n   = (tmp[0]*!RADEG + 360) mod 360  ;Ditto above.
  tmp = cv_coord( from_rect=B/Btot, /to_sphere)
  c[i].theta_b = 90.0 - tmp[1]*!RADEG           ;Ditto above. 
  c[i].phi_b   = (tmp[0]*!RADEG + 360) mod 360  ;Ditto above.
  
endfor
return
END


;*******************************************************************************
;Routine: getReciprocalVectors
;Purpose: Calculate the reciprocal vectors k1, k2, k3, k4 given in Equation 14.7
;         in "Analysis Methods for Multi-Spacecraft Data".
;Inputs:  pos = fltarr(3,4)   (x,y,z, c1,c2,c3,c4) S/C position (or offset from
;                             mesocenter, doesn't matter)
;Outputs: k = fltarr(3,4)  (x,y,z,  c1,c2,c3,c4) - the reciprocal vectors
;                          i.e. k1 = k[*,0]
;Notes:  Refer to Figure 14.1.  For k4:
;        r12 = base_segment1, r13 = base_segment2, r14 = side_segment,
;        r12 x r13 = base_normal
;*******************************************************************************

pro getReciprocalVectors, pos, k

k = fltarr(3,4)
for SC=0,3 do begin
  cpoi = shift( indgen(4), -SC)  ;"Cyclic Permutation of Indices" mentioned below Equation 14.7.
  base_segment1 = pos[*,cpoi[1]] - pos[*,cpoi[0]]
  base_segment2 = pos[*,cpoi[2]] - pos[*,cpoi[0]]
  side_segment  = pos[*,cpoi[3]] - pos[*,cpoi[0]]
  base_normal = crossp( base_segment1, base_segment2)
  k[*,cpoi[3]] = base_normal / total( side_segment * base_normal)
endfor
return
END


;*******************************************************************************
;Routine: raveCurvature
;Purpose: Display curvature related vectors and semi-circles in RAVE 3D display.
;Inputs:  state.magnetic_field_curvature     ;Show it or not.
;         state.event_time                   ;Time to show it at.
;         state.minTimeStr, state.maxTimeStr ;Time range of loaded data.
;         old_state                          ;Optimize what stuff needs doing.
;Method:  
; 1) On start-up, or when state.min/maxTimeStr changes, get inputs for curvature, and
;    calculate curvature time-series over entire time range.  
;    Save curvature time-series internally between calls.
; 2) Find the closest data to state.event_time, and display B, curvature and osculating plane
;    vectors at the mesocenter.  Display a semi-circle with the curvature radius which passes
;    through the mesocenter.  Leave old vectors and semi-circles(s) visible.
;    An interpolation method is also possible;  currently commented out.
; 3) When state.magnetic_field_curvature is 0, erase all semi-circles and vectors.
; 4) The routine creates and maintains this graphics sub-tree under the top-level model:
;
;    model---Curvature---Transform1 (uvalue = cv structure)
;                    |            |---B---arrowShaft  ;arrowShaft and arrowHead are both named
;                    |            |   |---arrowHead   ;"mesocenterB" to facilitate picking
;                    |            |
;                    |            |---C---arrowShaft  ;arrowShaft and arrowHead are both named
;                    |            |   |---arrowHead   ;"CurvatureVector" to facilitate picking
;                    |            |
;                    |            |---N---arrowShaft  ;arrowShaft and arrowHead are both named
;                    |            |   |---arrowHead   ;"OsculatingPlaneNormal" to facilitate picking
;                    |            |
;                    |            |---CurvatureArcRotation---CurvatureArc
;                    |
;                    |---Transform2---repeat above
;                    |---Repeat for all the curvature line/vector sets, i.e. all the event times
;                        since the "Show/Magnetic Field Curvature" checkbox was turned on.
;
;
;
;
;   Each transform object under the curvature object positions the origin of the vectors
;   and semi-circles at the mesocenter, and rotates them appropriately.
;*******************************************************************************

pro raveCurvature, state

common rave_common, pos, window, view, model, trackball, old_state, $
                    xs, ys, zs, xrange, yrange, zrange
common rave_curvature_common, T, D, gse_coords, oldMinTimeStr, oldMaxTimeStr

curvatureObj = model->GetByName( 'Curvature')
if not obj_valid( curvatureObj) then begin
  curvatureObj = obj_new( 'IDLgrModel', name='Curvature')
  model->add, curvatureObj
endif
;Bug Note: If set color attributes of arrow shaft and arrow head to 3-element RGB vector, 
;          then .png file shows green in place of lightBlue and lightRed, so using indexed color.
;          This could be an IDL bug worth tracking down.
window->GetProperty, palette=palette
lightBlue = palette->NearestColor( 120,120,255)
lightRed  = palette->NearestColor( 255,120,120)
foreground_color = palette->NearestColor( state.foreground_color[0], state.foreground_color[1], state.foreground_color[2])


if state.show.MagneticFieldCurvature eq 0 then begin
  ;Delete everything under the 'Curvature' object.
  transformArr = curvatureObj->Get( /ALL, count=count)
  for i=0,count-1 do begin
    curvatureObj->Remove, transformArr[i]
    for j=0,2 do begin
      vectorObj = transformArr[i]->Get()
      obj_destroy, vectorObj->Get()      ;Destroy arrowShaft object.
      obj_destroy, vectorObj->Get()      ;Destroy arrowHead object.
      obj_destroy, vectorObj             ;Destroy vector object.
    endfor
    obj_destroy, transformArr[i]->Get()  ;Destroy semicircle object.
    obj_destroy, transformArr[i]         ;Destroy Transform object.
  endfor
  return
endif

read_data = 0
if n_elements(T) eq 0 then begin
  read_data = 1
endif else begin
  if str_to_t( state.minTimeStr) lt str_to_t( oldMinTimeStr) or $
    str_to_t( state.maxTimeStr) gt str_to_t( oldMaxTimeStr) then begin
    read_data = 1
  endif
  if abs(T[1] - T[0] - state.averaging_interval/86400.) ge 0.1/86400. then begin
    print,'raveCurvature: state.averaging_interval has changed.'
    read_data = 1
  endif
endelse

if read_data then begin
  averaging_interval_in_minutes = strtrim( state.averaging_interval/60.,2)
  strip_trailing_zeros, averaging_interval_in_minutes
  print,'raveCurvature: Reading data and averaging every ' + strtrim( state.averaging_interval,2) + $
        ' seconds (' + averaging_interval_in_minutes + ' minutes)'
  getInputsForCurvature, state.minTimeStr, state.maxTimeStr, TT, gse_coords, B1, B2, B3, B4, success, $
                         averaging_interval=state.averaging_interval/86400.
  if not success then return
  oldMinTimeStr = state.minTimeStr
  oldMaxTimeStr = state.maxTimeStr
  calculateCurvature, TT, gse_coords, B1, B2, B3, B4, T, D  
endif

;Find the closest curvature data to state.event_time instead of interpolating.
;Ideally, we should re-do the averaging, centered on event_time, if it isn't already.
averaging_interval_T = state.averaging_interval/86400.
W = where( T ge (state.event_time - averaging_interval_T) and T lt (state.event_time + averaging_interval_T), co)
if co ne 1 then begin
  if co eq 0 then begin
    print,'raveCurvature: Error! No data found within state.event_time +/- averaging_interval!'
    return
  endif else begin
    ;print,'raveCurvature: Warning! Found ' + strtrim(co,2) + ' points within state.event_time +/- averaging_interval!'
    ;print,'raveCurvature: curvature is at: ' + t_to_str( T[W[0]])
  endelse
endif

T0          = T[W[0]]
cv          = D[W[0]]
mesocenter  = gse_coords[W[0]].pos_gse / 6378.0

if 0 then begin
  for i=0,2 do begin  ;interpolate x,y,z components independently
    cv.c[i]        = interpol( D[*].c[i],                T, state.event_time)
    cv.B[i]        = interpol( D[*].B[i],                T, state.event_time)
    cv.oplane_n[i] = interpol( D[*].oplane_n[i],         T, state.event_time)
    mesocenter[i] = interpol( gse_coords[*].pos_gse[i], T, state.event_time)
  endfor
endif

cv.T = T0
Bmag = sqrt( total( cv.B^2))
cv.B = cv.B / Bmag
cv.c = cv.c / sqrt(total(cv.c^2))
cv.oplane_n = cv.oplane_n / sqrt(total(cv.oplane_n^2))

;Standard cartesian to spherical coordinate conversions:
tmp = cv_coord( from_rect=cv.c, /to_sphere)
cv.theta_c = 90.0 - tmp[1]*!RADEG           ;Convert -90..90 to 0..180
cv.phi_c   = (tmp[0]*!RADEG + 360) mod 360  ;Convert -180..180 to 0..360
tmp = cv_coord( from_rect=cv.oplane_n, /to_sphere)
cv.theta_n = 90.0 - tmp[1]*!RADEG           ;Convert -90..90 to 0..180
cv.phi_n   = (tmp[0]*!RADEG + 360) mod 360  ;Convert -180..180 to 0..360
cv.B = cv.B / sqrt( total( cv.B^2))
tmp = cv_coord( from_rect=cv.B, /to_sphere)
cv.theta_b = 90.0 - tmp[1]*!RADEG           ;Convert -90..90 to 0..180
cv.phi_b   = (tmp[0]*!RADEG + 360) mod 360  ;Convert -180..180 to 0..360
cv.radius = interpol( D[*].radius, T, state.event_time)

;Create a new sub-tree starting with a new transform object.
cv.B = cv.B * Bmag
transformObj = obj_new( 'IDLgrModel', uvalue=cv)
cv.B = cv.B / Bmag
curvatureObj->Add, transformObj
vector_names = ['N','B','C']
help_names = ['OsculatingPlaneNormal', 'MesocenterB', 'CurvatureVector']
colors = [lightBlue, foreground_color, lightRed]
for i=0,2 do begin
  vectorObj = obj_new( 'IDLgrModel', name=vector_names[i])
  transformObj->add, vectorObj
  vectorObj->add, obj_new( 'IDLgrPolyline', color=0, name=help_names[i])
  vectorObj->add, obj_new( 'IDLgrPolygon',  color=0, name=help_names[i])
endfor

;Add the new vectors to the display.
vectors = [[cv.oplane_n],[cv.B],[cv.c]]
if xs[1] lt 0 then vectors[0,*] = -vectors[0,*]
if ys[1] lt 0 then vectors[1,*] = -vectors[1,*]
for i=0,2 do begin
  vectorObj = transformObj->GetByName( vector_names[i])
  arrow_start = [0,0,0]
  arrow_end   = vectors[*,i] * 0.1
  drawArrow, arrow_start, arrow_end, vectorObj, color=colors[i]
endfor

;Translate to the mesocenter position at the event time.
transformObj->Translate, xs[0]+xs[1]*mesocenter[0], $
                         ys[0]+ys[1]*mesocenter[1], $
                         zs[0]+zs[1]*mesocenter[2]

;Create the semi-circle graphics object and add it.
;The nominal reference orientation of the semi-circle is when theta_c and phi_c are 0.
;This means the curvature vector points up the +z axis, and the semi-circle is oriented
;such that it is in the xz plane, open towards the +z axis, with the midpoint on the
;semi-circle at the origin.  The parametric equations of a semi-circle oriented this way are:
;x = radius*cos(tt), y = 0, z = radius - radius*sin(tt), where tt = 0..pi

normalized_radius = cv.radius * abs( xs[1])
tt = findgen(101)/100 * !PI
x = normalized_radius * cos(tt)
y = tt*0
z = normalized_radius - normalized_radius * sin(tt)
polylineObj  = obj_new( 'IDLgrPolyline', x, y, z, color=state.foreground_color, name='CurvatureArc')

N = vectors[*,0]
B = vectors[*,1]
C = vectors[*,2]
print,'raveCurvature: ' + t_to_str( T0) + ', curvature radius = ' + strtrim( cv.radius,2)

semicircleRotation = obj_new( 'IDLgrModel', name='CurvatureArcRotation')
B = vectors[*,0]  ;B and N appear to be reversed here, but this is what works! (Why?)
N = vectors[*,1]
C = vectors[*,2]

rotMatrix = [[N[0], B[0], C[0], 0.0], $
             [N[1], B[1], C[1], 0.0], $
             [N[2], B[2], C[2], 0.0], $
             [0.0,  0.0,   0.0, 1.0]]
semicircleRotation->SetProperty, transform=rotMatrix  

semicircleRotation->Add, polylineObj
transformObj->Add, semicircleRotation

transformArr = curvatureObj->Get( /ALL, count=count)
for i=0,(count - state.curvature_lines)-1 do begin
  curvatureObj->Remove, transformArr[i]
  for j=0,2 do begin
    vectorObj = transformArr[i]->Get()
    obj_destroy, vectorObj->Get()      ;Destroy arrowShaft object.
    obj_destroy, vectorObj->Get()      ;Destroy arrowHead object.
    obj_destroy, vectorObj             ;Destroy vector object.
  endfor
  obj_destroy, transformArr[i]->Get()  ;Destroy semicircle object.
  obj_destroy, transformArr[i]         ;Destroy Transform object.
endfor
return
END
