;********************************************************************************
; Purpose: Create a text field with up and down arrow next to it:
;           _____
;          |     |  /\
;          | 2.0 | /__\  The text field actually has only one line, and the
;          |     | \  /  arrows both fit vertically within that line, to the
;          |_____|  \/   right of it.
;
;          For example, the value in the text field could be numeric, and clicking on
;          the arrows cause it to increment or decrement.
;
; Widget Hierarchy:
;
;          The following widgets are added to the input base:
;          base-----cw_base-----Label
;                            ---TextField (editable)
;                            ---Draw (contains clickable up and down arrows)
;
; Event Handling:
;          The compound widget has an internal event handler which processes events and
;          and forwards a new event to the user.  The compound widget generates the
;          following event structure:
;
;          {CW_TEXT_WITH_ARROWS, 
;           ID            : 0L,  ;id of the compound widget (the cw_base node in the widget tree)
;           TOP           : 0L,  ;top above the compound widget
;           HANDLER       : 0L,  ;id of the compound widget (the cw_base node in the widget tree)
;           UP_ARROW      : 0,   ;1 if event was caused by up arrow click, otherwise 0
;           DOWN_ARROW    : 0}   ;1 if event was caused by down arrow click, otherwise 0
;
; The uvalue of the compound widget is assigned by the user, and can be gotton from
; the ID or HANDLER fields in the user's event handler.  It is stored in the cw_base node.
;
; The value of the compound widget is the same as its cw_base: the text field value.
; Routines are provided so the GET_VALUE and SET_VALUE keywords to WIDGET_CONTROL will work 
; when passing the compound widget's id.
;
; Usage:   Any extra keyword arguments are passed to the WIDGET_BASE call which
;          creates the cw_base.
;********************************************************************************




;********************************************************************************
; Routine: cw_text_with_arrows__get_value
; Purpose: Makes the GET_VALUE keyword to WIDGET_CONTROL work on the compound widget
;          like it does on primitive widgets;  returns the text field value.
;********************************************************************************
function cw_text_with_arrows__get_value, id
label_widget = WIDGET_INFO( id, /CHILD)
text_widget  = WIDGET_INFO( label_widget, /SIBLING)
WIDGET_CONTROL, text_widget, GET_VALUE=text
return, text
END


;********************************************************************************
; Routine: cw_text_with_arrows__set_value
; Purpose: Makes the SET_VALUE keywords to WIDGET_CONTROL work on the compound widget
;          like it does on primitive widgets;  sets the text field value.
;********************************************************************************
pro cw_text_with_arrows__set_value, id, value
label_widget = WIDGET_INFO( id, /CHILD)
text_widget  = WIDGET_INFO( label_widget, /SIBLING)
WIDGET_CONTROL, text_widget, SET_VALUE=value
END


;********************************************************************************
; Routine: cw_text_with_arrows__realize
; Purpose: After the text field is realized (drawn), get its height and set the
;          draw widget's width and height to similar values.  
;          Draw the up and down arrows.
;********************************************************************************
pro cw_text_with_arrows__realize, base_widget, draw=draw
label_widget = WIDGET_INFO( base_widget, /CHILD)
text_widget = WIDGET_INFO( label_widget, /SIBLING)
draw_widget = WIDGET_INFO( text_widget, /SIBLING)
;print,'base ysize  = ',(WIDGET_INFO( base_widget, /GEOMETRY)).scr_ysize
;print,'label ysize = ',(WIDGET_INFO( label_widget, /GEOMETRY)).scr_ysize
;print,'text ysize  = ',(WIDGET_INFO( text_widget, /GEOMETRY)).scr_ysize
if not keyword_set( draw) then begin
  geom = WIDGET_INFO( base_widget, /GEOMETRY)
  xsize = geom.scr_ysize/2
  ysize = geom.scr_ysize-1
  WIDGET_CONTROL, draw_widget, scr_xsize=xsize, xsize=xsize, scr_ysize=ysize, ysize=ysize
endif

;Ideally the following would run during the one and only call to cw_text_with_arrows__realize.
;However testing has shown that:
;1) This routine gets called BEFORE the final window appears on the screen (a minimum size window
;   appears in the left-hand corner).
;2) At that time, IDL doesn't scale the axes/plot to the drawing area correctly, despite all the
;   system variables being set correctly (all the GEOM variables, !X.crange, !X.margin, etc).  
;   When run again with /DRAW, after the topLevelBase is realized with UPDATE=1, it does work.

WIDGET_CONTROL, draw_widget, get_value=window, get_uvalue=view
window->draw, view

END


;********************************************************************************
; Routine: cw_text_with_arrows__event_handler
; Purpose: Catch events from the components of the compound widget, and generate
;          a new event of type CW_TEXT_WITH_ARROWS.
;********************************************************************************
function cw_text_with_arrows__event_handler, event

new_event = {CW_TEXT_WITH_ARROWS,           $
             ID            : event.handler, $
             TOP           : event.top,     $
             HANDLER       : event.handler, $
             UP_ARROW      : 0,             $
             DOWN_ARROW    : 0}
event_type = tag_names( event, /structure_name)
case event_type of
  'WIDGET_TRACKING': begin
    event.id = event.handler   ;So cw_base's uname is used by handler (not Label's, Text's, or Draw's)
    return, event
  end
  'WIDGET_DRAW': begin
    if (event.type eq 4) then begin    ;Expose.
      WIDGET_CONTROL, event.id, get_value=window, get_uvalue=view
      window->draw, view
      return,0
    endif
    if event.type eq 0 then begin  ;button pressed
      ;Note: when switching between windows, the system variables used by convert_coord get
      ;messed up, use device coordinates to determine where click occurred.
      geom = WIDGET_INFO( event.id, /GEOMETRY)
      if event.y ge geom.draw_ysize/2.0 then begin
        new_event.up_arrow = 1
      endif else begin
        new_event.down_arrow = 1
      endelse
      return, new_event
    endif
  end
  'WIDGET_TEXT_CH': begin          ;user hit return in text field
    WIDGET_CONTROL, event.id, get_value=text
    message, /reset_error_state
    value = float( text[0])
    if !error ne 0 or text[0] eq '' then begin
      message = 'Value is invalid! ' + text[0]
      junk = WIDGET_MESSAGE( message, /ERROR)
      return, 0
    endif
    return, new_event
  end
  else: ;
endcase
return, 0
END


;********************************************************************************
; Routine: strip_trailing_zeros
; Purpose: Utility routine to remove 0's after decimal point from end of a string.
;********************************************************************************
pro strip_trailing_zeros, string_value
len = strlen( string_value)
decimal_position = strpos( string_value, '.')
if decimal_position eq -1 then return
while strmid( string_value, len-1, 1) eq '0' and (len-1) gt decimal_position do begin
  string_value = strmid( string_value, 0, len-1)
  len = len - 1
endwhile
END


;********************************************************************************
; Routine: cw_text_with_arrows
; Purpose: This function creates the compound widget.
;********************************************************************************
function cw_text_with_arrows, base, text, font=font, label=label, _extra=_extra
cw_base = WIDGET_BASE( base, /ROW, _extra=_extra, $
                       event_func='cw_text_with_arrows__event_handler', $
                       func_get_value='cw_text_with_arrows__get_value', $
                       pro_set_value='cw_text_with_arrows__set_value' , $
                       notify_realize='cw_text_with_arrows__realize')
if not keyword_set( label) then label=''
W = where( tag_names( _extra) eq 'TRACKING_EVENTS', count)
TRACKING_EVENTS = (count eq 1) ? _extra.tracking_events : 0
junk = WIDGET_LABEL( cw_base, value=label, font=font, TRACKING_EVENTS=TRACKING_EVENTS)
junk = WIDGET_TEXT( cw_base, value=text, xsize=strlen( text), font=font, $
                    /EDITABLE, TRACKING_EVENTS=TRACKING_EVENTS)

coords = [[0.2, 0.6], [0.5, 0.8], [0.8, 0.6], $ ;up arrow
          [0.2, 0.4], [0.5, 0.2], [0.8, 0.4]]   ;down arrow
polygons = [3,0,1,2,3,3,4,5,-1]
arrows = obj_new( 'IDLgrPolygon', coords, polygons=polygons, color=[255,255,255])
model = obj_new( 'IDLgrModel')
model->Add, arrows
view = obj_new( 'IDLgrView', color=[0,0,0], viewplane_rect=[0,0,1,1.0])  ;Black background
view->Add, model
junk = WIDGET_DRAW( cw_base, xsize=10, ysize=20, /BUTTON_EVENTS, GRAPHICS_LEVEL=2, $
                    RETAIN=0, /EXPOSE_EVENTS, TRACKING_EVENTS=TRACKING_EVENTS, uvalue=view)
return, cw_base
END

