;*******************************************************************************
; File:  pitch.pro
;
; Purpose:   Make pitch angle distribution plots available for RAVE.
;
; Routines:  
;            calculatePitchAngle
;            createPitchAnglePlot
;            drawPitchAnglePlot
;            getPitchAngleData
;            subdividePixel
;
; Author:  LASP/Steve Monk
;
; Notes:
; 1) The current release of RAVE uses Reiner's pitch angle files.
;    Routines here can be used to calculate the pitch angle distribution on-the-fly,
;    but I'm not confident the results are correct.
;*******************************************************************************



;*******************************************************************************
pro getPitchAngleData, SC, inputTime, averaging_interval, outputData, success, $
                  restoreStartTime=restoreStartTime, restoreStopTime=restoreStopTime, $
                  band=band, pa_positions=pa_positions
;*******************************************************************************
common RAPID_PA_DATA_COMMON, Ts, Ds, pa_positions_save, old_band, old_restore_start_stop_times
success = 0

if (n_elements(Ts) eq 0) or keyword_set( restoreStartTime) then begin
  if not (keyword_set( restoreStartTime) and keyword_set( restoreStopTime)) then $
    throw, 'restoreStart/StopTime keywords must both be set on first call!'
  if not keyword_set( band) then band = 0
  if n_elements(band) gt 1 then $
    throw, 'band must be a scalar between 0 and 8!'
  
  if n_elements(Ts) eq 0 then begin
    old_band = -1
    old_restore_start_stop_times = [0d,0d]
  endif
  
  ;This is for convenience during development - often I restart the program, but the data remains
  ;in common blocks, so don't take the time to re-read it.
  times_are_unchanged = (old_restore_start_stop_times[0] eq restoreStartTime) and $
                        (old_restore_start_stop_times[1] eq restoreStopTime)
  if times_are_unchanged and (old_band eq band) then begin
    print,'getPitchAngleData: Not re-loading data as requested, because it appears to be loaded already!'
    return
  endif
    
  old_band = band
  old_restore_start_stop_times = [restoreStartTime, restoreStopTime]
  
  T0=0 & T1=0 & T2=0 & T3=0  &  D0=0 & D1=0 & D2=0 & D3=0
  split_time, restoreStartTime, year, doy, start_hour, mm, ss, month, day
  yyyymmdd = strtrim( year,2) + string( format='(I2.2)', month) + string( format='(I2.2)', day)
  for SC=0,3 do begin
    file = getenv('PAPCO_DLNK') + '/cluster/rapid/pitch/C' + strtrim(SC+1,2) + '_' + yyyymmdd + $
          '_IES_CLEANBM_pitch.idl'
    files = findfile( file, count=count)
    if count eq 1 then begin
      print,'Restoring ' + file
      restore, files[0]
      D = input_data
      T = (D.endtime + D.time)/2.0
      pa_positions_save = input_header.pa_positions  ;fltarr(19,2)
      pa_positions = pa_positions_save
      strs = call_function( 'tai2utc', T, /ecs)  ;yyyy/mm/dd hh:mm:ss.sss
      N = n_elements( strs)
      T = dblarr(N)
      for i=0L,N-1 do begin
        tcad_time_string__parse_date_time,strs[i],dt
        T[i] = dt.days
      endfor
      success = 1
      case SC of
      0: begin &  T0=T & D0=reform( D.data[ band,*])  & end  ;fltarr(19,n)
      1: begin &  T1=T & D1=reform( D.data[ band,*])  & end
      2: begin &  T2=T & D2=reform( D.data[ band,*])  & end
      3: begin &  T3=T & D3=reform( D.data[ band,*])  & end
      endcase
    endif else begin
      print,"Couldn't find pitch angle file: " + file
    endelse
  endfor
  
  Ts = create_struct('t0',T0,'t1',T1,'t2',T2,'t3',T3)
  Ds = create_struct('d0',D0,'d1',D1,'d2',D2,'d3',D3)
    
  for SC=0,3 do begin
    W = where( Ts.(SC) ge restoreStartTime and Ts.(SC) lt restoreStopTime, co)
    if co gt 0 then begin
      Ts.(SC) = (Ts.(SC))[W]
      Ds.(SC) = (Ds.(SC))[*,W]
    endif
  endfor
  return
endif

success = 1
bin_width = averaging_interval/86400./2
W = where( Ts.(SC) ge inputTime-bin_width and Ts.(SC) lt inputTime+bin_width, count)
if count eq 0 then begin
  t1_str = strmid( t_to_str( inputTime-bin_width), 9)  ;"hh:mm:ss"
  t2_str = strmid( t_to_str( inputTime+bin_width), 9)  ;"hh:mm:ss"
  print, "Couldn't find any CLEANBM pitch angle data for Cluster " + strtrim(SC+1,2) + $
         ' between ' + t1_str + ' and ' + t2_str
  success = 0
  return
endif
D = (Ds.(SC))[*,W]
outputData = D[*,0]
for i=1,count-1 do begin
  outputData = outputData + D[*,i]
endfor
;outputData = outputData / count
pa_positions = pa_positions_save
return
END


;*******************************************************************************
; Routine:   calculatePitchAngle
; Notes:
; 1) Sychronizing Magnetic Field and Electron Data:
;    The one check I made showed both mag and electron data at 4 sec intervals,
;    but with centers off by 2 sec from each other.  The mag data source has a
;    single time-tag, probably the start of rapid data taking, and the rapid data
;    source has a start and stop time whose center is the time we use.  Line them
;    up as simply as possible.  Use +/- 8 sec in the read_rapid_B_papco call.
; 2) Common Block:
;    - pixels and pixel_index_map are calculated for a specific pixel subdivision
;      as specified by subdivision_factor, then used in all later calls.
;*******************************************************************************
pro calculatePitchAngle, state, SC, pa, success, pa_positions=pa_positions, $
                         pixel_info=pixel_info

common calculatePitchAngle_common, pixels, pixel_index_map

success = 0
minTime = str_to_t( state.minTimeStr)
maxTime = str_to_t( state.maxTimeStr)
bin_width = state.averaging_interval/86400./2
start_time_str = t_to_str( (state.event_time - bin_width - 4/86400.) > minTime)
stop_time_str  = t_to_str( (state.event_time + bin_width + 4/86400.) < maxTime)
read_rapid_B_papco, start_time_str, stop_time_str, SC, T_B, Bs, success
if not success then return

getRapidData, SC, state.event_time, state.averaging_interval, T, Ds, success, /NO_AVERAGE
if not success then return

for i=0L,n_elements(T_B)-1 do begin
  if T_B[i] ge T[0] then break
endfor
T_B = T_B[i:*]
Bs = Bs[i:*]
n = min( [n_elements(T_B), n_elements(T)])
T_B = T_B[0:n-1]
Bs = Bs[0:n-1]
T = T[0:n-1]
Ds = Ds[*,*,0:n-1]

FORCE_REINIT = 0
if n_elements( pixels) eq 0 or FORCE_REINIT then begin
  
  ;Subdivide pixels (a "pixel" is the trapazoidal area covered by each element of d[16,9]) so that 
  ;the pixels have nearly equal area, and so that the length of each pixel's lat edge is comparable
  ;to the length of its lon edge.
  
  subdivision_factor = -1  ;One equatorial pixel should be subdivided into 4^subdivision_factor pixels.
  area_of_an_equatorial_band_pixel = 4*!PI*sin( 10/!RADEG)/16   ;2*pi*r  * 2*sin(10)*r
  desired_area = area_of_an_equatorial_band_pixel / (4.0^subdivision_factor)
  
  ;print,'area_of_an_equatorial_band_pixel = ',area_of_an_equatorial_band_pixel,', desired_area = ',desired_area
  
  ;First subdivide each of the 9 pixels in the single north-south strip from d[0,8] to d[0,0].
  n_results = long( 9 * ((4.0^(subdivision_factor + 0.2)) > 1))
  ;Size result approximately big enough.
  result = replicate( {coords: fltarr(3), area: 0.0, counts: 0.0, cos: 0.0}, n_results)  
  print,'Allocated     ' + strtrim( n_results,2) + ' results for single latitude strip.'
  colat_bounds = [0.0, 20, 40, 60, 80, 100, 120, 140, 160, 180]   ;To optimize: don't calculate [120,140,160,180]
  lon_bounds = [0,360/16.] + 71.417   ;71.417 = longitude of center of D[0,*]
  n_results = 0L
  pixel_index_map = lonarr(2,16,9)
  ;pixel_index_map contains start/stop indices into the 'pixels' array for a given lat/lon.
  ;pixels[ pixel_index_map[0,lon,lat]:pixel_index_map[1,lon,lat]] are the subdivided pixels for given lat/lon.
  for i=0,n_elements( colat_bounds)-2 do begin
    old_n_results = n_results
    subdividePixel, colat_bounds[i:i+1], lon_bounds, desired_area, result, n_results, area=area
    result[ old_n_results:n_results-1].area = result[ old_n_results:n_results-1].area / area
    pixel_index_map[*,0,8-i] = [old_n_results, n_results - 1]
  endfor
  print,'Actually used ' + strtrim( n_results,2) + ' results for single latitude strip.'
  result = result[0:n_results-1]
  result0 = result
  
  ;Replicate and adjust the single north-south strip for all other lons.
  pixels = replicate( result[0], n_results * 16)
  for i=0,15 do begin
    result = result0
    result[*].coords[0] = (result[*].coords[0] + i*360/16.) mod 360.
    pixels[i*n_results] = result
    pixel_index_map[*,i,*] = pixel_index_map[*,0,*] + i*n_results
  endfor
  result  = 0
  result0 = 0
  
  pixels.coords[1] = 90.0 - pixels.coords[1]   ;Convert colatitude to latitude.
  pixels.coords = cv_coord( from_sphere=pixels.coords, /DEGREES, /TO_RECT) * (-1)
  ;lat/lon direction is the NEGATIVE of the electron's velocity vector V, and we need B dot V.
  
  TEST = 0
  if TEST then begin   ;Toss the coordinates as a wire-frame model into a 3-D viewer.
    model = obj_new( 'IDLgrModel')
    METHOD = 2
    if METHOD eq 1 then begin
      globe = obj_new( 'IDLgrPolygon', pixels.coords, style=1, /HIDDEN_LINES, color=[255,255,255])
    endif else begin
      mesh_obj, 0, v, p, pixels.coords
      globe = obj_new( 'IDLgrPolygon', v, polygons=p, style=1, /HIDDEN_LINES, color=[255,255,255])
    endelse    
    model->add, globe
    xobjview, model, /block, scale=1.0, background=[0,0,0]
    stop
  endif
endif  ;End initialization block.

pa = fltarr(19)
pa_positions = [[0,indgen(18)*10+5.0],[indgen(18)*10+5.0,180.0]]

;Pixels is a flat, 1-D array containing the areas for all the subdivided pixels.
;Pixel_info is the pixels array replicated as many time as there are different times.
;E.g. struct array (144,7) if subdivision_factor is -1, and there's data at 7 different times.
pixel_info = replicate( pixels[0], [n_elements( pixels), n_elements(T)])

;Loop through the different times:
;1) Distribute counts from each source pixel into the subdivided pixels according to each
;   subdivided pixels's area.
;2) Calculate the pitch angle of each subdivided pixel.
;3) Sort by pitch angle and accumulate counts in pa array.

for i=0,n_elements(T)-1 do begin
  D = Ds[*,*,i]
  ;D = transpose( rotate( D, 3))
  ;Note: don't fill data gaps here!  Messes up results.  E.g. for 2002/8/11-14:22:30 without fill,
  ;J0 = 16.7, A1=(-0.68, 0.68, -0.27) which is close to expected, but with fill, A1 is bad, and of
  ;course J0 is higher.
  
  ;Should reverse the calibration here, but need input_header and input_data.lut, and code doesn't exist
  ;to reverse all cals...
  
  ;Distribute the counts for each d[lon,lat] pixel into the subdivided pixels
  ;according to each pixel's area.

  pixels[*].counts = 0
  pixels[*].cos    = 0
  W = where( D gt 0, co)
  if co eq 0 then continue
  for j=0,co-1 do begin
    index_range = pixel_index_map[*, W[j] mod 16, W[j]/16]
    pixels[ index_range[0]:index_range[1]].counts = pixels[ index_range[0]:index_range[1]].area * d[W[j]]
  endfor
  
  B = [Bs[i].Bx, Bs[i].By, Bs[i].Bz]
  Bnorm = B / sqrt(total(B^2))
  
  nonzero_pixel_indices = where( pixels.counts ne 0, co)
  if co lt 2 then continue   ;histogram needs at least a 2-element array.
  dot_products = reform( pixels[ nonzero_pixel_indices].coords ## Bnorm)
  pixels[ nonzero_pixel_indices].cos = dot_products
  
  pixel_info[*,i] = pixels
  
  DEBUG = 0 ;(SC eq 2)
  if DEBUG then begin
    print,'Bnorm = ',Bnorm
    W = where( D ne 0, co)
    print,'Non-zero counts = ',co
    for j=0,co-1 do begin
      lon = W[j] mod 16
      lat = W[j]/16
      index_range = pixel_index_map[*, lon, lat]
      subpixels = pixels[ index_range[0]:index_range[1]]
      total_counts = total( subpixels.counts)
      sub_nonzero_pixel_indices = where( subpixels.counts ne 0, subpixel_counts)
      if subpixel_counts eq 1 then begin
        sub_dot_products = total( subpixels[ sub_nonzero_pixel_indices].coords * Bnorm)
      endif else begin
        sub_dot_products = subpixels[ sub_nonzero_pixel_indices].coords ## Bnorm
      endelse
      sub_pitch_angles = acos( sub_dot_products) * !RADEG
      average_pitch_angle = total( sub_pitch_angles) / n_elements( sub_pitch_angles)
      print,'D[',strtrim(lon,2),',',strtrim(lat,2),'] = ',strtrim(D[lon,lat],2), $
        ', average pitch angle = ',strtrim( average_pitch_angle,2)
      for k=index_range[0],index_range[1] do begin
        coords = cv_coord( from_rect=pixels[k].coords, /DEGREES, /TO_SPHER)
        print,'pixels[',strtrim(k,2),']:  lon = ',strtrim( (coords[0]+360) mod 360.,2),$
                                     ', colat = ',strtrim( 90-coords[1],2)
      endfor
    endfor
  endif  ;DEBUG
  
  ;Sort into pitch angle bins and add to the pa array.
  pitch_angles_of_pixels = acos( dot_products) * !RADEG
  
  pa = pa*0
  binsize = 10.0  ;degrees
  h = histogram( pitch_angles_of_pixels, binsize=binsize, min=-5.0, max=184.99, reverse_indices=r)
  for j=0,n_elements(h)-1 do begin
    num_elements_in_bin_j = r[j+1] - r[j]
    if num_elements_in_bin_j gt 0 then begin
      W = r[r[j]:r[j+1]-1]
      pa[j] = pa[j] + total( pixels[ nonzero_pixel_indices[ W]].counts)
    endif
  endfor
  
  if DEBUG then begin
    W = where( h gt 0,co)
    for j=0,co-1 do begin
      print,pa_positions[W[j],0],' ',pa_positions[W[j],1],'   ',h[W[j]],'  ',pa[W[j]], format='(I3,A,I3,A,I2,A,F6.2)'
    endfor
  endif
endfor

return

;NOTE: THIS CODE SHOULD IMPROVE THINGS, BUT DOESN'T!
;THE ONLY CHANGE WHICH MAKES IT COMPARABLE WITH THE CORRECT RESULT AT 2001/8/11-14:22:30
;IS RETURNING SUM OF COUNTS INSTEAD OF AVERAGED COUNTS:  A1 IS OK, BUT J0 IS WAY BIG...

;For each subpixel, average the (non-zero) counts in pixel_info over all the time samples, 
;and make the cos a count-weighted average of the cos's from the different time samples.
output_pixel_info = pixel_info[*,0]  ;E.g. [144,1], same subpixel resolution.
output_pixel_info.counts = 0
output_pixel_info.cos    = 0
for i=0L,n_elements( output_pixel_info)-1 do begin
  W = where( pixel_info[i,*].counts gt 0, n)
  if n eq 0 then continue
  counts = pixel_info[i,W].counts
  cos    = pixel_info[i,W].cos
  ;Make the output output_pixel_info.cos a weighted time average.  A subpixel's cos at one time
  ;contributes to the averaged cos in proportion to its corresponding counts, as a fraction of sum
  ;of the subpixels' counts at all times.
  output_pixel_info[i].cos = total( cos * counts / total( counts))
  output_pixel_info[i].counts = total( counts)
  output_pixel_info[i].coords = pixel_info[i,0].coords
endfor
W = where( output_pixel_info.counts gt 0, co)
if co gt 0 then begin
  output_pixel_info = output_pixel_info[W]
  success = 1
endif else begin
  success = 0
endelse
pixel_info = output_pixel_info
END



;*******************************************************************************
; Routine:   calculatePitchAngle2
; Notes:
; 1) Sychronizing Magnetic Field and Electron Data:
;    The one check I made showed both mag and electron data at 4 sec intervals,
;    but with centers off by 2 sec from each other.  The mag data source has a
;    single time-tag, probably the start of rapid data taking, and the rapid data
;    source has a start and stop time whose center is the time we use.  Line them
;    up as simply as possible.  Use +/- 8 sec in the read_rapid_B_papco call.
; 2) Common Block:
;    - pixels and pixel_index_map are calculated for a specific pixel subdivision
;      as specified by subdivision_factor, then used in all later calls.
;*******************************************************************************
pro calculatePitchAngle2, state, SC, pixel_info, B, success

common calculatePitchAngle2_common, pixels, pixel_index_map

success = 0
minTime = str_to_t( state.minTimeStr)
maxTime = str_to_t( state.maxTimeStr)
bin_width = state.averaging_interval/86400./2
start_time_str = t_to_str( (state.event_time - bin_width - 4/86400.) > minTime)
stop_time_str  = t_to_str( (state.event_time + bin_width + 4/86400.) < maxTime)
read_rapid_B_papco, start_time_str, stop_time_str, SC, T_B, Bs, success
if not success then return

getRapidData, SC, state.event_time, state.averaging_interval, T, Ds, success, /NO_AVERAGE
if not success then return

for i=0L,n_elements(T_B)-1 do begin
  if T_B[i] ge T[0] then break
endfor
T_B = T_B[i:*]
Bs = Bs[i:*]
n = min( [n_elements(T_B), n_elements(T)])
T_B = T_B[0:n-1]
Bs = Bs[0:n-1]
T = T[0:n-1]
Ds = Ds[*,*,0:n-1]

if n_elements( pixels) eq 0 then begin
  
  ;Subdivide pixels (a "pixel" is the trapazoidal area covered by each element of d[16,9]) so that 
  ;the pixels have nearly equal area, and so that the length of each pixel's lat edge is comparable
  ;to the length of its lon edge.
  
  subdivision_factor = -1  ;One equatorial pixel should be subdivided into 4^subdivision_factor pixels.
  area_of_an_equatorial_band_pixel = 4*!PI*sin( 10/!RADEG)/16   ;2*pi*r  * 2*sin(10)*r
  desired_area = area_of_an_equatorial_band_pixel / (4.0^subdivision_factor)
  
  ;print,'area_of_an_equatorial_band_pixel = ',area_of_an_equatorial_band_pixel,', desired_area = ',desired_area
  
  ;First subdivide each of the 9 pixels in the single north-south strip from d[0,8] to d[0,0].
  n_results = 9
  result = replicate( {coords: fltarr(3), area: 0.0, counts: 0.0, cos: 0.0}, n_results)  
  colat_bounds = [0.0, 20, 40, 60, 80, 100, 120, 140, 160, 180]
  lon_bounds = [0,360/16.] + 60.167   ;Longitude of D[0,*]
  n_results = 0
  pixel_index_map = lonarr(2,16,9)
  ;pixel_index_map contains start/stop indices into the 'pixels' array for a given lat/lon.
  ;pixels[ pixel_index_map[0,lon,lat]:pixel_index_map[1,lon,lat]] are the subdivided pixels for given lat/lon.
  for i=0,n_elements( colat_bounds)-2 do begin
    old_n_results = n_results
    subdividePixel, colat_bounds[i:i+1], lon_bounds, desired_area, result, n_results, area=area
    ;Set each subdivided pixel's area to a fraction between 0 and 1 of the source pixel's area.
    result[ old_n_results:n_results-1].area = result[ old_n_results:n_results-1].area / area
    pixel_index_map[*,0,8-i] = [old_n_results, n_results - 1]
  endfor
  result0 = result
  
  ;Replicate and adjust the single north-south strip for all other lons.
  pixels = replicate( result[0], n_results * 16)
  for i=0,15 do begin
    result = result0
    result[*].coords[0] = (result[*].coords[0] + i*360/16.) mod 360.
    pixels[i*n_results] = result
    pixel_index_map[*,i,*] = pixel_index_map[*,0,*] + i*n_results
  endfor
  result  = 0
  result0 = 0
  
  pixels.coords[1] = 90.0 - pixels.coords[1]   ;Convert colatitude to latitude.
  pixels.coords = cv_coord( from_sphere=pixels.coords, /DEGREES, /TO_RECT) * (-1)
  ;lat/lon direction is the NEGATIVE of the electron's velocity vector V, and we need B dot V.
  
endif  ;End initialization block.

;Pixels is a flat, 1-D array containing the areas for all the subdivided pixels.
;Pixel_info is the pixels array, replicated as many time as there are different times.
;E.g. struct array (144,7) if subdivision_factor is -1, and there's data at 7 different times.
pixel_info = replicate( pixels[0], [n_elements( pixels), n_elements(T)])

;Loop through the different times:
;1) Distribute counts from each source pixel into the subdivided pixels according to each
;   subdivided pixels's area.
;2) Calculate the pitch angle of each subdivided pixel.

for i=0,n_elements(T)-1 do begin
  D = Ds[*,*,i]
  ;Should reverse the calibration here, but need input_header, and code doesn't exist to reverse all cals...
  
  ;Distribute the counts for each d[lon,lat] pixel into the subdivided pixels
  ;according to each pixel's area.

  pixels[*].counts = 0
  pixels[*].cos    = 0
  W = where( D gt 0, co)
  if co eq 0 then continue
  for j=0,co-1 do begin
    index_range = pixel_index_map[*, W[j] mod 16, W[j]/16]
    pixels[ index_range[0]:index_range[1]].counts = pixels[ index_range[0]:index_range[1]].area * d[W[j]]
  endfor
  
  B = [Bs[i].Bx, Bs[i].By, Bs[i].Bz]
  Bnorm = B / sqrt(total(B^2))
  
  nonzero_pixel_indices = where( pixels.counts ne 0, co)
  if co eq 0 then continue
  if co eq 1 then begin
    dot_products = total( pixels[ nonzero_pixel_indices].coords * Bnorm)
  endif else begin
    dot_products = reform( pixels[ nonzero_pixel_indices].coords ## Bnorm)
  endelse
  pixels[ nonzero_pixel_indices].cos = dot_products
  pixel_info[*,i] = pixels
endfor

B = fltarr(3)                               ;Set output keyword B.
Bmag = sqrt( Bs.Bx^2 + Bs.By^2 + Bs.Bz^2)
count = n_elements( Bs)
BmagAve = total( Bmag) / count
for i=0,2 do begin                         
  B_unit_component = Bs.(i) / Bmag
  B[i] = total( B_unit_component) / count  ;Average the i'th component of the unit vectors.
  B[i] = B[i] * BmagAve                    ;Multiply by averaged magnitude.
endfor
END



;*******************************************************************************
; Routine: subdividePixel
; Inputs:  colat_bounds - fltarr(2), must be increasing
;          lon_bounds   - fltarr(2), must be increasing
;          desired_area - assume sphere with radius=1
; Outputs
;          result       - array of structures on input, added to on output
;          n_results    - on input, the number of already used structures in result;
;                         on output, incremented by the number of structures we added.
;          area         - Optional keyword for area of the input pixel.
;
; Notes:
; 1) This routine is recursive.  The recursion stops when the area of a pixel
;    reaches 'desired_area' or less;  the call then adds the pixel's colat, lon
;    and area to the 'result' array.
;
;*******************************************************************************
pro subdividePixel, colat_bounds, lon_bounds, desired_area, result, n_results, area=area

;Calculate area of pixel described by colat_bounds and lon_bounds as a trapazoid, with:
;A long base segment along a meridian of longitude at the lat closest to the equator.
;A shorter top segment along a meridian of longitude at the lat closest to a pole.
;Height along a meridian of latitude.

segs = (lon_bounds[1]-lon_bounds[0])/360. * 2 * !PI * sin( colat_bounds/!RADEG)
if segs[1] gt segs[0] then begin
  long_segment  = segs[1]
  short_segment = segs[0]
endif else begin
  long_segment  = segs[0]
  short_segment = segs[1]
endelse
lat_segment = 2*!PI*(colat_bounds[1]-colat_bounds[0])/360.
area = lat_segment * long_segment + (long_segment - short_segment)*lat_segment/2.0

;print,'area = ',area,', colat_bounds = ',colat_bounds

if area le desired_area then begin
  ;Assign colat/lon of center of trapazoid-shaped pixel.
  struct = {coords: [(lon_bounds[0]   + lon_bounds[1])  /2.0, $
                   (colat_bounds[0] + colat_bounds[1])  /2.0, $
                    1.0], $
            area: area, $
            counts: 0.0, $
            cos: 0.0}
  ;print,'Adding result[' + strtrim( n_results,2) + '] = ',struct
  result[ n_results] = struct
  n_results = n_results + 1
  return
endif

if lat_segment gt long_segment then begin  ;Subdivide lat.
  half_theta = acos( total( cos( colat_bounds/!RADEG))/2.0) * !RADEG
  subdividePixel, [colat_bounds[0], half_theta], lon_bounds, desired_area, result, n_results
  subdividePixel, [half_theta, colat_bounds[1]], lon_bounds, desired_area, result, n_results
endif else begin                           ;Subdivide lon.
  half_lon = total( lon_bounds)/2.0
  subdividePixel, colat_bounds, [lon_bounds[0], half_lon], desired_area, result, n_results
  subdividePixel, colat_bounds, [half_lon, lon_bounds[1]], desired_area, result, n_results
endelse
return
END

  
;*******************************************************************************
pro createPitchAnglePlot, state, palette
;*******************************************************************************

dimensions = [0.6,0.3]  ;In normalized coordinates, relative to graphics destination's rect, i.e.[0,1]
location   = [0,0]       ;Lower left corner, also relative to [0,1], (NOT the 3d view's viewplane_rect [-1,2])
view = obj_new( 'IDLgrView', color=state.background_color, viewplane_rect=[0.0, 0.0, 1.0, 1.0], $
                location=location, dimensions=dimensions, name='PitchAngleView', units=3)  ;3 = Normalized w.r.t. window
xAxisKeywords = {major: 7, minor: 2, exact: 1}
SC_colors = [state.foreground_color[0],244,137,73]   ;white/black,red,green,blue

;Note: Specify the color keyword as bytarr(3), because if you specify color as a scalar (color index),
;plotObj->GetProperty, color=color is bytarr(1), which is rejected by plotObj->SetProperty, color=color.
;(Actually, Rave_Xplot does plotObj->GetProperty, all=all, then plotObj->SetProperty, _extra=all, but
;I think it's the same problem.)

Rave_Xplot, [0,180], [-1,-1], pitchAnglePlotModel, xtitle='Pitch Angle', ytitle='Normalized Counts', $
            xrange=[0,180], yrange=[0,1], /EXACT, color=palette->GetRGB( SC_colors[0]), $
            xAxisKeywords=xAxisKeywords, axisColor=[255,255,0]
pitchAnglePlotModel->SetProperty, name='PitchAnglePlotModel'
for i=1,7 do begin
  Rave_Xplot, [0,180], [-1,-1], pitchAnglePlotModel, oplot=i, color=palette->GetRGB( SC_colors[i mod 4])
endfor
view->add, pitchAnglePlotModel
state.pitch_angle_plot_model = pitchAnglePlotModel
state.pitch_angle_view = view
WIDGET_CONTROL, state.rave_tlb, set_uvalue=state
END


;*******************************************************************************
pro drawPitchAnglePlot, state
;*******************************************************************************
for SC=0,3 do begin
  
  
  getPitchAngleData, SC, state.event_time, state.averaging_interval, y, success, $
                     pa_positions=pa_positions
  ;success = (success eq 1 and SC eq 2)
  if success then begin
    x = pa_positions[*,0] + (pa_positions[*,1] - pa_positions[*,0])/2
    y = y / max(y)
    Rave_XPlot, x, y, state.pitch_angle_plot_model, oplot=SC, thick=2
  endif else begin
    Rave_XPlot, [0,180], [-1,-1], state.pitch_angle_plot_model, oplot=SC
  endelse    
  
  CALCULATE = 0
  if CALCULATE then begin
    calculatePitchAngle, state, SC, y, success, pa_positions=pa_positions
    ;success = (success eq 1 and SC eq 2)
    if success then begin
      y = y / max(y)
      x = pa_positions[*,0] + (pa_positions[*,1] - pa_positions[*,0])/2
      Rave_XPlot, x, y, state.pitch_angle_plot_model, oplot=SC+4, thick=2, linestyle=2
    endif else begin
      Rave_XPlot, [0,180], [-1,-1], state.pitch_angle_plot_model, oplot=SC+4
    endelse    
  endif
endfor
END



